package gov.va.med.ccht.ui.common;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import gov.va.med.fw.model.AbstractKeyedEntity;
import gov.va.med.fw.model.lookup.Lookup;
import gov.va.med.fw.security.Role;
import gov.va.med.fw.security.SecurityContext;
import gov.va.med.fw.security.SecurityContextHelper;
import gov.va.med.fw.security.UserPrincipal;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.ui.conversion.ConversionServiceImpl;
import gov.va.med.fw.ui.model.TermType;

import gov.va.med.ccht.model.inventory.SimpleFacility;
import gov.va.med.ccht.model.inventory.SimpleVisn;
import gov.va.med.ccht.model.inventory.Vendor;
import gov.va.med.ccht.service.common.SecurityService;
import gov.va.med.ccht.service.common.TerminologyCache;
import gov.va.med.ccht.service.common.TerminologyException;
import gov.va.med.ccht.service.inventory.InventoryService;

public class AbstractConversionService extends ConversionServiceImpl {

	private TerminologyCache terminologyCache;
	private InventoryService inventoryService;
	private SecurityService securityService;

	public <T extends Lookup> T getByCode(Class<T> terminologyType, String code)
			throws TerminologyException {
		return terminologyCache.getTermByCode(terminologyType, code);
	}

	public <T extends Lookup> T getByName(Class<T> terminologyType, String name)
			throws TerminologyException {
		return terminologyCache.getTermByName(terminologyType, name);
	}

	public <T extends Lookup> T getById(Class<T> terminologyType, Long id)
			throws TerminologyException {
		return terminologyCache.getTermById(terminologyType, id);
	}

	public <T extends Lookup> List<T> findAll(Class<T> terminologyType) throws TerminologyException {
		return terminologyCache.getTerms(terminologyType);
	}

	public List<TermType> toTermTypeList(Set<? extends Lookup> lookups) {
		List<TermType> termTypes = new ArrayList<TermType>();
		if (lookups != null && lookups.size() > 0) {
			for (Lookup lookup : lookups) {
				if (lookup != null)
					termTypes.add(new TermType(lookup.getName(), lookup.getCode()));
			}
		}
		Collections.sort(termTypes);
		return termTypes;
	}

	public List<TermType> toTermTypeListFromValues(Set<String> values) {
		List<TermType> termTypes = new ArrayList<TermType>();
		if (values != null && values.size() > 0) {
			for (String value : values) {
				termTypes.add(new TermType(value, value));
			}
		}
		return termTypes;
	}

	public TermType toTermType(Lookup lookup) {
		return lookup != null ? new TermType(lookup.getName(), lookup.getCode()) : null;
	}

	public <T extends Lookup> T getLookup(Class<T> terminologyType, TermType termType)
		throws TerminologyException {
		return (termType != null) ? 
				terminologyCache.getTermByCode(terminologyType, termType.getValue()) : null;
	}
	
	public SimpleFacility getFacility(TermType termType) throws TerminologyException {
		try {
			return (termType != null) ? 
				inventoryService.getSimpleFacility(termType.getValue()) : null;
		}catch (ServiceException e){
			throw new TerminologyException("inventoryService.getFacility failed");
		}
	}
	
	public SimpleVisn getVisn(TermType termType) throws TerminologyException {
		try {
			return (termType != null) ? 
				inventoryService.getSimpleVisn(termType.getValue()) : null;
		}catch (Exception e){
			throw new TerminologyException("inventoryService.getSimpleVisn failed");
		}
	}
	
	public Role getRole(TermType termType) throws TerminologyException {
		try {
			return (termType != null) ? 
					securityService.getRole(termType.getValue()) : null;
		}catch (Exception e){
			throw new TerminologyException("securityService.getRole failed");
		}
	}
	
	public Vendor getVendor(TermType termType) throws TerminologyException {
		try {
			return (termType != null) ? 
				inventoryService.getVendor(termType.getLabel()) : null;
		}catch (ServiceException e){
			throw new TerminologyException("inventoryService.getFacility failed");
		}
	}
	
	public TerminologyCache getTerminologyCache() {
		return terminologyCache;
	}

	public void setTerminologyCache(TerminologyCache terminologyCache) {
		this.terminologyCache = terminologyCache;
	}

	public InventoryService getInventoryService() {
		return inventoryService;
	}

	public void setInventoryService(InventoryService inventoryService) {
		this.inventoryService = inventoryService;
	}

	public SecurityService getSecurityService() {
		return securityService;
	}

	public void setSecurityService(SecurityService securityService) {
		this.securityService = securityService;
	}

	public Map<String, AbstractKeyedEntity> getMapFromCollection(
			Set<? extends AbstractKeyedEntity> set) {
		Map<String, AbstractKeyedEntity> map = new HashMap<String, AbstractKeyedEntity>();
		if (set != null) {
			for (AbstractKeyedEntity keyedEntity : set) {
				map.put(keyedEntity.getEntityKey().getKeyValueAsString(), keyedEntity);
			}
		}
		return map;
	}

	protected UserPrincipal getCurrentUser() {
		SecurityContext securityContext = SecurityContextHelper.getSecurityContext();
		return securityContext.getUserPrincipal();
	}
}
