package gov.va.med.ccht.controller;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import gov.va.med.fw.model.ldap.SearchCriteria;
import gov.va.med.fw.model.lookup.AbstractActiveLookup;
import gov.va.med.fw.model.lookup.Lookup;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.ui.model.TermType;

import gov.va.med.ccht.model.CCHTRoles;
import gov.va.med.ccht.model.User;
import gov.va.med.ccht.model.inventory.Device;
import gov.va.med.ccht.model.inventory.DeviceType;
import gov.va.med.ccht.model.inventory.ReportWeek;
import gov.va.med.ccht.model.inventory.SimpleFacility;
import gov.va.med.ccht.model.inventory.SimpleVisn;
import gov.va.med.ccht.model.inventory.Vendor;
import gov.va.med.ccht.model.dmp.DMPToolUser;

import gov.va.med.ccht.ui.common.AbstractController;
import gov.va.med.ccht.ui.common.ControllerException;
import gov.va.med.ccht.ui.common.TermTypeWrapper;

public class TerminologyController extends AbstractController {
			
	public TermTypeWrapper getWrappedTermType(String termTypeName, String varName, boolean convert,
			String uid) throws ControllerException {
		try {
			List<? extends Lookup> terms = getTerminologyCache().getTerms(termTypeName);
			return new TermTypeWrapper(termTypeName, varName, uid,
					convert ? convertToTermType(terms) : terms);
		} catch (Exception ex) {
			throw handleException(ex, this);
		}
	}

	public List<? extends Lookup> getTermType(String termTypeName, String varName, boolean convert,
			String uid) throws ControllerException {
		try {
			return getTerminologyCache().getTerms(termTypeName);
		} catch (Exception ex) {
			throw handleException(ex, this);
		}
	}
	/**
	 * Get User assigned VISNs
	 * @param termTypeName
	 * @param varName
	 * @param convert
	 * @param uid
	 * @return
	 * @throws ControllerException
	 */
	public TermTypeWrapper getVisns(String termTypeName, String varName, boolean convert, String uid)
			throws ControllerException {
		try {
			List<SimpleVisn> terms = getUserVisns();
			return new TermTypeWrapper(termTypeName, varName, uid,
					convert ? convertVisnsToTermType(terms, false) : terms);
		} catch (Exception ex) {
			throw handleException(ex, this);
		}
	}

	public TermTypeWrapper getReportVisns(String termTypeName, String varName, boolean convert, String uid)
	throws ControllerException {
		try {
			List<SimpleVisn> terms = getUserVisns();
			return new TermTypeWrapper(termTypeName, varName, uid,
					convert ? convertVisnsToTermType(terms, true) : terms);
		} catch (Exception ex) {
			throw handleException(ex, this);
		}
	}
	
	private List<SimpleVisn> getUserVisns() throws ServiceException {
		User currentUser = (User)getCurrentUser();
		List<SimpleVisn> terms = new ArrayList<SimpleVisn>();
		if (currentUser.isPermissionGranted(NATIONAL_ADMIN) || 
				currentUser.isPermissionGranted(APPLICATION_ADMIN) || 
				currentUser.isPermissionGranted(MANAGEMENT)) {
			terms = getInventoryService().getSimpleVisns();
		}
		else if (currentUser.isPermissionGranted(VISN_ADMIN)) {
			if (currentUser.getVisn() != null) {
				terms.add(currentUser.getVisn());
			}
		}else {
			//none
		}
		return terms;
	}
	
	public TermTypeWrapper getFacilities(String termTypeName, String varName, boolean convert, String uid)
	throws ControllerException {
		try {
			List<SimpleFacility> terms = getUserFacilities();
			return new TermTypeWrapper(termTypeName, varName, uid,
					convert ? convertFacilitiesToTermType(terms,false) : terms);
		} catch (Exception ex) {
			throw handleException(ex, this);
		}		
	}
	
	public TermTypeWrapper getAllFacilities(String termTypeName, String varName, boolean convert, String uid)
	throws ControllerException {
		try {
			List<SimpleFacility> terms = getInventoryService().getSimpleFacilities();
			return new TermTypeWrapper(termTypeName, varName, uid,
					convert ? convertFacilitiesToTermType(terms,false) : terms);
		} catch (Exception ex) {
			throw handleException(ex, this);
		}		
	}
	public TermTypeWrapper getReportFacilities(String termTypeName, String varName, boolean convert, String uid)
	throws ControllerException {
		try {
			boolean allFacilities = false;
			User currentUser = (User)getCurrentUser();
			if (currentUser.isPermissionGranted(NATIONAL_ADMIN) || 
					currentUser.isPermissionGranted(APPLICATION_ADMIN) || 
					currentUser.isPermissionGranted(MANAGEMENT)) {
				allFacilities = true;
			}
			List<SimpleFacility> terms = getUserFacilities();
			return new TermTypeWrapper(termTypeName, varName, uid,
					convert ? convertFacilitiesToTermType(terms,allFacilities) : terms);				

		} catch (Exception ex) {
			throw handleException(ex, this);
		}
	}
	
	public List<SimpleFacility> getUserFacilities() throws ServiceException {
		User currentUser = (User)getCurrentUser();
		List<SimpleFacility> terms = new ArrayList<SimpleFacility>();
		if (currentUser.isPermissionGranted(NATIONAL_ADMIN) || 
				currentUser.isPermissionGranted(APPLICATION_ADMIN) || 
				currentUser.isPermissionGranted(MANAGEMENT)) {
			terms = getInventoryService().getSimpleFacilities();
			
		}else if (currentUser.isPermissionGranted(VISN_ADMIN)||currentUser.isPermissionGranted(QIR_ORIGINATOR)) {
			if (currentUser.getVisn() != null) {
				terms = getInventoryService().getSimpleFacilities(currentUser.getVisn());
			}
		}else {
			if (currentUser.getFacility() != null) {
				terms.add(currentUser.getFacility());
			}
			if (currentUser.getSecondaryFacility() != null) {
				terms.add(currentUser.getSecondaryFacility());
			}
		}
		return terms;
	}
	
	public TermTypeWrapper getDevices(String termTypeName, String varName, boolean convert, String uid)
	throws ControllerException {
		try {
			List<Device> terms = getInventoryService().getDevices();
			return new TermTypeWrapper(termTypeName, varName, uid,
					convert ? convertDevicesToTermType(terms) : terms);
		} catch (Exception ex) {
			throw handleException(ex, this);
		}
	}	
	
	public TermTypeWrapper getDeviceTypes(String termTypeName, String varName, boolean convert, String uid)
	throws ControllerException {
		try {
			List<? extends Lookup> terms = getTerminologyCache().getTerms(termTypeName);
			return new TermTypeWrapper(termTypeName, varName, uid,
					convert ? convertDeviceTypesToTermType(terms) : terms);
		} catch (Exception ex) {
			throw handleException(ex, this);
		}
	}
	
	public TermTypeWrapper getVendorByDevice(String termTypeName, String varName, boolean convert, String uid)
	throws ControllerException {
		try {
			List<? extends Lookup> terms = getTerminologyCache().getTerms(termTypeName);
			return new TermTypeWrapper(termTypeName, varName, uid,
					convert ? convertVendorDeviceTypesToTermType(terms) : terms);
		} catch (Exception ex) {
			throw handleException(ex, this);
		}
	}
	
	public TermTypeWrapper getReportWeeks(String termTypeName, String varName, boolean convert, String uid)
	throws ControllerException {
		try {
			List<ReportWeek> terms = getInventoryService().getReportWeeks();
			return new TermTypeWrapper(termTypeName, varName, uid,terms);					
		} catch (Exception ex) {
			throw handleException(ex, this);
		}
	}
	
	public TermTypeWrapper getVendors(String termTypeName, String varName, boolean convert, String uid)
	throws ControllerException {
		try {
			List<Vendor> terms = getInventoryService().getVendors();
			return new TermTypeWrapper(termTypeName, varName, uid,
					convert ? convertVendorsToTermType(terms) : terms);
		} catch (Exception ex) {
			throw handleException(ex, this);
		}
	}
	public TermTypeWrapper getCareCoordinators(String termTypeName, String varName, boolean convert, String uid)
	throws ControllerException {
		try {
			User currentUser = (User)getCurrentUser();
			if (currentUser.getFacility() != null) {
				SearchCriteria searchCriteria = new SearchCriteria();
				//Add role
				List<String> roles = new ArrayList<String>();
				roles.add(CCHTRoles.CARE_COORDINATOR);
				searchCriteria.setRoles(roles);
				//Add sites
				
				List<SimpleFacility> facilities = new ArrayList<SimpleFacility>(); //getUserFacilities();
				
					facilities.add(currentUser.getFacility());
				
				List<String> sites = new ArrayList<String>();
				for (SimpleFacility facility:facilities) {
					sites.add(facility.getStationNumber());
				}
				searchCriteria.setStations(sites);
				
				List<User> users = getSecurityService().findAppUsers(searchCriteria);
				return new TermTypeWrapper(termTypeName, varName, uid, convertUserToTermType(users));
			}
			else {
				//empty list for all other users
				return new TermTypeWrapper(termTypeName, varName, uid, convertUserToTermType(new ArrayList<User>()));
			}
		} catch (Exception ex) {
			throw handleException(ex, this);
		}
	}
	
	public List<TermType> getFacilityCareCoordinators(TermType facility)
	throws ControllerException {
		try {
				SearchCriteria searchCriteria = new SearchCriteria();
				//Add role
				List<String> roles = new ArrayList<String>();
				roles.add(CCHTRoles.CARE_COORDINATOR);
				searchCriteria.setRoles(roles);
				List<String> sites = new ArrayList<String>();
				sites.add(facility.getValue());
				searchCriteria.setStations(sites);
				
				List<User> users = getSecurityService().findAppUsers(searchCriteria);
				List<TermType> result = new ArrayList<TermType>();
				for (User o : users) {
						result.add(new TermType(o.getFullName(), o.getUsername()));			
				}
				return result;			
		} catch (Exception ex) {
			throw handleException(ex, this);
		}
	}
	public TermTypeWrapper getFacilitators(String termTypeName, String varName, boolean convert, String uid)
	throws ControllerException {
		try {
			List<DMPToolUser> terms = getDmpService().findDmpFacilitators();
			return new TermTypeWrapper(termTypeName, varName, uid,
					convert ? convertFacilitatorsToTermType(terms,false) : terms);
		} catch (Exception ex) {
			throw handleException(ex, this);
		}		
	}
	public TermTypeWrapper getAssignees(String termTypeName, String varName, boolean convert, String uid)
	throws ControllerException {
		try {
			List<DMPToolUser> terms = getDmpService().findAllDMPToolUsers();
			return new TermTypeWrapper(termTypeName, varName, uid,
					convert ? convertAssigneesToTermType(terms,false) : terms);
		} catch (Exception ex) {
			throw handleException(ex, this);
		}		
	}	
	private List<TermType> convertToTermType(List<? extends Lookup> list) {
		List<TermType> result = new ArrayList<TermType>();
		for (Lookup o : list) {
			if (o instanceof AbstractActiveLookup) {
				AbstractActiveLookup lookup = (AbstractActiveLookup) o;
				result.add(new TermType(o.getName(), o.getCode(), lookup.isActive()));
			} else {
				result.add(new TermType(o.getName(), o.getCode()));
			}
		}
		return result;
	}
	
	private List<TermType> convertVisnsToTermType(List<SimpleVisn> list, Boolean addAllVisns) {
		List<TermType> result = new ArrayList<TermType>();
		//Add all VIsns
		if (list.size() > 1 && addAllVisns) {
			SimpleVisn allVisns = SimpleVisn.ALL_VISNS;
			result.add(new TermType(allVisns.getName(), allVisns.getName()));		
		}
		for (SimpleVisn o : list) {
				result.add(new TermType(o.getName(), o.getCode()));			
		}
		return result;
	}
	
	private List<TermType> convertFacilitiesToTermType(List<SimpleFacility> list , Boolean addAllFacilities) {
		Collections.sort(list);
		List<TermType> result = new ArrayList<TermType>();
		//Add All Facilities
		if (list.size() > 1 && addAllFacilities) {
			SimpleFacility allFacilities = SimpleFacility.ALL_FACILITIES;
			result.add(new TermType(allFacilities.getName(), allFacilities.getName()));
		}
		for (SimpleFacility o : list) {
				result.add(new TermType(o.getDescription(), o.getCode()));			
		}
		return result;
	}
	
	private List<TermType> convertDevicesToTermType(List<Device> list) {
		List<TermType> result = new ArrayList<TermType>();
		//Add Null Device
		Device nullDevice = Device.NULL_DEVICE;
		result.add(new TermType(nullDevice.getDeviceName(), nullDevice.getDeviceName()));
		for (Device o : list) {
				result.add(new TermType(o.getDeviceName(), o.getDeviceName()));			
		}
		return result;
	}	
	
	private List<TermType> convertVendorsToTermType(List<Vendor> list) {
		List<TermType> result = new ArrayList<TermType>();
		//Add Null Device		
		for (Vendor o : list) {
				result.add(new TermType(o.getName(), String.valueOf(o.getId())));			
		}
		return result;
	}
	
	private List<TermType> convertDeviceTypesToTermType(List<? extends Lookup> list) {
		List<TermType> result = new ArrayList<TermType>();
		//Add Null Device Type
		result.add(new TermType(DeviceType.NULL_DEVICE_NAME, DeviceType.NULL_DEVICE_NAME));
		for (Lookup o : list) {
			result.add(new TermType(o.getName(), o.getCode()));			
		}
		return result;
	}
	
	private List<TermType> convertVendorDeviceTypesToTermType(List<? extends Lookup> list) {
		List<TermType> result = new ArrayList<TermType>();
		//Add Null Device Type
		result.add(new TermType(DeviceType.NULL_DEVICE_NAME, DeviceType.NULL_DEVICE_NAME));
		for (Lookup o : list) {
			if( o instanceof DeviceType ) {
				result.add(new TermType( ((DeviceType)o).getCode(), String.valueOf( ((DeviceType)o).getVendor().getId()) ));			
			}
		}
		return result;
	}
	
	private List<TermType> convertUserToTermType(List<User> list) {
		List<TermType> result = new ArrayList<TermType>();
		for (User o : list) {
				result.add(new TermType(o.getFullName(), o.getUsername()));			
		}
		return result;
	}
	private List<TermType> convertFacilitatorsToTermType(List<DMPToolUser> list , Boolean addDMPFacilitators) {
		List<TermType> result = new ArrayList<TermType>();
		//Add Null Device		
		for (DMPToolUser o : list) {
				result.add(new TermType(o.getAppUser().getFullName(), String.valueOf(o.getAppUser().getUsername())));			
		}
		return result;
	}
	private List<TermType> convertAssigneesToTermType(List<DMPToolUser> list , Boolean addDMPFacilitators) {
		List<TermType> result = new ArrayList<TermType>();
		//Add Null Device		
		for (DMPToolUser o : list) {
				result.add(new TermType(o.getAppUser().getFullName(), String.valueOf(o.getAppUser().getId())));			
		}
		return result;
	}	
	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.fw.service.AbstractComponent#afterPropertiesSet()
	 */
	@Override
	public void afterPropertiesSet() throws Exception {
		super.afterPropertiesSet();
	}
}
