package gov.va.med.ccht.controller;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import gov.va.med.fw.model.EntityKeyFactory;
import gov.va.med.fw.model.ldap.SearchCriteria;
import gov.va.med.fw.security.Permission;
import gov.va.med.fw.security.Role;
import gov.va.med.fw.security.SecurityContext;
import gov.va.med.fw.security.SecurityContextHelper;
import gov.va.med.fw.security.SimpleRole;
import gov.va.med.fw.security.UserPrincipal;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.service.config.EnvironmentParamService;
import gov.va.med.fw.ui.model.TermType;
import gov.va.med.fw.util.StringUtils;

import gov.va.med.ccht.model.CCHTRoles;
import gov.va.med.ccht.model.User;
import gov.va.med.ccht.model.UserPreference;
import gov.va.med.ccht.model.inventory.SimpleFacility;
import gov.va.med.ccht.model.inventory.SimpleVisn;
import gov.va.med.ccht.model.terminology.RegistrationDeniedReason;
import gov.va.med.ccht.model.terminology.RegistrationReason;
import gov.va.med.ccht.model.terminology.RegistrationStatus;

import gov.va.med.ccht.ui.common.AbstractController;
import gov.va.med.ccht.ui.common.ClientSessionParameters;
import gov.va.med.ccht.ui.common.ControllerException;
import gov.va.med.ccht.ui.common.ControllerResult;
import gov.va.med.ccht.ui.model.RoleForm;
import gov.va.med.ccht.ui.model.UserForm;
import gov.va.med.ccht.ui.model.UserPrincipalForm;

public class SecurityController extends AbstractController 	{
	
	public static final TermType NATIOANL_ADMIN_ROLE = new TermType(NATIONAL_ADMIN, NATIONAL_ADMIN);
	public static final TermType VISN_ADMIN_ROLE = new TermType(VISN_ADMIN, VISN_ADMIN);
	public static final TermType FACILITY_ADMIN_ROLE = new TermType(FACILITY_ADMIN, FACILITY_ADMIN);
	public static final TermType APPLICATION_ADMIN_ROLE = new TermType(APPLICATION_ADMIN, APPLICATION_ADMIN);
	public static final TermType MANAGEMENT_ROLE = new TermType(MANAGEMENT, MANAGEMENT);
	public static final TermType DMP_ADMIN_ROLE = new TermType(DMP_ADMIN, DMP_ADMIN);
	public static final TermType DMP_ROLE = new TermType(DMP, DMP);
	public static final TermType VENDOR_ROLE = new TermType(VENDOR, VENDOR);
	public static final TermType NAC_POC_ROLE = new TermType(NAC, NAC);
	public static final TermType SYSTEM_ADMIN_ROLE = new TermType(SYSTEM_ADMIN, SYSTEM_ADMIN);
	public static final TermType CARE_COORDINATOR_ROLE = new TermType(CARE_COORDINATOR, CARE_COORDINATOR);
	public static final TermType PROGRAM_ASSISTANT_ROLE = new TermType(PROGRAM_ASSISTANT, PROGRAM_ASSISTANT);
	public static final TermType QIR_ORIGINATOR_ROLE = new TermType(QIR_ORIGINATOR, QIR_ORIGINATOR);
	
	public Boolean isServerActive() {
		return true;
	}
	public ClientSessionParameters getClientSessionParameters() {
		if (clientSessionParameters == null)
		{
			clientSessionParameters = new ClientSessionParameters();
		}
		SecurityContext sc = SecurityContextHelper.getSecurityContext();
		UserPrincipalForm user = new UserPrincipalForm(sc);
		clientSessionParameters.setUser(user);
		return clientSessionParameters;
	}

	//Manage roles and Permissions
	public RoleForm getRole(String roleName) throws ControllerException { 
		try {
			Role role = getSecurityService().getRole(roleName);
			RoleForm rf = new RoleForm();
			rf.setId(String.valueOf(role.getId()));
			rf.setName(role.getName());
			rf.setDescription(role.getDescription());
			List<TermType> permissions = new ArrayList<TermType>();
			for (Permission permission:role.getPermissions()){
				permissions.add(new TermType(permission.getName(), permission.getName()));
			}
			rf.setPermissions(permissions);
			List<TermType> allPermissions = getAllPermissions();
			allPermissions.removeAll(permissions);
			rf.setAvailablePermissions(allPermissions);
			return rf;
		}catch (ServiceException e){
			throw handleException(e.getMessage(), e, this);
		}
	}
	
	public List<TermType> getAllRoles() throws ControllerException {
		List<TermType> allRoles = new ArrayList<TermType>();
		try {
			List<SimpleRole> roles = getSecurityService().getAllSimpleRoles();
			for (SimpleRole role:roles){
				allRoles.add(new TermType(role.getDescription(),role.getName()));
			}
			return allRoles;
		}catch (ServiceException e){
			throw handleException(e.getMessage(), e, this);
		}

	}
	
	public List<TermType> getAllPermissions() throws ControllerException {
		List<TermType> allPermissions = new ArrayList<TermType>();
		try {
			List<Permission> permissions = getSecurityService().getAllPermissions();
			for (Permission permission:permissions){
				allPermissions.add(new TermType(permission.getName(), permission.getName()));
			}			
			return allPermissions;
		}catch (ServiceException e){
			throw handleException(e.getMessage(), e, this);
		}

	}	
	
	public ControllerResult updateRole(RoleForm roleForm) throws ControllerException {
		try {
			
			Role role = getSecurityService().getRole(roleForm.getName());
			//new role check the name
			if (StringUtils.isEmpty(roleForm.getId()) && role != null){
				//new Role, validate the name for duplicates
				return new ControllerResult(ControllerResult.ERROR, roleForm.getName() + " - Role with same name already exists");
			}
			//existing role check for the name
			if (StringUtils.isNotEmpty(roleForm.getId()) && role != null) {
				Long roleId = Long.parseLong(roleForm.getId());
				if (!roleId.equals(role.getId())) {
					return new ControllerResult(ControllerResult.ERROR, roleForm.getName() + " - Role with same name already exists");
				}
			}
			//existing role
			if (StringUtils.isNotEmpty(roleForm.getId())) {
				Long roleId = Long.parseLong(roleForm.getId());
				role = getSecurityService().getRole(EntityKeyFactory.createEntityKey(roleId, Role.class));
			}
			//new role
			if (role == null){				
				role = new Role();
			}
			//update
			role.setName(roleForm.getName());
			role.setDescription(roleForm.getDescription());			
			Set<Permission> newPermissions = new HashSet<Permission>();
			List<Permission> allPermissions = getSecurityService().getAllPermissions();
			Map<String, Permission> permissionMap = new HashMap<String, Permission>();
			for (Permission permission:allPermissions) {
				permissionMap.put(permission.getName(), permission);
			}
			for (TermType termType:roleForm.getPermissions()){
				newPermissions.add(permissionMap.get(termType.getValue()));
			}
			role.setPermissions(newPermissions);
			getSecurityService().updateRole(role);
		}catch (Exception e){
			throw handleException(e.getMessage(), e, this);
		}
		return new ControllerResult(ControllerResult.SUCCESS);
	}
	
	public UserForm getUser(String userName) throws ControllerException {
		try {
			UserPrincipal currentUser = getCurrentUser();
			User user = getSecurityService().getUser(userName);
			UserForm userForm = new UserForm();
			securityConversionService.convert(user, userForm);
			List<TermType> assignedRoles = userForm.getRoles();
			List<TermType> availableRoles = new ArrayList<TermType>(); //getAllRoles();	
			List<TermType> allRoles = new ArrayList<TermType>(); //getAllRoles();	
			List<SimpleRole> roles;
			if(currentUser.isPermissionGranted(CCHTRoles.DMP_ADMIN)&&currentUser.isPermissionGranted(CCHTRoles.NATIONAL_ADMIN))
			{
				roles = getSecurityService().getAllSimpleRoles();
			}
			else if(currentUser.isPermissionGranted(CCHTRoles.DMP_ADMIN))
			{
				roles = getSecurityService().getAllDMPRoles();
			}
			else
			{		
			    roles = getSecurityService().getAllSimpleRoles();
			}
			for (SimpleRole role:roles){
				allRoles.add(new TermType(role.getName(),role.getName()));
			}
			//filter the available roles based on the user role
			if (currentUser.isPermissionGranted(CCHTRoles.NATIONAL_ADMIN)) {
				//no filtering
				availableRoles = allRoles;
			}else {
				//availableRoles.remove(NATIOANL_ADMIN_ROLE);
				//availableRoles.remove(APPLICATION_ADMIN_ROLE);
				//availableRoles.remove(MANAGEMENT_ROLE);
				User loginUser = (User)currentUser;
				boolean canAddRole = false;
				if (currentUser.isPermissionGranted(CCHTRoles.FACILITY_ADMIN)) {
					
					if (loginUser.getFacility() != null) {
						if (user.getFacility() != null && (loginUser.getFacility().getId().longValue() == user.getFacility().getId().longValue())) {
							canAddRole = true;
						} else if (user.getSecondaryFacility()!= null && (loginUser.getFacility().getId().longValue() == user.getSecondaryFacility().getId().longValue())) {
							canAddRole = true;
						}
					} else if (loginUser.getSecondaryFacility()!= null) {
						if (user.getFacility() != null && (loginUser.getSecondaryFacility().getId().longValue() == user.getFacility().getId().longValue())) {
							canAddRole = true;
						} else if (user.getSecondaryFacility()!= null && (loginUser.getSecondaryFacility().getId().longValue() == user.getSecondaryFacility().getId().longValue())) {
							canAddRole = true;
						}
					}
					if (canAddRole) {
						availableRoles.add(FACILITY_ADMIN_ROLE);
						availableRoles.add(CARE_COORDINATOR_ROLE);
						availableRoles.add(PROGRAM_ASSISTANT_ROLE);
						availableRoles.add(APPLICATION_ADMIN_ROLE);
					}
				}
				canAddRole = false;
				if (currentUser.isPermissionGranted(CCHTRoles.VISN_ADMIN)) {
					//keep Facility Administrator, Care Coordinator, QIR Originator, Program Support Assistant, Application Administrator
					// if user and current user are in the same VISN, otherwise, remove all
					if (loginUser.getVisn() != null && user.getVisn() != null && (loginUser.getVisn().getId().longValue() == user.getVisn().getId().longValue())) {
						canAddRole = true;
					}
					if (canAddRole) {
						if (! availableRoles.contains(VISN_ADMIN_ROLE)) availableRoles.add(VISN_ADMIN_ROLE);
						if (! availableRoles.contains(FACILITY_ADMIN_ROLE)) availableRoles.add(FACILITY_ADMIN_ROLE);
						if (! availableRoles.contains(CARE_COORDINATOR_ROLE)) availableRoles.add(CARE_COORDINATOR_ROLE);
						if (! availableRoles.contains(QIR_ORIGINATOR_ROLE)) availableRoles.add(QIR_ORIGINATOR_ROLE);
						if (! availableRoles.contains(PROGRAM_ASSISTANT_ROLE)) availableRoles.add(PROGRAM_ASSISTANT_ROLE);						
					}
				}
				if(currentUser.isPermissionGranted(CCHTRoles.DMP_ADMIN)) {
					if (! availableRoles.contains(DMP_ADMIN_ROLE)) availableRoles.add(DMP_ADMIN_ROLE);
					if (! availableRoles.contains(DMP_ROLE)) availableRoles.add(DMP_ROLE);
				}
			}
							
			availableRoles.removeAll(assignedRoles);
			userForm.setAvailableRoles(availableRoles);		
			
			//Preferences
			// Set a default value of MDI to false for HTIE_CodeCR436
			userForm.setMultipleWindows(getBooleanValue(user.getPreferenceValue(UserPreference.MULTIPLE_WINDOWS),false));
			
			return userForm;			
		}catch (Exception e){
			throw handleException(e.getMessage(), e, this);
		}
	}
	
	public UserForm getUser(UserForm userForm) throws ControllerException {
		try {			
			return getUser(userForm.getUserName());
		}catch (Exception e){
			throw handleException(e.getMessage(), e, this);
		}
	}
	
	public List<UserForm> findUsers(SearchCriteria searchCriteria) throws ControllerException {
		try {
			List<UserForm> users = new ArrayList<UserForm>();
			List<User> cchtUsers = getSecurityService().findAppUsers(searchCriteria);
			// if only single user is found return the object
			if (cchtUsers != null && cchtUsers.size() > 0) {
				// found users
				if (cchtUsers.size() > 1) {
					for (User user:cchtUsers) {			
						UserForm userForm = new UserForm();
						securityConversionService.convert(user, userForm);
						users.add(userForm);
					}
				} else {
					User user = cchtUsers.get(0);
					UserForm userForm = getUser(user.getName());
					users.add(userForm);
				}
			}
			return users;
		} catch (Exception e) {
			throw handleException("LDAP Search for Users Failed", e, this);
		}
	}
	
	public ControllerResult updateUser(UserForm userForm) throws ControllerException {
		try {
			
			User user = getSecurityService().getUser(userForm.getUserName());
			String oldRegStatusCode = 
				user.getRegistrationStatus() == null ? "" : user.getRegistrationStatus().getCode();
			securityConversionService.convert(userForm, user);
			getSecurityService().updateUser(user);
			String newRegStatusCode = 
				user.getRegistrationStatus() == null ? "" : user.getRegistrationStatus().getCode();
			if (!oldRegStatusCode.equals(newRegStatusCode)) {
				if (RegistrationStatus.APPROVED.equals(newRegStatusCode)) {
					getRegistrationService().sendWelcomeMail(user);
				}else if (RegistrationStatus.REJECTED.equals(newRegStatusCode)) {
					getRegistrationService().sendDeniedMail(user);
				}else {
					//submitted nothing to do moved to pending status
				}
			}
			return new ControllerResult(ControllerResult.SUCCESS);			
		}catch (Exception e){
			throw handleException(e.getMessage(), e, this);
		}
	}
	
	public ControllerResult updateUserProfile(UserForm userForm) throws ControllerException {
		try {			
			User user = getSecurityService().getUser(userForm.getUserName());
			user.addOrUpdatePreference(UserPreference.MULTIPLE_WINDOWS,String.valueOf(userForm.getMultipleWindows()));
			getSecurityService().updateUser(user);
			return new ControllerResult(ControllerResult.SUCCESS);			
		}catch (Exception e){
			throw handleException(e.getMessage(), e, this);
		}
	}
	
	public List<UserForm> getSubmittedRegistrations() throws ControllerException {
		try {			
			List<UserForm> users = new ArrayList<UserForm>();
			Long visnId = null;
			Long facilityId = null;			
			UserPrincipal currentUser = getCurrentUser();
			List<User> registrations; 
			if (currentUser.isPermissionGranted(DMP_ADMIN)) {
				registrations = getSecurityService().getSubmittedRegistrationsForDmp(visnId,facilityId);
			}
			else if (currentUser.isPermissionGranted(NATIONAL_ADMIN)) {
				registrations = getSecurityService().getSubmittedRegistrations(visnId,facilityId);
				//all visns and sites
			}else if (currentUser.isPermissionGranted(VISN_ADMIN)) {
					visnId = ((User) currentUser).getVisn().getId();
					registrations = getSecurityService().getSubmittedRegistrations(visnId,facilityId);
			}else if (currentUser.isPermissionGranted(FACILITY_ADMIN)) {
				facilityId = ((User) currentUser).getFacility().getId();
				registrations = getSecurityService().getSubmittedRegistrations(visnId,facilityId);
			}else {
				return users;
			}
			//convert
			for (User registration:registrations) {
				UserForm userForm = new UserForm();
				userForm.setUserName(registration.getName());
				userForm.setGivenName(registration.getGivenName());
				userForm.setFamilyName(registration.getFamilyName());
				userForm.setMiddleName(registration.getMiddleName());
				userForm.setTelephoneNumber(registration.getTelephoneNumber());
				RegistrationStatus registrationStatus = registration.getRegistrationStatus();
				if (registrationStatus != null) {
					userForm.setRegistrationStatus(new TermType(registrationStatus.getName(),registrationStatus.getCode()));
				}
				SimpleVisn visn = registration.getVisn();
				if (visn != null) {
					userForm.setVisn(new TermType(visn.getName(),visn.getId().toString()));
				}
				SimpleFacility facility = registration.getFacility();
				if (facility != null) {
					userForm.setFacility(new TermType(facility.getName(),facility.getStationNumber()));
				}
				RegistrationReason registrationReason = registration.getRegistrationReason();
				if (registrationReason != null) {
					userForm.setRegistrationReason(new TermType(registrationReason.getName(), registrationReason.getCode()));
				}
				RegistrationDeniedReason registrationDeniedReason = registration.getRegistrationDeniedReason();
				if (registrationDeniedReason != null) {
					userForm.setRegistrationDeniedReason(
						new TermType(registrationDeniedReason.getName(),registrationDeniedReason.getCode()));
				}
				users.add(userForm);
			}
			return users;			
		}catch (Exception e){
			throw handleException(e.getMessage(), e, this);
		}
	}
	
	public List<UserForm> approveRegistration(UserForm userForm) throws ControllerException {
		try {
			User user = getSecurityService().getUser(userForm.getUserName());
			user.setRegistrationStatus(getTerminologyCache().getTermByCode(RegistrationStatus.class, RegistrationStatus.APPROVED));
			//update
			getRegistrationService().approveUser(user);
			return getSubmittedRegistrations();
		}catch (Exception e){
			throw handleException(e.getMessage(), e, this);
		}
	}
	
	public List<UserForm> rejectRegistration(UserForm userForm) throws ControllerException {
		try {
			User user = getSecurityService().getUser(userForm.getUserName());
			user.setRegistrationStatus(getTerminologyCache().getTermByCode(RegistrationStatus.class, RegistrationStatus.REJECTED));
			getRegistrationService().rejectUser(user);
			return getSubmittedRegistrations();
		}catch (Exception e){
			throw handleException(e.getMessage(), e, this);
		}
	}
	
	@Override
	public void afterPropertiesSet() throws Exception {		
		super.afterPropertiesSet();
	}

	
	private SecurityConversionService securityConversionService = null;
	private ClientSessionParameters clientSessionParameters = null;
	private EnvironmentParamService environmentParamService = null;
	
	public SecurityConversionService getSecurityConversionService() {
		return securityConversionService;
	}
	public void setSecurityConversionService(
			SecurityConversionService securityConversionService) {
		this.securityConversionService = securityConversionService;
	}
	public EnvironmentParamService getEnvironmentParamService() {
		return environmentParamService;
	}
	public void setEnvironmentParamService(
			EnvironmentParamService environmentParamService) {
		this.environmentParamService = environmentParamService;
	}
	public void setClientSessionParameters(
			ClientSessionParameters clientSessionParameters) {
		this.clientSessionParameters = clientSessionParameters;
	}
	private Boolean getBooleanValue(String value, Boolean defaultValue) {
		try {
			if (value != null) {
				return Boolean.parseBoolean(value);
			}
		}
		catch (Exception e){logger.error("getBooleanValue error:  "+e.getMessage());}
		return defaultValue;
	}
}
