/********************************************************************
 * Copyright � 2010 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.ccht.controller;

import gov.va.med.ccht.model.inventory.DIReportData;
import gov.va.med.ccht.model.inventory.DeviceFailureReportData;
import gov.va.med.ccht.model.inventory.DeviceShortageReportData;
import gov.va.med.ccht.model.inventory.DeviceStatusReportData;
import gov.va.med.ccht.model.report.CompletedReport;
import gov.va.med.ccht.model.report.CompletedReportFieldType;
import gov.va.med.ccht.model.report.ReportSchedule;
import gov.va.med.ccht.model.report.ReportSetup;
import gov.va.med.ccht.model.report.ScheduledReport;
import gov.va.med.ccht.model.report.SimpleCompletedReport;
import gov.va.med.ccht.model.terminology.ReportFormat;
import gov.va.med.ccht.model.terminology.ReportType;
import gov.va.med.ccht.model.terminology.StandardReport;
import gov.va.med.ccht.service.report.CompletedReportsSearchQueryInfo;
import gov.va.med.ccht.service.report.ReportConstants;
import gov.va.med.ccht.service.report.ReportFilterSearchQueryInfo;
import gov.va.med.ccht.service.report.ReportParameterConversionService;
import gov.va.med.ccht.service.report.StandardReportService;
import gov.va.med.ccht.ui.common.AbstractController;
import gov.va.med.ccht.ui.common.ControllerException;
import gov.va.med.ccht.ui.common.ControllerResult;
import gov.va.med.ccht.ui.model.CompletedReportDisplayParametersForm;
import gov.va.med.ccht.ui.model.CompletedReportForm;
import gov.va.med.ccht.ui.model.CompletedReportUniqueFilterValuesForm;
import gov.va.med.ccht.ui.model.FormattedDateForm;
import gov.va.med.ccht.ui.model.ReportForm;
import gov.va.med.ccht.ui.model.ReportScheduleForm;
import gov.va.med.ccht.ui.model.ReportSetupForm;
import gov.va.med.ccht.ui.model.ScheduledReportForm;
import gov.va.med.ccht.ui.model.StandardReportForm;
import gov.va.med.fw.model.EntityKey;
import gov.va.med.fw.model.EntityKeyFactory;
import gov.va.med.fw.security.Permission;
import gov.va.med.fw.security.SecurityContextHelper;
import gov.va.med.fw.security.UserPrincipal;
import gov.va.med.fw.service.InvalidScheduleException;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.ui.model.TermType;
import gov.va.med.fw.util.StringUtils;
import gov.va.med.fw.util.date.DateWithTimeZone;
import gov.va.med.fw.util.date.TimeZoneUtils;

import java.io.Serializable;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @author vhaisakatikm
 */
public class ReportController extends AbstractController implements
		ReportConstants {

	private StandardReportService standardReportService = null;
	private ReportConversionService reportConversionService = null;
	private ReportParameterConversionService reportParameterConversionService = null;

	// list if reports
	public List<StandardReportForm> getStandardReports()
			throws ControllerException {
		try {
			List<StandardReportForm> standardReports = new ArrayList<StandardReportForm>();
			UserPrincipal user = getCurrentUser();
			List<StandardReport> reportsList = getTerminologyCache().getTerms(
					StandardReport.class);
			// convert to standard report form
			for (StandardReport stdReport : reportsList) {
				// ccr 629 Remove Irrelevant Inventory Tracker Menu Items
				if (isAllowedRunScheduleReport(user, stdReport)) {
					StandardReportForm stdRepForm = getStandardReportFormFromStandardReport(stdReport);
					if (stdRepForm.getStandardReport().getValue().equalsIgnoreCase("DI1"))
					{
						standardReports.add(stdRepForm);
					}
				}
			}
			return standardReports;

		} catch (Exception ex) {
			throw handleException(
					"Error occurred while getting the standard reports.", ex);
		}
	}

	private StandardReportForm getStandardReportFormFromStandardReport(
			StandardReport stdReport) {
		StandardReportForm stdRepForm = new StandardReportForm();
		ReportType reportType = stdReport.getReportType();
		stdRepForm.setStandardReport(new TermType(stdReport.getName(),
				stdReport.getCode()));
		stdRepForm.setReportType(new TermType(reportType.getName(), reportType
				.getCode()));
		stdRepForm.setDetailedReportPermission(stdReport
				.getDetailedReportPermission());
		stdRepForm.setSummaryReportPermission(stdReport
				.getSummaryReportPermission());
		return stdRepForm;
	}

	public StandardReportForm getStandardReport(String code)
			throws ControllerException {
		try {
			StandardReport sr = getTerminologyCache().getTermByCode(
					StandardReport.class, code);
			StandardReportForm stdRepForm = getStandardReportFormFromStandardReport(sr);
			return stdRepForm;
		} catch (Exception ex) {
			throw handleException(
					"Error occurred while getting a standard report", ex);
		}
	}

	public CompletedReportUniqueFilterValuesForm getCompletedReportUniqueFilterValues(
			String field) {
		CompletedReportFieldType fieldType = CompletedReportFieldType
				.getRequiredWithCode(field);
		LinkedHashMap<String, String> uniqueVals = standardReportService
				.getUniqueFilterValues(fieldType);
		return new CompletedReportUniqueFilterValuesForm(field, uniqueVals);
	}

	public List<CompletedReportForm> getCompletedReports(
			CompletedReportDisplayParametersForm params)
			throws ControllerException {
		try {
			List<CompletedReportForm> completedReports = new ArrayList<CompletedReportForm>();
			// create search criteria
			CompletedReportsSearchQueryInfo searchQueryInfo = new CompletedReportsSearchQueryInfo();
			if (params.getSpecificReportID() != null)
				searchQueryInfo.setReportIds(Arrays.asList(params
						.getSpecificReportID()));
			// searchQueryInfo.setUser(getCurrentUser());

			// Get logged in user permissions, sites, ..
			// User user = (User) getCurrentUser();

			Set<String> permissions = new HashSet<String>(SecurityContextHelper
					.getGrantedPermissions());

			List<SimpleCompletedReport> reportsList = standardReportService
					.searchCompletedReports(searchQueryInfo);
			/*
			 * doesn't contain report content
			 */
			for (SimpleCompletedReport completedReport : reportsList) {
				Map<String, String> repParameters = importFromCSV(completedReport
						.getReportParameterText());
				String reportFormatCode = getReportFormat(repParameters);

				StandardReport standardReport = completedReport
						.getStandardReport();
				Boolean canView = false;
				if (ReportFormat.DETAIL.equals(reportFormatCode)) {

					String detailedPermission = standardReport
							.getDetailedReportPermission();
					if (detailedPermission == null
							|| permissions.contains(detailedPermission)) {
						canView = true;
					}
				} else if (ReportFormat.SUMMARY.equals(reportFormatCode)) {
					String summaryPermission = standardReport
							.getSummaryReportPermission();
					if (summaryPermission == null
							|| permissions.contains(summaryPermission)) {
						canView = true;
					}
				}

				// User doesn't have permission or doesn't have access to the
				// sites
				if (!canView)
					continue;

				CompletedReportForm completedRepForm = new CompletedReportForm();
				reportConversionService.convert(completedReport,
						completedRepForm);
				ReportFormat reportFormat = getTerminologyCache()
						.getTermByCode(ReportFormat.class, reportFormatCode);
				if (reportFormat != null) {
					completedRepForm.setReportFormat(new TermType(reportFormat
							.getName(), reportFormat.getCode()));
				}
				completedReports.add(completedRepForm);
			}
			return completedReports;
		} catch (Exception ex) {
			throw handleException(
					"Error occured while getting the completed reports.", ex);
		}
	}

	public List<ScheduledReportForm> getScheduledReports()
			throws ControllerException {
		try {
			ReportFilterSearchQueryInfo searchQueryInfo = new ReportFilterSearchQueryInfo();
			searchQueryInfo.setUser(getCurrentUser());
			// build search query info
			List<ScheduledReportForm> scheduledReports = new ArrayList<ScheduledReportForm>();
			List<ScheduledReport> reports = standardReportService
					.searchScheduledReports(searchQueryInfo);

			// convert
			if (reports != null && reports.size() > 0) {
				for (ScheduledReport scheduledReport : reports) {
					ScheduledReportForm scheduledReportForm = new ScheduledReportForm();
					reportParameterConversionService
							.convertFromText(scheduledReport);
					reportConversionService.convert(scheduledReport,
							scheduledReportForm);
					scheduledReports.add(scheduledReportForm);
				}
			}
			return scheduledReports;
		} catch (Exception ex) {
			throw handleException(
					"Error occurred while getting the scheduled reports.", ex);
		}
	}

	/**
	 * Generates the report with a delay of 1 minute
	 * 
	 * @param reportSetupForm
	 * @throws ControllerException
	 */
	public ReportForm generateReport(ReportSetupForm reportSetupForm)
			throws ControllerException {
		try {
			ReportSetup reportSetup = new ReportSetup();
			reportConversionService.convert(reportSetupForm, reportSetup);
			reportSetup.setSetupUser(getCurrentUser().getUserCredentials()
					.getUserID());
			ReportForm reportForm = new ReportForm();
			reportForm.reportSetup = reportSetupForm;
			if (reportSetup.getFileType() == null
					|| reportSetup.getFileType().getCode().equals("9")) {
				getReportData(reportForm, reportSetup);
			} else {
				// reportParameterConversionService.convertToText(reportSetup);
				standardReportService.generateReport(reportSetup);
			}
			return reportForm;
		} catch (Exception ex) {
			throw handleException("Error occured while generating the report.",
					ex);
		}
	}

	public ReportForm generateReportData(String reportId)
			throws ControllerException {
		try {
			EntityKey<SimpleCompletedReport> identifier = EntityKeyFactory
					.createEntityKey(new Long(reportId),
							SimpleCompletedReport.class);
			ReportSetup reportSetup = standardReportService
					.getCompletedReportParameters(identifier);
			ReportSetupForm reportSetupForm = new ReportSetupForm();
			reportConversionService.convert(reportSetup, reportSetupForm);
			ReportForm reportForm = new ReportForm();
			reportForm.reportSetup = reportSetupForm;
			getReportData(reportForm, reportSetup);
			return reportForm;
		} catch (Exception e) {
			throw new ControllerException(e.getMessage(), e);
		}
	}

	private void getReportData(ReportForm reportForm, ReportSetup reportSetup)
			throws ServiceException {

		List<Object[]> data = getInventoryService().generateInventoryReports(
				reportSetup);

		List<Serializable> results = new ArrayList<Serializable>();
		// conversion
		if (reportSetup.getStandardReport().getCode().equals("DI1")) {
			for (Object[] objs : data) {
				if (0 == (Byte) objs[5]) {
					DIReportData record = new DIReportData();
					record.visnName = (String) objs[0];
					record.facilityName = (String) objs[1];
					record.vendorName = (String) objs[2];
					record.deviceName = (String) objs[3];
					record.deviceCount = (Integer) objs[4];
					results.add(record);
				}
			}
		} else if (reportSetup.getStandardReport().getCode().equals("DI2")) {
			for (Object[] objs : data) {
				// Summaries eliminated for view
				if (0 == (Byte) objs[9]) {
					DeviceStatusReportData record = new DeviceStatusReportData();
					// need to remove location
					record.location = (String) objs[0];
					record.visnName = (String) objs[0];
					record.facility = (String) objs[0];
					record.vendorName = (String) objs[1];
					record.deviceName = (String) objs[2];
					record.assigned = (Integer) objs[3];
					record.available = (Integer) objs[4];
					;
					record.refurbish = (Integer) objs[5];
					record.repair = (Integer) objs[6];
					record.decommissioned = (Integer) objs[7];
					record.total = (Integer) objs[8];
					record.glevel = (Byte) objs[9];
					results.add(record);
				}
			}
		} else if (reportSetup.getStandardReport().getCode().equals("DI3")) {
			for (Object[] objs : data) {
				DeviceFailureReportData record = new DeviceFailureReportData();
				record.visnName = (String) objs[0];
				record.facility = (String) objs[1];
				record.vendorName = (String) objs[2];
				record.deviceName = (String) objs[3];
				record.totalDevices = (Integer) objs[4];
				record.failedDevices = (Integer) objs[5];
				record.failureRate = ((BigDecimal) objs[6]).intValue();
				results.add(record);
			}
		} else if (reportSetup.getStandardReport().getCode().equals("DI4")) {
			for (Object[] objs : data) {
				DeviceShortageReportData record = new DeviceShortageReportData();
				record.visnName = (String) objs[0];
				record.facility = (String) objs[1];
				record.vendorName = (String) objs[2];
				record.deviceName = (String) objs[3];
				record.totalDevices = (Integer) objs[4];
				record.requiredDevices = objs[5] == null ? 0
						: (Integer) objs[5];
				record.shortage = objs[6] == null ? 0 : (Integer) objs[6];
				results.add(record);
			}
		}
		reportForm.reportData = results;
	}

	public void deleteCompletedReport(String completedReportId)
			throws ControllerException {
		try {
			// alertService.deleteAlertByReportId(new Long(completedReportId));
			EntityKey<CompletedReport> identifier = EntityKeyFactory
					.createEntityKey(new Long(completedReportId),
							CompletedReport.class);
			standardReportService.deleteCompletedReport(identifier,
					getCurrentUser());
		} catch (Exception e) {
			throw handleException("Error deleting the completed report", e);
		}
	}

	public CompletedReportForm getCompletedReport(String reportId)
			throws ControllerException {
		try {
			EntityKey<SimpleCompletedReport> identifier = EntityKeyFactory
					.createEntityKey(new Long(reportId),
							SimpleCompletedReport.class);
			SimpleCompletedReport simpleCompletedReport = standardReportService
					.getSimpleCompletedReport(identifier);
			CompletedReportForm completedReportForm = new CompletedReportForm();
			reportConversionService.convert(simpleCompletedReport,
					completedReportForm);
			return completedReportForm;
		} catch (Exception e) {
			throw handleException("Error getting the Simple completed report",
					e);
		}
	}

	public ReportSetupForm getCompletedReportParameters(
			String simpleCompletedReportId) throws ControllerException {
		try {
			EntityKey<SimpleCompletedReport> identifier = EntityKeyFactory
					.createEntityKey(new Long(simpleCompletedReportId),
							SimpleCompletedReport.class);
			ReportSetup reportSetup = standardReportService
					.getCompletedReportParameters(identifier);
			ReportSetupForm reportSetupForm = new ReportSetupForm();
			reportConversionService.convert(reportSetup, reportSetupForm);
			return reportSetupForm;
		} catch (Exception e) {
			throw handleException("Error getting the Simple completed report",
					e);
		}
	}

	public void extendCompletedReport(String reportId)
			throws ControllerException {
		try {
			// delete the alerts
			// alertService.deleteAlertByReportId(new Long(reportId));

			// extend the report by one year from today
			EntityKey<SimpleCompletedReport> identifier = EntityKeyFactory
					.createEntityKey(new Long(reportId),
							SimpleCompletedReport.class);
			SimpleCompletedReport simpleCompletedReport = standardReportService
					.getSimpleCompletedReport(identifier);
			Calendar cal = Calendar.getInstance();
			cal.add(Calendar.YEAR, 1);
			simpleCompletedReport.setExpirationDate(cal.getTime());
			standardReportService.updateCompletedReport(simpleCompletedReport);
		} catch (Exception e) {
			throw handleException("Error deleting the completed report", e);
		}
	}

	// schedule of reports
	public ControllerResult scheduleReport(ReportSetupForm reportSetupForm)
			throws ControllerException {

		// if it is an update delete and cancel the old scheduled report and
		// schedule a new one.
		if (StringUtils.isNotEmpty(reportSetupForm.getId())) {
			cancelScheduledReport(reportSetupForm.getId());
		}
		reportSetupForm.setId(null); // remove the id to create a new schedule
		ScheduledReport scheduledReport = new ScheduledReport();

		try {
			reportConversionService.convert(reportSetupForm, scheduledReport);
			scheduledReport.setUserName(getCurrentUser().getUserCredentials()
					.getUserID());
			// save setup before scheduling
			if (logger.isDebugEnabled()) {
				logger.debug("Scheduled report before being saved");
				logger.debug(scheduledReport);
			}
			ScheduledReport persisted = standardReportService
					.saveScheduledReport(scheduledReport, true);
			if (logger.isDebugEnabled()) {
				logger.debug("Scheduled report after being saved");
				logger.debug(persisted);
			}
			if (scheduledReport.getId() == null && persisted.getId() != null) {
				scheduledReport.setId(persisted.getId());
			}
			standardReportService.scheduleReport(scheduledReport);
			// save the scheduled report again to save the next scheduled date
			standardReportService.saveScheduledReport(scheduledReport, false);
			return new ControllerResult(ControllerResult.SUCCESS);
		} catch (InvalidScheduleException e) {
			// delete the saved report setup
			cancelScheduledReport(String.valueOf(scheduledReport.getId()));
			return new ControllerResult(ControllerResult.ERROR, e.getMessage());
		} catch (ServiceException e) {
			// Always delete the schedule report if a report failed. CCR 407
			// Check transaction details here
			if (logger.isDebugEnabled()) {
				logger.debug("Exception message: " + e.getMessage());
			}
			// delete the saved report setup
			cancelScheduledReport(String.valueOf(scheduledReport.getId()));
			return new ControllerResult(ControllerResult.ERROR, e.getMessage());
		}
	}

	public void cancelScheduledReport(String reportId)
			throws ControllerException {
		try {
			EntityKey<ScheduledReport> identifier = EntityKeyFactory
					.createEntityKey(new Long(reportId), ScheduledReport.class);
			standardReportService.cancelScheduledReport(identifier);
		} catch (Exception ex) {
			throw handleException(
					"Error occured while canceling the scheduled Report.", ex);
		}
	}

	public ReportSetupForm getScheduledReport(String scheduledReportId)
			throws ControllerException {
		try {
			EntityKey<ScheduledReport> identifier = EntityKeyFactory
					.createEntityKey(new Long(scheduledReportId),
							ScheduledReport.class);
			ScheduledReport scheduledReport = standardReportService
					.getScheduledReport(identifier);
			ReportSetupForm reportSetupForm = new ReportSetupForm();
			reportConversionService.convert(scheduledReport, reportSetupForm);
			return reportSetupForm;
		} catch (Exception e) {
			throw handleException("Error getting the scheduled report", e);
		}
	}

	public void saveReportCriteria(ReportSetupForm reportSetupForm)
			throws ControllerException {
		try {
			ReportSetup reportSetup = new ReportSetup();
			reportConversionService.convert(reportSetupForm, reportSetup);
			reportSetup.setSetupUser(getCurrentUser().getUserCredentials()
					.getUserID());
			reportParameterConversionService.convertToText(reportSetup);
			standardReportService.saveReportCriteria(reportSetup);
		} catch (Exception e) {
			throw handleException("Error saving Report Criteria", e);
		}

	}

	public ReportScheduleForm validateSchedule(
			ReportScheduleForm reportScheduleForm) throws ControllerException {
		try {
			ReportSchedule reportSchedule = new ReportSchedule();
			reportConversionService.convert(reportScheduleForm, reportSchedule);
			reportScheduleForm.setCronExpression(reportSchedule
					.getCronExpression());
			reportScheduleForm.setFriendlyDescription(reportSchedule
					.getFriendlyDescription());
			if (reportSchedule.getNextScheduledDate() != null) {
				reportScheduleForm.setNextFireTime(new FormattedDateForm(
						new DateWithTimeZone(reportSchedule
								.getNextScheduledDate(), TimeZoneUtils
								.getTimeZone())));
			} else {
				reportScheduleForm.setNextFireTime(null);
			}
			return reportScheduleForm;
		} catch (Exception e) {
			throw handleException("Error converting ReportScheduleForm", e);
		}
	}

	public StandardReportService getStandardReportService() {
		return standardReportService;
	}

	public void setStandardReportService(
			StandardReportService standardReportService) {
		this.standardReportService = standardReportService;
	}

	public ReportConversionService getReportConversionService() {
		return reportConversionService;
	}

	public void setReportConversionService(
			ReportConversionService reportConversionService) {
		this.reportConversionService = reportConversionService;
	}

	public ReportParameterConversionService getReportParameterConversionService() {
		return reportParameterConversionService;
	}

	public void setReportParameterConversionService(
			ReportParameterConversionService reportParameterConversionService) {
		this.reportParameterConversionService = reportParameterConversionService;
	}

	private ControllerException handleException(String errorMessage,
			Exception ex) {
		return handleException(errorMessage, ex, this);
	}

	private boolean isAllowedRunScheduleReport(UserPrincipal user,
			StandardReport stdReport) {
		Set<Permission> rptCaps = stdReport.getPermissions();
		if (rptCaps != null && rptCaps.size() > 0) {
			for (Permission p : rptCaps) {
				String code = p.getName();
				if (user.isPermissionGranted(code)) {
					return true;
				}
			}
		} else {
			return true;
		}
		return false;
	}

	private String getReportFormat(Map<String, String> map) {
		return map.get(ReportConstants.REPORT_PARAM_REPORT_FORMAT);
	}

	private Map<String, String> importFromCSV(String csv) {
		Map<String, String> data = new HashMap<String, String>();
		if (StringUtils.isNotEmpty(csv)) {
			String[] vals = csv.split(",");
			String[] val = null;
			for (String val2 : vals) {
				val = val2.split("=");
				if (val.length == 2)
					data.put(val[0].trim(), val[1]);
			}
		}
		return data;
	}
}
