/**
 * 
 */
package gov.va.med.ccht.controller;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Set;

import gov.va.med.ccht.model.qir.QIR;
import gov.va.med.ccht.model.qir.QIRAttachment;
import gov.va.med.ccht.model.qir.QIRSearchParameters;
import gov.va.med.ccht.model.qir.QIRSearchResult;
import gov.va.med.ccht.model.qir.QIRSimpleAttachment;
import gov.va.med.ccht.model.qir.QIRStatusType;
import gov.va.med.ccht.model.terminology.DocumentType;
import gov.va.med.ccht.ui.common.AbstractController;
import gov.va.med.ccht.ui.common.ControllerException;
import gov.va.med.ccht.ui.common.ControllerResult;
import gov.va.med.ccht.ui.model.AttachmentForm;
import gov.va.med.ccht.ui.model.QIRForm;
import gov.va.med.fw.model.EntityKey;
import gov.va.med.fw.model.EntityKeyFactory;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.service.ServiceOptimisticLockException;
import gov.va.med.fw.util.StringUtils;

/**
 * @author vhaisakatikm
 *
 */
public class QIRController extends AbstractController {

	private QIRConversionService qirConversionService = null;

	public QIRConversionService getQirConversionService() {
		return qirConversionService;
	}

	public void setQirConversionService(QIRConversionService qirConversionService) {
		this.qirConversionService = qirConversionService;
	}
	
	public List<QIRSearchResult> searchQIR(QIRSearchParameters qirSearchParameters)
	throws ControllerException {
		try {
			List<Object[]> data = getQirService().searchQIR(qirSearchParameters);
			
			
			List<QIRSearchResult> results = new ArrayList<QIRSearchResult>();
			for (Object[] objs:data){
				
				QIRSearchResult record = new QIRSearchResult();
				record.id = String.valueOf(objs[0]);
				record.qirType = (String)objs[1];
				record.qirStatusType = (String)objs[2];
				record.visn = (String)objs[3];
				record.facilityCode = (String)objs[4];
				record.facility = (String)objs[5];
				record.deviceType = (String)objs[6];
				record.vendor = (String)objs[7];			
				record.complaint = (String)objs[8];	
				record.headline = (String)objs[9];				
				record.submittedDate = (Date)objs[10];
				record.submittedBy = (String)objs[11];
				record.submittedByName = (String)objs[12];
				record.attachmentCount = ((Integer)objs[13]).toString();				
				if((Integer)objs[13]>0)
				{
					record.hasAttachments="true";
				}
			
				results.add(record);
			}		
			return results;
		}catch (Exception e){
			throw handleException(e.getMessage(),e);
		}
	}
	
		
	public QIRForm getQIR(String id) throws ControllerException {
		EntityKey<QIR> entityKey = EntityKeyFactory.createEntityKey(new Long(id), QIR.class);
		try {
			QIR qir = getQirService().getQIR(entityKey);
			QIRForm qirForm = new QIRForm();
			getQirConversionService().convert(qir, qirForm);
			return qirForm;
		}catch (Exception e){
			throw new ControllerException(e.getMessage(),e);
		}
	}
	
	public QIRForm getQIRById(String id) throws ControllerException {
		EntityKey<QIR> entityKey = EntityKeyFactory.createEntityKey(new Long(id), QIR.class);
		try {
			QIR qir = getQirService().getQIR(entityKey);
			QIRForm qirForm = new QIRForm();
			getQirConversionService().convert(qir, qirForm);
			return qirForm;
		}catch (Exception e){
			throw new ControllerException(e.getMessage(),e);
		}
	}
	

	public List<QIRForm> getQIRs(List<QIRSearchResult> qirs) throws ControllerException {
		
		List<QIRForm> qirList =  new ArrayList<QIRForm>();	
		
		for(QIRSearchResult qir : qirs)
		{
			String id = qir.id;
			QIRForm newQirForm = getQIRById(id);
			
			try
			{
				
				String label = newQirForm.getQirStatusType().getLabel();
				
				boolean isNew = label.equalsIgnoreCase("New");
				boolean withDrawn = label.equalsIgnoreCase("Withdrawn");
				
				if(newQirForm.getVendorResponseDueDate() == null && !(isNew || withDrawn))
				{
					Date vendorResponseDueDate = getQirService().getDueDateFromHistory(id);	
					newQirForm.setVendorResponseDueDate(vendorResponseDueDate);
				}
			}
			catch(Exception e)
			{
				throw new ControllerException(e.getMessage(),e);
			}
		
			qirList.add(newQirForm);			
		}		 
		return qirList;
	}
	
	public void addQIR(QIRForm qirForm) throws ControllerException {
		try {
			QIR qir = new QIR();
			getQirConversionService().convert(qirForm, qir);
			getQirService().updateQIR(qir);
			addAttachments(qir, qirForm);
		}catch (Exception e){
			throw handleException(e.getMessage(),e);
		}
	}
	
	
	
	
	public ControllerResult updateQIR(QIRForm qirForm) throws ControllerException{
		return this.updateQIR(qirForm, false);
	}
	
	public ControllerResult updateQIR(QIRForm qirForm, boolean updatedDueDate ) throws ControllerException{
		try {
			EntityKey<QIR> entityKey = 
				EntityKeyFactory.createEntityKey(new Long(qirForm.getId()), new Integer(qirForm.getVersionNumber()), QIR.class);
			QIR qir = getQirService().getQIR(entityKey);
			if (qir != null) {
				// copy QIR data excluding attachments
				getQirConversionService().convert(qirForm, qir);
				getQirService().updateQIR(qir,updatedDueDate);
				addAttachments(qir, qirForm);
				removeAttachments(qir, qirForm);
				return new ControllerResult(ControllerResult.SUCCESS);
			}else {
				throw new ControllerException ("QIR Not found");
			}
		}catch (ServiceOptimisticLockException ople){
			return new ControllerResult(ControllerResult.DATA_CHANGED);
		}catch (ServiceException e){
			throw handleException(e.getMessage(),e);
		}
	}
	
	private void removeAttachments(QIR qir, QIRForm qirForm) throws ServiceException {
		
		// Remove attachments
		Set<QIRSimpleAttachment> persistedList = qir.getAttachments();
		List<AttachmentForm>  incomingList = qirForm.getAttachments();
		
		for (QIRSimpleAttachment persisted:persistedList) {
			String persistedID = String.valueOf( persisted.getId() );
			
			// If no incoming attachments, remove all persisted attachments
			if( incomingList == null ) {
				this.getQirService().deleteQIRAttachment(persisted);
			}
			else {
				// Look for an attachment in database that is not on incoming list
				boolean found = false;
				for (AttachmentForm incoming:incomingList) {
					String incomingID = incoming.getId();
					if (StringUtils.isNotEmpty(incomingID) && persistedID.equals(incomingID)) {
						found = true;
						break;
					}
				}
				if( !found ) {
					this.getQirService().deleteQIRAttachment(persisted);
				}
			}
		}
	}
	private void addAttachments(QIR qir, QIRForm qirForm) throws ServiceException {
		//Add attachments
		List<AttachmentForm>  attachments = qirForm.getAttachments();
		if (attachments == null) attachments = new ArrayList<AttachmentForm>();
		for (AttachmentForm attachment:attachments) {
			if (StringUtils.isEmpty(attachment.getId())) {
				QIRAttachment newAttachment = new QIRAttachment();
				newAttachment.setDate(attachment.getDate());
				newAttachment.setDocumentName(attachment.getName());
				newAttachment.setDocumentType(
					getQirConversionService().getLookup(DocumentType.class, attachment.getType()));
				newAttachment.setLength(attachment.getLength());
				newAttachment.setData(attachment.getData());
				newAttachment.setQirId(qir.getId());
				getQirService().updateQIRAttachment(newAttachment);
			}
		}
	}
	public void generateVendorResponseDueNotifications() throws ControllerException{
		try {
			getQirService().generateVendorResponseDueNotifications();
		}catch (Exception e){
			throw handleException(e.getMessage(),e);
		}
	}
}
