package gov.va.med.ccht.controller;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;

import gov.va.med.ccht.model.dmp.DMPReportParameters;
import gov.va.med.ccht.model.dmpreport.DMPReportSearchResult;
import gov.va.med.ccht.model.dmpreport.DmpReportEntry;
import gov.va.med.ccht.model.inventory.Facility;
import gov.va.med.ccht.model.inventory.Vendor;
import gov.va.med.ccht.model.inventory.Visn;
import gov.va.med.ccht.service.dmpreport.DmpReportService;
import gov.va.med.ccht.ui.common.AbstractController;
import gov.va.med.ccht.ui.common.ControllerException;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.ui.model.TermType;

public class HTReportController extends AbstractController {
	private DmpReportService dmpReportService;
	
	private List<Visn> ALL_VISNS = null;
	private List<Facility> ALL_FACILITIES = null;
	private HashMap<String, Facility> ALL_FACILITY_MAP = new HashMap<String, Facility>();
	private List<Vendor> ALL_VENDORS = null;
	
	private static final TermType ITEM_BLANK = new TermType("", "0");
	private static final TermType ITEM_ALL = new TermType("ALL", "-1");
	private static final String ITEM_ALL_VALUE = "-1";
	
	/*
	 * General Report Functionality
	 */
	public List<TermType> getModalityTypes(int reportType) throws ControllerException
	{
		try
		{
			return getInventoryService().getModalityType(reportType);
		}
		catch(Exception e)
		{
			throw new ControllerException(e.getMessage(), e);
		}
	}
	
	/*
	 * DMP Reports
	*/
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	public DmpReportService getDmpReportService() {
		return dmpReportService;
	}
	public void setDmpReportService(DmpReportService dmpReportService) {
		this.dmpReportService = dmpReportService;
	}
	
	//get all reported dmps names from DMP Report service
	public List<String> findAllStandardDmpNames() throws ControllerException {
		List<String> results = null; //new ArrayList<String>();
		try {
			results = getDmpReportService().findAllStandardDmpNames();
			 Collections.sort(results);
			 results.add(0, "");
			return results;
		} catch (ServiceException e) {
			logger.error(e.getMessage());
			results=new ArrayList<String>();
			results.add("DMP HDR Service Unavailable");
			return results;
		}
	}
	
	public List<TermType> getVisns(boolean withBlankAllItem) throws ControllerException {
		List<TermType> results = new ArrayList<TermType>();
		TermType termType  = null;
		try {
			if (ALL_VISNS == null) {
				ALL_VISNS = getInventoryService().getVisns();
				//sort results
				Collections.sort(ALL_VISNS, new Comparator<Visn>() {
					@Override
					public int compare(Visn entry1, Visn entry2) {
						return entry1.getId().compareTo(entry2.getId());
					}
				});
			}
			//List<Visn> visns = getInventoryService().getVisns();
			if (ALL_VISNS != null && ALL_VISNS.size() > 0) {
				for (Visn visn : ALL_VISNS) {
					termType = new TermType(visn.getName(), visn.getId().toString());
					results.add(termType);
				}
			}
						
			//termType = new TermType("", "0");
			if (withBlankAllItem) {
				 results.add(0, ITEM_BLANK);
				 //results.add(1, ITEM_ALL);
			}
			 return results;
		} catch (Exception e) {
			throw new ControllerException(e.getMessage(),e);
		}
	}
	
	public List<TermType> getVisns() throws ControllerException {
		return getVisns(true);
	}
	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	public List<TermType> getFacilitiys(boolean withBlankAllItem) throws ControllerException {
		List<TermType> results = new ArrayList<TermType>(); 
		//List<Facility> returnedList = null;
		TermType termType  = null;
		try {
			if (ALL_FACILITIES == null) {
				ALL_FACILITIES = getInventoryService().getFacilities();
				fillFacilityMap();
				Collections.sort(ALL_FACILITIES, new Comparator<Facility>() {
					@Override
					public int compare(Facility entry1, Facility entry2) {
						String e1 = String.format("%06d",entry1.getVisn().getId()) + entry1.getStationNumber() + entry1.getName();
						String e2 = String.format("%06d",entry2.getVisn().getId()) + entry2.getStationNumber() + entry2.getName();
						return e1.compareTo(e2);
					}
				});
			}
			if (ALL_FACILITIES != null && ALL_FACILITIES.size() > 0) {
				for (Facility facility : ALL_FACILITIES) {
					termType = new TermType(facility.getVisn().getName() + " Facility " + facility.getStationNumber() + " " + facility.getName(), facility.getId().toString());
					results.add(termType);
				}
			}
			//sort results
			/*
			Collections.sort(results, new Comparator<TermType>() {
				@Override
				public int compare(TermType entry1, TermType entry2) {
					return entry1.getLabel().compareTo(entry2.getLabel());
				}
			});
			*/
			if (withBlankAllItem) {
				results.add(0, ITEM_BLANK);
				//results.add(1, ITEM_ALL);
			}
			return results;
		} catch (Exception e) {
			logger.error("DMPController Exception:  " + e.getMessage());
			throw handleException(e.getMessage(), e);
		}
	}
	
	public List<TermType> getFacilitiys() throws ControllerException {
		return getFacilitiys(true);
	}
	////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	public List<TermType> getVendors(boolean withBlankAllItem) throws ControllerException {
		List<TermType> results = new ArrayList<TermType>(); 
		List<Vendor> returnedList = null;
		TermType termType  = null;
		try {
			if (ALL_VENDORS == null) {
				ALL_VENDORS = getInventoryService().getVendors();
			}
			if (ALL_VENDORS != null && ALL_VENDORS.size() > 0) {
				for (Vendor vendor : ALL_VENDORS) {
					termType = new TermType(vendor.getName(), vendor.getNumber());
					results.add(termType);
				}
			}
			//sort results
			Collections.sort(results, new Comparator<TermType>() {
				@Override
				public int compare(TermType entry1, TermType entry2) {
					return entry1.getLabel().compareTo(entry2.getLabel());
				}
			});
			if (withBlankAllItem) {
				results.add(0, ITEM_BLANK);
				results.add(1, ITEM_ALL);
			}
			return results;
		} catch (Exception e) {
			logger.error("DMPController Exception:  " + e.getMessage());
			throw handleException(e.getMessage(), e);
		}
	}
	
	public List<TermType> getVendors() throws ControllerException {
		return getVendors(true);
	}

	public DMPReportSearchResult findDmpReports(DMPReportParameters dmpReportParameters) throws ControllerException {
		DMPReportSearchResult searchResult = new DMPReportSearchResult();
		checkAndReplace(dmpReportParameters);
		if (dmpReportParameters.getSearchBy() == DMPReportParameters.SEARCH_BY_DMP) {
			searchResult = this.findDmpReportsByDmp(dmpReportParameters);
		}
		else if (dmpReportParameters.getSearchBy() == DMPReportParameters.SEARCH_BY_DMP_VISN) {
			searchResult = this.findDmpReportsByDmpAndVisn(dmpReportParameters);
		} else if (dmpReportParameters.getSearchBy() == DMPReportParameters.SEARCH_BY_DMP_FACILITY) {
			searchResult = this.findDmpReportsByDmpAndFacility(dmpReportParameters);
		} else if (dmpReportParameters.getSearchBy() == DMPReportParameters.SEARCH_BY_DMP_VENDOR) {
			searchResult = this.findDmpReportsByDmpAndVendor(dmpReportParameters);
		}  else if (dmpReportParameters.getSearchBy() == DMPReportParameters.SEARCH_BY_DMP_VISN_VENDOR) {
			searchResult = this.findDmpReportsByDmpVisnAndVendor(dmpReportParameters);
		}  else if (dmpReportParameters.getSearchBy() == DMPReportParameters.SEARCH_BY_DMP_FACILITY_VENDOR) {
			searchResult = this.findDmpReportsByDmpFacilityAndVendor(dmpReportParameters);
		}
		return searchResult;
	}
	
	//get all reported dmps/patient names from DMP Report service
	public DMPReportSearchResult findDmpReportsByDmp(DMPReportParameters dmpReportParameters) throws ControllerException {
		DMPReportSearchResult searchResult = new DMPReportSearchResult();
		List<DmpReportEntry> dmpReportEntries = new ArrayList<DmpReportEntry>();
		int totalPatients=0;
		try {
			List<String> dmpNameList = dmpReportParameters.getDmpNameList();
			if (dmpNameList != null) {
				for (int i = 0; i < dmpNameList.size(); i++) {
					String name = dmpNameList.get(i);
					if (name != null && name.trim().length() > 0) {
						DmpReportEntry dmpReportEntry = new DmpReportEntry();
						dmpReportEntry.setDmp(name);
						dmpReportEntry.setStartDate(dmpReportParameters.getSubmittedFromDate());
						dmpReportEntry.setEndDate(dmpReportParameters.getSubmittedToDate());
						dmpReportEntries.add(dmpReportEntry);
					}
				}
				dmpReportEntries=getDmpReportService().findDmpReportsByDmp(dmpReportEntries);
				for(DmpReportEntry dmpReportEntry: dmpReportEntries)
				{
					totalPatients=totalPatients+dmpReportEntry.getNumberOfPatients();
				}
			}
			//sort results
			Collections.sort(dmpReportEntries, new Comparator<DmpReportEntry>() {
				@Override
				public int compare(DmpReportEntry entry1, DmpReportEntry entry2) {
					return entry1.getDmp().compareTo(entry2.getDmp());
				}
			});
			searchResult.setDmpReportEntryList(dmpReportEntries);
			searchResult.setTotalPatients(totalPatients);
			return searchResult;
		} catch (Exception e) {
			logger.error("DMPController Exception:  " + e.getMessage());
			throw handleException(e.getMessage(), e);
		}
	}
	///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	//get all reported dmps visns from DMP Report service
	public DMPReportSearchResult findDmpReportsByDmpAndVisn(DMPReportParameters dmpReportParameters) throws ControllerException {
		DMPReportSearchResult searchResult = new DMPReportSearchResult();
		List<DmpReportEntry> dmpReportEntries = new ArrayList<DmpReportEntry>();
		boolean isAllVisns=false;
		int totalPatients = 0;
		try {
			List<String> dmpNameList = dmpReportParameters.getDmpNameList();
			if (dmpNameList != null) {
				for (int i = 0; i < dmpNameList.size(); i++) {
					String dmpName = dmpNameList.get(i);
					if (dmpName != null && dmpName.trim().length() > 0) {
						//fill in visn
						if (dmpReportParameters.getVisnList() != null && !dmpReportParameters.isAllVisns()) {
							for (TermType visn : dmpReportParameters.getVisnList()) {
								DmpReportEntry dmpReportEntry = new DmpReportEntry();
								dmpReportEntry.setDmp(dmpName);
								dmpReportEntry.setVisn(visn.getValue());
								dmpReportEntry.setVisnName(visn.getLabel());
								dmpReportEntry.setStartDate(dmpReportParameters.getSubmittedFromDate());
								dmpReportEntry.setEndDate(dmpReportParameters.getSubmittedToDate());
								dmpReportEntries.add(dmpReportEntry);
							}
						}
						else if(dmpReportParameters.getVisnList() != null && dmpReportParameters.isAllVisns())
						{
							isAllVisns=true;
							DmpReportEntry dmpReportEntry = new DmpReportEntry();
							dmpReportEntry.setDmp(dmpName);
							dmpReportEntry.setVisn("National");
							dmpReportEntry.setVisnName("ALL");
							dmpReportEntry.setStartDate(dmpReportParameters.getSubmittedFromDate());
							dmpReportEntry.setEndDate(dmpReportParameters.getSubmittedToDate());
							dmpReportEntries.add(dmpReportEntry);
							
						}
						dmpReportEntries=getDmpReportService().findDmpReportsByDmpAndVisn(dmpReportEntries);
						for(DmpReportEntry dmpReportEntry: dmpReportEntries)
						{
							totalPatients=totalPatients+dmpReportEntry.getNumberOfPatients();
						}
					}
				}
			}
			//sort results
			//String.format("%06d",entry1.getVisn().getId())
			if(isAllVisns)
			{
				Collections.sort(dmpReportEntries, new Comparator<DmpReportEntry>() {
					@Override
					public int compare(DmpReportEntry entry1, DmpReportEntry entry2) {
						return entry1.getDmp().compareTo(entry2.getDmp());
					}
				});
				
			}
			else
			{
			Collections.sort(dmpReportEntries, new Comparator<DmpReportEntry>() {
				@Override
				public int compare(DmpReportEntry entry1, DmpReportEntry entry2) {
					String visnId1 = String.format("%06d", Integer.valueOf(entry1.getVisn()));
					String visnId2 = String.format("%06d", Integer.valueOf(entry2.getVisn()));
					return (entry1.getDmp() + visnId1).compareTo(entry2.getDmp() + visnId2);
				}
			});
			}
			searchResult.setDmpReportEntryList(dmpReportEntries);
			searchResult.setTotalPatients(totalPatients);
			return searchResult;
		} catch (Exception e) {
			logger.error("DMPController Exception:  " + e.getMessage());
			throw handleException(e.getMessage(), e);
		}
	}

	public DMPReportSearchResult findDmpReportsByDmpAndFacility(DMPReportParameters dmpReportParameters) throws ControllerException {
		DMPReportSearchResult searchResult = new DMPReportSearchResult();
		List<DmpReportEntry> dmpReportEntries = new ArrayList<DmpReportEntry>();
		int totalPatients = 0;
		boolean isAllFacilities=false;
		try {
			List<String> dmpNameList = dmpReportParameters.getDmpNameList();
			if (dmpNameList != null) {
				for (int i = 0; i < dmpNameList.size(); i++) {
					String dmpName = dmpNameList.get(i);
					if (dmpName != null && dmpName.trim().length() > 0) {
						//fill in facility
						if(dmpReportParameters.isAllFacilities()&&dmpReportParameters.isAllVisns())
						{
							isAllFacilities=true;
							DmpReportEntry dmpReportEntry = new DmpReportEntry();
							dmpReportEntry.setDmp(dmpName);
							dmpReportEntry.setFacilityStationNumber("National");
							dmpReportEntry.setAllFacilities(true);
							dmpReportEntry.setVisnName("ALL");
							dmpReportEntry.setAllVisns(true);
							dmpReportEntry.setStartDate(dmpReportParameters.getSubmittedFromDate());
							dmpReportEntry.setEndDate(dmpReportParameters.getSubmittedToDate());
							dmpReportEntries.add(dmpReportEntry);
						}
						else if (dmpReportParameters.getFacilityList() != null) {
							for (TermType facility : dmpReportParameters.getFacilityList()) {
								DmpReportEntry dmpReportEntry = new DmpReportEntry();
								dmpReportEntry.setDmp(dmpName);
								Facility theFacility = (Facility)ALL_FACILITY_MAP.get(facility.getValue());
								//String facilityName = theFacility.getName();
								dmpReportEntry.setFacility(theFacility.getName());
								dmpReportEntry.setFacilityStationNumber(theFacility.getStationNumber());
								dmpReportEntry.setFacilityId(new Integer(facility.getValue()));
								dmpReportEntry.setStartDate(dmpReportParameters.getSubmittedFromDate());
								dmpReportEntry.setEndDate(dmpReportParameters.getSubmittedToDate());
								Visn visn = getVisn(facility.getValue());
								if (visn != null) {
									dmpReportEntry.setVisnName(visn.getName());
									dmpReportEntry.setVisn(visn.getId().toString());
								}
								else {
									dmpReportEntry.setVisnName("");
									dmpReportEntry.setVisn("0");
								}
								dmpReportEntries.add(dmpReportEntry);
							}
						}
					}
				}
				dmpReportEntries=getDmpReportService().findDmpReportsByDmpAndFacility(dmpReportEntries);
				for(DmpReportEntry dmpReportEntry: dmpReportEntries)
				{
					if(dmpReportEntry.isAllFacilities()&& dmpReportEntry.isAllVisns())
					{
						dmpReportEntry.setFacility("ALL");
					}
					else
					{
						dmpReportEntry.setFacility("Facility " + dmpReportEntry.getFacilityStationNumber() + " " + dmpReportEntry.getFacility());
					}
					totalPatients=totalPatients+dmpReportEntry.getNumberOfPatients();
				}
			}
			//sort results
			if(isAllFacilities)
			{
				Collections.sort(dmpReportEntries, new Comparator<DmpReportEntry>() {
					@Override
					public int compare(DmpReportEntry entry1, DmpReportEntry entry2) {
						return entry1.getDmp().compareTo(entry2.getDmp());
					}
				});
				
			}
			else
			{
			Collections.sort(dmpReportEntries, new Comparator<DmpReportEntry>() {
				@Override
				public int compare(DmpReportEntry entry1, DmpReportEntry entry2) {
					String visnId1 = String.format("%06d", Integer.valueOf(entry1.getVisn()));
					String visnId2 = String.format("%06d", Integer.valueOf(entry2.getVisn()));
					return (entry1.getDmp() + entry1.getFacility() + visnId1).compareTo(entry2.getDmp() + entry1.getFacility() + visnId2);
				}
			});
			}
			searchResult.setDmpReportEntryList(dmpReportEntries);
			searchResult.setTotalPatients(totalPatients);
			return searchResult;
		} catch (Exception e) {
			logger.error("DMPController Exception:  " + e.getMessage());
			throw handleException(e.getMessage(), e);
		}
	}
	///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	//get all reported dmps visns from DMP Report service
	public DMPReportSearchResult findDmpReportsByDmpAndVendor(DMPReportParameters dmpReportParameters) throws ControllerException {
		DMPReportSearchResult searchResult = new DMPReportSearchResult();
		List<DmpReportEntry> dmpReportEntries = new ArrayList<DmpReportEntry>();
		int totalPatients = 0;
		boolean isAllVendors=false;
		try {
			List<String> dmpNameList = dmpReportParameters.getDmpNameList();			
			if (dmpNameList != null) {
				for (int i = 0; i < dmpNameList.size(); i++) {
					String dmpName = dmpNameList.get(i);
					if (dmpName != null && dmpName.trim().length() > 0) {
						//fill in vendor
						if (dmpReportParameters.getVendorList() != null) {
							if(!dmpReportParameters.isAllVendors())
							{
							for (TermType vendor : dmpReportParameters.getVendorList()) {
								DmpReportEntry dmpReportEntry = new DmpReportEntry();
								dmpReportEntry.setDmp(dmpName);
								dmpReportEntry.setVendorNumber(vendor.getValue());
								dmpReportEntry.setVendor(vendor.getLabel());
								dmpReportEntry.setStartDate(dmpReportParameters.getSubmittedFromDate());
								dmpReportEntry.setEndDate(dmpReportParameters.getSubmittedToDate());
								dmpReportEntries.add(dmpReportEntry);
							}
							}
							else
							{
								isAllVendors=true;
								DmpReportEntry dmpReportEntry = new DmpReportEntry();
								dmpReportEntry.setDmp(dmpName);
								dmpReportEntry.setVendor("ALL");
								dmpReportEntry.setStartDate(dmpReportParameters.getSubmittedFromDate());
								dmpReportEntry.setEndDate(dmpReportParameters.getSubmittedToDate());
								dmpReportEntries.add(dmpReportEntry);
								
							}
						}
					}
				}
				dmpReportEntries=getDmpReportService().findDmpReportsByDmpAndVendor(dmpReportEntries);
				for(DmpReportEntry dmpReportEntry: dmpReportEntries)
				{
					totalPatients=totalPatients+dmpReportEntry.getNumberOfPatients();
				}

			}
			
			//sort results
			if(!isAllVendors)
			{
			Collections.sort(dmpReportEntries, new Comparator<DmpReportEntry>() {
				@Override
				public int compare(DmpReportEntry entry1, DmpReportEntry entry2) {
					return (entry1.getDmp() + entry1.getVendor()).compareTo(entry2.getDmp()+ entry2.getVendor());
				}
			});
			}
			else
			{
				Collections.sort(dmpReportEntries, new Comparator<DmpReportEntry>() {
					@Override
					public int compare(DmpReportEntry entry1, DmpReportEntry entry2) {
						return entry1.getDmp().compareTo(entry2.getDmp());
					}
				});				
			}
			searchResult.setDmpReportEntryList(dmpReportEntries);
			searchResult.setTotalPatients(totalPatients);
			return searchResult;
		} catch (Exception e) {
			logger.error("DMPController Exception:  " + e.getMessage());
			throw handleException(e.getMessage(), e);
		}
	}
	///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	//get all reported dmps visns from DMP Report service
	public DMPReportSearchResult findDmpReportsByDmpVisnAndVendor(DMPReportParameters dmpReportParameters) throws ControllerException {
		DMPReportSearchResult searchResult = new DMPReportSearchResult();
		List<DmpReportEntry> dmpReportEntries = new ArrayList<DmpReportEntry>();
		boolean isAllVisns=false;
		int totalPatients = 0;
		try {
			List<String> dmpNameList = dmpReportParameters.getDmpNameList();
			if (dmpNameList != null) {
				for (int i = 0; i < dmpNameList.size(); i++) {
					String dmpName = dmpNameList.get(i);
					if (dmpName != null && dmpName.trim().length() > 0) {
						//fill in visn
						if (dmpReportParameters.getVisnList() != null  && !dmpReportParameters.isAllVisns()) {
							for (TermType visn : dmpReportParameters.getVisnList()) {
								//fill in vendor
								if (dmpReportParameters.getVendorList() != null) {
									for (TermType vendor : dmpReportParameters.getVendorList()) {
										DmpReportEntry dmpReportEntry = new DmpReportEntry();
										dmpReportEntry.setDmp(dmpName);
										dmpReportEntry.setVisn(visn.getValue());
										dmpReportEntry.setVisnName(visn.getLabel());
										dmpReportEntry.setVendor(vendor.getLabel());
										dmpReportEntry.setVendorNumber(vendor.getValue());
										dmpReportEntry.setStartDate(dmpReportParameters.getSubmittedFromDate());
										dmpReportEntry.setEndDate(dmpReportParameters.getSubmittedToDate());
										dmpReportEntries.add(dmpReportEntry);
									}
								}
							}
						}
						else if(dmpReportParameters.getVisnList() != null && dmpReportParameters.isAllVisns())
						{
							isAllVisns=true;
							if (dmpReportParameters.getVendorList() != null) {
								for (TermType vendor : dmpReportParameters.getVendorList()) {
									DmpReportEntry dmpReportEntry = new DmpReportEntry();
									dmpReportEntry.setDmp(dmpName);
									dmpReportEntry.setVisn("National");
									dmpReportEntry.setVisnName("ALL");
									dmpReportEntry.setVendor(vendor.getLabel());
									dmpReportEntry.setVendorNumber(vendor.getValue());
									dmpReportEntry.setStartDate(dmpReportParameters.getSubmittedFromDate());
									dmpReportEntry.setEndDate(dmpReportParameters.getSubmittedToDate());
									dmpReportEntries.add(dmpReportEntry);
								}
							}
						}
					}
				}
				dmpReportEntries=getDmpReportService().findDmpReportsByDmpVisnAndVendor(dmpReportEntries);
				for(DmpReportEntry dmpReportEntry: dmpReportEntries)
				{
					totalPatients=totalPatients+dmpReportEntry.getNumberOfPatients();
				}

			}
			//sort results
			if(isAllVisns)
			{
				Collections.sort(dmpReportEntries, new Comparator<DmpReportEntry>() {
					@Override
					public int compare(DmpReportEntry entry1, DmpReportEntry entry2) {
						return entry1.getDmp().compareTo(entry2.getDmp());
					}
				});
				
			}
			else
			{
			Collections.sort(dmpReportEntries, new Comparator<DmpReportEntry>() {
				@Override
				public int compare(DmpReportEntry entry1, DmpReportEntry entry2) {
					String visnId1 = String.format("%06d", Integer.valueOf(entry1.getVisn()));
					String visnId2 = String.format("%06d", Integer.valueOf(entry2.getVisn()));
					return (entry1.getDmp() + visnId1).compareTo(entry2.getDmp() + visnId2);
				}
			});
			}
			searchResult.setDmpReportEntryList(dmpReportEntries);
			searchResult.setTotalPatients(totalPatients);
			return searchResult;
		} catch (Exception e) {
			logger.error("DMPController Exception:  " + e.getMessage());
			throw handleException(e.getMessage(), e);
		}
	}
	///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	//get all reported dmps visns from DMP Report service
	public DMPReportSearchResult findDmpReportsByDmpFacilityAndVendor(DMPReportParameters dmpReportParameters) throws ControllerException {
		DMPReportSearchResult searchResult = new DMPReportSearchResult();
		List<DmpReportEntry> dmpReportEntries = new ArrayList<DmpReportEntry>();
		int totalPatients = 0;
		try {
			List<String> dmpNameList = dmpReportParameters.getDmpNameList();
			if (dmpNameList != null) {
				for (int i = 0; i < dmpNameList.size(); i++) {
					String dmpName = dmpNameList.get(i);
					if (dmpName != null && dmpName.trim().length() > 0) {
						//fill in visn
						if (dmpReportParameters.getFacilityList() != null) {
							for (TermType facility : dmpReportParameters.getFacilityList()) {
								//fill in facility
								Visn visn = getVisn(facility.getValue());
								if (dmpReportParameters.getVendorList() != null) {
									for (TermType vendor : dmpReportParameters.getVendorList()) {
										DmpReportEntry dmpReportEntry = new DmpReportEntry();
										dmpReportEntry.setDmp(dmpName);
										Facility theFacility = (Facility)ALL_FACILITY_MAP.get(facility.getValue());
										dmpReportEntry.setFacility(theFacility.getName());
										dmpReportEntry.setFacilityStationNumber(theFacility.getStationNumber());
										dmpReportEntry.setFacilityId(new Integer(facility.getValue()));
										dmpReportEntry.setVendor(vendor.getLabel());
										dmpReportEntry.setVendorNumber(vendor.getValue());
										dmpReportEntry.setStartDate(dmpReportParameters.getSubmittedFromDate());
										dmpReportEntry.setEndDate(dmpReportParameters.getSubmittedToDate());
										if (visn != null) {
											dmpReportEntry.setVisnName(visn.getName());
											dmpReportEntry.setVisn(visn.getId().toString());
										}
										else {
											dmpReportEntry.setVisnName("");
											dmpReportEntry.setVisn("0");
										}
										dmpReportEntries.add(dmpReportEntry);
									}
								}
							}
						}
					}
				}
			}
			dmpReportEntries=getDmpReportService().findDmpReportsByDmpFacilityAndVendor(dmpReportEntries);
			for(DmpReportEntry dmpReportEntry: dmpReportEntries)
			{
				dmpReportEntry.setFacility("Facility " + dmpReportEntry.getFacilityStationNumber() + " " + dmpReportEntry.getFacility());
				totalPatients=totalPatients+dmpReportEntry.getNumberOfPatients();
			}
			//sort results
			Collections.sort(dmpReportEntries, new Comparator<DmpReportEntry>() {
				@Override
				public int compare(DmpReportEntry entry1, DmpReportEntry entry2) {
					String visnId1 = String.format("%06d", Integer.valueOf(entry1.getVisn()));
					String visnId2 = String.format("%06d", Integer.valueOf(entry2.getVisn()));
					return (entry1.getDmp() + entry1.getFacility() + visnId1 + entry1.getVendor()).compareTo(entry2.getDmp() + entry2.getFacility() + visnId2 + entry2.getVendor());
				}
			});
			searchResult.setDmpReportEntryList(dmpReportEntries);
			searchResult.setTotalPatients(totalPatients);
			return searchResult;
		} catch (Exception e) {
			logger.error("DMPController Exception:  " + e.getMessage());
			throw handleException(e.getMessage(), e);
		}
	}
	///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	
	private Visn getVisn(String FacilityID) throws ControllerException {
		Visn visn = null;
		try {
			if (ALL_FACILITIES == null) {
				ALL_FACILITIES = getInventoryService().getFacilities();
				fillFacilityMap();
			}
			for (Facility facility : ALL_FACILITIES) {
				if (facility.getId().toString().equals(FacilityID)) {
					visn = facility.getVisn();
					break;
				}
				
			}
		} catch (Exception e) {
			logger.error("DMPController Exception:  " + e.getMessage());
			throw handleException(e.getMessage(), e);
		}
		return visn;
	}
	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	private void checkAndReplace(DMPReportParameters dmpReportParameters) throws ControllerException {
		try {
			//visn
			if (dmpReportParameters.getVisnList() != null && dmpReportParameters.getVisnList().size() > 0) {
				if (itemAllInList(dmpReportParameters.getVisnList())) {
					List<TermType> visns = this.getVisns(false);
					dmpReportParameters.setVisnList(visns);
				}
			}
			//facility
			if (dmpReportParameters.getFacilityList() != null && dmpReportParameters.getFacilityList().size() > 0) {
				if (itemAllInList(dmpReportParameters.getFacilityList())) {
					List<TermType> facilities = this.getFacilitiys(false);
					dmpReportParameters.setFacilityList(facilities);
				}
			}
			//vendor
			if (dmpReportParameters.getVendorList() != null && dmpReportParameters.getVendorList().size() > 0) {
				if (itemAllInList(dmpReportParameters.getVendorList())) {
					List<TermType> vendors = this.getVendors(false);
					dmpReportParameters.setVendorList(vendors);
				}
			}
		} catch (Exception e) {
			logger.error("DMPController Exception:  " + e.getMessage());
			throw handleException(e.getMessage(), e);
		}
	}
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	private boolean itemAllInList(List<TermType> list) {
		boolean isIn = false;
		if (list != null && list.size() > 0) {
			for (TermType item : list) {
				if (item.getValue().equals(ITEM_ALL_VALUE)) {
					isIn = true;
					break;
				}
			}
		}
		return isIn;
	}
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	private void fillFacilityMap() throws ControllerException {
		try {
			if (ALL_FACILITIES == null || ALL_FACILITIES.size() == 0) return;
			for (Facility facility : ALL_FACILITIES) {
				ALL_FACILITY_MAP.put(facility.getId().toString(), facility);
			}
		} catch (Exception e) {
			logger.error("DMPController Exception:  " + e.getMessage());
			throw handleException(e.getMessage(), e);
		}
	}
}
