package gov.va.med.ccht.controller;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.Iterator;

import gov.va.med.ccht.model.User;
import gov.va.med.ccht.model.dmp.DMP;
import gov.va.med.ccht.model.dmp.DMPFindReplaceCriteria;
import gov.va.med.ccht.model.dmp.DMPQuestion;
import gov.va.med.ccht.model.dmp.DMPResponse;
import gov.va.med.ccht.model.dmp.DMPReview;
import gov.va.med.ccht.model.dmp.DMPSearch;
import gov.va.med.ccht.model.dmp.DMPSearchParameters;
import gov.va.med.ccht.model.dmp.DMPStandLibEntry;
import gov.va.med.ccht.model.dmp.DMPToolUser;
import gov.va.med.ccht.model.dmp.DMPVendorDeployment;
import gov.va.med.ccht.model.inventory.Vendor;
import gov.va.med.ccht.model.terminology.RegistrationStatus;
import gov.va.med.ccht.ui.common.AbstractController;
import gov.va.med.ccht.ui.common.ControllerException;
import gov.va.med.ccht.ui.model.DMPForm;
import gov.va.med.ccht.ui.model.DMPSearchForm;
import gov.va.med.ccht.ui.model.DmpAssignee;
import gov.va.med.ccht.ui.model.UserForm;
import gov.va.med.ccht.util.DMPExport;
import gov.va.med.fw.model.EntityKey;
import gov.va.med.fw.model.EntityKeyFactory;
import gov.va.med.fw.model.ldap.SearchCriteria;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.ui.model.TermType;

import org.apache.commons.lang.StringUtils;

public class DMPController extends AbstractController {

	private DMPConversionService dmpConversionService;
	private SecurityConversionService securityConversionService;
	private SecurityController securityController = null;
	
	private final int LOCKED_INDICATOR = 1;
	private final int UNLOCKED_INDICATOR = 0;

	public List<DMP> searchDMP(DMP dmp) throws ControllerException {
		List<DMP> results = new ArrayList<DMP>();
		try {
			results = getDmpService().searchDMP(dmp);

			return results;
		} catch (Exception e) {
			throw handleException(e.getMessage(), e);
		}
	}

	public DMPFindReplaceCriteria findTermInDMP(DMPFindReplaceCriteria criteria) throws ControllerException {
		try {

			return getDmpService().findTermInDMP(criteria);
		} catch (Exception e) {
			throw handleException(e.getMessage(), e);
		}
	}
	public DMPFindReplaceCriteria replaceTermInDMP(DMPFindReplaceCriteria criteria) throws ControllerException {
		try {

			return getDmpService().updateDMPwithReplaceTerm(criteria);
		} catch (Exception e) {
			if(e.getMessage().indexOf("Column size limit exceeded")>-1)
			{
				criteria.setException("Column size limit exceeded");
				return criteria;
			}
			throw handleException(e.getMessage(), e);
		}
	}

	public DMPForm getDMP(String id) throws ControllerException {
		EntityKey<DMP> entityKey = EntityKeyFactory.createEntityKey(
				new Long(id), DMP.class);
		try {
			DMP dmp = getDmpService().getDMP(entityKey);
			DMPForm dmpForm = new DMPForm();
			getDmpConversionService().convert(dmp, dmpForm);
			dmpForm = this.refreshDmpForm(dmpForm, dmp);
			return dmpForm;
		} catch (Exception e) {
			throw new ControllerException(e.getMessage(), e);
		}
	}

	public List<DMPForm> getAssignedDmps(String userId)
			throws ControllerException {
		List<DMPForm> results = new ArrayList<DMPForm>();
		try {
			List<DMP> rawResults = getDmpService().findAssigedDmps(userId);

			for (DMP dmp : rawResults) {
				DMPForm dmpForm = new DMPForm();
				dmpForm = this.refreshDmpForm(dmpForm, dmp);
				///////// fchen 10/14/12: HTCE_CodeCR805 /////
				dmpForm.setModifiedBy(dmp.getModifiedByName());
				dmpForm.setModifiedDate(dmp.getModifiedOn().toString());
				//////// end of HTCE_CodeCR805 /////////////
				results.add(dmpForm);
			}
			return results;
		} catch (Exception e) {
			logger.error("******:  DMPController Exception:  " + e.getMessage());
			throw handleException(e.getMessage(), e);
		}
	}

	@SuppressWarnings("unchecked")
	public List<DMPForm> getAdminViewMyDmps(String userId)
			throws ControllerException {
		List<DMPForm> results = new ArrayList<DMPForm>();
		try {
			List<DMP> rawResults = getDmpService().findAdminViewMyDmps(userId);
			Collections.sort(rawResults, new Comparator()

			{
				public int compare(Object obj1, Object obj2){
					DMP dmp1 = (DMP)obj1;
					DMP dmp2 = (DMP)obj2;
					return (dmp1.getName().compareToIgnoreCase(dmp2.getName()));
				}
			});
//			Collections.reverse(rawResults);
			for (DMP dmp : rawResults) {
				DMPForm dmpForm = new DMPForm();
				dmpForm = this.refreshDmpForm(dmpForm, dmp);
				results.add(dmpForm);
			}
			return results;
		} catch (Exception e) {
			logger.error("***VAKA***:  DMPController Exception:  "
					+ e.getMessage());
			throw handleException(e.getMessage(), e);
		}
	}

	public List<DMPForm> getAllAssignedDmps(String loginUserId)
			throws ControllerException {
		List<DMPForm> results = new ArrayList<DMPForm>();
		try {
			List<DMP> rawResults = getDmpService().findAllDmps();
			for (DMP dmp : rawResults) {
				DMPForm dmpForm = new DMPForm();
				dmpForm.setDmp(dmp);
				dmpForm.setId(String.valueOf(dmp.getId()));
				getDmpConversionService().processDMPTypes(dmpForm);
				dmpForm.setDmpToolUsers(getDmpService().findDMPToolUsers(dmp));
				ArrayList<DMPQuestion> dmpQuestions = new ArrayList<DMPQuestion>(
						dmp.getQuestions());
				for (DMPQuestion dmpQuestion : dmpQuestions) {
					getDmpConversionService().processQuestionResponse(
							dmpQuestion);
					getDmpConversionService().processQuestionContent(
							dmpQuestion);
				}
				dmpForm.setQuestions(dmpQuestions);
				try {
					if (dmp.getModifiedBy() != null) {
						if (dmp.getModifiedBy().getChangeUser() != null) {
							dmpForm.setModifiedBy(getUser(
									dmp.getModifiedBy().getChangeUser())
									.getFullName());
						}
					}
				} catch (Exception e) {
					dmpForm.setModifiedBy(dmp.getModifiedBy().getChangeUser());
					logger.error("DMP modifiedBy not available for user:  "
							+ dmp.getModifiedBy().getChangeUser());
				}
				try {

					if (dmp.getCreatedBy() != null) {
						if (dmp.getCreatedBy().getChangeUser() != null) {
							dmpForm.setCreatedBy(getUser(
									dmp.getCreatedBy().getChangeUser())
									.getFullName());
						}
					}
				} catch (Exception e) {
					dmpForm.setCreatedBy(dmp.getCreatedBy().getChangeUser());
					logger.error("DMP createdBy not available for user:  "
							+ dmp.getCreatedBy().getChangeUser());
				}
				results.add(dmpForm);
			}
			return results;
		} catch (Exception e) {
			logger.error("***VAKA***:  DMPController Exception:  "
					+ e.getMessage());
			throw handleException(e.getMessage(), e);
		}
	}


	public DMPForm assignDmp(DMPForm dmpForm) throws ControllerException {
		try {
			List<DmpAssignee> dmpAssignees = dmpForm.getAssignees();
			DMP dmp = getDmpService().findDMPById(dmpForm.getId());
			for (DmpAssignee dmpAssignee : dmpAssignees) {
				if (dmpAssignee.getIsAssigned()
						&& !dmpAssignee.isDisableNewUserAssignment()) {
					DMPToolUser newAssignee = new DMPToolUser();
					User assingedUser = getDmpService().getUserByLoginId(
							dmpAssignee.getUserForm().getUserName());
					newAssignee.setAppUser(assingedUser);
					newAssignee.setAssigned(true);
					newAssignee.setDmp(dmp);
					newAssignee.setIsAssigned(true);
					getDmpService().saveDMPToolUser(newAssignee, true);
				}
			}
			dmp = getDmpService().saveDMP(dmp);
			dmp = getDmpService().findDMPById(dmpForm.getId());
			dmpForm = this.refreshDmpForm(dmpForm, dmp);
		} catch (Exception e) {
			throw handleException(e.getMessage(), e);
		}
		return dmpForm;
	}
	public DMPExport exportDmp(DMPForm dmpForm)throws ControllerException {
		DMPExport dmpExport = new DMPExport();
		try
		{
			dmpExport=getDmpService().createDmpExportContent(dmpForm.getId());
		} catch (Exception e) {
			throw handleException(e.getMessage(), e);
		}
		return dmpExport;
	}
	public DMPForm saveResponse(DMPForm dmpForm) throws ControllerException {
		try {

			// dmpForm.getSelectedResponse();
			EntityKey<DMPQuestion> entityKey = EntityKeyFactory
					.createEntityKey(dmpForm.getSelectedQuestion().getId(),
							DMPQuestion.class);
			DMPQuestion question = getDmpService().getDMPQuestion(entityKey);
			question.setQuestionResponseTermType(dmpForm.getSelectedQuestion()
					.getQuestionResponseTermType());
			question.setQuestionContentTermType(dmpForm.getSelectedQuestion()
					.getQuestionContentTermType());
			getDmpConversionService().processQuestionResponseTermType(question);
			getDmpConversionService().processQuestionContentTermType(question);
			question = getDmpService().saveDMPQuestion(question);
			EntityKey<DMP> dmpEntityKey = EntityKeyFactory.createEntityKey(Long
					.parseLong(dmpForm.getId()), DMP.class);
			DMP dmp = getDmpService().getDMP(dmpEntityKey);
			ArrayList<DMPResponse> savedResponses = new ArrayList<DMPResponse>();
			for (DMPResponse aResponse : dmpForm.getSelectedResponses()) {
				DMPResponse response = null;
				if (aResponse.getId() != null && aResponse.getId() > 0) {
					EntityKey<DMPResponse> responseEntityKey = EntityKeyFactory
							.createEntityKey(aResponse.getId(),
									DMPResponse.class);
					response = getDmpService()
							.getDMPResponse(responseEntityKey);
				}
				if (response==null) {
					response = new DMPResponse();
				}
				response.setResponseContent(aResponse.getResponseContent());
				response.setVendorInstrContent(aResponse.getVendorInstrContent());
				response.setEduContent(aResponse.getEduContent());
				response.setQuestion(question);
				if(aResponse.getAlertLevel()!=null&& aResponse.getAlertLevel()==0)
				{
					response.setAlertLevel(null);
				}
				else
				{
					response.setAlertLevel(aResponse.getAlertLevel());
				}

				response.setBranchId(aResponse.getBranchId());
				response = getDmpService().saveDMPResponse(response);

				savedResponses.add(response);
			}
			question.setResponses(new HashSet<DMPResponse>(savedResponses));
			question = getDmpService().saveDMPQuestion(question);
			dmp = getDmpService().saveDMP(dmp);
			dmp = getDmpService().findDMPById(dmpForm.getId());
			dmpForm.setDmp(dmp);
			dmpForm.setSelectedQuestion(question);
			// dmpForm.setSelectedResponse(response);
			dmpForm.setSelectedResponses(savedResponses);
			dmpForm = refreshDmpForm(dmpForm, dmp);

		} catch (Exception e) {
			throw handleException(e.getMessage(), e);
		}
		return dmpForm;
	}

	public DMPForm saveQuestion(DMPForm dmpForm) throws ControllerException {
		try {
			DMPQuestion question = dmpForm.getSelectedQuestion();
			getDmpConversionService().processQuestionResponseTermType(question);
			getDmpConversionService().processQuestionContentTermType(question);
			DMP dmp = getDmpService().findDMPById(dmpForm.getId());
			if (question.getContentOrder() == -1) {
				question.setContentOrder(dmp.getQuestions().size()+1);
			}
			question.setDmp(dmp);
			question.setResponses(null);
			question = getDmpService().saveDMPQuestion(question);
			getDmpConversionService().processQuestionContentTermType(question);
			dmp = getDmpService().saveDMP(dmp);
			dmp = getDmpService().findDMPById(dmpForm.getId());
			dmpForm.setDmp(dmp);
			dmpForm.setSelectedQuestion(question);
			dmpForm = refreshDmpForm(dmpForm, dmp);

		} catch (Exception e) {
			throw handleException(e.getMessage(), e);
		}
		return dmpForm;
	}

	public DMPForm unAssignDmp(DMPForm dmpForm) throws ControllerException {
		try {
			DMP dmp = getDmpService().findDMPById(dmpForm.getId());
			Set<DMPToolUser> assignees = dmp.getAssignees();
			List<DmpAssignee> dmpUnassignees = dmpForm.getAssignees();
			for (DMPToolUser potentialUnassignee : assignees) {
				for (DmpAssignee dmpAssignee : dmpUnassignees) {
					if (dmpAssignee.getIsAssigned()) {
						if (Long.parseLong(dmpAssignee.getDmpToolUserId()) == potentialUnassignee
								.getId().longValue()) {
							potentialUnassignee.setIsAssigned(false);
							potentialUnassignee.setAssigned(false);
							getDmpService().saveDMPToolUser(potentialUnassignee, true);
							//getDmpService().deleteDMPToolUser(
							//		potentialUnassignee.getEntityKey());
						}
					}
				}
			}
			dmp = getDmpService().findDMPById(dmpForm.getId());
			dmp = getDmpService().saveDMP(dmp);
			dmp = getDmpService().findDMPById(dmpForm.getId());
			dmpForm = refreshDmpForm(dmpForm, dmp);
		} catch (Exception e) {
			throw handleException(e.getMessage(), e);
		}
		return dmpForm;
	}

	public List<DmpAssignee> findUsers(SearchCriteria searchCriteria,
			DMPForm dmp) throws ControllerException {
		try {
			List<DmpAssignee> dmpAssignees = new ArrayList<DmpAssignee>();
			List<DMPToolUser> dmpToolUsers = getDmpService().findDMPToolUsers(
					dmp.getId());
			List<User> cchtUsers = getSecurityService().findAppUsers(
					searchCriteria);
			if (cchtUsers != null && cchtUsers.size() > 0) {
				// found users
				for (User user : cchtUsers) {
					DmpAssignee dmpAssignee = new DmpAssignee();
					UserForm userForm = securityController.getUser(user
							.getName());
					user.setGivenName(userForm.getGivenName()
							+ " - USER ALREADY ASSIGNED");
					dmpAssignee.setUser(user);
					dmpAssignee.setUserForm(userForm);
					for (DMPToolUser dmpToolUser : dmpToolUsers) {
						if (dmpToolUser.getAppUser().getName().equalsIgnoreCase(user.getName())) {
							if (dmpToolUser.isAssigned())
							{
								dmpAssignee.setDisableNewUserAssignment(true);
								dmpAssignee.setIsAssigned(true);
							}
							else
							{
								dmpAssignee.setDisableNewUserAssignment(false);
								dmpAssignee.setIsAssigned(false);
							}
							if (dmpToolUser.isReviewer())
							{
								dmpAssignee.setDisableNewUserReviewerAssignment(true);
								dmpAssignee.setReviewer(true);
							}
							else
							{
								dmpAssignee.setDisableNewUserReviewerAssignment(false);
								dmpAssignee.setReviewer(false);
							}
							dmpAssignee.setDmpToolUserId(dmpToolUser.getId().toString());
						//	dmpAssignee.setDisableNewUserAssignment(true);
						//	dmpAssignee.setIsAssigned(true);
							if (!dmpToolUser.getAppUser().getRegistrationStatus().getCode().equals(RegistrationStatus.APPROVED)){
								dmpAssignee.setDisableNewUserAssignment(true);
								dmpAssignee.setDisableNewUserReviewerAssignment(true);
							}
							break;
						}
						
					}
					dmpAssignees.add(dmpAssignee);
				}
			}
			return dmpAssignees;
		} catch (Exception e) {
			throw handleException("DMP Search for Users Failed", e, this);
		}
	}

	public List<DmpAssignee> findDmpAssignees(DMPForm dmp)
			throws ControllerException {
		try {
			List<DmpAssignee> dmpAssignees = new ArrayList<DmpAssignee>();
			List<DMPToolUser> dmpToolUsers = getDmpService().findDMPToolUsers(
					dmp.getId());
			if (dmpToolUsers != null && dmpToolUsers.size() > 0) {
				for (DMPToolUser dmpToolUser : dmpToolUsers) {
					if (dmpToolUser.isAssigned())
					{
						UserForm userForm = securityController.getUser(dmpToolUser
								.getAppUser().getName());
						DmpAssignee dmpAssignee = new DmpAssignee();
						dmpAssignee.setDmpToolUserId(dmpToolUser.getId().toString());
						dmpAssignee.setUser(dmpToolUser.getAppUser());
						dmpAssignee.setUserForm(userForm);
						dmpAssignee.setAssigned(dmpToolUser.isAssigned());
						dmpAssignees.add(dmpAssignee);
					}
				}
			}
			//sort dmpAssignees		
			Collections.sort(dmpAssignees, new Comparator<DmpAssignee>() {
				@Override
				public int compare(DmpAssignee entry1, DmpAssignee entry2) {
					String e1 = formatName(entry1.getUserForm()).toLowerCase();
					String e2 = formatName(entry2.getUserForm()).toLowerCase();
					/*
					String e1 = formatName(entry1.getUserForm()) +
						entry1.getVisnFacility() == null ? "" : entry1.getVisnFacility() + 
						entry1.getUserForm() == null || entry1.getUserForm().getTelephoneNumber() == null ? "" : entry1.getUserForm().getTelephoneNumber() +
						entry1.getUserForm() == null || entry1.getUserForm().getEmail() == null ? "" : entry1.getUserForm().getEmail();
					String e2 = formatName(entry2.getUserForm()) +
						entry2.getVisnFacility() == null ? "" : entry2.getVisnFacility() + 
						entry2.getUserForm() == null || entry2.getUserForm().getTelephoneNumber() == null ? "" : entry2.getUserForm().getTelephoneNumber() +
						entry2.getUserForm() == null || entry2.getUserForm().getEmail() == null ? "" : entry2.getUserForm().getEmail();
					*/
					return e1.compareTo(e2);
				}
			});
			
			return dmpAssignees;
		} catch (Exception e) {
			throw handleException("DMP Search for Users Failed", e, this);
		}
	}

	public List<DMPQuestion> findDmpQuestions(DMPForm dmpForm) {
		List<DMPQuestion> questions = dmpForm.getQuestions();
		for (int i = 0; i < questions.size(); i++) {
			DMPQuestion question = questions.get(i);
			question.setQuestionOrder(i + "");
		}
		return questions;
	}

	public List<DMPQuestion> getAllStandardizedLibs(DMPForm dmpForm) {
		List<DMPQuestion> questions = dmpForm.getQuestions();
		for (int i = 0; i < questions.size(); i++) {
			DMPQuestion question = questions.get(i);
			question.setQuestionOrder(i + "");
		}
		return questions;
	}

	public DMPForm deleteQuestions(DMPForm dmpForm) throws ControllerException {
		try {
			DMP dmp = getDmpService().findDMPById(dmpForm.getId());
			List<DMPQuestion> deletedQuestions = dmpForm.getDeletedQuestions();
			for (DMPQuestion deletedQuestion : deletedQuestions) {
				if (deletedQuestion.getMarkAsDeleted()) {
					getDmpService().deleteDMPQuestion(
							deletedQuestion.getEntityKey());
				}
			}
			dmp = getDmpService().findDMPById(dmpForm.getId());
			dmp = getDmpService().saveDMP(dmp);
				logger.debug("setting new orderfor all questions");
				ArrayList<DMPQuestion> questions = new ArrayList<DMPQuestion>(dmp.getQuestions());
				int qCount=1;
				for(DMPQuestion aQuestion:questions)
				{
					aQuestion.setContentOrder(qCount);
						logger.debug("setting new order for qid: "+aQuestion.getId().toString()+", qCount:  "+qCount+", order: "+aQuestion.getContentOrder());
						aQuestion = getDmpService().saveDMPQuestion(aQuestion);
					qCount++;
				}
			dmp = getDmpService().findDMPById(dmpForm.getId());
			dmpForm = refreshDmpForm(dmpForm, dmp);
		} catch (Exception e) {
			throw handleException(e.getMessage(), e);
		}
		return dmpForm;
	}

	public DMPForm deleteQuestion(DMPForm dmpForm) throws ControllerException {
		try {
			DMP dmp = getDmpService().findDMPById(dmpForm.getId());
			DMPQuestion deletedQuestion = dmpForm.getSelectedQuestion();
			getDmpService().deleteDMPQuestion(deletedQuestion.getEntityKey());
			dmp = getDmpService().findDMPById(dmpForm.getId());
			dmp = getDmpService().saveDMP(dmp);
			logger.debug("setting new orderfor all questions");
			ArrayList<DMPQuestion> questions = new ArrayList<DMPQuestion>(dmp.getQuestions());
			int qCount=1;
			for(DMPQuestion aQuestion:questions)
			{
				aQuestion.setContentOrder(qCount);
					logger.debug("setting new order for qid: "+aQuestion.getId().toString()+", qCount:  "+qCount+", order: "+aQuestion.getContentOrder());
					aQuestion = getDmpService().saveDMPQuestion(aQuestion);
				qCount++;
			}
			dmp = getDmpService().findDMPById(dmpForm.getId());
			dmpForm = refreshDmpForm(dmpForm, dmp);
		} catch (Exception e) {
			throw handleException(e.getMessage(), e);
		}
		return dmpForm;
	}

	public DMPConversionService getDmpConversionService() {
		return dmpConversionService;
	}

	public void addDMP(DMPForm dmpForm) throws ControllerException {
		try {
			getDmpConversionService().processTermTypes(dmpForm);
			getDmpService().saveDMP(dmpForm.getDmp());
		} catch (Exception e) {
			throw handleException(e.getMessage(), e);
		}
	}
	public Boolean isDmpStandLibEntryDuplicate(DMPStandLibEntry dmpStandLibEntry) throws ControllerException
	{
		try {
			getDmpConversionService().processDmpStandLibEntryTermTypes(dmpStandLibEntry);
			return getDmpService().isDmpStandLibEntryDuplicate(dmpStandLibEntry);
		} catch (Exception e) {
			throw handleException(e.getMessage(), e);
		}
	}
	public DMPStandLibEntry addStandLibEntry(DMPStandLibEntry dmpStandLibEntry) throws ControllerException {
		try {
			getDmpConversionService().processDmpStandLibEntryTermTypes(dmpStandLibEntry);
			dmpStandLibEntry=getDmpService().saveDMPStandLibEntry(dmpStandLibEntry);
			getDmpConversionService().processDmpStandLibEntryTypes(dmpStandLibEntry);
			return dmpStandLibEntry;
		} catch (Exception e) {
			throw handleException(e.getMessage(), e);
		}
	}
	public DMPStandLibEntry updateStandLibEntry(DMPStandLibEntry dmpStandLibEntry) throws ControllerException {
		try {
			getDmpConversionService().processDmpStandLibEntryTermTypes(dmpStandLibEntry);
			DMPStandLibEntry persitedDmpStandLibEntry = getDmpService().findDMPStanLibEntryById(dmpStandLibEntry.getId());
			persitedDmpStandLibEntry.setCeOtherText(dmpStandLibEntry.getCeOtherText());
			persitedDmpStandLibEntry.setClinicalEvidenceOther(dmpStandLibEntry.getClinicalEvidenceOther());
			persitedDmpStandLibEntry.setClinicalEvidenceVA(dmpStandLibEntry.getClinicalEvidenceVA());
			persitedDmpStandLibEntry.setClinicalEvidenceAhrq(dmpStandLibEntry.getClinicalEvidenceAhrq());
			persitedDmpStandLibEntry.setModifiedBy(dmpStandLibEntry.getModifiedBy());
			persitedDmpStandLibEntry.setModifiedInTimeZone(dmpStandLibEntry.getModifiedInTimeZone());
			persitedDmpStandLibEntry.setModifiedOn(dmpStandLibEntry.getModifiedOn());
			persitedDmpStandLibEntry.setReasonModTermType(dmpStandLibEntry.getReasonModTermType());
			persitedDmpStandLibEntry.setReasonModType(dmpStandLibEntry.getReasonModType());
			persitedDmpStandLibEntry.setRsnModComment(dmpStandLibEntry.getRsnModComment());
			persitedDmpStandLibEntry.setSlCategoryTermType(dmpStandLibEntry.getSlCategoryTermType());
			persitedDmpStandLibEntry.setSlCategoryType(dmpStandLibEntry.getSlCategoryType());
			persitedDmpStandLibEntry.setSlContent(dmpStandLibEntry.getSlContent());
			persitedDmpStandLibEntry.setSlContentTermType(dmpStandLibEntry.getSlCategoryTermType());
			persitedDmpStandLibEntry.setSlContentType(dmpStandLibEntry.getSlContentType());
			dmpStandLibEntry=getDmpService().saveDMPStandLibEntry(persitedDmpStandLibEntry);
			getDmpConversionService().processDmpStandLibEntryTypes(persitedDmpStandLibEntry);
			return persitedDmpStandLibEntry;
		} catch (Exception e) {
			throw handleException(e.getMessage(), e);
		}
	}

	public List<DMPStandLibEntry> getAllStandLibEntries() throws ControllerException {
		try {
			List<DMPStandLibEntry> results=getDmpService().getAllStandardLibs();
			List<DMPStandLibEntry> processedResults=new ArrayList<DMPStandLibEntry>();
			for(DMPStandLibEntry libEntry:results)
			{
				getDmpConversionService().processDmpStandLibEntryTypes(libEntry);
				processedResults.add(libEntry);
			}
			return processedResults;
		} catch (Exception e) {
			throw handleException(e.getMessage(), e);
		}
	}

	public DMPForm updateDmp(DMPForm dmpForm) throws ControllerException {
		try {
			getDmpConversionService().processTermTypes(dmpForm);
			EntityKey<DMP> entityKey = EntityKeyFactory.createEntityKey(Long
					.parseLong(dmpForm.getId()), DMP.class);
			DMP dmp = getDmpService().getDMP(entityKey);
			dmp.setTag(dmpForm.getDmp().getTag());
			dmp.setName(dmpForm.getDmp().getName());
			dmp.setObjective(dmpForm.getDmp().getObjective());
			dmp.setDescription(dmpForm.getDmp().getDescription());
			dmp.setDmpStatusType(dmpForm.getDmp().getDmpStatusType());
			dmp.setDmpVersion(dmpForm.getDmp().getDmpVersion());
			dmp = getDmpService().saveDMP(dmp);
			dmpForm = this.refreshDmpForm(dmpForm, dmp);
		} catch (Exception e) {
			throw handleException(e.getMessage(), e);
		}
		return dmpForm;
	}
	public DMPForm updateQuestionListLocks(List<DMPQuestion> lockChangedQs,DMPForm dmpForm ) throws ControllerException {
		DMPForm updatedDMP=null;
		try {
			List<Long> lockedQuestions = new ArrayList<Long>();
			List<Long> unLockedQuestions = new ArrayList<Long>();
			
			// Separate the questions into a locked or unlocked list
			for(DMPQuestion aQuestion: lockChangedQs)
			{				
				boolean locked = aQuestion.getLocked();				
				
				if(locked)
				{
					lockedQuestions.add(aQuestion.getId());
				}
				else
				{
					unLockedQuestions.add(aQuestion.getId());
				}
				
			}
			
			
			if(lockedQuestions.size() > 0)
			{
				getDmpService().updateDmpQuestionsLockStatus(lockedQuestions, LOCKED_INDICATOR);
			}
			
			if(unLockedQuestions.size() > 0)
			{
				getDmpService().updateDmpQuestionsLockStatus(unLockedQuestions, UNLOCKED_INDICATOR);
			}
						
			DMP dmp = getDmpService().findDMPById(dmpForm.getId());
			updatedDMP = refreshDmpForm(dmpForm, dmp);
			updatedDMP.setSearchScreen(dmpForm.getSearchScreen());
		} catch (Exception e) {
			throw handleException(e.getMessage(), e);
		}
		return updatedDMP;
	}

	public DMPForm updateQuestion(DMPForm dmpForm) throws ControllerException {
		try {
			// getDmpConversionService().processTermTypes(dmpForm);
			EntityKey<DMPQuestion> entityKey = EntityKeyFactory
					.createEntityKey(dmpForm.getSelectedQuestion().getId(),
							DMPQuestion.class);
			DMPQuestion question = getDmpService().getDMPQuestion(entityKey);
			question.setQuestionContent(dmpForm.getSelectedQuestion()
					.getQuestionContent());
			question.setFrequency(dmpForm.getSelectedQuestion().getFrequency());
			int newOrder = dmpForm.getSelectedQuestion().getContentOrder();
			int oldOrder=question.getContentOrder();
			question.setContentOrder(dmpForm.getSelectedQuestion()
					.getContentOrder());
			question.setQuestionContentTermType(dmpForm.getSelectedQuestion()
					.getQuestionContentTermType());
			question.setQuestionResponseTermType(dmpForm.getSelectedQuestion()
					.getQuestionResponseTermType());
			question.setLocked(dmpForm.getSelectedQuestion().getLocked());

			if (question.getQuestionResponseTermType().getLabel()
					.equalsIgnoreCase("No response")|| !question.getQuestionContentTermType().getLabel().equalsIgnoreCase("Question/Statement")||
					!question.getDmpResponseType().getDescription().equalsIgnoreCase(dmpForm.getSelectedQuestion().getQuestionResponseTermType().getLabel())) {
				Iterator<DMPResponse> iterator = question.getResponses().iterator();
				while (iterator.hasNext())
				{
					DMPResponse deletedResponse =iterator.next();
					deletedResponse.setAlertLevel(null);
					deletedResponse.setBranchId(null);
					deletedResponse.setEduContent(null);
					deletedResponse.setResponseContent(null);
					deletedResponse.setVendorInstrContent(null);
				}
			}
			getDmpConversionService().processQuestionResponseTermType(question);
			getDmpConversionService().processQuestionContentTermType(question);
			boolean changeDmpQuestionOrder=false;
			if(newOrder!=oldOrder)
			{
				changeDmpQuestionOrder=true;
			}
			//question.setContentOrder(newOrder);
			EntityKey<DMP> dmpEntityKey = EntityKeyFactory.createEntityKey(Long
					.parseLong(dmpForm.getId()), DMP.class);
			DMP dmp = getDmpService().getDMP(dmpEntityKey);
			if(changeDmpQuestionOrder)
			{
				logger.debug("setting new orderfor all questions");
				//getDmpConversionService().processTermTypes(dmpForm);
				ArrayList<DMPQuestion> questions = new ArrayList<DMPQuestion>(dmp.getQuestions());
				questions.remove(oldOrder-1);
				questions.add(newOrder-1, question);
				int qCount=1;
				for(DMPQuestion aQuestion:questions)
				{
/*					EntityKey<DMPQuestion> dmpQentityKey = EntityKeyFactory
					.createEntityKey(aQuestion.getId(),
							DMPQuestion.class);
						DMPQuestion dmpQ = getDmpService().getDMPQuestion(dmpQentityKey);
*/
					aQuestion.setContentOrder(qCount);
						logger.debug("setting new order for qid: "+aQuestion.getId().toString()+", qCount:  "+qCount+", order: "+aQuestion.getContentOrder());
						aQuestion = getDmpService().saveDMPQuestion(aQuestion);
						//getDmpService().updateDMPQuestion(aQuestion);
					qCount++;
				}

			}
			else
			{
				question = getDmpService().saveDMPQuestion(question);
			}
			dmp = getDmpService().getDMP(dmpEntityKey);
			dmpForm = this.refreshDmpForm(dmpForm, dmp);
			//dmpForm.setQuestions(new ArrayList<DMPQuestion>(dmp.getQuestions()));
			dmpForm.setSelectedQuestion(question);
			//dmpForm.setSelectedResponses(new ArrayList<DMPResponse>(question.getResponses()));
		} catch (Exception e) {
			throw handleException(e.getMessage(), e);
		}
		return dmpForm;
	}

	public void setDmpConversionService(
			DMPConversionService dmpConversionService) {
		this.dmpConversionService = dmpConversionService;
	}

	public User getUser(String userName) throws ControllerException {
		try {
			return this.getDmpService().getUser(userName);
		} catch (ServiceException e) {
			throw handleException(e.getMessage(), e);
		}
	}

	public SecurityController getSecurityController() {
		return securityController;
	}

	public void setSecurityController(SecurityController securityController) {
		this.securityController = securityController;
	}

	public SecurityConversionService getSecurityConversionService() {
		return securityConversionService;
	}

	public void setSecurityConversionService(
			SecurityConversionService securityConversionService) {
		this.securityConversionService = securityConversionService;
	}

	private DMPForm refreshDmpForm(DMPForm dmpForm, DMP dmp) {
		dmpForm.setDmp(dmp);
		dmpForm.setId(String.valueOf(dmp.getId()));
		try {
			getDmpConversionService().processDMPTypes(dmpForm);
			dmpForm.setDmpToolUsers(new ArrayList<DMPToolUser>(dmp
					.getAssignees()));

			ArrayList<DMPQuestion> dmpQuestions = new ArrayList<DMPQuestion>(
					dmp.getQuestions());
			for (int i = 0; i < dmpQuestions.size(); i++) {
				DMPQuestion dmpQuestion = dmpQuestions.get(i);
				getDmpConversionService().processQuestionResponse(dmpQuestion);
				getDmpConversionService().processQuestionContent(dmpQuestion);
				dmpQuestion.setQuestionOrder(i + 1 + "");
			}
			dmpForm.setQuestions(dmpQuestions);

			if (dmp.getModifiedBy() != null) {
				if (dmp.getModifiedBy().getChangeUser() != null) {
					dmpForm.setModifiedBy(getUser(
							dmp.getModifiedBy().getChangeUser()).getFullName());
				}
			}
			
			//sort assignees
			ArrayList<DMPToolUser> assigneeList = new ArrayList<DMPToolUser>(dmp.getAssignees());
			Collections.sort(assigneeList, new Comparator<DMPToolUser>() {
				@Override
				public int compare(DMPToolUser dmpToolUser1, DMPToolUser dmpToolUser2) {
					return dmpToolUser1.getAppUser().getFullName().compareToIgnoreCase(dmpToolUser2.getAppUser().getFullName());
				}
			});
			
			String dmpAssignees = "";
			int count = 0;
			for (DMPToolUser dmpToolUser : assigneeList) {
				if (dmpToolUser.isAssigned())
				{
					if (count == 0) {
						dmpAssignees = dmpAssignees
								+ dmpToolUser.getAppUser().getFullName();
					} else {
						dmpAssignees = dmpAssignees + "; "
								+ dmpToolUser.getAppUser().getFullName() + " ";
					}
					if (count > 1 && count % 6 == 0) {
						dmpAssignees = dmpAssignees + "\n";
					}
					/**
					 * REMOVED FOR BACKING OUT CCR 1056 CHANGES
					dmpForm.setIsAssignee(true);
					dmpForm.setCanEdit(true);
					*/
					count++;
				}
			}
			dmpForm.setDmpAssignees(dmpAssignees);
			
		} catch (Exception e) {
			dmpForm.setModifiedBy(dmp.getModifiedBy().getChangeUser());
			logger.error("DMP modifiedBy not available for user:  "
					+ dmp.getModifiedBy().getChangeUser());
		}
		try {

			if (dmp.getCreatedBy() != null) {
				if (dmp.getCreatedBy().getChangeUser() != null) {
					dmpForm.setCreatedBy(getUser(
							dmp.getCreatedBy().getChangeUser()).getFullName());
				}
			}
		} catch (Exception e) {
			dmpForm.setCreatedBy(dmp.getCreatedBy().getChangeUser());
			logger.error("DMP createdBy not available for user:  "
					+ dmp.getCreatedBy().getChangeUser());
		}
		dmpForm.setHasComments(dmp.isHasComments());
		return dmpForm;

	}
	public List<DMPSearchForm> getAllDmps() throws ControllerException {
		List<DMPSearchForm> results = new ArrayList<DMPSearchForm>();
		try {
			List<DMPSearch> rawResults = getDmpService().getAllDmps();
			for (DMPSearch dmp : rawResults) {
				DMPSearchForm dmpForm = new DMPSearchForm();
				dmpForm = this.refreshDmpSearchForm(dmpForm, dmp);
				results.add(dmpForm);
			}
			return results;
		} catch (Exception e) {
			logger.error("DMPController Exception:  " + e.getMessage());
			throw handleException(e.getMessage(), e);
		}
	}
	private DMPSearchForm refreshDmpSearchForm(DMPSearchForm dmpForm, DMPSearch dmp) {
		dmpForm.setDmp(dmp);
		dmpForm.setId(String.valueOf(dmp.getId()));
		try {
			getDmpConversionService().processDMPTypes(dmpForm);
			String dmpAssignees = "";
			dmpForm.setDmpAssignees(dmpAssignees);
			dmpForm.setHasComments(dmp.isHasComments());
		} catch (Exception e) {
			logger.error("DMP modifiedBy not available for user:  "
					+ dmp.getModifiedBy().getChangeUser());
		}

		return dmpForm;

	}
	public List<DMPSearchForm> searchAllDmps(DMPSearchParameters dmpSearchParameters) throws ControllerException {
		List<DMPSearchForm> results = new ArrayList<DMPSearchForm>();
		try {
			List<DMPSearch> rawResults = getDmpService().searchAllDmps(dmpSearchParameters);
			for (DMPSearch dmp : rawResults) {
				DMPSearchForm dmpForm = new DMPSearchForm();
				dmpForm = this.refreshDmpSearchForm(dmpForm, dmp);
				results.add(dmpForm);
			}
			return results;
		} catch (Exception e) {
			logger.error("DMPController Exception:  " + e.getMessage());
			throw handleException(e.getMessage(), e);
		}
	}
	public DMPForm getSelectedDmp(String dmpId ) throws ControllerException {
		DMPForm dmpForm = new DMPForm();
		try {
			DMP dmp = getDmpService().findDMPById(dmpId);
			dmpForm = refreshDmpForm(dmpForm, dmp);
		} catch (Exception e) {
			logger.error("DMPController Exception:  " + e.getMessage());
			throw handleException(e.getMessage(), e);
		}
		return dmpForm;
	}
	public DMPForm getSelectedDmp(String dmpId, String userId ) throws ControllerException {
		DMPForm dmpForm = new DMPForm();
		try {
			DMP dmp = getDmpService().findDMPById(dmpId);
			ArrayList<DMPToolUser> utl = new ArrayList<DMPToolUser>(dmp.getAssignees());
			dmpForm.setDmpToolUserReviewers(this.getDMPToolUserReviewers(utl));
			dmpForm.setDmpReviewerNames(this.findDmpReviewerNames(dmpForm));
			dmpForm = refreshDmpForm(dmpForm, dmp);
			//set canEditComment property
			boolean isReviewer = this.isUserInDMPReviewers(dmpForm, userId);
			dmpForm.setIsReviewer(isReviewer);
			
			/**
			 * CCR 1056 NEXT TWO LINES ADDED BACK IN TO REVERT BACK TO PRE-1056
			 */
			boolean isAssignee = this.isUserInAssignees(utl, userId);
			dmpForm.setIsAssignee(isAssignee);
			
			if (dmp.getCreatedBy() != null && dmp.getCreatedBy().getChangeUser() != null) {
				if (dmp.getCreatedBy().getChangeUser().equalsIgnoreCase(userId)) {
					dmpForm.setIsAssignee(true);
					
					/**
					 * CCR 1056 REMOVED FOR REVERSION BACK TO PRE-1056
					dmpForm.setCanEdit(true);
					return dmpForm;
					*/
				}
			}
			
			/**
			 * CCR 1056 REMOVED FOR REVERSION BACK TO PRE-1056	
			boolean isAssignee = this.isUserInAssignees(utl, userId);
			dmpForm.setIsAssignee(isAssignee);
			dmpForm.setCanEdit(isAssignee);
			*/
		} catch (Exception e) {
			logger.error("DMPController Exception:  " + e.getMessage());
			throw handleException(e.getMessage(), e);
		}
		return dmpForm;
	}
	public DMPStandLibEntry getSelectedDmpSL(Long dmpSlId ) throws ControllerException {
		DMPStandLibEntry dmpStandLibEntry = new DMPStandLibEntry();
		try {
			dmpStandLibEntry = getDmpService().findDMPStanLibEntryById(dmpSlId);
			getDmpConversionService().processDmpStandLibEntryTypes(dmpStandLibEntry);
		} catch (Exception e) {
			logger.error("DMPController Exception:  " + e.getMessage());
			throw handleException(e.getMessage(), e);
		}
		return dmpStandLibEntry;
	}
	public List<DMPForm> getAdminViewMyDmps2(String userId)
	throws ControllerException {

		List<DMPForm> results = new ArrayList<DMPForm>();
		try {
			List<DMP> rawResults = getDmpService().findAdminViewMyDmps(userId);
			Collections.sort(rawResults, new Comparator()

			{
				public int compare(Object obj1, Object obj2){
					DMP dmp1 = (DMP)obj1;
					DMP dmp2 = (DMP)obj2;
					return (dmp1.getName().compareToIgnoreCase(dmp2.getName()));
				}
			});
		//	Collections.reverse(rawResults);
			for (DMP dmp : rawResults) {
				DMPForm dmpForm = new DMPForm();
				dmpForm = this.refreshDmpForm(dmpForm, dmp);
				dmpForm.setModifiedBy(dmp.getModifiedByName());
				dmpForm.setModifiedDate(dmp.getModifiedOn().toString());
				results.add(dmpForm);
			}
			return results;
		} catch (Exception e) {
			logger.error("DMPController Exception:  "
					+ e.getMessage());
			throw handleException(e.getMessage(), e);
		}
}
	public DMPForm saveDmpComments(DMPForm dmpForm) throws ControllerException {
		try {
			DMPReview review = dmpForm.getSelectedDMPReview();
			review = getDmpService().saveDMPReview(review);
			DMP dmp = getDmpService().findDMPById(dmpForm.getId());
			dmpForm.setDmp(dmp);
			dmpForm = refreshDmpForm(dmpForm, dmp);
			List<DMPReview> reviewList = getDmpService().getDMPQuestionComments(dmpForm.getId(), review.getDmpContentId());
			dmpForm.getSelectedQuestion().setReviewComments(new HashSet(reviewList));
			dmpForm.setSelectedDMPReview(review);
			
			return dmpForm;
			
		} catch (Exception e) {
			throw handleException(e.getMessage(), e);
		}
	}

	public DMPForm deleteDmpComments(DMPForm dmpForm) throws ControllerException {
		try {
			DMPReview review = dmpForm.getSelectedDMPReview();
			review = getDmpService().deleteDMPReview(review);
			dmpForm.setSelectedDMPReview(review);
			List<DMPReview> reviewList = getDmpService().getDMPQuestionComments(dmpForm.getId(), review.getDmpContentId());
			dmpForm.getSelectedQuestion().setReviewComments(new HashSet(reviewList));
			return dmpForm;
		} catch (Exception e) {
			throw handleException(e.getMessage(), e);
		}
	}
	
	public Boolean checkIfAlreadyDeployedToVendor(DMPForm deployedDMP, String vendor)
	{
		
		return true;
	}
	public Boolean deployToVendor(DMPForm deployedDMP, String vendorName,byte[] byteme, String name) throws ControllerException {
		try {
			EntityKey<DMP> entityKey = EntityKeyFactory.createEntityKey(
					new Long(deployedDMP.getId()), DMP.class);
				DMP dmp = getDmpService().getDMP(entityKey);
				DMPVendorDeployment dmpVendorDeployment = new DMPVendorDeployment();
				dmpVendorDeployment.setDeployedDate(new Date(System.currentTimeMillis()));
				dmpVendorDeployment.setDmp(dmp);
				Integer version = dmp.getVersion();
				if (version!=null)
				{
				  dmpVendorDeployment.setDmpVersion(dmp.getDmpVersion().toString());
				}
				dmpVendorDeployment.setFacilitator(dmp.getCreatedById());
				dmpVendorDeployment.setStatusType(dmp.getDmpStatusType());
				Vendor vendor = getDmpService().getInventoryDAO().getVendor(vendorName);
				dmpVendorDeployment.setVendor(vendor.getId());
				
			getDmpService().sendVendorMail(dmp, vendorName, byteme,name);
			getDmpService().saveDMPVendorDeployment(dmpVendorDeployment);
			return true;
		} catch (Exception e) {
			throw handleException(e.getMessage(), e);
		}
	}
	public Boolean isDmpAlreadyDeployedToVendor(DMPForm deployedDMP, String vendorName) throws ControllerException {
		try {
			EntityKey<DMP> entityKey = EntityKeyFactory.createEntityKey(
					new Long(deployedDMP.getId()), DMP.class);
				DMP dmp = getDmpService().getDMP(entityKey);
			
			return getDmpService().checkIfAlreadyDeployedToVendor(dmp, vendorName);
		} catch (Exception e) {
			throw handleException(e.getMessage(), e);
		}
	}
	public List<DMPReview> getDMPQuestionComments(String dmpId, String dmpContentId) throws ControllerException {
		try {
			List<DMPReview> dmpReview=getDmpService().getDMPQuestionComments(dmpId, dmpContentId);
			return dmpReview;
		} catch (Exception e) {
			throw handleException(e.getMessage(), e);
		}
	}
	
	public List<DmpAssignee> findDmpReviewers(DMPForm dmp) throws ControllerException {
		try {
			List<DmpAssignee> dmpReviewers = new ArrayList<DmpAssignee>();
			List<DMPToolUser> dmpToolUsers = getDmpService().findDMPToolUsers(dmp.getId());
			//List<DMPToolUser> dmpReviewerList = dmp.getDmpToolUserReviewers();
			if (dmpToolUsers != null && dmpToolUsers.size() > 0) {
				for (DMPToolUser dmpToolUser : dmpToolUsers) {
					if (dmpToolUser.isReviewer()) {
						UserForm userForm = securityController.getUser(dmpToolUser.getAppUser().getName());
						if (userForm.getTelephoneNumber() == null) userForm.setTelephoneNumber("");
						if (userForm.getEmail() == null) userForm.setEmail("");
						if (userForm.getRegistrationStatus() == null) {
							TermType tm = new TermType("","");
							userForm.setRegistrationStatus(tm);
						}
						DmpAssignee dmpReviewer = new DmpAssignee();
						dmpReviewer.setDmpToolUserId(dmpToolUser.getId().toString());
						dmpReviewer.setUser(dmpToolUser.getAppUser());
						dmpReviewer.setUserForm(userForm);
						dmpReviewer.setAssigned(dmpToolUser.isAssigned());
						dmpReviewer.setReviewer(dmpToolUser.isReviewer());
						dmpReviewers.add(dmpReviewer);
					}
				}
			}
			
			Collections.sort(dmpReviewers, new Comparator<DmpAssignee>() {
				@Override
				public int compare(DmpAssignee entry1, DmpAssignee entry2) {
					String e1 = formatName(entry1.getUserForm()).toLowerCase();
					String e2 = formatName(entry2.getUserForm()).toLowerCase();
					return e1.compareTo(e2);
				}
			});
			
			return dmpReviewers;
		} catch (Exception e) {
			throw handleException("DMP Search for Users Failed", e, this);
		}
	}
	
	private List<DMPToolUser> getDMPToolUserReviewers(List<DMPToolUser> dmpToolUsers) throws ControllerException {
		try {
			List<DMPToolUser> dmpReviewers = new ArrayList<DMPToolUser>();
			if (dmpToolUsers != null && dmpToolUsers.size() > 0) {
				for (DMPToolUser dmpToolUser : dmpToolUsers) {
					if (dmpToolUser.isReviewer()) {
						dmpReviewers.add(dmpToolUser);
					}
				}
			}
			return dmpReviewers;
		} catch (Exception e) {
			throw handleException("DMP Search for reviewers Failed", e, this);
		}
	}
	
	private String findDmpReviewerNames(DMPForm dmp) throws ControllerException {
		StringBuffer reviewersb = new StringBuffer();
		final String delimiter = "; ";
		try {
			List<DMPToolUser> dmpReviewerList = dmp.getDmpToolUserReviewers();
			//sort the dmp reviwers
			Collections.sort(dmpReviewerList, new Comparator<DMPToolUser>() {
				@Override
				public int compare(DMPToolUser dmpToolUser1, DMPToolUser dmpToolUser2) {
					return dmpToolUser1.getAppUser().getFullName().compareToIgnoreCase(dmpToolUser2.getAppUser().getFullName());
				}
			});

			int count = 0;
			if (dmpReviewerList != null && dmpReviewerList.size() > 0) {
				for (DMPToolUser reviewer : dmpReviewerList) {
					if (reviewer.isReviewer()) {
						if (count > 0) {
							reviewersb.append(delimiter);
						}
						reviewersb.append(reviewer.getAppUser().getFullName());
						if (count > 1 && count % 6 == 0) {
							reviewersb.append("\n");
						}
						count++;
					}
				}
			}
			return reviewersb.toString();
			
		} catch (Exception e) {
			throw handleException("DMP Search for Reviewer Names Failed", e, this);
		}
	}
	
	public DMPForm assignDmpReviewer(DMPForm dmpForm) throws ControllerException {
		try {
			List<DmpAssignee> dmpReviewers = dmpForm.getReviewers();
			DMP dmp = getDmpService().findDMPById(dmpForm.getId());
			for (DmpAssignee dmpReviewer : dmpReviewers) {
				if (dmpReviewer.isReviewer() && !dmpReviewer.isDisableNewUserReviewerAssignment()) {
					DMPToolUser newReviewer = new DMPToolUser();
					User assingedUser = getDmpService().getUserByLoginId(dmpReviewer.getUserForm().getUserName());
					newReviewer.setAppUser(assingedUser);
					newReviewer.setReviewer(true);
					newReviewer.setDmp(dmp);
					getDmpService().saveDMPToolUser(newReviewer, false);
				}
			}
			dmp = getDmpService().saveDMP(dmp);
			dmp = getDmpService().findDMPById(dmpForm.getId());
			ArrayList<DMPToolUser> utl = new ArrayList<DMPToolUser>(dmp.getAssignees());
			dmpForm.setDmpToolUserReviewers(this.getDMPToolUserReviewers(utl));
			dmpForm.setDmpReviewerNames(this.findDmpReviewerNames(dmpForm));
			dmpForm = this.refreshDmpForm(dmpForm, dmp);
		} catch (Exception e) {
			throw handleException(e.getMessage(), e);
		}
		return dmpForm;
	}
	
	public DMPForm unAssignDmpReviewer(DMPForm dmpForm) throws ControllerException {
		ArrayList<DMPToolUser> utl = null;
		try {
			DMP dmp = getDmpService().findDMPById(dmpForm.getId());
			utl = new ArrayList<DMPToolUser>(dmp.getAssignees());
			List<DMPToolUser> reviewers = this.getDMPToolUserReviewers(utl);
			List<DmpAssignee> dmpNonReviewers = dmpForm.getReviewers();
			for (DMPToolUser potentialNonReviewer : reviewers) {
				for (DmpAssignee dmpNonReviewer : dmpNonReviewers) {
					if (dmpNonReviewer.isReviewer()) {
						if (Long.parseLong(dmpNonReviewer.getDmpToolUserId()) == potentialNonReviewer.getId().longValue()) {
							potentialNonReviewer.setReviewer(false);
							getDmpService().saveDMPToolUser(potentialNonReviewer, false);
						}
					}
				}
			}
			dmp = getDmpService().findDMPById(dmpForm.getId());
			dmp = getDmpService().saveDMP(dmp);
			utl = new ArrayList<DMPToolUser>(dmp.getAssignees());
			dmpForm.setDmpToolUserReviewers(this.getDMPToolUserReviewers(utl));
			dmpForm.setDmpReviewerNames(this.findDmpReviewerNames(dmpForm));
			dmpForm = refreshDmpForm(dmpForm, dmp);
		} catch (Exception e) {
			throw handleException(e.getMessage(), e);
		}
		return dmpForm;
	}
	
	private boolean isUserInDMPReviewers(DMPForm dmpForm, String userId) throws ControllerException {
		boolean isIn = false;
		if (userId != null) {
			try {
				User loginUser = getDmpService().getUserByLoginId(userId);
				List<DMPToolUser> reviewers = dmpForm.getDmpToolUserReviewers();
				for (DMPToolUser reviewer : reviewers) {
					if (reviewer.getAppUser().getId().longValue() == loginUser.getId().longValue()) {
						isIn = true;
						break;
					}
				}
				
			} catch (Exception e) {
				throw handleException(e.getMessage(), e);
			}
		}
		return isIn;
	}
	
	private boolean isUserInAssignees(ArrayList<DMPToolUser> utl, String userId) throws ControllerException {
		boolean isIn = false;
		if (userId != null) {
			try {
				User loginUser = getDmpService().getUserByLoginId(userId);
				for (DMPToolUser tooluser : utl) {
					if (tooluser.getAppUser().getId().longValue() == loginUser.getId().longValue()) {
						if (tooluser.isAssigned()) {
							isIn = true;
						}
						break;
					}
				}				
			} catch (Exception e) {
				throw handleException(e.getMessage(), e);
			}
		}
		return isIn;
	}
	
	private String formatName(UserForm userForm) {
		if (userForm == null) return "";
		String firstName = userForm.getGivenName();
		String middleName = userForm.getMiddleName();
		String lastName =userForm.getFamilyName();
		String fullName = "";
		try {
			if(lastName != null && !StringUtils.isEmpty(lastName)) {
				fullName = fullName + lastName;
				if(firstName != null && !StringUtils.isEmpty(firstName)) {
					fullName = fullName + ", " + firstName;
				}
				if(middleName != null && !StringUtils.isEmpty(middleName)) {
					if(firstName == null || StringUtils.isEmpty(firstName)) {
						fullName = fullName + ", " + middleName;
					} else {
						fullName = fullName + " " + middleName;
					}
				}
			}
		}
		catch (Exception e) {
			logger.error("formatName:  " + e.getMessage());
		}
		return fullName;
	}
}
