/********************************************************************
 * Copyright � 2010 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.ccht.service.report.impl;

import java.text.ParseException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;

import gov.va.med.fw.model.lookup.AbstractLookup;
import gov.va.med.fw.model.lookup.Lookup;
import gov.va.med.fw.service.AbstractComponent;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.util.DateUtils;
import gov.va.med.fw.util.StringUtils;
import gov.va.med.fw.util.date.DateWithTimeZone;

import gov.va.med.ccht.model.Schedule;
import gov.va.med.ccht.model.inventory.DIReportParameters;
import gov.va.med.ccht.model.inventory.DeviceStatusType;
import gov.va.med.ccht.model.inventory.ReportWeek;
import gov.va.med.ccht.model.report.ReportParameters;
import gov.va.med.ccht.model.report.ReportSchedule;
import gov.va.med.ccht.model.report.ReportSetup;
import gov.va.med.ccht.model.report.ScheduledReport;
import gov.va.med.ccht.model.terminology.DayOfMonth;
import gov.va.med.ccht.model.terminology.DayOfQuarter;
import gov.va.med.ccht.model.terminology.DayOfWeek;
import gov.va.med.ccht.model.terminology.Month;
import gov.va.med.ccht.model.terminology.ReportFormat;
import gov.va.med.ccht.model.terminology.RunFrequency;
import gov.va.med.ccht.model.terminology.ScheduleType;
import gov.va.med.ccht.model.terminology.WeekOfMonth;
import gov.va.med.ccht.service.common.TerminologyException;
import gov.va.med.ccht.service.common.TerminologyService;
import gov.va.med.ccht.service.inventory.InventoryService;
import gov.va.med.ccht.service.report.ReportConstants;
import gov.va.med.ccht.service.report.ReportParameterConversionService;

/**
 * 
 * @author vhaisakatikm
 */
public class ReportParameterConversionServiceImpl extends AbstractComponent implements
		ReportParameterConversionService, ReportConstants {

	private static final long serialVersionUID = -4247795731608103914L;

	public void convertToText(ScheduledReport scheduledReport) throws ServiceException {

			if (scheduledReport.getReportSchedule() != null) {
				scheduledReport.setReportScheduleText(toText(scheduledReport.getReportSchedule()));
			} else {
				scheduledReport.setReportScheduleText(null);
			}
			if (scheduledReport.getReportParameters() != null) {
				scheduledReport.setReportParameterText(convertReportParametersToText(scheduledReport.getReportParameters()));
			}else {
				scheduledReport.setReportParameterText(null);
			}			
	}
	
	public void convertFromText(ScheduledReport scheduledReport) throws ServiceException, ParseException  {
			scheduledReport.setReportSchedule(convertTextToReportSchedule(scheduledReport.getReportScheduleText()));
			scheduledReport.setReportParameters(convertTextToReportParameters(scheduledReport.getReportParameterText()));

	}
	public void convertToText(ReportSetup reportSetup) throws ServiceException {
			if (reportSetup.getReportSchedule() != null) {
				reportSetup.setReportScheduleText(toText(reportSetup.getReportSchedule()));
			} else {
				reportSetup.setReportScheduleText(null);
			}
			if (reportSetup.getReportParameters() != null) {
				reportSetup.setReportParametersText(convertReportParametersToText(reportSetup.getReportParameters()));
			}else {
				reportSetup.setReportParametersText(null);
			}			
	}


	
	public void convertFromText(ReportSetup reportSetup) throws ServiceException, ParseException {
			reportSetup.setReportSchedule(convertTextToReportSchedule(reportSetup.getReportScheduleText()));
			reportSetup.setReportParameters(convertTextToReportParameters(reportSetup.getReportParametersText()));
	}

	public String toText(Schedule reportSchedule) throws ServiceException{
			Map<String, String> map = new HashMap<String, String>();
			map.put("runFrequency", fromLookup(reportSchedule.getRunFrequency()));
			map.put("scheduleType", fromLookup(reportSchedule.getScheduleType()));
			map.put("dayOfMonth", fromLookup(reportSchedule.getDayOfMonth()));
			map.put("weekOfMonth", fromLookup(reportSchedule.getWeekOfMonth()));
			map.put("dayOfWeek", fromLookup(reportSchedule.getDayOfWeek()));
			map.put("dayOfQuarter", fromLookup(reportSchedule.getDayOfQuarter()));
			map.put("month", fromLookup(reportSchedule.getMonth()));
			map.put("numericDayOfMonth", fromInteger(reportSchedule.getNumericDayOfMonth()));
			map.put("dateToGenerate", fromDateWithTimeZone(reportSchedule.getDateToGenerate()));
			map.put("hour", fromInteger(reportSchedule.getHour()));
			map.put("minute", fromInteger(reportSchedule.getMinute()));
			map.put("daysOfWeek", fromLookupSet(reportSchedule.getDaysOfWeek()));
			return exportAsCSV(map);

	}
	
	public Schedule toSchedule(String scheduleText) throws ServiceException, ParseException{
		Schedule schedule = new Schedule();
		convertTextToSchedule(scheduleText, schedule);
		return schedule;
	}
	private ReportSchedule convertTextToReportSchedule(String reportScheduleText) throws TerminologyException, ParseException {
		ReportSchedule reportSchedule = new ReportSchedule();
		convertTextToSchedule(reportScheduleText, reportSchedule);
		return reportSchedule;
	}	
	private void convertTextToSchedule(String reportScheduleText, Schedule reportSchedule) throws TerminologyException, ParseException{
		
		if (StringUtils.isNotEmpty(reportScheduleText)) {
			Map<String, String> map = importFromCSV(reportScheduleText);
			reportSchedule.setRunFrequency(toLookup(RunFrequency.class, map.get("runFrequency")));
			reportSchedule.setScheduleType(toLookup(ScheduleType.class, map.get("scheduleType")));
			reportSchedule.setDayOfMonth(toLookup(DayOfMonth.class, map.get("dayOfMonth")));
			reportSchedule.setWeekOfMonth(toLookup(WeekOfMonth.class, map.get("weekOfMonth")));
			reportSchedule.setDayOfWeek(toLookup(DayOfWeek.class, map.get("dayOfWeek")));
			reportSchedule.setDayOfQuarter(toLookup(DayOfQuarter.class, map.get("dayOfQuarter")));
			reportSchedule.setMonth(toLookup(Month.class, map.get("month")));
			reportSchedule.setNumericDayOfMonth(toInteger(map.get("numericDayOfMonth")));
			reportSchedule.setDateToGenerate(toDateWithTimeZone(map.get("dateToGenerate")));
			reportSchedule.setHour(toInteger(map.get("hour")));
			reportSchedule.setMinute(toInteger(map.get("minute")));
			reportSchedule.setDaysOfWeek(toLookupSet(DayOfWeek.class, map.get("daysOfWeek")));			
		}
	}
	
	private String convertReportParametersToText(ReportParameters reportParameters) {
		DIReportParameters diReportParameters = (DIReportParameters)reportParameters;
		Map<String, String> map = new HashMap<String, String>();
		map.put(REPORT_PARAM_USER_TIMEZONE, fromTimeZone(diReportParameters.getUserTimeZone()));
		map.put(REPORT_PARAM_FROM_DATE, fromDate(diReportParameters.getFromDate()));
		map.put(REPORT_PARAM_TO_DATE, fromDate(diReportParameters.getToDate()));
		map.put(REPORT_PARAM_AS_OF_DATE, fromDate(diReportParameters.getAsOfDate()));
		map.put(REPORT_PARAM_REPORT_FORMAT, fromLookup(diReportParameters.getReportFormat()));		
		map.put(REPORT_SUMMARY_TYPE, diReportParameters.getReportSummaryType());
		map.put(REPORT_PARAM_REPORT_WEEK, fromReportWeek(diReportParameters.getReportWeek()));
		map.put(REPORT_PARAM_SUBTOTAL_TYPE, diReportParameters.getSubTotalType());
		map.put(REPORT_PARAM_ALL_VISNS, fromBoolean(diReportParameters.getAllVisns()));
		map.put(REPORT_PARAM_VISN, diReportParameters.getVisn() == null ? "" : diReportParameters.getVisn().getName());
		map.put(REPORT_PARAM_ALL_FACILITIES, fromBoolean(diReportParameters.getAllFacilities()));
		map.put(REPORT_PARAM_FACILITY, diReportParameters.getFacility() == null ? "" : diReportParameters.getFacility().getStationNumber());
		map.put(REPORT_PARAM_ALL_DEVICE_STATUSES, fromBoolean(diReportParameters.getAllDeviceStatuses()));
		map.put(REPORT_PARAM_DEVICE_STATUSES, fromLookupSet(diReportParameters.getDeviceStatuses()));
		map.put(DAYS_TO_KEEP_COMPLETED_REPORTS, fromInteger(diReportParameters.getDaysToKeepCompletedReports()));
		return exportAsCSV(map);
	}
	
	private ReportParameters convertTextToReportParameters(String reportParametersText) throws ServiceException {
		DIReportParameters reportParameters = new DIReportParameters();
		if (StringUtils.isNotEmpty(reportParametersText)) {
			Map<String, String> map = importFromCSV(reportParametersText);
			reportParameters.setUserTimeZone(toTimeZone(map.get(REPORT_PARAM_USER_TIMEZONE)));
			reportParameters.setFromDate(toDate(map.get(REPORT_PARAM_FROM_DATE)));
			reportParameters.setToDate(toDate(map.get(REPORT_PARAM_TO_DATE)));
			reportParameters.setAsOfDate(toDate(map.get(REPORT_PARAM_AS_OF_DATE)));			
			reportParameters.setReportFormat(toLookup(ReportFormat.class, map.get(REPORT_PARAM_REPORT_FORMAT)));
			reportParameters.setReportSummaryType(map.get(REPORT_SUMMARY_TYPE));
			reportParameters.setReportWeek(toReportWeek(map.get(REPORT_PARAM_REPORT_WEEK)));
			reportParameters.setSubTotalType(map.get(REPORT_PARAM_SUBTOTAL_TYPE));
			reportParameters.setAllVisns(toBoolean(map.get(REPORT_PARAM_ALL_VISNS)));
			String visn = map.get(REPORT_PARAM_VISN);
			if (StringUtils.isNotEmpty(visn)) {
				reportParameters.setVisn(inventoryService.getSimpleVisn(visn));
			}
			reportParameters.setAllFacilities(toBoolean(map.get(REPORT_PARAM_ALL_FACILITIES)));
			String facility = map.get(REPORT_PARAM_FACILITY);			
			if (StringUtils.isNotEmpty(facility)) {
				reportParameters.setFacility(inventoryService.getFacility(facility));
			}
			reportParameters.setAllDeviceStatuses(toBoolean(map.get(REPORT_PARAM_ALL_DEVICE_STATUSES)));
			reportParameters.setDeviceStatuses(toLookupSet(DeviceStatusType.class,map.get(REPORT_PARAM_DEVICE_STATUSES)));
			reportParameters.setDaysToKeepCompletedReports(toInteger(map.get(DAYS_TO_KEEP_COMPLETED_REPORTS)));
		}
		return reportParameters;
	}
	
	protected Date toDate(String date) {
		return StringUtils.isEmpty(date) ? null : new Date(Long.parseLong(date));
	}
	
	protected ReportWeek toReportWeek(String reportWeek){
		ReportWeek week = null;
		if (StringUtils.isEmpty(reportWeek)) {
			return null;
		}else {
			week = new ReportWeek();
			String[] dates = reportWeek.split(":");
			week.setStartDate(toDate(dates[0]));
			week.setEndDate(toDate(dates[1]));
		}
		return week;
	}

	protected DateWithTimeZone toDateWithTimeZone(String date) throws ParseException  {
		if (StringUtils.isEmpty(date))
			return null;

		Date d = null;
		TimeZone tz = null;

		if (date.contains(";")) {
			/* new format */
			String[] pieces = date.split(";");
			if (pieces.length != 2)
				return null;

			d = new Date(Long.parseLong(pieces[0]));
			tz = TimeZone.getTimeZone(pieces[1]);
		} else {
			/* transition from old format */
			String[] pieces = date.split(" ");
			d = DateUtils.parseDate(pieces[0], DateUtils.MMDDYYYY);
			tz = TimeZone.getTimeZone(pieces[2]);
		}
		DateWithTimeZone dt = new DateWithTimeZone(d, tz);
		return dt;
	}

	protected TimeZone toTimeZone(String timeZone){
		return StringUtils.isEmpty(timeZone) ? null : TimeZone.getTimeZone(timeZone);
	}

	protected <T extends Lookup> T toLookup(Class<T> clazz, String code) throws TerminologyException {
		return StringUtils.isEmpty(code) ? null : terminologyService.getByCode(clazz, code);
	}

	protected Set<String> toSet(String codes){
		Set<String> set = new HashSet<String>();
		if (!StringUtils.isEmpty(codes))
			for (String val : codes.split(":"))
				set.add(val);
		return set;
	}

	protected <T extends Lookup> Set<T> toLookupSet(Class<T> clazz, String codes) throws TerminologyException {
		Set<T> lookups = new HashSet<T>();
		if (!StringUtils.isEmpty(codes)) {
			List<String> vals = Arrays.asList(codes.split(":"));
			lookups.addAll(terminologyService.getByCodes(clazz, vals).values());
		}

		return lookups;
	}

	protected Integer toInteger(String intValue) {
		return StringUtils.isEmpty(intValue) ? null : Integer.parseInt(intValue);
	}

	protected Boolean toBoolean(String booleanValue) {
		return Boolean.parseBoolean(booleanValue);
	}

	protected Set<Boolean> toBooleanSet(String codes){
		Set<Boolean> set = new HashSet<Boolean>();
		if (!StringUtils.isEmpty(codes)) {
			String[] vals = codes.split(":");
			for (String val : vals) {
				set.add(Boolean.parseBoolean(val));
			}
		}
		return set;
	}

	protected String fromBoolean(Boolean booleanValue) {
		return booleanValue == null ? "" : String.valueOf(booleanValue);
	}

	protected String fromBooleanSet(Set<Boolean> set) {
		if (set == null || set.size() == 0) {
			return "";
		} else {
			StringBuilder list = new StringBuilder();
			for (Iterator<Boolean> i = set.iterator(); i.hasNext();) {
				Boolean booleanValue = i.next();
				list.append(booleanValue == null ? "" : booleanValue.toString());
				if (i.hasNext())
					list.append(":");
			}
			return list.toString();
		}
	}

	protected String fromDate(Date date) {
		return date == null ? "" : String.valueOf(date.getTime());
	}
	protected String fromReportWeek(ReportWeek reportWeek) {
		return reportWeek == null ? "" : 
			(reportWeek.getStartDate() == null ? "" : String.valueOf(reportWeek.getStartDate().getTime())) + ":" + 
			(reportWeek.getEndDate() == null ? "" : String.valueOf(reportWeek.getEndDate().getTime()));
	}
	protected String fromDateWithTimeZone(DateWithTimeZone date) {
		return date == null || date.getDate() == null || date.getTimeZone() == null ? "" : date
				.getDate().getTime()
				+ ";" + date.getTimeZone().getID();
	}

	protected String fromTimeZone(TimeZone timeZone) {
		return timeZone == null ? null : timeZone.getID();
	}

	protected String fromLookup(AbstractLookup lookup) {
		return lookup == null ? "" : lookup.getCode();
	}

	protected String fromSet(Set<String> set) {
		if (set == null || set.size() == 0) {
			return "";
		} else {
			StringBuilder list = new StringBuilder();
			for (Iterator<String> i = set.iterator(); i.hasNext();) {
				String value = i.next();
				list.append(value);
				if (i.hasNext())
					list.append(":");
			}
			return list.toString();
		}
	}

	protected String fromLookupSet(Set<? extends Lookup> lookupSet) {
		if (lookupSet == null || lookupSet.size() == 0) {
			return "";
		} else {
			StringBuilder list = new StringBuilder();
			for (Iterator<? extends Lookup> i = lookupSet.iterator(); i.hasNext();) {
				Lookup lookup = i.next();
				list.append(lookup.getCode());
				if (i.hasNext())
					list.append(":");
			}
			return list.toString();
		}
	}

	protected String fromInteger(Integer intValue) {
		return intValue == null ? "" : String.valueOf(intValue);
	}

	/**
	 * Exports the map as comma-separated values, with each entry as a
	 * name=value pair. If any value is null, we simply append "name=" (with
	 * nothing after the equals sign).
	 * 
	 * @param map
	 * @return
	 */
	public final String exportAsCSV(Map<String, String> map) {
		StringBuilder sb = new StringBuilder();
		for (Map.Entry<String, String> entry : map.entrySet()) {
			String key = entry.getKey();
			String value = entry.getValue();
			if (sb.length() > 0)
				sb.append(",");
			sb.append(key).append("=").append(value != null ? value : "");
		}

		return sb.toString();
	}

	public final Map<String, String> importFromCSV(String csv)  {
		Map<String, String> data = new HashMap<String, String>();
		String[] vals = csv.split(",");
		String[] val = null;
		for (String val2 : vals) {
			val = val2.split("=");
			if (val.length == 2)
				data.put(val[0].trim(), val[1]);
		}
		return data;
	}
	

	private TerminologyService terminologyService = null;
	private InventoryService inventoryService = null;
	
	public TerminologyService getTerminologyService() {
		return terminologyService;
	}
	public void setTerminologyService(TerminologyService terminologyService) {
		this.terminologyService = terminologyService;
	}
	public InventoryService getInventoryService() {
		return inventoryService;
	}
	public void setInventoryService(InventoryService inventoryService) {
		this.inventoryService = inventoryService;
	}


}
