/********************************************************************
 * Copyright � 2010 VHA. All rights reserved
 ********************************************************************/
// Package
package gov.va.med.ccht.service.report.impl;

import org.apache.commons.lang.Validate;
import org.springframework.mail.SimpleMailMessage;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.TimeZone;

import gov.va.med.ccht.model.report.CompletedReport;
import gov.va.med.ccht.model.report.ReportParameters;
import gov.va.med.ccht.model.report.ReportSetup;
import gov.va.med.ccht.persistent.ReportDAO;
import gov.va.med.ccht.service.common.impl.AbstractBusinessService;
import gov.va.med.ccht.service.report.ReportPersistenceService;
import gov.va.med.ccht.service.report.StandardReportCriteria;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.fw.report.ReportConfiguration;
import gov.va.med.fw.report.ReportException;
import gov.va.med.fw.report.data.QueryCriteria;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.util.StringUtils;
import gov.va.med.fw.util.date.TimeZoneUtils;


/**
 * 
 * 
 * Project: Common</br> Created on: 11:10:56 AM </br>
 * 
 * @author VHAISALEV
 */
public class DatabaseReportPersister extends AbstractBusinessService implements ReportPersistenceService {

	private static final long serialVersionUID = 8051680457792536251L;

	private ReportDAO reportDAO = null;
	private SimpleMailMessage messageTemplate = null;

	public DatabaseReportPersister() {
		super();
	}

	/**
	 * @see gov.va.med.fw.service.AbstractComponent#afterPropertiesSet()
	 */
	public void afterPropertiesSet() throws Exception {
		super.afterPropertiesSet();
		Validate.notNull(reportDAO, "A report DAO must be configured");
		Validate.notNull(messageTemplate, "A message template must be configured");
	}

	/**
	 * @param reportDAO
	 *            The reportDAO to set.
	 */
	public void setReportDAO(ReportDAO reportDAO) {
		this.reportDAO = reportDAO;
	}

	/**
	 * @param reportDAO
	 *            The reportDAO to set.
	 */
	public void setMessageTemplate(SimpleMailMessage template) {
		this.messageTemplate = template;
	}

	/**
	 * @see gov.va.med.fw.report.ReportPersister#persist(gov.va.med.fw.report.ReportConfiguration,
	 *      java.io.ByteArrayOutputStream)
	 */
	public void persist(ReportConfiguration config, ByteArrayOutputStream stream)
			throws ReportException {

		// Extract data from a report configuration
		QueryCriteria criteria = config.getQueryCriteria();
		StandardReportCriteria reportCriteria = criteria instanceof StandardReportCriteria ? (StandardReportCriteria) criteria
				: null;
		ReportSetup reportSetup = reportCriteria != null ? reportCriteria.getReportSetup() : null;

		CompletedReport report = createCompletedReport(reportSetup);
		
		try {
			// Set a generated report to a CompletedReport for persistence
			stream.flush();
			report.setReportFileContent(stream.toByteArray());
			stream.close();
		} catch (IOException e) {
			logger.error("Failed to write a report to a stream ", e);
			throw new ReportException("Failed to write a report to a stream ", e);
		}

		try {
			reportDAO.saveCompletedReport(report);
		} catch (DAOException e) {
			logger.error("Failed to persist a report", e);
			throw new ReportException("Failed to persist a report", e);
		}

		try {
			sendNotificationMail(reportSetup, report.getCompletedDate());
		} catch (Exception e) {
			// Do nothing here just log exception
			if (logger.isInfoEnabled()) {
				logger.info("Failed to send notification message", e);
			}
		}
	}

	@Override
	public void saveReportCriteria(ReportSetup reportSetup)
			throws ServiceException {
		CompletedReport report = createCompletedReport(reportSetup);
		
		try {
			reportDAO.saveCompletedReport(report);
		} catch (DAOException e) {
			logger.error("Failed to persist a report", e);
			throw new ReportException("Failed to persist a report", e);
		}
		
		try {
			sendNotificationMail(reportSetup, report.getCompletedDate());
		} catch (Exception e) {
			// Do nothing here just log exception
			if (logger.isInfoEnabled()) {
				logger.info("Failed to send notification message", e);
			}
		}
	}
	
	private void sendNotificationMail(ReportSetup reportSetup, Date completedDate) throws ServiceException {
		String email = reportSetup.getEmailAddress();
		if (!StringUtils.isBlank(email)) {
			Map<String, Object> data = new HashMap<String, Object>();
	    	data.put("report", reportSetup.getStandardReport());
	    	data.put("runDate", TimeZoneUtils.convertDateToTimeZone(completedDate, TimeZone.getDefault()));
	    	data.put("webAppUrl", getWebAppUrl());
	        SimpleMailMessage template = processMailTemplate(data,messageTemplate);
	        template.setTo(StringUtils.split(email, ","));
	    	send(template);
		}
	}
	
	private CompletedReport createCompletedReport(ReportSetup reportSetup) {

		// Populate a completed report object
		CompletedReport report = new CompletedReport();
		Date createdDate = Calendar.getInstance().getTime();

		ReportParameters parameters = reportSetup.getReportParameters();
		Integer daysToKeepCompletedReports = parameters.getDaysToKeepCompletedReports();

		Calendar cal = Calendar.getInstance();
		cal.setTime(createdDate);
		if (daysToKeepCompletedReports != null) {
			cal.add(Calendar.DATE, daysToKeepCompletedReports);
		} else {
			/* For manual reports, keep default to 1 year */
			cal.add(Calendar.YEAR, 1);
		}
		Date expireDate = cal.getTime();
		report.setExpirationDate(expireDate);
		report.setCompletedDate(createdDate);
		//report.setCompletedDate(new DateWithTimeZone(createdDate, parameters.getUserTimeZone()));
		report.setUserName(reportSetup.getSetupUser());
		report.setStandardReport(reportSetup.getStandardReport());
		report.setFileType(reportSetup.getFileType());
		report.setReportParameterText(reportSetup.getReportParametersText());
		if (StringUtils.isEmpty(parameters.getReportName())) {
			report.setReportName(reportSetup.getStandardReport().getName());
		}else {
			report.setReportName(parameters.getReportName());
		}
		return report;
	}
}