/********************************************************************
 * Copyright � 2010 VHA. All rights reserved
 ********************************************************************/
// Package
package gov.va.med.ccht.service.report;

import java.io.Serializable;
import java.text.ParseException;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import gov.va.med.fw.model.EntityKey;
import gov.va.med.fw.persistent.NoRecordFoundException;
import gov.va.med.fw.report.ReportException;
import gov.va.med.fw.security.UserPrincipal;
import gov.va.med.fw.service.ServiceException;

import gov.va.med.ccht.model.report.CompletedReport;
import gov.va.med.ccht.model.report.CompletedReportFieldType;
import gov.va.med.ccht.model.report.ReportSetup;
import gov.va.med.ccht.model.report.ScheduledReport;
import gov.va.med.ccht.model.report.SimpleCompletedReport;
import gov.va.med.ccht.model.terminology.StandardReport;

/**
 * Provides methods for scheduling, generating, and viewing reports, report
 * schedules, and report setups. These methods are mainly invoked by the ESR UI
 * layer to display reports, report setups on the screen.
 * 
 * Project: Common</br> Created on: 11:25:09 AM </br>
 * 
 * @author VHAISALEV
 */
public interface StandardReportService extends Serializable {

	/**
	 * Gets a list of standard reports. This method is intended for a Scheduled
	 * Reports screen.
	 */
	@SuppressWarnings("unchecked")
	List getStandardReports() throws ServiceException;

	/**
	 * Gets a list of standard reports accessible for the user. This method is
	 * intended for a Scheduled Reports screen.
	 * 
	 * @param user
	 *            The user who setup the reports.
	 */
	List<StandardReport> getStandardReports(UserPrincipal user) throws ServiceException;

	/**
	 * 
	 * @param identifier
	 * @return ReportSetup with the specified id or null if not found
	 * @throws ServiceException
	 */
	ScheduledReport getScheduledReport(EntityKey<ScheduledReport> identifier) throws ServiceException;

	/**
	 * Add or update a CompletedReport
	 * 
	 * @param reportSetup
	 * @throws ServiceException
	 */
	ScheduledReport saveScheduledReport(ScheduledReport scheduledReport, boolean convertToText) throws ServiceException;
	void updateScheduledReport(ScheduledReport scheduledReport)
			throws ServiceException;

	/**
	 * 
	 * @param identifier
	 * @return CompletedReport with the specified id or null if not found.
	 * @throws ServiceException
	 */
	CompletedReport getCompletedReport(EntityKey<CompletedReport> identifier)
			throws ServiceException;

	void saveReportCriteria(ReportSetup reportSetup) throws ServiceException;

	/**
	 * Generates a report asynchronously with the specific report setup. This
	 * method is intended for a Report setup screen when a user presses a button
	 * to generate a report asynchronously.
	 * 
	 * @param user
	 *            A current logged-in user
	 * @param setup
	 *            A report setup to generate a report
	 */
	void generateReport(ReportSetup setup) throws ServiceException;

	/**
	 * Generates a report asynchronously with the specific report data in the
	 * Map. Useful for persistence of generateReport arguments into a database.
	 * 
	 * @param reportData
	 *            ReportData abiding by keys in StandardReportService
	 */
	void buildReport(ReportSetup setup) throws ServiceException;

	/**
	 * Generates a report asynchronously with the specific report setup. This
	 * method is intended for a Report setup screen when a user presses a button
	 * to generate a report asynchronously.
	 * 
	 * @param user
	 *            A current logged-in user
	 * @param criteria
	 *            An OPP report criteria to generate an OPP report
	 */

	/**
	 * Schedules a report generation. This method is intended for a Report
	 * schedule screen when a user presses a button to schedule a report
	 * generation.
	 * 
	 * @param user
	 *            A current logged-in user
	 * @param setup
	 *            A report schedule to schedule for a report generation
	 */
	void scheduleReport(ScheduledReport scheduledReport) throws ServiceException;

	/**
	 * Cancels a previously scheduled report. The ReportSetup will be deleted,
	 * and any scheduled triggers will be unscheduled.
	 * 
	 * @param identifier
	 * @throws ServiceException
	 */
	void cancelScheduledReport(EntityKey<ScheduledReport> identifier)
			throws NoRecordFoundException, ServiceException;

	/**
	 * Returns the SimpleCompletedReport for the given identifier.
	 * 
	 * @param identifier
	 * @return
	 * @throws ServiceException
	 */
	SimpleCompletedReport getSimpleCompletedReport(EntityKey<SimpleCompletedReport> identifier)
			throws ServiceException;

	/**
	 * Returns Completed Report parameters as ReportSetup
	 * 
	 * @param identifier
	 * @return
	 * @throws ServiceException
	 * @throws ParseException 
	 */
	ReportSetup getCompletedReportParameters(EntityKey<SimpleCompletedReport> identifier)
			throws ServiceException, ParseException;

	/**
	 * This method deleted completed report.
	 * 
	 * @param identifier
	 * @param user
	 * @throws NoRecordFoundException
	 * @throws ServiceException
	 */

	void deleteCompletedReport(EntityKey<CompletedReport> identifier, UserPrincipal user)
			throws NoRecordFoundException, ServiceException;

	/**
	 * This method inactivates a completed report.
	 * 
	 * @param identifier
	 * @param user
	 * @throws NoRecordFoundException
	 * @throws ServiceException
	 */
	void inactivateCompletedReport(EntityKey<CompletedReport> identifier, UserPrincipal user)
			throws NoRecordFoundException, ServiceException, ReportException;

	/**
	 * This method purges all those inactive reports which are older than
	 * specified system parameter. At present, only batch process uses this
	 * method.
	 * 
	 * @return
	 * @throws ServiceException
	 */
	Integer purgeCompletedReports() throws ServiceException;

	Map<String, String> getExpiringReportIdUserMap() throws ServiceException;

	/**
	 * This method returns all the information required to display filter
	 * dropdown menus on completed reports screen.
	 * 
	 * @return
	 * @throws ServiceException
	 */
	ReportFilter getCompletedReportFilter() throws ServiceException;

	void updateCompletedReport(SimpleCompletedReport simpleCompletedReport) throws ServiceException;

	/**
	 * Returns a table containing the next <numDates> occurrences of the report
	 * execution after (and not including) the specified <startingDate>. Each
	 * array in the returned list will contain three values:
	 * <ol>
	 * <li>The report execution date
	 * <li>the calculated start date of the report data on that date
	 * <li>the calculated end date of the report data on that date
	 * </ol>
	 * The returned List will always be non-null, although it may have zero
	 * number of rows.
	 * 
	 * @param user
	 *            The user requesting the preview
	 * @param reportSetup
	 *            The ReportSetup whose schedule and parameters we consider when
	 *            calculating the preview text
	 * @param numDates
	 *            The number of dates to calculate
	 * @param startingDate
	 *            The starting date after which the results will be populated
	 * @return See description
	 * @throws ServiceException
	 *             If there were any Quartz scheduling errors or other issues
	 */
	List<Date[]> getSchedulePreview(UserPrincipal user, ReportSetup reportSetup, int numDates,
			Date startingDate) throws ServiceException;
	List<ScheduledReport> searchScheduledReports(ReportFilterSearchQueryInfo searchQueryInfo) throws ServiceException;
	List<SimpleCompletedReport> searchCompletedReports(CompletedReportsSearchQueryInfo searchQueryInfo) throws ServiceException;
	LinkedHashMap<String, String> getUniqueFilterValues(CompletedReportFieldType fieldType);
}