/**
 * 
 */
package gov.va.med.ccht.service.qir.impl;

import gov.va.med.ccht.model.CCHTRoles;
import gov.va.med.ccht.model.User;
import gov.va.med.ccht.model.inventory.SimpleFacility;
import gov.va.med.ccht.model.inventory.SimpleVisn;
import gov.va.med.ccht.model.inventory.Vendor;
import gov.va.med.ccht.model.qir.QIR;
import gov.va.med.ccht.model.qir.QIRAttachment;
import gov.va.med.ccht.model.qir.QIRSearchParameters;
import gov.va.med.ccht.model.qir.QIRSimpleAttachment;
import gov.va.med.ccht.model.qir.QIRStatusType;
import gov.va.med.ccht.model.terminology.FederalHoliday;
import gov.va.med.ccht.persistent.QIRDAO;
import gov.va.med.ccht.service.common.SecurityService;
import gov.va.med.ccht.service.common.impl.AbstractBusinessService;
import gov.va.med.ccht.service.qir.QIRService;
import gov.va.med.fw.model.EntityKey;
import gov.va.med.fw.model.EntityKeyFactory;
import gov.va.med.fw.model.VersionedEntityKey;
import gov.va.med.fw.model.ldap.SearchCriteria;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.fw.security.UserPrincipal;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.service.ServiceOptimisticLockException;
import gov.va.med.fw.service.config.RecursivePropertyPlaceholderConfigurer;
import gov.va.med.fw.service.jms.JMSPayload;
import gov.va.med.fw.service.jms.MessageProducerService;
import gov.va.med.fw.util.DateUtils;
import gov.va.med.fw.util.StringUtils;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.springframework.mail.SimpleMailMessage;

/**
 * @author vhaisakatikm
 *
 */
public class QIRServiceImpl extends AbstractBusinessService implements QIRService {

	public QIR getQIR(EntityKey<QIR> entityKey) throws ServiceException {
		try {			
			QIR qir = qirDAO.getByKey(entityKey);
			if (VersionedEntityKey.class.isAssignableFrom(entityKey.getClass())) {
				VersionedEntityKey<?> vKey = (VersionedEntityKey<?>) entityKey;
				if (qir != null && vKey.getVersion() != null
						&& !vKey.getVersion().equals(qir.getVersion())) {
					ServiceOptimisticLockException optEx = new ServiceOptimisticLockException(
							"Version number specified in QIR EntityKey does not match on file QIR");
					optEx.setEntity(qir);
					optEx.setErrorType("QIR Data changed");
					throw optEx;
				}
			}
			return qir;
		}catch (DAOException e){
			throw new ServiceException(e.getMessage(), e);	
		}
	}
	
	public Date getDueDateFromHistory(String qirId) throws ServiceException
	{		
		try
		{
			Date vendorResponseDueDate = null;
			
			if(qirId != null)
			{
				vendorResponseDueDate = qirDAO.getVendorResponseDueDate(qirId);				
			}
			return vendorResponseDueDate;
		}
		catch (DAOException e)
		{
			throw new ServiceException(e.getMessage(), e);	
		}
	}

	public QIRAttachment getQIRAttachment(EntityKey<QIRAttachment> entityKey)
			throws ServiceException {
		try {
			return qirDAO.getByKey(entityKey);
		}catch (DAOException e){
			throw new ServiceException(e.getMessage(), e);	
		}
	}

	public List<Object[]> searchQIR(QIRSearchParameters qirSearchParameters)
			throws ServiceException {
		try {
			return qirDAO.searchQIR(qirSearchParameters);
		}catch (DAOException e){
			throw new ServiceException(e.getMessage(), e);
		}
	}

	public void updateQIR(QIR qir) throws ServiceException {
		updateQIR(qir,false);
	}
	
	public void updateQIR(QIR qir, boolean updatedDueDate ) throws ServiceException {
		try {
			if( updatedDueDate ) {
				/**
				 * Starts a vendor response due date clock on the next day
				 * email remind sent date to null
				 */
				Date start = DateUtils.getWeekDate();
				qir.setVendorResponseDueDate( DateUtils.forwardBusinessDays(start, 10) );
				qir.setEmailReminderSentDate(null);
				Date end = qir.getVendorResponseDueDate();
				
				/**
				 * If a federal holiday falls between a date when a vendor response due date is set and a day when a vendor response is due, forward a vendor response due date by one day.
				 * If a vendor response due date falls on either Saturday or Sunday, forward a vendor response due date by one day.
				 */
				Calendar cld = Calendar.getInstance();
				List<FederalHoliday> holidays = this.getSecurityService().getHolidays(cld.get( Calendar.YEAR ));
				
				for( FederalHoliday holiday : holidays ) {
					Date fdate = DateUtils.truncate( holiday.getDate(), Calendar.DAY_OF_MONTH );
										
					if( DateUtils.isDateBetweenIgnoreTime(fdate, start, end) ) {
						cld.setTime( qir.getVendorResponseDueDate() );
						cld.roll(Calendar.DATE, true);
						if( cld.get( Calendar.DAY_OF_WEEK ) == Calendar.SATURDAY ) {
							cld.roll(Calendar.DATE, 2);
						}
						if( cld.get( Calendar.DAY_OF_WEEK ) == Calendar.SUNDAY ) {
							cld.roll(Calendar.DATE, true);
						}
						qir.setVendorResponseDueDate(cld.getTime());
					}
				}
			}
			if (qir.getEntityKey() != null) {
				QIR onFile = (QIR) qirDAO.getByKey(qir.getEntityKey());
				getMergeService().mergeQIR(qir, onFile);
				QIR persisted = this.saveQIR(onFile);
				if( persisted.getId() != null && onFile.getId() == null ) {
					onFile.setId( persisted.getId() );
				}
				sendQIRNotificationMail(onFile);
			}else {
				QIR persisted = this.saveQIR(qir);
				if( persisted.getId() != null && qir.getId() == null ) {
					qir.setId( persisted.getId() );
				}
				sendQIRNotificationMail(qir);
			}
		}catch (DAOException e){
			throw new ServiceException(e.getMessage(), e);	
		}		
	}
	
	public QIR saveQIR(QIR qir) throws ServiceException {
		try {
			return qirDAO.saveQIR(qir);
		}
		catch( DAOException e ) {
			throw new ServiceException(e.getMessage(), e);	
		}
	}
	
	public void updateQIRAttachment(QIRAttachment qirAttachment)
			throws ServiceException {
		try {
			qirDAO.update(qirAttachment);		
		}catch (DAOException e){
			throw new ServiceException(e.getMessage(), e);	
		}
	}

	public void deleteQIRAttachment(QIRSimpleAttachment qirAttachment) throws ServiceException {
		try {
			EntityKey<QIRAttachment> entityKey = 
				EntityKeyFactory.createEntityKey(new Long(qirAttachment.getId()), qirAttachment.getVersion(), QIRAttachment.class);
			qirDAO.removeObject( entityKey );
		}
		catch (DAOException e){
			throw new ServiceException(e.getMessage(), e);	
		}
	}
	
	public Integer generateVendorResponseDueNotifications() throws ServiceException {
		try {
			List<String> qirIds = qirDAO.getVendorResponseDueQIRIds();	
			for (String id:qirIds) {
				//send jms message
				Map<String, Object> map = new HashMap<String, Object>();
				map.put("QIR_ID", id);
				JMSPayload payload = (JMSPayload) getApplicationContext().getBean(
						getQirNotificationJMSPayLoadName());
				payload.setPayload((Serializable) map);
				messageProducerService.send(payload);
			}
			return qirIds.size();
		}catch (Exception e){
			throw new ServiceException(e.getMessage(), e);	
		}
	}

	public void sendVendorResponseDueEMail(Map<String,?> qirData)
			throws ServiceException {
		try {
			String qirId = (String) qirData.get("QIR_ID");
			EntityKey<QIR> entityKey = EntityKeyFactory.createEntityKey(new Long(qirId), QIR.class);
			QIR qir = qirDAO.getByKey(entityKey);
			sendVendorResponseDueNotification(qir);
			qir.setEmailReminderSentDate(new Date());
			qirDAO.update(qir);
			//update sent date
		}catch (DAOException e){
			throw new ServiceException(e.getMessage(), e);	
		}
	}

	private void sendQIRNotificationMail(QIR qir) {
		try {
			List<User> userList = new ArrayList<User>();
			String qirStatusCode = qir.getQirStatusType().getCode();
			
			List<User> nationalAdmins = getNationalAdmins();
			List<User> visnAdmins = getVisnAdmins(qir.getVisn());
			List<User> facilityAdmins = getFacilityAdmins(qir.getFacility());
			List<User> submitters = getSubmitter(qir.getSubmittedBy());
						
			if (nationalAdmins != null) userList.addAll(nationalAdmins);
			if (visnAdmins != null) userList.addAll(visnAdmins);
			if (facilityAdmins != null) userList.addAll(facilityAdmins);			
			if (submitters != null) userList.addAll(submitters);
						
			if (QIRStatusType.NEW.equals(qirStatusCode) || QIRStatusType.WITHDRAWN.equals(qirStatusCode)){
				//default list				
			}
			else {
				List<User> vendors = getVendors(qir.getVendor());			
				List<User> nac = getNAC();		
				if (vendors != null) userList.addAll(vendors);
				if (nac != null) userList.addAll(nac);
			}
							
			//Only active users will receive e-mails
			List<String> emailList = getEmailList(userList);
			if (emailList.size() > 0) {			   	
		    	Map<String, Object> data = new HashMap<String, Object>();
		    	data.put("qir", qir);
		    	data.put("webAppUrl", getWebAppUrl());
		    	if (submitters != null && submitters.size() > 0) {
			    	User submitter = submitters.get(0);
			    	qir.setSubmittedByName(submitter.getFullName());
			    	qir.setEmailSubmittedByName(submitter.getGivenName()+" "+submitter.getFamilyName());
		    	}
		    	UserPrincipal currentUser = getCurrentUser();
		    	if (currentUser != null) {
		    		qir.setModifiedByName(getCurrentUser().getFullName());
		    	}
		    	SimpleMailMessage template = processMailTemplate(data,qirNotificationMessage);
		    	String[] toAddress = new String[emailList.size()];
		        for (int i=0; i<emailList.size(); i++) {
		        	toAddress[i] = emailList.get(i);
		        }
		    	template.setTo(toAddress);		    	
		    	send(template);
			}			
		}catch (ServiceException e){
			logger.error("QIR Service : sendQIRNotificationMail Failed", e);
		}
	}
	
	private void sendVendorResponseDueNotification(QIR qir) {
		try {
			List<User> userList = new ArrayList<User>();
			List<User> nationalAdmins = getNationalAdmins();
			List<User> vendors = getVendors(qir.getVendor());			
			List<User> nac = getNAC();	
			List<User> visnAdmins = getVisnAdmins(qir.getVisn());
			List<User> facilityAdmins = this.getFacilityAdmins(qir.getFacility());
			List<User> submitter = this.getSubmitter(qir.getSubmittedBy());
			
			if (nationalAdmins != null) userList.addAll(nationalAdmins);
			if (vendors != null) userList.addAll(vendors);
			if (nac != null) userList.addAll(nac);
			if (visnAdmins != null) userList.addAll(visnAdmins);
			if (facilityAdmins != null) userList.addAll(facilityAdmins);
			if (submitter != null) userList.addAll(submitter);
										
			//Only active users will receive e-mails
			List<String> emailList = getEmailList(userList);
			
			if( logger.isDebugEnabled() ) {
				logger.debug("Sending e-mail to the following: " + emailList + " for QIR " + qir.getId() );
			}
			if (emailList.size() > 0) {			   	
		    	Map<String, Object> data = new HashMap<String, Object>();
		    	data.put("qir", qir);
		    	data.put("webAppUrl", getWebAppUrl());
		    	SimpleMailMessage template = processMailTemplate(data,qirVendorResponseDueMailMessage);
		    	String[] toAddress = new String[emailList.size()];
		        for (int i=0; i<emailList.size(); i++) {
		        	toAddress[i] = emailList.get(i);
		        }
		    	template.setTo(toAddress);
		    	
		    	// Send a message with no attachment
		    	send(template,null,Priority.HIGH);
				if( logger.isDebugEnabled() ) {
					logger.debug("E-mail was sent successfully to the following: " + emailList + " for QIR " + qir.getId() );
				}
			}			
		}catch (ServiceException e){
			logger.error("QIR Service : sendVendorResponseDueNotification Failed", e);
		}
	}
	
	private List<User> getNAC() throws ServiceException {
		SearchCriteria searchCriteria = new SearchCriteria();
		List<String> roles = new ArrayList<String> ();
		roles.add(CCHTRoles.NAC);
		searchCriteria.setRoles(roles);
		return securityService.findAppUsers(searchCriteria);
    }
	
	private List<User> getNationalAdmins() throws ServiceException {
		SearchCriteria searchCriteria = new SearchCriteria();
		List<String> roles = new ArrayList<String> ();
		roles.add(CCHTRoles.NATIONAL_ADMIN);
		searchCriteria.setRoles(roles);
		return securityService.findAppUsers(searchCriteria);
    }
	
	private List<User> getVisnAdmins(SimpleVisn visn) throws ServiceException{
		if (visn != null) {
			SearchCriteria searchCriteria = new SearchCriteria();
			List<String> roles = new ArrayList<String> ();
			roles.add(CCHTRoles.VISN_ADMIN);
			searchCriteria.setRoles(roles);
			List<String> visns = new ArrayList<String>();
			visns.add(visn.getCode());
			searchCriteria.setVisns(visns);
			List<User> admins = securityService.findAppUsers(searchCriteria);		
			return admins;
		}else {
			return new ArrayList<User>();
		}
    }

    private List<User> getFacilityAdmins(SimpleFacility facility) throws ServiceException{
    	if (facility != null) {
			SearchCriteria searchCriteria = new SearchCriteria();
			List<String> roles = new ArrayList<String> ();
			roles.add(CCHTRoles.FACILITY_ADMIN);
			searchCriteria.setRoles(roles);
			List<String> facilities = new ArrayList<String>();
			facilities.add(facility.getCode());
			searchCriteria.setStations(facilities);
			List<User> admins = securityService.findAppUsers(searchCriteria);		
			return admins;
    	}else {
    		return new ArrayList<User>();
    	}
    }
	
	private List<User> getVendors(Vendor vendor) throws ServiceException {
		if (vendor != null) {
			SearchCriteria searchCriteria = new SearchCriteria();
			List<String> roles = new ArrayList<String> ();
			roles.add(CCHTRoles.VENDOR);
			searchCriteria.setRoles(roles);
			List<String> vendors = new ArrayList<String>();
			vendors.add(vendor.getName());
			searchCriteria.setVendors(vendors);
			List<User> vendorpocs = securityService.findAppUsers(searchCriteria);
			return vendorpocs;
		}else {
			return new ArrayList<User>();
		}
    }
	
	private List<User> getSubmitter(String username) throws ServiceException {
		if (StringUtils.isNotEmpty(username)) {
			SearchCriteria searchCriteria = new SearchCriteria();
			searchCriteria.setSAMAccountName(username);
			return securityService.findAppUsers(searchCriteria);    
		}else {
			return new ArrayList<User>();
		}
    }	
	private MessageProducerService messageProducerService;
	private String qirNotificationJMSPayLoadName;
	private SimpleMailMessage qirNotificationMessage;
	private SimpleMailMessage qirVendorResponseDueMailMessage;
	private SecurityService securityService;
	private QIRDAO qirDAO;
	
	public MessageProducerService getMessageProducerService() {
		return messageProducerService;
	}

	public void setMessageProducerService(
			MessageProducerService messageProducerService) {
		this.messageProducerService = messageProducerService;
	}

	public String getQirNotificationJMSPayLoadName() {
		return qirNotificationJMSPayLoadName;
	}

	public void setQirNotificationJMSPayLoadName(
			String qirNotificationJMSPayLoadName) {
		this.qirNotificationJMSPayLoadName = qirNotificationJMSPayLoadName;
	}

	public SimpleMailMessage getQirNotificationMessage() {
		return qirNotificationMessage;
	}

	public void setQirNotificationMessage(SimpleMailMessage qirNotificationMessage) {
		this.qirNotificationMessage = qirNotificationMessage;
	}

	public SimpleMailMessage getQirVendorResponseDueMailMessage() {
		return qirVendorResponseDueMailMessage;
	}

	public void setQirVendorResponseDueMailMessage(
			SimpleMailMessage qirVendorResponseDueMailMessage) {
		this.qirVendorResponseDueMailMessage = qirVendorResponseDueMailMessage;
	}

	public SecurityService getSecurityService() {
		return securityService;
	}

	public void setSecurityService(SecurityService securityService) {
		this.securityService = securityService;
	}	
	public QIRDAO getQirDAO() {
		return qirDAO;
	}

	public void setQirDAO(QIRDAO qirDAO) {
		this.qirDAO = qirDAO;
	}
}
