package gov.va.med.ccht.service.inventory.impl;

import gov.va.med.ccht.model.inventory.Device;
import gov.va.med.ccht.model.inventory.DeviceDetail;
import gov.va.med.ccht.model.inventory.DeviceRequirement;
import gov.va.med.ccht.model.inventory.DeviceSearchParameters;
import gov.va.med.ccht.model.inventory.DeviceTrackerLog;
import gov.va.med.ccht.model.inventory.DeviceType;
import gov.va.med.ccht.model.inventory.Facility;
import gov.va.med.ccht.model.inventory.ReportWeek;
import gov.va.med.ccht.model.inventory.SimpleFacility;
import gov.va.med.ccht.model.inventory.SimpleVisn;
import gov.va.med.ccht.model.inventory.Vendor;
import gov.va.med.ccht.model.inventory.Visn;
import gov.va.med.ccht.model.report.ReportSetup;
import gov.va.med.ccht.persistent.InventoryDAO;
import gov.va.med.ccht.service.common.impl.AbstractBusinessService;
import gov.va.med.ccht.service.inventory.InventoryService;
import gov.va.med.fw.model.EntityKey;
import gov.va.med.fw.model.VersionedEntityKey;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.fw.persistent.DAOOperations;
import gov.va.med.fw.scheduling.SchedulingService;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.service.ServiceOptimisticLockException;
import gov.va.med.fw.ui.model.TermType;
import gov.va.med.fw.util.DateUtils;

import java.awt.Color;
import java.awt.GradientPaint;
import java.awt.image.BufferedImage;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jfree.chart.ChartFactory;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.NumberAxis;
import org.jfree.chart.plot.CategoryPlot;
import org.jfree.chart.plot.PlotOrientation;
import org.jfree.chart.renderer.category.BarRenderer;
import org.jfree.data.category.DefaultCategoryDataset;
import org.quartz.CronTrigger;
import org.quartz.Trigger;
import org.springframework.mail.SimpleMailMessage;

public class InventoryServiceImpl extends AbstractBusinessService implements
		InventoryService {

	private InventoryDAO inventoryDAO;
	private Date reportingStartDate;
	private long numberWeeksToDisplay = 0;
	private SimpleMailMessage weeklyVCReportMessage;
	private NumberFormat percentFormat = new DecimalFormat("#00.00");
	private SchedulingService schedulingService;
	private DAOOperations genericDao;

	public InventoryDAO getInventoryDAO() {
		return inventoryDAO;
	}

	public void setInventoryDAO(InventoryDAO inventoryDAO) {
		this.inventoryDAO = inventoryDAO;
	}

	public Date getReportingStartDate() {
		if (reportingStartDate == null) {
			Calendar startDate = Calendar.getInstance();
			startDate.set(2006, 8, 3, 0, 0, 0);
			startDate.set(Calendar.MILLISECOND, 0);
			reportingStartDate = startDate.getTime();
		}
		return reportingStartDate;
	}

	public void setReportingStartDate(Date reportingStartDate) {
		this.reportingStartDate = reportingStartDate;
	}

	public long getNumberWeeksToDisplay() {
		if (numberWeeksToDisplay > 0) {
			return numberWeeksToDisplay;
		} else {
			return (new Date().getTime() - getReportingStartDate().getTime())
					/ (24 * 60 * 60 * 1000) / 7;
		}
	}

	public void setNumberWeeksToDisplay(int numberWeeksToDisplay) {
		this.numberWeeksToDisplay = numberWeeksToDisplay;
	}

	public SimpleMailMessage getWeeklyVCReportMessage() {
		return weeklyVCReportMessage;
	}

	public void setWeeklyVCReportMessage(SimpleMailMessage weeklyVCReportMessage) {
		this.weeklyVCReportMessage = weeklyVCReportMessage;
	}

	public List<Device> getDevices() throws ServiceException {
		try {
			return inventoryDAO.getDevices();
		} catch (Exception e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}

	public List<Facility> getFacilities() throws ServiceException {
		try {
			return inventoryDAO.getFacilities();
		} catch (Exception e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}

	public List<Facility> getFacilities(String visnName)
			throws ServiceException {
		try {
			return inventoryDAO.getFacilities(visnName);
		} catch (Exception e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}

	public List<Vendor> getVendors() throws ServiceException {
		try {
			return inventoryDAO.getVendors();
		} catch (Exception e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}

	public List<Visn> getVisns() throws ServiceException {
		try {
			return inventoryDAO.getVisns();
		} catch (Exception e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}

	public Device getDevice(String name) throws ServiceException {
		try {
			return inventoryDAO.getDevice(name);
		} catch (Exception e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}

	public DeviceType getDeviceType(String code) throws ServiceException {
		try {
			return inventoryDAO.getDeviceType(code);
		} catch (Exception e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}

	public Facility getFacility(String stationNumber) throws ServiceException {
		try {
			return inventoryDAO.getFacility(stationNumber);
		} catch (Exception e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}

	public Vendor getVendor(String name) throws ServiceException {
		try {
			return inventoryDAO.getVendor(name);
		} catch (Exception e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}

	public Visn getVisn(String name) throws ServiceException {
		try {
			return inventoryDAO.getVisn(name);
		} catch (Exception e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}

	public SimpleVisn getSimpleVisn(String name) throws ServiceException {
		try {
			return inventoryDAO.getSimpleVisn(name);
		} catch (Exception e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}

	public List<SimpleVisn> getSimpleVisns() throws ServiceException {
		try {
			List<SimpleVisn> list = inventoryDAO.getSimpleVisns();
			return list;
		} catch (Exception e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}

	public SimpleFacility getSimpleFacility(String stationNumber)
			throws ServiceException {
		try {
			return inventoryDAO.getSimpleFacility(stationNumber);
		} catch (Exception e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}

	public List<SimpleFacility> getSimpleFacilities() throws ServiceException {
		try {
			List<SimpleFacility> list = inventoryDAO.getSimpleFacilities();
			return list;
		} catch (Exception e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}

	public List<SimpleFacility> getSimpleFacilities(SimpleVisn visn)
			throws ServiceException {
		try {
			List<SimpleFacility> list = inventoryDAO.getSimpleFacilities(visn);
			return list;
		} catch (Exception e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}

	public List<ReportWeek> getReportWeeks() throws ServiceException {

		long weeks = getNumberWeeksToDisplay();

		// last weekend (Recent Saturday)
		Calendar weekend = getLastWeekend();

		// Week start
		Calendar weekstart = Calendar.getInstance();
		weekstart.setTime(weekend.getTime());
		weekstart.add(Calendar.DAY_OF_MONTH, -6);

		List<ReportWeek> reportWeeks = new ArrayList<ReportWeek>();

		for (int i = 0; i < weeks; i++) {
			ReportWeek reportWeek = new ReportWeek(weekstart.getTime(), weekend
					.getTime());
			reportWeeks.add(reportWeek);
			weekstart.add(Calendar.DAY_OF_MONTH, -7);
			weekend.add(Calendar.DAY_OF_MONTH, -7);
		}

		return reportWeeks;
	}

	public List<Object[]> generateInventoryReports(ReportSetup reportSetup)
			throws ServiceException {
		try {
			return inventoryDAO.generateInventoryReports(reportSetup);
		} catch (DAOException e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}

	public List<Object[]> searchDevice(
			DeviceSearchParameters deviceSearchParameters)
			throws ServiceException {
		try {
			return inventoryDAO.searchDevice(deviceSearchParameters);
		} catch (DAOException e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}

	public List<Object[]> generateVCCurrentReport(
			DeviceSearchParameters deviceSearchParameters)
			throws ServiceException {
		try {
			return inventoryDAO.generateVCCurrentReport(deviceSearchParameters);
		} catch (DAOException e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}

	public List<Object[]> generateVCHistoryReport(
			DeviceSearchParameters deviceSearchParameters)
			throws ServiceException {
		try {
			return inventoryDAO.generateVCHistoryReport(deviceSearchParameters);
		} catch (DAOException e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}

	public List<Object[]> generateVCReportCharts(
			DeviceSearchParameters deviceSearchParameters)
			throws ServiceException {
		try {
			return inventoryDAO.generateVCReportCharts(deviceSearchParameters);
		} catch (DAOException e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}

	public DeviceDetail getDeviceDetail(EntityKey<DeviceDetail> entityKey)
			throws ServiceException {
		try {
			DeviceDetail dd = inventoryDAO.getByKey(entityKey);
			if (VersionedEntityKey.class.isAssignableFrom(entityKey.getClass())) {
				VersionedEntityKey<?> vKey = (VersionedEntityKey<?>) entityKey;
				if (dd != null && vKey.getVersion() != null
						&& !vKey.getVersion().equals(dd.getVersion())) {
					ServiceOptimisticLockException optEx = new ServiceOptimisticLockException(
							"Version number specified in Device EntityKey does not match on file Device");
					optEx.setEntity(dd);
					optEx.setErrorType("Device Data changed");
					throw optEx;
				}
			}
			return dd;
		} catch (DAOException e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}

	public DeviceDetail getDeviceDetail(
			DeviceSearchParameters deviceSearchParameters)
			throws ServiceException {
		try {
			return inventoryDAO.getDeviceDetail(deviceSearchParameters);
		} catch (DAOException e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}

	public void updateDeviceDetail(DeviceDetail deviceDetail)
			throws ServiceException {
		try {
			if (deviceDetail.getEntityKey() != null) {
				DeviceDetail onFile = (DeviceDetail) inventoryDAO
						.getByKey(deviceDetail.getEntityKey());
				getMergeService().mergeDeviceDetail(deviceDetail, onFile);
				inventoryDAO.update(onFile);
			} else {
				inventoryDAO.update(deviceDetail);
			}
		} catch (DAOException e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}

	public DeviceRequirement getDeviceRequirement(
			EntityKey<DeviceRequirement> entityKey) throws ServiceException {
		try {
			DeviceRequirement dr = inventoryDAO.getByKey(entityKey);
			if (VersionedEntityKey.class.isAssignableFrom(entityKey.getClass())) {
				VersionedEntityKey<?> vKey = (VersionedEntityKey<?>) entityKey;
				if (dr != null && vKey.getVersion() != null
						&& !vKey.getVersion().equals(dr.getVersion())) {
					ServiceOptimisticLockException optEx = new ServiceOptimisticLockException(
							"Version number specified in Device Requirement EntityKey does not match on file Device Requirement");
					optEx.setEntity(dr);
					optEx.setErrorType("Device Data changed");
					throw optEx;
				}
			}
			return dr;

		} catch (DAOException e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}

	public List<DeviceRequirement> getDeviceRequirements(
			SimpleFacility simpleFacility) throws ServiceException {
		try {
			return inventoryDAO.getDeviceRequirements(simpleFacility);
		} catch (Exception e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}

	public List<DeviceRequirement> updateDeviceRequirements(
			List<DeviceRequirement> deviceRequirements) throws ServiceException {
		try {
			for (DeviceRequirement deviceRequirement : deviceRequirements) {
				inventoryDAO.update(deviceRequirement);
			}
		} catch (DAOException e) {
			throw new ServiceException(e.getMessage(), e);
		}
		return null;
	}

	public Integer generateWeeklyVendorComplianceReport()
			throws ServiceException {
		logger.debug("generateWeeklyVendorComplianceReport");
		List<DeviceTrackerLog> deviceTrackerLogs = this
				.processDeviceTrackerLogs();
		if (deviceTrackerLogs.size() > 0) {
			logger
					.debug("generateWeeklyVendorComplianceReport, deviceTrackerLogs.size = "
							+ deviceTrackerLogs.size());
			Calendar weekend = getLastWeekend();
			Calendar weekstart = Calendar.getInstance();
			weekstart.setTime(weekend.getTime());
			// 52 weeks
			weekstart.add(Calendar.DAY_OF_MONTH, -(51 * 7 + 6));
			ReportWeek reportWeek = new ReportWeek(weekstart.getTime(), weekend
					.getTime());
			DeviceSearchParameters deviceSearchParameters = new DeviceSearchParameters();
			deviceSearchParameters.setReportWeek(reportWeek);
			StringBuilder htmlReport = new StringBuilder();
			BufferedImage chartImage = null;
			try {
				List<Object[]> data = inventoryDAO
						.generateVCHistoryReport(deviceSearchParameters);
				// create chart from data
				chartImage = createChart(data);
				String headerrow = "<tr><th>Week Ending</th><th>Vendor</th><th>% with Device Model</th><th>% with Serial Number</th></tr>";
				htmlReport.append(headerrow);

				// generate html table
				for (Object[] objs : data) {
					// Date reportEndDate = (Date)objs[0];
					// String vendorName = (String)objs[1];
					Integer deviceCount = (Integer) objs[2];
					// Integer modelNameCount = (Integer)objs[3];
					// Integer serialNumberCount = (Integer)objs[4];
					String row = "<tr>"
							+ "<td>"
							+ DateUtils.format((Date) objs[0], null)
							+ "</td>"
							+ "<td>"
							+ (String) objs[1]
							+ "</td>"
							+ "<td align=\"right\">"
							+ percentFormat.format(100.0 * (Integer) objs[3]
									/ deviceCount)
							+ "</td>"
							+ "<td align=\"right\">"
							+ percentFormat.format(100.0 * (Integer) objs[4]
									/ deviceCount) + "</td>" + "</tr>";
					htmlReport.append(row);
				}
			} catch (DAOException e) {
				logger
						.error("generateWeeklyVendorComplianceReport had a dao exception:"
								+ e.getMessage());
				throw new ServiceException(e.getMessage(), e);
			}

			SimpleMailMessage template = new SimpleMailMessage(
					weeklyVCReportMessage);
			Map<String, Object> data = new HashMap<String, Object>();
			data
					.put("reportEndDate", DateUtils.format(weekend.getTime(),
							null));
			data.put("reportStartDate", DateUtils.format(weekstart.getTime(),
					null));
			data.put("webAppUrl", getWebAppUrl());
			data.put("dataTable", htmlReport);
			template.setText(merge(weeklyVCReportMessage.getText(), data));
			template
					.setSubject(merge(weeklyVCReportMessage.getSubject(), data));
			Map<String, Object> attachments = new HashMap<String, Object>();
			attachments.put("vcchart_"
					+ DateUtils.format(weekend.getTime(), "MMddyyyy") + ".png",
					chartImage);
			send(template, attachments);
			updateVendorComplianceReportCompletion(deviceTrackerLogs);
			updateWeeklyVendorComplianceJob(true);
			logger
					.debug("generateWeeklyVendorComplianceReport updateWeeklyVendorComplianceJob");
			return new Integer(1);
		} else {
			logger.debug("deviceTrackerLogs <1, updating cron for weekly vendor compliance");
			updateWeeklyVendorComplianceJob(false);
			return new Integer(0);
		}
	}

	public List<TermType> getModalityType(int type) throws ServiceException
	{
		try
		{
			return inventoryDAO.getModalityType(type);			
		}
		catch(DAOException e)
		{
			throw new ServiceException(e.getMessage(), e);
		}
	}
	private BufferedImage createChart(List<Object[]> data) {
		// create chart for the lat week only
		String MODEL_NAME = "Device Model";
		String SERIAL_NUMBER = "Serial Number";

		DefaultCategoryDataset dataset = new DefaultCategoryDataset();
		Date reportEndDate = (Date) data.get(0)[0];
		for (Object[] objs : data) {
			if (reportEndDate.equals((Date) objs[0])) {
				String vendorName = (String) objs[1];
				Integer deviceCount = (Integer) objs[2];
				Integer modelNameCount = (Integer) objs[3];
				Integer serialNumberCount = (Integer) objs[4];
				dataset.addValue(100.0 * modelNameCount / deviceCount,
						MODEL_NAME, vendorName);
				dataset.addValue(100.0 * serialNumberCount / deviceCount,
						SERIAL_NUMBER, vendorName);
			} else {
				// we have collected last weeks data
				break;
			}
		}

		JFreeChart chart = ChartFactory.createBarChart(
				"Device Model and Serial Number Vendor Compliance for the week ending "
						+ DateUtils.format(reportEndDate, null), // chart title
				"Vendor", // domain axis label
				"Percent of Devices Reporting Serial Number/Device Model",
				dataset, // data
				PlotOrientation.VERTICAL, // orientation
				true, // include legend
				true, // tooltips?
				false // URLs?
				);

		// set the background color for the chart...
		chart.setBackgroundPaint(Color.white);

		// get a reference to the plot for further customisation...
		CategoryPlot plot = (CategoryPlot) chart.getPlot();

		// set the range axis to display integers only...
		NumberAxis rangeAxis = (NumberAxis) plot.getRangeAxis();
		rangeAxis.setStandardTickUnits(NumberAxis.createIntegerTickUnits());

		// disable bar outlines...
		BarRenderer renderer = (BarRenderer) plot.getRenderer();
		renderer.setDrawBarOutline(false);

		// set up gradient paints for series...
		GradientPaint gp0 = new GradientPaint(0.0f, 0.0f, Color.blue, 0.0f,
				0.0f, new Color(0, 0, 64));
		GradientPaint gp1 = new GradientPaint(0.0f, 0.0f, Color.green, 0.0f,
				0.0f, new Color(0, 64, 0));
		renderer.setSeriesPaint(0, gp0);
		renderer.setSeriesPaint(1, gp1);

		// CategoryAxis domainAxis = plot.getDomainAxis();
		// /domainAxis.setCategoryLabelPositions(
		// /CategoryLabelPositions.createUpRotationLabelPositions(
		// Math.PI / 6.0));
		return chart.createBufferedImage(700, 700);
	}

	private Calendar getLastWeekend() {
		Calendar weekend = Calendar.getInstance();
		weekend.set(Calendar.MILLISECOND, 0);
		weekend.set(Calendar.SECOND, 0);
		weekend.set(Calendar.MINUTE, 0);
		weekend.set(Calendar.HOUR_OF_DAY, 0);
		weekend.add(Calendar.DATE, -weekend.get(Calendar.DAY_OF_WEEK));
		return weekend;
	}

	private List<DeviceTrackerLog> processDeviceTrackerLogs()
			throws ServiceException {
		List<DeviceTrackerLog> results = new ArrayList<DeviceTrackerLog>();
		logger.debug("processDeviceTrackerLogs list size:  "+results.size());
		try {
			List<Vendor> activeVendors = inventoryDAO
					.getActiveVendorsForComplianceReport();
			List<DeviceTrackerLog> deviceTrackerLogs = inventoryDAO
					.getUnprocessedDeviceTrackerLogs();
			if (deviceTrackerLogs.size() >= activeVendors.size()) {
				results = deviceTrackerLogs;
			} else {
				this.logger
						.error("Active vendors have not reported in device tracker log table");
			}

		} catch (Exception e) {
			throw new ServiceException(e.getMessage(), e);
		}
		return results;
	}

	private void updateWeeklyVendorComplianceJob(boolean reset)
			throws ServiceException {
		try {
			String defaultExp = "0 0 7 ? * MON *";
			Trigger trigger = schedulingService.getTrigger(
					"weeklyVendorComplianceReportCronTrigger",
					"common.static.triggers");
			String currentExp = ((CronTrigger) trigger).getCronExpression();
			if (reset && !currentExp.equalsIgnoreCase(defaultExp)) {
				logger.debug("resetting weeekly vendor compliance cron exp to:  "+defaultExp);
				((CronTrigger) trigger).setCronExpression(defaultExp);
				schedulingService.reschedule(
						"weeklyVendorComplianceReportCronTrigger",
						"common.static.triggers", (CronTrigger) trigger);
			} else if (!reset) {
				String nextCronExp = this.rescheduleCron(currentExp);
				logger.debug("changing weeekly vendor compliance cron exp to:  "+nextCronExp);
				((CronTrigger) trigger).setCronExpression(nextCronExp);
				schedulingService.reschedule(
						"weeklyVendorComplianceReportCronTrigger",
						"common.static.triggers", (CronTrigger) trigger);
			}
		} catch (Exception e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}

	private void updateVendorComplianceReportCompletion(
			List<DeviceTrackerLog> results) throws ServiceException {
		try {
			inventoryDAO.setVendorComplianceReportCompletion(results);
		} catch (Exception e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}

	public SchedulingService getSchedulingService() {
		return schedulingService;
	}

	public void setSchedulingService(SchedulingService schedulingService) {
		this.schedulingService = schedulingService;
	}

	private String rescheduleCron(String cron) {
		String[] expArray = cron.split(" ");
		int hour = Integer.parseInt(expArray[2]);
		String day= expArray[5];
		if(hour<23)
			{
			  hour = hour+1;
			}
		else{
			hour=0;
			day=getNextDay(day);
		}
		expArray[2]=hour+"";
		expArray[5]=day;
		String newCron = java.util.Arrays.toString(expArray);
		newCron=newCron.replaceAll(",", "");
		newCron=newCron.replaceAll("\\[", "");
		newCron=newCron.replaceAll("\\]", "");
		return newCron;
	}

	private String getNextDay(String day)
	{
		if(day.equalsIgnoreCase("MON"))
		{
			day="TUE";
			return day;
		}
		else if(day.equalsIgnoreCase("TUE"))
		{
			day="WED";
			return day;
		}
		else if(day.equalsIgnoreCase("WED"))
		{
			day="THU";
			return day;
		}
		else if(day.equalsIgnoreCase("THU"))
		{
			day="FRI";
			return day;
		}
		else if(day.equalsIgnoreCase("FRI"))
		{
			day="SAT";
			return day;
		}
		else if(day.equalsIgnoreCase("SAT"))
		{
			day="SUN";
			return day;
		}
		else if(day.equalsIgnoreCase("SUN"))
		{
			day="MON";
			return day;
		}
		else
		{
			day="MON";
		}
		return day;
	}
	public DAOOperations getGenericDao() {
		return genericDao;
	}

	public void setGenericDao(DAOOperations genericDao) {
		this.genericDao = genericDao;
	}
}