/********************************************************************
 * Copyright � 2010 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.ccht.service.common.impl;

import gov.va.med.ccht.model.UserRole;
import gov.va.med.ccht.model.SimpleUser;
import gov.va.med.ccht.model.User;
import gov.va.med.ccht.model.terminology.FederalHoliday;
import gov.va.med.ccht.persistent.SecurityDAO;
import gov.va.med.ccht.service.common.MergeService;
import gov.va.med.ccht.service.common.SecurityService;
import gov.va.med.fw.model.EntityKey;
import gov.va.med.fw.model.VersionedEntityKey;
import gov.va.med.fw.model.ldap.LdapPerson;
import gov.va.med.fw.model.ldap.SearchCriteria;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.fw.security.Application;
import gov.va.med.fw.security.Permission;
import gov.va.med.fw.security.Role;
import gov.va.med.fw.security.SimpleRole;
import gov.va.med.fw.security.UserPrincipal;
import gov.va.med.fw.service.AbstractComponent;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.service.ServiceOptimisticLockException;
import gov.va.med.fw.service.ldap.LdapPersonService;
import gov.va.med.fw.service.transaction.TransactionTimestampManager;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.lang.Validate;
import org.springframework.ldap.CommunicationException;

/**
 * Implements the security services to manage system users and user profiles
 * 
 * @author vhaisakatikm
 */
public class SecurityServiceImpl extends AbstractComponent implements SecurityService {

	private LdapPersonService ldapPersonService = null;
	private SecurityDAO securityDAO = null;
	private TransactionTimestampManager timestampManager;
	private MergeService mergeService = null;

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.ccht.service.security.SecurityService#getRole(gov.va.med.fw
	 * .model.EntityKey)
	 */
	public Role getRole(EntityKey<?> entityKey) throws ServiceException {
		Validate.notNull(entityKey);
		try {
			Role role = (Role) securityDAO.getByKey(entityKey);
			if (VersionedEntityKey.class.isAssignableFrom(entityKey.getClass())) {
				VersionedEntityKey<?> vKey = (VersionedEntityKey<?>) entityKey;
				if (role != null && vKey.getVersion() != null
						&& !vKey.getVersion().equals(role.getVersion())) {
					ServiceOptimisticLockException optEx = new ServiceOptimisticLockException(
							"Version number specified in Role EntityKey does not match on file Role");
					optEx.setEntity(role);
					optEx.setErrorType(ServiceOptimisticLockException.ROLE_DATA_CHANGED_ERROR);
					throw optEx;
				}
			}
			return role;
		} catch (DAOException e) {
			throw new ServiceException("SecurityServiceImpl.getRole: Error while retrieving Role",
					e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.ccht.service.security.SecurityService#getRole(java.lang.String
	 * )
	 */
	public Role getRole(String roleName) throws ServiceException {
		Validate.notNull(roleName);
		try {
			return (Role) securityDAO.getRoleByName(roleName);
		} catch (DAOException e) {
			throw new ServiceException("SecurityServiceImpl.getRole: Error while retrieving Role",
					e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.ccht.service.security.SecurityService#getUser(gov.va.med.fw
	 * .model.EntityKey)
	 */
	public User getUser(EntityKey entityKey) throws ServiceException {
		Validate.notNull(entityKey);
		try {
			// Get the user from ccht database
			User user = securityDAO.getUserByKey(entityKey);
			if (VersionedEntityKey.class.isAssignableFrom(entityKey.getClass())) {
				VersionedEntityKey<?> vKey = (VersionedEntityKey<?>) entityKey;
				if (user != null && vKey.getVersion() != null
						&& !vKey.getVersion().equals(user.getVersion())) {
					ServiceOptimisticLockException optEx = new ServiceOptimisticLockException(
							"Version number specified in User EntityKey does not match on file User");
					optEx.setEntity(user);
					optEx.setErrorType(ServiceOptimisticLockException.USER_DATA_CHANGED_ERROR);
					throw optEx;
				}
			}
			if (user != null) {
				try {
					// Get the user from LDAP (roles)
					LdapPerson ldapPerson = ldapPersonService.findBySamAccountName(user.getName());
					if (ldapPerson != null) {
						//onFile.setDisabled(ldapPerson.isDisabled()); 
						user.setTelephoneNumber(ldapPerson.getTelephoneNumber());
						user.setEmail(ldapPerson.getEmail());
						user.setLdapPerson(ldapPerson);
					}
				} catch(CommunicationException e) {
					logger.error("LDAP is down, unable to retrieve full user info from LDAP for: " + user.getName());
					// since this is not authenticating at this point, still allow if LDAP is down
					user.setUserDn(user.getName());
				}
			}
			return user;
		} catch (DAOException e) {
			throw new ServiceException(
					"SecurityServiceImpl.getUser: Error while retrieving User", e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.ccht.service.security.SecurityService#getUser(java.lang.String
	 * )
	 */
	public User getUser(String userName) throws ServiceException {
		Validate.notNull(userName);
		try {
			// Get the user from ccht database and ldap
			User user = securityDAO.getUserByName(userName);
			if (user != null) /* check for user group to be internal  */ {
				try {
					LdapPerson ldapPerson = ldapPersonService.findBySamAccountName(userName);
					user.setLdapPerson(ldapPerson);
				} catch(CommunicationException e) {
					logger.error("LDAP is down, unable to retrieve full user info from LDAP for: " + userName);
					// since this is not authenticating at this point, still allow if LDAP is down
					user.setUserDn(userName);
				}
			}
			return user;
		} catch (DAOException e) {
			throw new ServiceException(
					"SecurityServiceImpl.getUser: Error while retrieving User", e);
		}
	}

	public LdapPerson getLdapUser(String samAcountName) throws ServiceException {
		return ldapPersonService.findBySamAccountName(samAcountName);
	}

	public Boolean authenticateLdapUser(String userDn, String password)
			throws ServiceException {
		try {
			if (ldapPersonService.authenticate(userDn, password)) {
				return true;
			}else {
				return false;
			}
		}catch (CommunicationException e) {
			throw new ServiceException("Authentication Failed",e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.ccht.service.security.SecurityService#createUser(gov.va.med
	 * .ccht.service.security.User)
	 */
	public void createUser(User user) throws ServiceException {
		// save user in the ccht databae
		try {
			securityDAO.createUser(user);
		} catch (DAOException e) {
			throw new ServiceException(
					"SecurityServiceImpl.createUser: Error while creating the User", e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.ccht.service.security.SecurityService#updateUser(gov.va.med
	 * .ccht.service.security.User)
	 */
	public void updateUser(User user) throws ServiceException {
		try {
			// Get the user as it is currently persisted in the CCHT database
			User onFile = getUser(user.getEntityKey());
			
			for( UserRole role : user.getRoles() ) {
				logger.debug("These are the userids from the incoming roles: " + role.getUser().getId() );
				role.setUser( user );
			}
			
			for( UserRole role : onFile.getRoles() ) {
				logger.debug("These are the userids from the onfile roles " +  role.getUser().getId());
				role.setUser( onFile );
				
			}

			// Iterate through the roles currently assigned to the user
			for (UserRole role: onFile.getRoles())
			{
				// If current role from the iteration isn't in the Set of new roles assigned
				// delete the role. This needs to be done as a work around of a hibernate bug
				// that has issues with cascading deletes.
				if(!user.getRoles().contains(role))
				{
					securityDAO.deleteRole(role.getEntityKey());
				}
			}
			
			
			mergeService.mergeUser(user, onFile);			
			for( UserRole role : onFile.getRoles() ) {
				logger.debug("These are the new userids from the onfile roles " +  role.getUser().getId());
				role.setUser( onFile );
			}
						

			// update user in ccht database
			securityDAO.updateUser(onFile);
		} catch (DAOException e) {
			logger.error("Error while saving User " + user.getUsername() + " ID: " + user.getId(),e);
			throw new ServiceException("Error while saving User " + user.getUsername() + " ID: " + user.getId(), e);
		}
	}	

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.ccht.service.security.SecurityService#updateRole(gov.va.med
	 * .ccht.service.security.cchtRole)
	 */
	public void updateRole(Role role) throws ServiceException {
		try {
			if (role.getEntityKey() == null) {
				securityDAO.updateRole(role);
			}else {
				Role onFile = getRole(role.getEntityKey());
				mergeService.mergeRole(role, onFile);
				securityDAO.updateRole(onFile);
			}
		} catch (DAOException e) {
			throw new ServiceException("SecurityServiceImpl.updateRole: failed", e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.ccht.service.security.SecurityService#deleteRole(gov.va.med
	 * .ccht.model.security.cchtRole)
	 */
	public void deleteRole(Role role) throws ServiceException {
		// TODO
		try {
			securityDAO.remove(role);
		} catch (Exception e) {
			throw new ServiceException("SecurityServiceImpl.deletRole failed", e);
		}
	}

	public List<Role> getAllRoles(Application application) throws ServiceException {
		try {
			return securityDAO.findAllRoles(application);

		} catch (DAOException e) {
			throw new ServiceException("getAvailableRoleNames failed", e);
		}
	}

	public Set<String> getAvailableRoleNames(Application application) throws ServiceException {
		try {
			List<String> roles = securityDAO.getAvailableRoleNames(application);
			Set<String> roleNames = new HashSet<String>();
			roleNames.addAll(roles);
			return roleNames;

		} catch (DAOException e) {
			throw new ServiceException("getAvailableRoleNames failed", e);
		}
	}
	
	public List<FederalHoliday> getHolidays(int year) throws ServiceException {
		try {
			return securityDAO.findHolidays(year);
		} catch (DAOException e) {
			throw new ServiceException("getHolidays failed", e);
		}
	}

	public List<SimpleRole> getAllSimpleRoles() throws ServiceException {
		try {
			return securityDAO.findAllSimpleRoles();
		} catch (DAOException e) {
			throw new ServiceException("getAvailableRoles failed", e);
		}
	}

	public List<SimpleRole> getAllDMPRoles() throws ServiceException {
		try {
			return securityDAO.findAllDMPRoles();
		} catch (DAOException e) {
			throw new ServiceException("getAvailableRoles failed", e);
		}
	}

	/**
	 * Get Permission names associated with the roles
	 */
	public Set<String> getRolePermissionNames(Set<Role> roles) throws ServiceException {
		Set<String> permissionNames = new HashSet<String>();
		if (roles == null)
			return permissionNames;

		try {
			for (Role role : roles) {
				List<String> permissions = securityDAO.getRolePermissionNames(role.getName());
				permissionNames.addAll(permissions);
			}
			return permissionNames;
		} catch (DAOException e) {
			throw new ServiceException("getAvailableRoleNames failed", e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.ccht.service.security.SecurityService#getPermissions(gov.va
	 * .med.fw.security.Application)
	 */
	public List<Permission> getAllPermissions(Application application) throws ServiceException {
		try {
			return securityDAO.findAllPermissions(application);
		} catch (DAOException e) {
			throw new ServiceException("Get All Permissions Failed", e);
		}
	}

	public Application getApplication(String applicationName) throws ServiceException {
		Validate.notNull(applicationName);
		try {
			return (Application) securityDAO.getApplicationByName(applicationName);
		} catch (DAOException e) {
			throw new ServiceException(
					"SecurityServiceImpl.getapplicationName: Error while retrieving Application", e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.ccht.service.security.SecurityService#updateFailedLogin(java
	 * .lang.String)
	 */
	public boolean updateFailedLogin(String userName) throws ServiceException {
		try {
			// get the user and update
			User user = securityDAO.getUserByName(userName);
			if (user != null && user.getId() != null) {
				// set failed login date
				//TimeZone currentTimeZone = TimeZoneUtils.getThreadCurrentTimezone();
				//DateWithTimeZone dt = new DateWithTimeZone(getTransactinTime(), currentTimeZone);
				user.setLoginFailedDate(new Date());
				// increment failed login count
				int failedCount = user.getLoginFailedCount() == null ? 1 : user
						.getLoginFailedCount().intValue() + 1;
				user.setLoginFailedCount(failedCount);

				// lock the account if the failed count >=3
				if (user.getLoginFailedCount().intValue() >= 3) {
					if (!user.isAccountLocked()) {
						user.setAccountLockDate(new Date());
						//user.setAccountLocked(true);
					}
				}
				securityDAO.updateUser(user);
				return user.isAccountLocked();
			}
			return false;

		} catch (DAOException e) {
			throw new ServiceException("Get All Permissions Failed", e);
		}
	}

	public void updateSuccessFulLogin(String userName) throws ServiceException {
		updateSuccessFulLogin(userName, null);
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.ccht.service.security.SecurityService#updateSuccessFulLogin
	 * (java.lang.String)
	 */
	public void updateSuccessFulLogin(String userName, String credential) throws ServiceException {
		try {
			// get the user and update
			User user = securityDAO.getUserByName(userName);
			if(user == null) {
				user = new User(userName);
			}
			if(credential != null) {
				user.setPassword(credential);
				user.getUserCredentials().hashUserCredentials();
			}
			updateLoginDate(user);

			securityDAO.updateUser(user);
		} catch (Exception e) {
			throw new ServiceException("updateSuccessFulLogin Failed", e);
		}
	}
		
		
	private void updateLoginDate(User user) {
		// set success login date
		user.setLastLoginDate(new Date());
		// set failed login count to 0
		user.setLoginFailedCount(0);		
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.fw.security.SecurityService#updateSuccessFulLogin(gov.va.med
	 * .fw.security.UserPrincipal)
	 */
	public void updateSuccessFulLogin(UserPrincipal userPrincipal, boolean updateRoles) throws ServiceException {
		Validate.notNull(userPrincipal);
		try {
			// get the user and update
			User user = securityDAO.getUserByName(userPrincipal.getName());
			if(user == null && userPrincipal instanceof User) {
				user = (User) userPrincipal;
			} else {
				user.getUserCredentials().setPassword(userPrincipal.getPassword());
				user.setGivenName(userPrincipal.getGivenName());
				user.setMiddleName(userPrincipal.getMiddleName());
				user.setFamilyName(userPrincipal.getFamilyName());
			}
			
			user.getUserCredentials().hashUserCredentials();
			updateLoginDate(user);

			securityDAO.updateUser(user);
		} catch (Exception e) {
			throw new ServiceException("updateSuccessFulLogin Failed", e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.ccht.service.security.SecurityService#getAllPermissions(java
	 * .lang.String)
	 */
	public List<Permission> getAllPermissions(String applicationName) throws ServiceException {
		return getAllPermissions(getApplication(applicationName));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.ccht.service.security.SecurityService#getAllRoles(java.lang
	 * .String)
	 */
	public List<Role> getAllRoles(String applicationName) throws ServiceException {
		return getAllRoles(getApplication(applicationName));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.ccht.service.security.SecurityService#getAvailableRoleNames
	 * (java.lang.String)
	 */
	public Set<String> getAvailableRoleNames(String applicationName) throws ServiceException {
		return getAvailableRoleNames(getApplication(applicationName));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.ccht.service.security.SecurityService#getAllPermissions()
	 */
	public List<Permission> getAllPermissions() throws ServiceException {
		return getAllPermissions((Application) null);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.ccht.service.security.SecurityService#getAllRoles()
	 */
	public List<Role> getAllRoles() throws ServiceException {
		return getAllRoles((Application) null);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.ccht.service.security.SecurityService#getAvailableRoleNames()
	 */
	public Set<String> getAvailableRoleNames() throws ServiceException {
		return getAvailableRoleNames((Application) null);
	}

	public List<User> findAppUsers(SearchCriteria searchCriteria) throws ServiceException {
		try {
			List<User> Users = new ArrayList<User>();
			List<SimpleUser> users = securityDAO.findAppUsers(searchCriteria);
			if (users != null && users.size() > 0) {
				for (SimpleUser person : users) {
					Users.add(new User(person));
				}
			}
			return Users;
		}catch (DAOException e){
			throw new ServiceException("Securityservice:findAppUsers failed", e);
		}
	}

	public UserPrincipal getAuditUser(String userName) throws ServiceException {
		try {
			return securityDAO.getAuditUser(userName);
		} catch (DAOException e) {
			throw new ServiceException("Securityservice:getAuditUser failed", e);
		}
	}

	public List<String> getUserIds() throws ServiceException {
		try {
			return securityDAO.getUserIds();
		} catch (Exception e) {
			logger.error("SecurityService.getUserIds() failed", e);
			throw new ServiceException("SecurityService.getUserIds() failed", e);
		}
	}
	
	public int updateUsers(List<String> userIds) throws ServiceException {
		int updateCount = 0;
		try {
			if(userIds != null && userIds.size() > 0) {
				for (String userId : userIds) {
					User user = getUser(userId);
					if(user != null) {
						updateUser(user);
						updateCount = updateCount + 1;
					} else {
						logger.info("SecurityService.updateUsers(): User not found for UserId = " + userId);
					}
				}
			}
		} catch (Exception e) {
			logger.error("SecurityService.updateUsers(): failed", e);
			throw new ServiceException("SecurityService.updateUsers(): failed", e);
		}
		return updateCount;
	}

	public List<User> getSubmittedRegistrations(Long visnId, Long facilityId) throws ServiceException {
		try {
			return securityDAO.getSubmittedRegistrations(visnId, facilityId);
		}catch (Exception e){
			logger.error("SecurityService.getSubmittedRegistrations(): failed", e);
			throw new ServiceException("SecurityService.getSubmittedRegistrations(): failed", e);
		}
	}
	public List<User> getSubmittedRegistrationsForDmp(Long visnId, Long facilityId) throws ServiceException {
		try {
			return securityDAO.getSubmittedRegistrationsForDmp(visnId, facilityId);
		}catch (Exception e){
			logger.error("SecurityService.getSubmittedRegistrationsForDmp(): failed", e);
			throw new ServiceException("SecurityService.getSubmittedRegistrationsForDmp(): failed", e);
		}
	}
	public SecurityDAO getSecurityDAO() {
		return securityDAO;
	}

	public void setSecurityDAO(SecurityDAO securityDAO) {
		this.securityDAO = securityDAO;
	}

	public LdapPersonService getLdapPersonService() {
		return ldapPersonService;
	}

	public void setLdapPersonService(LdapPersonService ldapPersonService) {
		this.ldapPersonService = ldapPersonService;
	}

	public TransactionTimestampManager getTimestampManager() {
		return timestampManager;
	}

	public void setTimestampManager(TransactionTimestampManager timestampManager) {
		this.timestampManager = timestampManager;
	}

	public MergeService getMergeService() {
		return mergeService;
	}

	public void setMergeService(MergeService mergeService) {
		this.mergeService = mergeService;
	}
}
