package gov.va.med.ccht.service.common.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.springframework.mail.SimpleMailMessage;

import gov.va.med.fw.model.ldap.SearchCriteria;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.util.StringUtils;

import gov.va.med.ccht.model.CCHTRoles;
import gov.va.med.ccht.model.User;
import gov.va.med.ccht.model.inventory.SimpleFacility;
import gov.va.med.ccht.model.inventory.SimpleVisn;
import gov.va.med.ccht.service.common.RegistrationService;
import gov.va.med.ccht.service.common.SecurityService;

public class RegistrationServiceImpl extends AbstractBusinessService implements
		RegistrationService {

	public void addPatient(User user) throws ServiceException {
	}

	public void addVAUser(User user) throws ServiceException {
		// set user group and registration status
		securityService.createUser(user);
		sendConfirmationMail(user);
		List<User> admins = getAdmins(user);
		sendAdminMail(user, admins);
	}

	public void approveUser(User user) throws ServiceException {
		// set user group and registration status
		securityService.updateUser(user);
		sendWelcomeMail(user);
	}

	public void rejectUser(User user) throws ServiceException {
		// set user group and registration status
		securityService.updateUser(user);
		sendDeniedMail(user);
	}

	public void addVendor(User user) throws ServiceException {
	}

	public void sendAdminMail(User user, List<User> admins)
			throws ServiceException {
		// verify the existance of email addresses
		if (user == null || admins == null || admins.size() == 0) {
			return;
		}
		List<String> adminEMailList = getEmailList(admins);

		if (adminEMailList.size() < 1)
			return;

		SimpleMailMessage template = new SimpleMailMessage(adminMailMessage);
		Map<String, Object> data = new HashMap<String, Object>();
		data.put("user", user);
		data.put("webAppUrl", getWebAppUrl());
		String[] toAddress = new String[adminEMailList.size()];
		for (int i = 0; i < adminEMailList.size(); i++) {
			toAddress[i] = adminEMailList.get(i);
		}
		template.setTo(toAddress);
		template.setText(merge(template.getText(), data));
		send(template);
	}

	public void sendConfirmationMail(User user) throws ServiceException {
		// copy template and process
		if (user != null && StringUtils.isNotEmpty(user.getEmail())) {
			Map<String, Object> data = new HashMap<String, Object>();
			data.put("user", user);
			data.put("webAppUrl", getWebAppUrl());
			String toAddress = user.getEmail();
			SimpleMailMessage template = processMailTemplate(data,
					confirmMailMessage);
			template.setTo(toAddress);
			send(template);
		}
	}

	public void sendWelcomeMail(User user) throws ServiceException {
		if (user != null && StringUtils.isNotEmpty(user.getEmail())) {
			SimpleMailMessage template = new SimpleMailMessage(
					welcomeMailMessage);
			Map<String, Object> data = new HashMap<String, Object>();
			data.put("user", user);
			data.put("webAppUrl", getWebAppUrl());
			String toAddress = user.getEmail();
			template.setTo(toAddress);
			template.setText(merge(template.getText(), data));
			send(template);
		}
	}

	public void sendDeniedMail(User user) throws ServiceException {
		if (user != null && StringUtils.isNotEmpty(user.getEmail())) {
			SimpleMailMessage template = new SimpleMailMessage(
					deniedMailMessage);
			Map<String, Object> data = new HashMap<String, Object>();
			data.put("user", user);
			data.put("webAppUrl", getWebAppUrl());
			String toAddress = user.getEmail();
			template.setTo(toAddress);
			template.setText(merge(template.getText(), data));
			send(template);
		}
	}

	public static String preserveCarriageReturns(String text) {
		if (text != null) {
			text = text.replaceAll("\r\n", "\n"); // standardizes line
			// separators
			return text.replaceAll("\n", "\t\n");
		} else {
			return text;
		}
	}

	private List<User> getAdmins(User user) throws ServiceException {
		List<User> adminUsers = new ArrayList<User>();
		List<User> facilityAdmins = null;
		List<User> visnAdmins = null;
		List<User> nationalAdmins = null;
		List<User> dmpAdmins = null;

		if (user.getRegistrationReason().getDescription().indexOf("DMP") != -1
				|| user.getRegistrationReason().getDescription().indexOf(
						"Training Center Staff") != -1) {
			dmpAdmins = getDmpAdmins();
			if (dmpAdmins != null)
				adminUsers.addAll(dmpAdmins);
		}
		// if user has not specified either VISN or Facility mail is sent to
		// National Admin
		else if (user.getFacility() != null) {
			// send mail to Facility and VISN admins
			facilityAdmins = getFacilityAdmins(user.getFacility());
			visnAdmins = getVisnAdmins(user.getFacility().getVisn());
			// if facility and visn admins don't exist send an e-mail to
			// national
			if ((facilityAdmins == null || facilityAdmins.size() == 0)
					&& (visnAdmins == null || visnAdmins.size() == 0)) {
				nationalAdmins = getNationalAdmins();
			}
		} else if (user.getVisn() != null) {
			// send mail to National and VISN admin
			visnAdmins = getVisnAdmins(user.getVisn());
			nationalAdmins = getNationalAdmins();
		} else {
			// send mail to National Admin
			nationalAdmins = getNationalAdmins();
		}

		if (facilityAdmins != null)
			adminUsers.addAll(facilityAdmins);
		if (visnAdmins != null)
			adminUsers.addAll(visnAdmins);
		if (nationalAdmins != null)
			adminUsers.addAll(nationalAdmins);
		return adminUsers;
	}

	/**
	 * Get National Admins
	 * 
	 * @return
	 * @throws ServiceException
	 */
	private List<User> getNationalAdmins() throws ServiceException {
		SearchCriteria searchCriteria = new SearchCriteria();
		List<String> roles = new ArrayList<String>();
		roles.add(CCHTRoles.NATIONAL_ADMIN);
		searchCriteria.setRoles(roles);
		List<User> admins = securityService.findAppUsers(searchCriteria);
		return admins;
	}

	/**
	 * Get VISN admins
	 * 
	 * @param visn
	 * @return
	 * @throws ServiceException
	 */
	private List<User> getVisnAdmins(SimpleVisn visn) throws ServiceException {
		SearchCriteria searchCriteria = new SearchCriteria();
		List<String> roles = new ArrayList<String>();
		roles.add(CCHTRoles.VISN_ADMIN);
		searchCriteria.setRoles(roles);
		List<String> visns = new ArrayList<String>();
		visns.add(visn.getCode());
		searchCriteria.setVisns(visns);
		List<User> admins = securityService.findAppUsers(searchCriteria);
		return admins;
	}

	/**
	 * Get Facility Admins
	 * 
	 * @param facility
	 * @return
	 * @throws ServiceException
	 */
	private List<User> getFacilityAdmins(SimpleFacility facility)
			throws ServiceException {
		SearchCriteria searchCriteria = new SearchCriteria();
		List<String> roles = new ArrayList<String>();
		roles.add(CCHTRoles.FACILITY_ADMIN);
		searchCriteria.setRoles(roles);
		List<String> facilities = new ArrayList<String>();
		facilities.add(facility.getCode());
		searchCriteria.setStations(facilities);
		List<User> admins = securityService.findAppUsers(searchCriteria);
		return admins;
	}

	/**
	 * Get VISN admins
	 * 
	 * @param visn
	 * @return
	 * @throws ServiceException
	 */
	private List<User> getDmpAdmins() throws ServiceException {
		SearchCriteria searchCriteria = new SearchCriteria();
		List<String> roles = new ArrayList<String>();
		roles.add(CCHTRoles.DMP_ADMIN);
		searchCriteria.setRoles(roles);
		List<User> admins = securityService.findAppUsers(searchCriteria);
		return admins;
	}

	private SimpleMailMessage welcomeMailMessage;
	private SimpleMailMessage adminMailMessage;
	private SimpleMailMessage confirmMailMessage;
	private SimpleMailMessage deniedMailMessage;
	private SecurityService securityService;
	private List<String> adminRoles;

	public SecurityService getSecurityService() {
		return securityService;
	}

	public void setSecurityService(SecurityService securityService) {
		this.securityService = securityService;
	}

	public SimpleMailMessage getWelcomeMailMessage() {
		return welcomeMailMessage;
	}

	public void setWelcomeMailMessage(SimpleMailMessage welcomeMailMessage) {
		this.welcomeMailMessage = welcomeMailMessage;
	}

	public SimpleMailMessage getAdminMailMessage() {
		return adminMailMessage;
	}

	public void setAdminMailMessage(SimpleMailMessage adminMailMessage) {
		this.adminMailMessage = adminMailMessage;
	}

	public SimpleMailMessage getConfirmMailMessage() {
		return confirmMailMessage;
	}

	public void setConfirmMailMessage(SimpleMailMessage confirmMailMessage) {
		this.confirmMailMessage = confirmMailMessage;
	}

	public SimpleMailMessage getDeniedMailMessage() {
		return deniedMailMessage;
	}

	public void setDeniedMailMessage(SimpleMailMessage deniedMailMessage) {
		this.deniedMailMessage = deniedMailMessage;
	}

	public List<String> getAdminRoles() {
		if (adminRoles == null)
			adminRoles = new ArrayList<String>();
		return adminRoles;
	}

	public void setAdminRoles(List<String> adminRoles) {
		this.adminRoles = adminRoles;
	}
}
