/********************************************************************
 * Copyright � 2010 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.ccht.service.common.impl;

import gov.va.med.ccht.model.terminology.ApplicationName;
import gov.va.med.ccht.service.common.ExceptionLogEntry;
import gov.va.med.ccht.service.common.ExceptionService;
import gov.va.med.fw.model.EntityKey;
import gov.va.med.fw.model.lookup.SeverityType;
import gov.va.med.fw.persistent.DAOOperations;
import gov.va.med.fw.service.AbstractComponent;
import gov.va.med.fw.util.InvalidConfigurationException;

import java.io.PrintWriter;
import java.io.StringWriter;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;

/**
 * IHTA - IHTA_common TODO: add content here
 * 
 * Jul 21, 2010
 * 
 * @author VHAISWBOHMEG
 */
public class ExceptionServiceImpl extends AbstractComponent implements ExceptionService {

	private static final int VARCHAR_COL_SIZE = 8000;
	private boolean logExceptions = true;

	private boolean persistExceptions = true;

	private boolean invokeEELS = false;

	private DAOOperations dao;

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.IHTA.service.common.ExceptionService#processException(java
	 * .lang.Throwable, java.lang.String, java.lang.Object)
	 */
	public EntityKey processException(Throwable exception, String message, Object producer,
			ApplicationName appName) {
		return processException(exception, message, producer, SeverityType.ERROR, appName);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.IHTA.service.common.ExceptionService#processException(java
	 * .lang.Throwable, java.lang.String, java.lang.Object)
	 */
	public EntityKey processException(Throwable exception, String message, Object producer,
			SeverityType severityType, ApplicationName appName) {
		Validate.notNull(producer, "producer object cannot be null");

		ExceptionLogEntry logEntry = new ExceptionLogEntry();
		logEntry.setApplicationName(appName != null ? appName.getCode() : null);
		logEntry.setSeverityType(severityType != null ? severityType.getCode() : null);
		logEntry.setException(exception);
		logEntry.setMessage(message);
		String detail = this.getExceptionStackTrace(exception);
		if (detail != null) {
			if (detail.length() > VARCHAR_COL_SIZE) {
				String detailExt = detail.substring(VARCHAR_COL_SIZE);
				detail = detail.substring(0, VARCHAR_COL_SIZE);
				detailExt = detailExt.length() > VARCHAR_COL_SIZE ? detail.substring(0,
						VARCHAR_COL_SIZE) : detailExt;
				logEntry.setDetailExt(detailExt);
			}
			logEntry.setDetail(detail);
		}

		logEntry.setProducerClass(producer.getClass());
		return doProcessException(logEntry);
	}

	private EntityKey doProcessException(ExceptionLogEntry logEntry) {
		EntityKey key = null;

		try {
			if (logExceptions) {

				if (logEntry.getProducerClass() == null && logEntry.getProducerClassName() == null) {
					throw new IllegalArgumentException(
							"ExeptionLogEntry must have knowledge of the producer class");
				}

				Logger producerClazzLogger = null;
				if (logEntry.getProducerClass() != null) {
					producerClazzLogger = Logger.getLogger(logEntry.getProducerClass());
				} else {
					producerClazzLogger = Logger.getLogger(logEntry.getProducerClassName());
				}

				if (producerClazzLogger.isEnabledFor(Level.ERROR)) {
					if (!StringUtils.isEmpty(logEntry.getMessage()))
						producerClazzLogger.error(logEntry.getMessage(), logEntry.getException());
					else
						producerClazzLogger.error(logEntry.getException());
				}
			}

			if (persistExceptions) {
				ExceptionLogEntry updatedLogEntry = (ExceptionLogEntry) dao.saveObject(logEntry);
				key = updatedLogEntry.getEntityKey();
			}

			if (invokeEELS) {
				logger.debug("TODO: Invoke EELS for Exception");
			}

		} catch (Throwable t) {
			String message = "Unable to complete doProcessException";
			logger.error(message, t);
			throw new RuntimeException(message, t);
		}
		return key;
	}

	protected String getExceptionStackTrace(Throwable exception) {
		/*
		 * StackTraceElement[] trace = exception.getStackTrace(); StringBuilder
		 * stackTrace = new StringBuilder(); if(stackTrace != null) { for(int
		 * i=0; i<trace.length; i++) { if(trace[i] != null) {
		 * stackTrace.append(trace[i].toString()); stackTrace.append("\n"); } }
		 * }
		 * 
		 * Throwable ourCause = exception.getCause(); while(ourCause != null) {
		 * if (ourCause != null) { stackTrace.append(ourCause.getMessage() +
		 * "\n"); ourCause = ourCause.getCause(); } } return
		 * stackTrace.toString();
		 */
		StringWriter sw = new StringWriter();
		PrintWriter pw = new PrintWriter(sw);
		pw.close();
		return sw.getBuffer().toString();
	}

	/**
	 * @return the invokeEELS
	 */
	public boolean isInvokeEELS() {
		return invokeEELS;
	}

	/**
	 * @param invokeEELS
	 *            the invokeEELS to set
	 */
	public void setInvokeEELS(boolean invokeEELS) {
		this.invokeEELS = invokeEELS;
	}

	/**
	 * @return the logExceptions
	 */
	public boolean isLogExceptions() {
		return logExceptions;
	}

	/**
	 * @param logExceptions
	 *            the logExceptions to set
	 */
	public void setLogExceptions(boolean logExceptions) {
		this.logExceptions = logExceptions;
	}

	/**
	 * @return the persistExceptions
	 */
	public boolean isPersistExceptions() {
		return persistExceptions;
	}

	/**
	 * @param persistExceptions
	 *            the persistExceptions to set
	 */
	public void setPersistExceptions(boolean persistExceptions) {
		this.persistExceptions = persistExceptions;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.IHTA.service.common.ExceptionService#processException(gov.
	 * va.med.IHTA.service.common.ExceptionLogEntry)
	 */
	public EntityKey processException(ExceptionLogEntry logEntry) {
		return doProcessException(logEntry);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.fw.service.AbstractComponent#afterPropertiesSet()
	 */
	@Override
	public void afterPropertiesSet() throws Exception {
		super.afterPropertiesSet();
		if (persistExceptions && dao == null) {
			throw new InvalidConfigurationException(
					"DAO must be non-null when configuring to force persistence of exceptions");
		}
	}

	/**
	 * @return the dao
	 */
	public DAOOperations getDao() {
		return dao;
	}

	/**
	 * @param dao
	 *            the dao to set
	 */
	public void setDao(DAOOperations dao) {
		this.dao = dao;
	}

}
