package gov.va.med.ccht.persistent.hibernate;

import edu.emory.mathcs.backport.java.util.Collections;
import gov.va.med.ccht.model.dmp.PatientSatSurveyReportParameters;
import gov.va.med.ccht.model.dmpreport.FacilityNameComparator;
import gov.va.med.ccht.model.dmpreport.PSSQuestion;
import gov.va.med.ccht.model.dmpreport.PSSQuestionChoice;
import gov.va.med.ccht.model.dmpreport.PSSQuestionManager;
import gov.va.med.ccht.model.dmpreport.SurveyDistributionEntry;
import gov.va.med.ccht.model.dmpreport.SurveyDistributionSearchResult;
import gov.va.med.ccht.model.dmpreport.VisnIdComparator;
import gov.va.med.ccht.persistent.SurveyDistributionReportDAO;
import gov.va.med.ccht.persistent.utils.PSSReportUtils;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.fw.persistent.hibernate.GenericDAOImpl;
import gov.va.med.fw.persistent.hibernate.QueryAndParams;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class SurveyDistributionReportDAOImpl extends GenericDAOImpl implements
		SurveyDistributionReportDAO {

	private String selectedModality = null;



	@Override
	public SurveyDistributionSearchResult getSurveyDistributionReport(
			PatientSatSurveyReportParameters reportParams) throws DAOException {

		SurveyDistributionSearchResult returnResult = new SurveyDistributionSearchResult();
		int totalNumberOfSurveys = 0;

		int selectedReportType = reportParams.getSelectedReportType();

		try {

			selectedModality = reportParams.getModality();
			switch (selectedReportType) {
			case 0:
				returnResult = getNationalSurveyDistributionReport(reportParams);
				break;
			case 1:
				returnResult = getSurveyDistributionReportByVisn(reportParams);
				break;
			case 2:
				returnResult = getSurveyDistributionReportByFacility(reportParams);
				break;
			}

		} catch (Exception e) {
			throw new DAOException("getSurveyDistributionReport failed " + e);
		}

		return returnResult;
	}

	private SurveyDistributionSearchResult getNationalSurveyDistributionReport(
			PatientSatSurveyReportParameters reportParams) throws DAOException {
		Map<String, List<SurveyDistributionEntry>> tempRecords = new HashMap<String, List<SurveyDistributionEntry>>();
		SurveyDistributionSearchResult retVal = new SurveyDistributionSearchResult();
		StringBuilder qString = new StringBuilder();
		int surveyId = reportParams.getSelectedSurvey();
		int totalNumberOfSurveys = 0;

		// BUILD SELECT SECTION OF SQL
		qString
				.append("SELECT '' AS [Breakout], v.vendor_name, COUNT(*) AS [NumSurveys], ");
		
		// Added to handle DO NOT INCLUDE
		String selectedModality = reportParams.getModality();
		if(selectedModality.equalsIgnoreCase(PSSReportUtils.DO_NOT_INCLUDE))
		{
			qString.append(" CASE Device.device_name WHEN 'IVR' THEN Device.device_name ELSE 'Home Device' END ");
		}
		else
		{
			qString.append(PSSReportUtils.getModalityString(selectedModality));
		}

		// BUILD FROM SECTION OF SQL
		qString.append(getCommonFromSQL(reportParams));
		qString.append(PSSReportUtils.getModalityInnerJoinString(reportParams));

		// BUILD WHERE SECTION OF SQL
		qString.append(getCommonWhereSQL(reportParams));

		String modalityInString = PSSReportUtils
				.getModalityInString(reportParams);
		qString.append(modalityInString);

		// BUILD GROUP BY SECTION OF SQL
		qString.append(" GROUP BY v.vendor_name");
		
		if(selectedModality.equalsIgnoreCase(PSSReportUtils.DO_NOT_INCLUDE))
		{
			qString.append(", CASE Device.device_name WHEN 'IVR' THEN Device.device_name ELSE 'Home Device' END ");
		}
		else
		{
			qString.append(PSSReportUtils
					.getModalityStringForGroupByClause(reportParams));
		}



		qString.append(" ORDER BY v.vendor_name ASC ");

		QueryAndParams queryAndParams = new QueryAndParams(qString.toString());

		queryAndParams.append(qString.toString());
		queryAndParams
				.addParam("fromDate", reportParams.getSubmittedFromDate());

		Date toDate = getModifiedToDate(reportParams.getSubmittedToDate());
		reportParams.setSubmittedToDate(toDate);

		queryAndParams.addParam("toDate", reportParams.getSubmittedToDate());
		queryAndParams.addParam("surveyId", reportParams.getSelectedSurvey());

		// TODO: IMPLEMENT METHOD FOR GETTING THE NEEDED MODALITY_TYPE ID'S
		if (selectedModality != null && selectedModality.length() > 0) {
			List modalityList = PSSReportUtils.getModalityIds(selectedModality);
			queryAndParams.addParam("modalityIdList", modalityList);
		}

		List<?> report = executeSQLQuery(queryAndParams);

		int numIterations = report.size();

		for (int j = 0; j < numIterations; j++) {
			Object[] record = (Object[]) report.get(j);
			if (record != null) {

				SurveyDistributionEntry newEntry = new SurveyDistributionEntry();
				Integer visnId = 0;
				String breakOut = (String) record[0];
				String vendorName = (String) record[1];
				Integer numSurveys = (Integer) record[2];

				String modality = (String) record[3];

				newEntry.setVendorName(vendorName);

				if (modality.equalsIgnoreCase("IVR")) {
					newEntry.setTotalIVRSurveys(numSurveys);
				} else if (modality.equalsIgnoreCase("Home Device")) {
					newEntry.setTotalHomeDeviceSurveys(numSurveys);
				} else if (modality.equalsIgnoreCase("Browser")) {
					newEntry.setTotalBrowserSurveys(numSurveys);
				}
				else
				{
					newEntry.setTotalNumberOfSurveys(numSurveys);
				}

				String key = breakOut + "-" + vendorName;

				List<SurveyDistributionEntry> entries = tempRecords.get(key);

				if (entries == null) {
					entries = new ArrayList<SurveyDistributionEntry>();
					entries.add(newEntry);
				} else {
					entries.add(newEntry);
				}
				totalNumberOfSurveys += numSurveys;
				tempRecords.put(key, entries);

			}
		}
		List<SurveyDistributionEntry> returnRecords = mergeByNationalRecords(tempRecords);

		retVal.setSurveyDistributionEntries(returnRecords);
		retVal.setTotalNumOfSurveys(totalNumberOfSurveys);

		return retVal;
	}

	private SurveyDistributionSearchResult getSurveyDistributionReportByVisn(
			PatientSatSurveyReportParameters reportParams) throws DAOException {
		SurveyDistributionSearchResult retVal = new SurveyDistributionSearchResult();
		StringBuilder qString = new StringBuilder();
		QueryAndParams queryAndParams = new QueryAndParams(qString.toString());
		int surveyId = reportParams.getSelectedSurvey();
		Map<String, List<SurveyDistributionEntry>> tempRecords = new HashMap<String, List<SurveyDistributionEntry>>();
		int totalNumberOfSurveys = 0;
		qString
				.append("SELECT sa.VISN_ID, cast(vi.VISN_Name as varchar) AS [Breakout], ");
		qString.append("v.vendor_name, COUNT(*) AS [NumSurveys], ");
		qString.append(PSSReportUtils.getModalityString(reportParams));

		qString.append(getCommonFromSQL(reportParams));
		qString.append(" INNER JOIN VISN AS vi ON sa.VISN_ID = vi.VISN_ID ");
		qString.append(PSSReportUtils.getModalityInnerJoinString(reportParams));

		// BUILD WHERE SECTION OF SQL
		qString.append(getCommonWhereSQL(reportParams));

		if (reportParams.getVisn().getValue().equalsIgnoreCase("-1")) {
			qString.append(" AND sa.VISN_ID >= 0");
		} else {
			qString.append(" AND sa.VISN_ID = :visnId ");
			queryAndParams
					.addParam("visnId", reportParams.getVisn().getValue());
		}

		PSSReportUtils.getModalityInString(reportParams);

		qString.append(" GROUP BY sa.VISN_id, vi.Visn_Name, v.vendor_name");
		qString.append(PSSReportUtils
				.getModalityStringForGroupByClause(reportParams));

		qString.append(" ORDER BY sa.VISN_id, vi.Visn_Name, v.vendor_name ASC");

		Date toDate = getModifiedToDate(reportParams.getSubmittedToDate());
		reportParams.setSubmittedToDate(toDate);

		queryAndParams
				.addParam("fromDate", reportParams.getSubmittedFromDate());
		queryAndParams.addParam("toDate", reportParams.getSubmittedToDate());
		queryAndParams.addParam("surveyId", reportParams.getSelectedSurvey());

		// TODO: IMPLEMENT METHOD FOR GETTING THE NEEDED MODALITY_TYPE ID'S
		if (selectedModality != null && selectedModality.length() > 0) {
			List modalityList = PSSReportUtils.getModalityIds(selectedModality);
			queryAndParams.addParam("modalityIdList", modalityList);
		}

		queryAndParams.append(qString.toString());
		List<?> report = executeSQLQuery(queryAndParams);

		int numIterations = report.size();

		for (int j = 0; j < numIterations; j++) {
			Object[] record = (Object[]) report.get(j);
			if (record != null) {

				SurveyDistributionEntry newEntry = new SurveyDistributionEntry();
				Integer visnId = 0;
				String breakOut = "";
				String vendorName = "";// (String)record[1];
				Integer numSurveys = 0;// (Integer)record[2];
				String modality = "";// (String)record[3];

				visnId = (Integer) record[0];
				breakOut = (String) record[1];
				vendorName = (String) record[2];
				numSurveys = (Integer) record[3];
				modality = (String) record[4];
				newEntry.setVisnId(visnId);
				newEntry.setVisnName(breakOut);

				newEntry.setVendorName(vendorName);

				if (modality.equalsIgnoreCase(PSSReportUtils.IVR)) {
					newEntry.setTotalIVRSurveys(numSurveys);
				} else if (modality
						.equalsIgnoreCase(PSSReportUtils.HOME_DEVICE)) {
					newEntry.setTotalHomeDeviceSurveys(numSurveys);
				} else if (modality.equalsIgnoreCase(PSSReportUtils.BROWSER)) {
					newEntry.setTotalBrowserSurveys(numSurveys);
				}
				else
				{
					newEntry.setTotalNumberOfSurveys(numSurveys);
				}

				String key = breakOut + "-" + vendorName;

				List<SurveyDistributionEntry> entries = tempRecords.get(key);

				if (entries == null) {
					entries = new ArrayList<SurveyDistributionEntry>();
					entries.add(newEntry);
				} else {
					entries.add(newEntry);
				}
				totalNumberOfSurveys += numSurveys;
				tempRecords.put(key, entries);
			}
		}
		List<SurveyDistributionEntry> returnRecords = mergeByVisnRecords(tempRecords);
		Collections.sort(returnRecords, new VisnIdComparator());
		retVal.setSurveyDistributionEntries(returnRecords);
		retVal.setTotalNumOfSurveys(totalNumberOfSurveys);
		return retVal;
	}

	private SurveyDistributionSearchResult getSurveyDistributionReportByFacility(
			PatientSatSurveyReportParameters reportParams) throws DAOException {
		SurveyDistributionSearchResult retVal = new SurveyDistributionSearchResult();
		StringBuilder qString = new StringBuilder();
		int surveyId = reportParams.getSelectedSurvey();
		Map<String, List<SurveyDistributionEntry>> tempRecords = new HashMap<String, List<SurveyDistributionEntry>>();
		int totalNumberOfSurveys = 0;

		QueryAndParams queryAndParams = new QueryAndParams(qString.toString());

		qString.append("SELECT f.facility_name AS [Breakout], ");
		qString.append("v.vendor_name, COUNT(*) AS [NumSurveys], ");
		qString.append(PSSReportUtils.getModalityString(reportParams));

		qString.append(getCommonFromSQL(reportParams));
		qString.append(" INNER JOIN Facilities as f ON f.ID = sa.facility_id ");
		qString.append(PSSReportUtils.getModalityInnerJoinString(reportParams));

		// BUILD WHERE SECTION OF SQL
		qString.append(getCommonWhereSQL(reportParams));

		String facilityValue = reportParams.getFacility().getValue();
		if (facilityValue.equalsIgnoreCase("All Facilities")
				|| facilityValue.equalsIgnoreCase("-1")) {
			qString.append(" AND sa.facility_ID >= 0 ");
		} else {
			qString.append(" AND sa.facility_ID = :facilityId ");
			queryAndParams.addParam("facilityId", reportParams.getFacility()
					.getValue());
		}

		qString.append(PSSReportUtils.getModalityInString(reportParams));

		qString.append(" GROUP BY f.Facility_name, v.vendor_name");

		qString.append(PSSReportUtils
				.getModalityStringForGroupByClause(reportParams));

		qString.append(" ORDER BY f.Facility_name, v.vendor_name ASC");

		Date toDate = getModifiedToDate(reportParams.getSubmittedToDate());
		reportParams.setSubmittedToDate(toDate);

		queryAndParams
				.addParam("fromDate", reportParams.getSubmittedFromDate());
		queryAndParams.addParam("toDate", reportParams.getSubmittedToDate());
		queryAndParams.addParam("surveyId", reportParams.getSelectedSurvey());

		// TODO: IMPLEMENT METHOD FOR GETTING THE NEEDED MODALITY_TYPE ID'S
		if (selectedModality != null && selectedModality.length() > 0) {
			List modalityList = PSSReportUtils.getModalityIds(selectedModality);
			queryAndParams.addParam("modalityIdList", modalityList);
		}
		queryAndParams.append(qString.toString());

		List<?> report = executeSQLQuery(queryAndParams);

		int numIterations = report.size();

		for (int j = 0; j < numIterations; j++) {
			Object[] record = (Object[]) report.get(j);
			if (record != null) {

				SurveyDistributionEntry newEntry = new SurveyDistributionEntry();
				Integer visnId = 0;
				String breakOut = "";
				String vendorName = "";
				Integer numSurveys = 0;
				String modality = "";
				breakOut = (String) record[0];
				vendorName = (String) record[1];
				numSurveys = (Integer) record[2];
				modality = (String) record[3];
				newEntry.setFacilityName(breakOut);
				newEntry.setVendorName(vendorName);

				if (modality.equalsIgnoreCase(PSSReportUtils.IVR)) {
					newEntry.setTotalIVRSurveys(numSurveys);
				} else if (modality
						.equalsIgnoreCase(PSSReportUtils.HOME_DEVICE)) {
					newEntry.setTotalHomeDeviceSurveys(numSurveys);
				} else if (modality.equalsIgnoreCase(PSSReportUtils.BROWSER)) {
					newEntry.setTotalBrowserSurveys(numSurveys);
				}
				else
				{
					newEntry.setTotalNumberOfSurveys(numSurveys);
				}

				String key = breakOut + "-" + vendorName;

				List<SurveyDistributionEntry> entries = tempRecords.get(key);

				if (entries == null) {
					entries = new ArrayList<SurveyDistributionEntry>();
					entries.add(newEntry);
				} else {
					entries.add(newEntry);
				}
				totalNumberOfSurveys += numSurveys;
				tempRecords.put(key, entries);
			}
		}
		List<SurveyDistributionEntry> returnRecords = mergeByFacilityRecords(tempRecords);
		Collections.sort(returnRecords, new FacilityNameComparator());
		retVal.setSurveyDistributionEntries(returnRecords);
		retVal.setTotalNumOfSurveys(totalNumberOfSurveys);

		return retVal;
	}

	private String getCommonFromSQL(
			PatientSatSurveyReportParameters reportParams) {
		StringBuilder sql = new StringBuilder();
		sql.append(" FROM Surveyed_Activity AS sa ");
		sql
				.append("INNER JOIN Surveyed_Activity_Response AS sar ON sa.Surveyed_Activity_ID = sar.Surveyed_Activity_ID ");
		sql.append("INNER JOIN Vendor AS v ON sa.Vendor_ID = v.Vendor_ID ");
		sql.append("INNER JOIN Device ON sa.Device_ID = Device.Device_ID ");

		return sql.toString();
	}

	private String getCommonWhereSQL(
			PatientSatSurveyReportParameters reportParams) {
		StringBuilder sql = new StringBuilder();
		sql.append("WHERE sar.completed_date >= :fromDate ");
		sql.append("AND sar.completed_date <= :toDate ");

		if (reportParams.getSelectedSurvey() == 0) {
			sql
					.append(" AND sar.survey_accepted_status=1 AND (sar.survey_id>=:surveyId) ");
		} else {
			sql
					.append(" AND sar.survey_accepted_status=1 AND (sar.survey_id=:surveyId) ");
		}

		return sql.toString();
	}

	private List<SurveyDistributionEntry> mergeByNationalRecords(
			Map<String, List<SurveyDistributionEntry>> tempRecords) {
		List<SurveyDistributionEntry> returnVal = new ArrayList<SurveyDistributionEntry>();
		for (String key : tempRecords.keySet()) 
		{
			List<SurveyDistributionEntry> temp = tempRecords.get(key);

			SurveyDistributionEntry mergedEntry = new SurveyDistributionEntry();
			for (SurveyDistributionEntry entry : temp) 
			{
				mergedEntry.setFacilityName(entry.getFacilityName());
				mergedEntry.setVendorName(entry.getVendorName());
				if (entry.getTotalIVRSurveys() != null) 
				{
					mergedEntry.setTotalIVRSurveys(entry.getTotalIVRSurveys());
				} 
				else if (entry.getTotalHomeDeviceSurveys() != null) 
				{
					mergedEntry.setTotalHomeDeviceSurveys(entry
							.getTotalHomeDeviceSurveys());
				} 
				else if (entry.getTotalBrowserSurveys() != null) 
				{
					mergedEntry.setTotalBrowserSurveys(entry
							.getTotalBrowserSurveys());
				}
			}
			returnVal.add(mergedEntry);
		}

		return returnVal;
	}

	private List<SurveyDistributionEntry> mergeByFacilityRecords(
			Map<String, List<SurveyDistributionEntry>> tempRecords) {
		List<SurveyDistributionEntry> returnVal = new ArrayList<SurveyDistributionEntry>();
		for (String key : tempRecords.keySet()) {
			List<SurveyDistributionEntry> temp = tempRecords.get(key);

			SurveyDistributionEntry mergedEntry = new SurveyDistributionEntry();
			for (SurveyDistributionEntry entry : temp) {
				mergedEntry.setFacilityName(entry.getFacilityName());
				mergedEntry.setVendorName(entry.getVendorName());
				if (entry.getTotalIVRSurveys() != null) {
					mergedEntry.setTotalIVRSurveys(entry.getTotalIVRSurveys());
				} else if (entry.getTotalHomeDeviceSurveys() != null) {
					mergedEntry.setTotalHomeDeviceSurveys(entry
							.getTotalHomeDeviceSurveys());
				} else if (entry.getTotalBrowserSurveys() != null) {
					mergedEntry.setTotalBrowserSurveys(entry
							.getTotalBrowserSurveys());
				}
			}
			returnVal.add(mergedEntry);
		}

		return returnVal;
	}

	private List<SurveyDistributionEntry> mergeByVisnRecords(
			Map<String, List<SurveyDistributionEntry>> tempRecords) {
		List<SurveyDistributionEntry> returnVal = new ArrayList<SurveyDistributionEntry>();
		for (String key : tempRecords.keySet()) {
			List<SurveyDistributionEntry> temp = tempRecords.get(key);

			SurveyDistributionEntry mergedEntry = new SurveyDistributionEntry();
			for (SurveyDistributionEntry entry : temp) {
				mergedEntry.setVisnId(entry.getVisnId());
				mergedEntry.setVisnName(entry.getVisnName());
				mergedEntry.setVendorName(entry.getVendorName());
				if (entry.getTotalIVRSurveys() != null) {
					mergedEntry.setTotalIVRSurveys(entry.getTotalIVRSurveys());
				} else if (entry.getTotalHomeDeviceSurveys() != null) {
					mergedEntry.setTotalHomeDeviceSurveys(entry
							.getTotalHomeDeviceSurveys());
				} else if (entry.getTotalBrowserSurveys() != null) {
					mergedEntry.setTotalBrowserSurveys(entry
							.getTotalBrowserSurveys());
				}
			}
			returnVal.add(mergedEntry);
		}

		return returnVal;
	}

	private Date getModifiedToDate(Date toDate) {
		Date modifiedToDate = null;
		GregorianCalendar toDateAsCal = new GregorianCalendar();
		toDateAsCal.setTime(toDate);
		toDateAsCal.set(Calendar.HOUR_OF_DAY, 23);
		toDateAsCal.set(Calendar.MINUTE, 59);
		toDateAsCal.set(Calendar.SECOND, 59);
		toDateAsCal.set(Calendar.MILLISECOND, 999);
		modifiedToDate = toDateAsCal.getTime();
		return modifiedToDate;
	}

	public PSSQuestionManager getQuestionsForReportVersion(Integer version)
			throws DAOException {
		PSSQuestionManager questionManager = new PSSQuestionManager();
		List<PSSQuestion> returnQuestions = new ArrayList<PSSQuestion>();
		QueryAndParams query = new QueryAndParams();

		StringBuilder qString = new StringBuilder();

		qString
				.append("select sq.Question_Number, sq.Question_Text, sq.Question_Text_Short, sc.Choice_Number, sc.Choice_Text ");
		qString.append("from Survey_Question as sq ");
		qString
				.append("inner join Survey_Choice as sc on sc.Survey_Question_ID = sq.Survey_Question_ID ");
		qString.append("where Survey_ID = :surveyId ");
		qString.append("order by sq.Question_Number");

		query.append(qString.toString());
		query.addParam("surveyId", version);

		try {
			List<?> report = executeSQLQuery(query);
			Map<Integer, PSSQuestion> questions = new HashMap<Integer, PSSQuestion>();
			for (Object object : report) {
				Object[] record = (Object[]) object;

				Integer questionNumber = (Integer) record[0];
				String questionText = (String) record[1];
				String questionTextShort = (String) record[2];
				Integer choiceNumber = (Integer) record[3];
				String choiceText = (String) record[4];

				if (questions.containsKey(questionNumber)) {
					PSSQuestion currentQuestion = questions
							.remove(questionNumber);
					PSSQuestionChoice choice = new PSSQuestionChoice();
					choice.setChoiceNumber(choiceNumber);
					choice.setChoiceText(choiceText);
					currentQuestion.addChoice(choice);
					questions.put(questionNumber, currentQuestion);
				} else {
					PSSQuestion newPSSQuestion = new PSSQuestion();
					newPSSQuestion.setQuestionNumber(questionNumber);

					if (questionTextShort == null
							|| questionTextShort.length() < 1) {
						newPSSQuestion.setQuestionText(questionText);
					} else {
						newPSSQuestion.setQuestionText(questionTextShort);
					}

					PSSQuestionChoice choice = new PSSQuestionChoice();
					choice.setChoiceNumber(choiceNumber);
					choice.setChoiceText(choiceText);
					newPSSQuestion.addChoice(choice);

					questions.put(questionNumber, newPSSQuestion);
				}
			}

			returnQuestions.addAll(questions.values());
		} catch (Exception e) {
			throw new DAOException("getQuestionsForReportVersion failed " + e);
		}
		questionManager.setVersion(version);
		questionManager.setQuestionList(returnQuestions);

		return questionManager;
	}
}
