/********************************************************************
 * Copyright � 2010 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.ccht.persistent.hibernate;

import gov.va.med.ccht.model.SimpleUser;
import gov.va.med.ccht.model.User;
import gov.va.med.ccht.model.UserRole;
import gov.va.med.ccht.model.inventory.SimpleFacility;
import gov.va.med.ccht.model.inventory.SimpleVisn;
import gov.va.med.ccht.model.inventory.Vendor;
import gov.va.med.ccht.model.terminology.FederalHoliday;
import gov.va.med.ccht.model.terminology.RegistrationStatus;
import gov.va.med.ccht.persistent.SecurityDAO;
import gov.va.med.fw.model.EntityKey;
import gov.va.med.fw.model.UserPrincipalImpl;
import gov.va.med.fw.model.ldap.SearchCriteria;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.fw.persistent.hibernate.GenericDAOImpl;
import gov.va.med.fw.persistent.hibernate.QueryAndParams;
import gov.va.med.fw.security.Application;
import gov.va.med.fw.security.Permission;
import gov.va.med.fw.security.Role;
import gov.va.med.fw.security.SimpleRole;
import gov.va.med.fw.security.UserPrincipal;
import gov.va.med.fw.util.StringUtils;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.Validate;

/**
 * 
 * 
 * @author vhaisakatikm
 */
public class SecurityDAOImpl extends GenericDAOImpl implements SecurityDAO {

	private static final long serialVersionUID = 4812681291498401077L;

	/**
	 * Get user by name
	 * 
	 * @param userId
	 * @return
	 * @throws DAOException
	 */
	@SuppressWarnings("unchecked")
	public User getUserByName(String userId) throws DAOException {
		try {
			String query = "select l from " + User.class.getName()
					+ " l where upper(l.userCredentials.userID) = :userId";
			List<User> list = getJpaTemplate().findByNamedParams(query,
					Collections.singletonMap("userId", userId.toUpperCase()));
			if (list != null && list.size() > 0) {
				return list.get(0);
			}
			return null;
		} catch (Exception e) {
			throw new DAOException("getUserByName failed", e);
		}
	}

	/**
	 * Get user by primary key
	 * 
	 * @param entityKey
	 * @return
	 * @throws DAOException
	 */
	public User getUserByKey(EntityKey<User> entityKey) throws DAOException {
		try {
			return super.getByKey(entityKey);
		} catch (Exception e) {
			throw new DAOException("getUserByKey failed", e);
		}
	}
	
	/**
	 * Delete a role
	 * 
	 * @param identifier
	 * @return
	 * @throws DAOException
	 */
	public void deleteRole(EntityKey<UserRole> identifier) throws DAOException {
		Validate.notNull(identifier, "Role identifier can not be null.");
		Validate.notNull(identifier.getKeyValue(), "Role identifier key can not be null.");
		removeObject(identifier);
	}

	@SuppressWarnings("unchecked")
	public List<User> getSubmittedRegistrations(Long visnId, Long facilityId) throws DAOException {
		try {
			Map<String, Object> params = new HashMap<String, Object>();
			params.put("registrationStatus", RegistrationStatus.APPROVED);	
			
			String query = "select l from " + User.class.getName() + " l where l.registrationStatus.code != :registrationStatus";
			if (visnId != null) {
				query += " and l.visn.id = :visnId";
				params.put("visnId", visnId);
			}else if (facilityId != null) {
				query += " and l.facility.id = :facilityId";
				params.put("facilityId", facilityId);
			}
			return getJpaTemplate().findByNamedParams(query, params);
		} catch (Exception e) {
			throw new DAOException("getSubmittedRegistrations failed: ", e);
		}
	}

	
	@SuppressWarnings("unchecked")
	public List<User> getSubmittedRegistrationsForDmp(Long visnId, Long facilityId) throws DAOException {
		try {
			Map<String, Object> params = new HashMap<String, Object>();
			params.put("registrationStatus", RegistrationStatus.APPROVED);	
			
			String query = "select l from " + User.class.getName() + " l where l.registrationStatus.code != :registrationStatus and (l.registrationReason.name like '%DMP%' or l.registrationReason.name = 'Training Center Staff')";
			if (visnId != null) {
				query += " and l.visn.id = :visnId";
				params.put("visnId", visnId);
			}else if (facilityId != null) {
				query += " and l.facility.id = :facilityId";
				params.put("facilityId", facilityId);
			}
			return getJpaTemplate().findByNamedParams(query, params);
		} catch (Exception e) {
			throw new DAOException("getSubmittedRegistrationsForDmp failed: ", e);
		}
	}
	
	/**
	 * Create user in the IHTA database
	 * 
	 * @param user
	 * @throws DAOException
	 */
	public void createUser(User user) throws DAOException {
		try {
			super.persist(user);
		} catch (Exception e) {
			throw new DAOException("createUser failed", e);
		}
	}

	/**
	 * Update user in the IHTA database
	 * 
	 * @param user
	 * @throws DAOException
	 */
	public void updateUser(User user) throws DAOException {
		try {
			super.saveObject(user, user);
		} catch (Exception e) {
			throw new DAOException("updateUser failed", e);
		}
	}

	/**
	 * Update role in the IHTA database
	 * 
	 * @param user
	 * @throws DAOException
	 */
	public void updateRole(Role role) throws DAOException {
		try {
			super.saveObject(role, role);
		} catch (Exception e) {
			throw new DAOException("updateUser failed", e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.IHTA.persistent.SecurityDAO#findAllRoles()
	 */
	@SuppressWarnings("unchecked")
	public List<Role> findAllRoles(Application application) throws DAOException {
		try {
			String query = "select l from " + Role.class.getName() + " l";
			Map<String, Object> params = new HashMap<String, Object>();
			if (application != null) {
				query = query + " where l.application.applicationName = :applicationName";
				params.put("applicationName", application.getApplicationName());
			}
			return getJpaTemplate().findByNamedParams(query, params);
		} catch (Exception e) {
			throw new DAOException("findAllRoles failed: ", e);
		}
	}

	@SuppressWarnings("unchecked")
	public List<SimpleRole> findAllSimpleRoles() throws DAOException {
		try {
			String query = "select l from " + SimpleRole.class.getName() + " l where l.id != 2020 order by l.name";
			Map<String, Object> params = new HashMap<String, Object>();
			return getJpaTemplate().findByNamedParams(query, params);
		} catch (Exception e) {
			throw new DAOException("findAllRoles failed: ", e);
		}
	}

	@SuppressWarnings("unchecked")
	public List<SimpleRole> findAllDMPRoles() throws DAOException {
		try {
			String query = "select l from " + SimpleRole.class.getName() + " l where l.name like '%DMP%' order by l.name";
			Map<String, Object> params = new HashMap<String, Object>();
			return getJpaTemplate().findByNamedParams(query, params);
		} catch (Exception e) {
			throw new DAOException("findAllDMPRoles failed: ", e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.IHTA.persistent.SecurityDAO#findAllPermissions(gov.va.med.
	 * fw.security.Application)
	 */
	@SuppressWarnings("unchecked")
	public List<Permission> findAllPermissions(Application application) throws DAOException {
		try {
			String query = "select l from " + Permission.class.getName() + " l";
			Map<String, Object> params = new HashMap<String, Object>();
			if (application != null) {
				query = query + " where l.application.applicationName = :applicationName";
				params.put("applicationName", application.getApplicationName());
			}
			return getJpaTemplate().findByNamedParams(query, params);
		} catch (Exception e) {
			throw new DAOException("findAllPermissions failed: ", e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.IHTA.persistent.SecurityDAO#getRoleByName(java.lang.String)
	 */
	@SuppressWarnings("unchecked")
	public Role getRoleByName(String roleName) throws DAOException {
		try {
			String query = "select l from " + Role.class.getName()
					+ " l where UPPER(l.name) = :roleName";
			List<Role> list = getJpaTemplate().findByNamedParams(query,
					Collections.singletonMap("roleName", roleName.toUpperCase()));
			if (list != null && list.size() > 0)
				return list.get(0);
			return null;
		} catch (Exception e) {
			throw new DAOException("getRoleByName Failed ", e);
		}
	}

	@SuppressWarnings("unchecked")
	public SimpleRole getSimpleRoleByName(String roleName) throws DAOException {
		try {
			String query = "select l from " + SimpleRole.class.getName()
					+ " l where l.name = :roleName";
			List<SimpleRole> list = getJpaTemplate().findByNamedParams(query,
					Collections.singletonMap("roleName", roleName));
			if (list != null && list.size() > 0)
				return list.get(0);
			return null;
		} catch (Exception e) {
			throw new DAOException("getSimpleRoleByName Failed ", e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.IHTA.persistent.SecurityDAO#getAvailableRoleNames()
	 */
	@SuppressWarnings("unchecked")
	public List<String> getAvailableRoleNames(Application application) throws DAOException {
		try {
			String query = "select l.name from " + Role.class.getName() + " l ";
			Map<String, Object> params = new HashMap<String, Object>();
			if (application != null) {
				query = query + " where l.application.applicationName = :applicationName";
				params.put("applicationName", application.getApplicationName());
			}
			return getJpaTemplate().findByNamedParams(query, params);
		} catch (Exception e) {
			throw new DAOException("getAvailableRoleNames failed", e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.IHTA.persistent.SecurityDAO#getRolePermissions(java.lang.String
	 * )
	 */
	@SuppressWarnings("unchecked")
	public List<String> getRolePermissionNames(String roleName) throws DAOException {
		try {
			String query = "select r.internalPermissions.permission.name from "
					+ Role.class.getName() + " r where r.name = :roleName";
			return getJpaTemplate().findByNamedParams(query,
					Collections.singletonMap("roleName", roleName));
		} catch (Exception e) {
			throw new DAOException("getRolePermissionNames failed", e);
		}
	}

	@SuppressWarnings("unchecked")
	public UserPrincipal getAuditUser(String userName) throws DAOException {
		Validate.notNull(userName);
		try {
			QueryAndParams queryAndParams = new QueryAndParams();
			queryAndParams
					.append("select USER_NAME, FIRST_NAME, MIDDLE_NAME, LAST_NAME");
			queryAndParams.append(" from ht.app_user");
			queryAndParams.append(" where upper(USER_NAME) = :userName");
			queryAndParams.addParam("userName", userName.toUpperCase());

			List<Object[]> results = (List<Object[]>) executeSQLQuery(queryAndParams);

			if (results != null) {
				for (Object[] result : results) {
					UserPrincipalImpl user = new UserPrincipalImpl((String) result[0]);
					user.setGivenName((String) result[1]);
					user.setMiddleName((String) result[2]);
					user.setFamilyName((String) result[3]);
					return user;
				}
			}
			return new UserPrincipalImpl(userName);

		} catch (Exception e) {
			throw new DAOException("getAuditUser failed", e);
		}
	}

	public Application getApplicationByName(String applicationName)
			throws DAOException {
		// TODO Auto-generated method stub
		return null;
	}

	@SuppressWarnings("unchecked")
	public List<FederalHoliday> findHolidays(int year) throws DAOException {
		
		try {
			String query = "select h from " + FederalHoliday.class.getName() + " h where year(h.activeDate) = :year";
			Map<String,Object> params = new HashMap<String,Object>();
			params.put("year", Integer.valueOf(year));
			return getJpaTemplate().findByNamedParams(query, params);
		} catch (Exception e) {
			throw new DAOException("findHolidays failed: ", e);
		}
	}

	@SuppressWarnings("unchecked")
	public List<String> getUserIds() throws DAOException {
		try {
			QueryAndParams queryAndParams = new QueryAndParams();
			queryAndParams.append("select USER_NAME from ht.app_user");

			return (List<String>) executeSQLQuery(queryAndParams);
			
		} catch (Exception e) {
			throw new DAOException("getAuditUser failed", e);
		}
	}

	@SuppressWarnings("unchecked")
	public List<SimpleUser> findAppUsers(SearchCriteria searchCriteria) throws DAOException {
		try {
			Map<String,Object> map = new HashMap<String,Object>();
			
			//user name search
			if (StringUtils.isNotEmpty(searchCriteria.getSAMAccountName())){
				String query = "select l from " + SimpleUser.class.getName()
						+ " l where upper(l.userCredentials.userID) = :userId";
				map.put("userId", searchCriteria.getSAMAccountName());
					
				return getJpaTemplate().findByNamedParams(query,map);							
			}//Last name and first name search
			else if (StringUtils.isNotEmpty(searchCriteria.getLastName())){				
				map.put("familyName", searchCriteria.getLastName());
				String query = "select l from " + SimpleUser.class.getName()
					+ " l where l.familyName = :familyName";
				if (StringUtils.isNotEmpty(searchCriteria.getFirstName())) {
					query = query + " and l.givenName like :givenName";
					map.put("givenName", searchCriteria.getFirstName());
				}
				//return the result set				
				return getJpaTemplate().findByNamedParams(query, map);	
			}//Role and VISN or facility search 
			else if (searchCriteria.getRoles().size() > 0) {	
				map.put("roles", searchCriteria.getRoles());
				String query = "select l from " + SimpleUser.class.getName()
				+ " l where l.id in (select distinct ur.user.id from " + UserRole.class.getName()
				+ " ur where ur.role.name in (:roles))";
				if (searchCriteria.getVisns() != null && searchCriteria.getVisns().size() > 0) {
					query += " and l.visn.id in (select v.id from " + SimpleVisn.class.getName() +
					" v where v.name in (:visns))";
					map.put("visns", searchCriteria.getVisns());
				}
				if (searchCriteria.getStations() != null && searchCriteria.getStations().size() > 0) {
					query += " and l.facility.id in (select f.id from " + SimpleFacility.class.getName() +
					" f where f.stationNumber in (:stations))";
					map.put("stations", searchCriteria.getStations());
				}					
				if (searchCriteria.getVendors() != null && searchCriteria.getVendors().size() > 0) {
					query += " and l.vendor.id in (select ven.id from " + Vendor.class.getName() +
					" ven where ven.name in (:vendors))";
					map.put("vendors", searchCriteria.getVendors());
				}
				return getJpaTemplate().findByNamedParams(query,map);
			} else {
				throw new Exception("findAppUsers failed: Invalid Search criteria");
			}
		}catch (Exception e){
			throw new DAOException("findAppUsers failed", e);
		}
	}
}
