package gov.va.med.ccht.persistent.hibernate;

import gov.va.med.ccht.model.dmp.PatientSatSurveyReportParameters;
import gov.va.med.ccht.model.dmpreport.PSSQuestion;
import gov.va.med.ccht.model.dmpreport.PSSQuestionChoice;
import gov.va.med.ccht.model.dmpreport.PSSQuestionManager;
import gov.va.med.ccht.model.dmpreport.SatisfactionSurveyQuestion;
import gov.va.med.ccht.model.dmpreport.SatisfactionSurveyQuestionResult;
import gov.va.med.ccht.persistent.SatisfactionSurveyReportDAO;
import gov.va.med.ccht.persistent.utils.PSSReportUtils;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.fw.persistent.hibernate.GenericDAOImpl;
import gov.va.med.fw.persistent.hibernate.QueryAndParams;
import gov.va.med.fw.util.DateUtils;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class SatisfactionSurveyReportDAOImpl extends GenericDAOImpl implements
		SatisfactionSurveyReportDAO {

	private String selectedModality = null;

	@Override
	public List<SatisfactionSurveyQuestionResult> getSatisfactionSurveyReport(
			PatientSatSurveyReportParameters reportParams) throws DAOException {

		List<SatisfactionSurveyQuestionResult> satisfactionSurveyQuestionResults = null;
		QueryAndParams queryAndParams = null;
		selectedModality = reportParams.getModality();
		switch (reportParams.getSelectedReportType()) {
		case 0:
			queryAndParams = getNationalSatisfactionSurveyReport(reportParams);
			break;
		case 1:
			queryAndParams = getSatisfactionSurveyReportByVisn(reportParams);
			break;
		case 2:
			queryAndParams = getSatisfactionSurveyReportByFacilities(reportParams);
			break;
		}

		List<?> report = executeSQLQuery(queryAndParams);

		int numIterations = report.size();

		satisfactionSurveyQuestionResults = new ArrayList<SatisfactionSurveyQuestionResult>();

		Map<Integer, SatisfactionSurveyQuestionResult> resultsByQuestion = new HashMap<Integer, SatisfactionSurveyQuestionResult>();
		for (int j = 0; j < numIterations; j++) {
			Object[] record = (Object[]) report.get(j);
			if (record != null) {
				Integer questionNumber = (Integer) record[0];
				String questionText = (String) record[1];
				int month = (Integer) record[2];
				int year = (Integer) record[3];
				Integer answerCount = (Integer) record[4];
				
				BigDecimal avgAnswerAsBD = (BigDecimal) record[5];
				double avgAnswer = 0;
				
				if(avgAnswerAsBD != null)
				{
					avgAnswer = avgAnswerAsBD.doubleValue();	
				}
				 
				String modality = (String) record[6];
				SatisfactionSurveyQuestion ssq = new SatisfactionSurveyQuestion();
				ssq.setQuestionNumber(questionNumber);
				ssq.setQuestionText(questionText);
				ssq.setModality(modality);
				ssq.setAvgAnswer(avgAnswer);
				ssq.setMonthAsInt(month);
				ssq.setYear(year);
				ssq.setAnswerCount(answerCount);
				ssq.setMonthAsString();
				ssq.setHeaderString(false);
				ssq.setTheDate();
				if (resultsByQuestion.containsKey(questionNumber)) {
					SatisfactionSurveyQuestionResult tempResult = resultsByQuestion
							.remove(questionNumber);
					tempResult.addToResultsKeyedByMonthMap(ssq);

					resultsByQuestion.put(questionNumber, tempResult);
				} else {
					SatisfactionSurveyQuestionResult newResult = new SatisfactionSurveyQuestionResult();
					newResult.setQuestionNumber(questionNumber);
					newResult.setQuestionText(questionText);
					newResult.setModality(reportParams.getModality());

					newResult.addToResultsKeyedByMonthMap(ssq);
					resultsByQuestion.put(questionNumber, newResult);
				}
			}

		}

		satisfactionSurveyQuestionResults.addAll(resultsByQuestion.values());

		if (numIterations > 0) {
			satisfactionSurveyQuestionResults.get(0).sortResults();
		}

		return satisfactionSurveyQuestionResults;

	}

	private QueryAndParams getNationalSatisfactionSurveyReport(
			PatientSatSurveyReportParameters reportParams) throws DAOException {
		StringBuilder qString = new StringBuilder();
		List<SatisfactionSurveyQuestionResult> satisfactionSurveyQuestionResults = null;
		QueryAndParams queryAndParams = new QueryAndParams(qString.toString());
		SimpleDateFormat dateFormat = new SimpleDateFormat(DateUtils.MMDDYYYYHHMMSS);

		// BUILD SELECT SECTION OF SQL
		qString.append(getCommonSelectSQL());
		qString.append(getAverageClause(reportParams));
		qString.append(PSSReportUtils.getModalityString(reportParams));

		// BUILD FROM SECTION OF SQL
		qString.append(getCommonFromSQL(reportParams));
		qString.append(PSSReportUtils.getModalityInnerJoinString(reportParams));

		// BUILD WHERE SECTION OF SQL
		qString.append(getCommonWhereSQL(reportParams));

		queryAndParams.addParam("fromDate", dateFormat.format(reportParams
				.getSubmittedFromDate()));
		queryAndParams.addParam("toDate", dateFormat.format(reportParams
				.getSubmittedToDate()));

		if (reportParams.getVendor() != null) {
			queryAndParams.addParam("vendorNumber", reportParams.getVendor()
					.getValue());
		}
		List<Integer> selectedQuestions = reportParams.getSelectedQuestions();

		queryAndParams.addParam("selectedQuestionList", selectedQuestions);

		String modalityInString = PSSReportUtils
				.getModalityInString(reportParams);
		qString.append(modalityInString);

		// TODO: IMPLEMENT METHOD FOR GETTING THE NEEDED MODALITY_TYPE ID'S
		if (selectedModality != null && selectedModality.length() > 0) {
			List modalityList = PSSReportUtils.getModalityIds(selectedModality);
			queryAndParams.addParam("modalityIdList", modalityList);
		}

		qString
				.append("GROUP BY DATEPART(yy, b.Completed_Date), DATEPART(mm, b.Completed_Date), ");
		qString.append(" c.Question_Number, c.Question_Text_Short ");
		qString.append(PSSReportUtils
				.getModalityStringForGroupByClause(reportParams));

		qString.append("ORDER BY Yr, Mon, c.Question_Number, Modality");

		queryAndParams.append(qString.toString());

		return queryAndParams;
	}

	private QueryAndParams getSatisfactionSurveyReportByVisn(
			PatientSatSurveyReportParameters reportParams) throws DAOException {
		StringBuilder qString = new StringBuilder();
		List<SatisfactionSurveyQuestionResult> satisfactionSurveyQuestionResults = null;
		QueryAndParams queryAndParams = new QueryAndParams(qString.toString());
		SimpleDateFormat dateFormat = new SimpleDateFormat(DateUtils.MMDDYYYYHHMMSS);

		// BUILD SELECT SECTION OF SQL
		qString.append(getCommonSelectSQL());
		qString.append(getAverageClause(reportParams));
		qString.append(PSSReportUtils.getModalityString(reportParams));

		// BUILD FROM SECTION OF SQL
		qString.append(getCommonFromSQL(reportParams));
		qString.append(PSSReportUtils.getModalityInnerJoinString(reportParams));

		// BUILD WHERE SECTION OF SQL
		qString.append(getCommonWhereSQL(reportParams));

		queryAndParams.addParam("fromDate", dateFormat.format(reportParams
				.getSubmittedFromDate()));
		queryAndParams.addParam("toDate", dateFormat.format(reportParams
				.getSubmittedToDate()));

		if (reportParams.getVendor() != null) {
			queryAndParams.addParam("vendorNumber", reportParams.getVendor()
					.getValue());
		}

		if (reportParams.getVisn().getLabel().equalsIgnoreCase("All Visns")) {
			qString.append(" AND d.VISN_ID >= 0");
		} else {
			qString.append("AND d.visn_id = :visnId ");
			queryAndParams
					.addParam("visnId", reportParams.getVisn().getValue());
		}

		List<Integer> selectedQuestions = reportParams.getSelectedQuestions();
		queryAndParams.addParam("selectedQuestionList", selectedQuestions);

		String modalityInString = PSSReportUtils
				.getModalityInString(reportParams);
		qString.append(modalityInString);

		// TODO: IMPLEMENT METHOD FOR GETTING THE NEEDED MODALITY_TYPE ID'S
		if (selectedModality != null && selectedModality.length() > 0) {
			List modalityList = PSSReportUtils.getModalityIds(selectedModality);
			queryAndParams.addParam("modalityIdList", modalityList);
		}

		qString
				.append("GROUP BY DATEPART(yy, b.Completed_Date), DATEPART(mm, b.Completed_Date), ");
		qString.append(" c.Question_Number, c.Question_Text_Short ");
		qString.append(PSSReportUtils
				.getModalityStringForGroupByClause(reportParams));

		qString.append("ORDER BY Yr, Mon, c.Question_Number, Modality");

		queryAndParams.append(qString.toString());

		return queryAndParams;

	}

	private QueryAndParams getSatisfactionSurveyReportByFacilities(
			PatientSatSurveyReportParameters reportParams) throws DAOException {
		StringBuilder qString = new StringBuilder();

		QueryAndParams queryAndParams = new QueryAndParams(qString.toString());
		SimpleDateFormat dateFormat = new SimpleDateFormat(DateUtils.MMDDYYYYHHMMSS);

		// BUILD SELECT SECTION OF SQL
		qString.append(getCommonSelectSQL());
		qString.append(getAverageClause(reportParams));
		qString.append(PSSReportUtils.getModalityString(reportParams));

		// BUILD FROM SECTION OF SQL
		qString.append(getCommonFromSQL(reportParams));
		qString.append(PSSReportUtils.getModalityInnerJoinString(reportParams));

		// BUILD WHERE SECTION OF SQL
		qString.append(getCommonWhereSQL(reportParams));

		queryAndParams.addParam("fromDate", dateFormat.format(reportParams
				.getSubmittedFromDate()));
		queryAndParams.addParam("toDate", dateFormat.format(reportParams
				.getSubmittedToDate()));

		if (reportParams.getVendor() != null) {
			queryAndParams.addParam("vendorNumber", reportParams.getVendor()
					.getValue());
		}

		if (reportParams.getFacility().getLabel().equalsIgnoreCase(
				"All Facilities")) {
			qString.append(" AND d.facility_id >= 0 ");
		} else {
			qString.append(" AND d.facility_id = :facilityId ");
			queryAndParams.addParam("facilityId", reportParams.getFacility()
					.getValue());
		}

		List<Integer> selectedQuestions = reportParams.getSelectedQuestions();
		queryAndParams.addParam("selectedQuestionList", selectedQuestions);

		String modalityInString = PSSReportUtils
				.getModalityInString(reportParams);
		qString.append(modalityInString);

		// TODO: IMPLEMENT METHOD FOR GETTING THE NEEDED MODALITY_TYPE ID'S
		if (selectedModality != null && selectedModality.length() > 0) {
			List modalityList = PSSReportUtils.getModalityIds(selectedModality);
			queryAndParams.addParam("modalityIdList", modalityList);
		}

		qString
				.append("GROUP BY DATEPART(yy, b.Completed_Date), DATEPART(mm, b.Completed_Date), ");
		qString.append(" c.Question_Number, c.Question_Text_Short ");
		qString.append(PSSReportUtils
				.getModalityStringForGroupByClause(reportParams));

		qString.append("ORDER BY Yr, Mon, c.Question_Number, Modality");

		queryAndParams.append(qString.toString());

		return queryAndParams;

	}

	private String getCommonSelectSQL() {
		StringBuilder sb = new StringBuilder();
		sb.append("SELECT c.Question_Number, c.Question_Text_Short, ");
		sb
				.append("DATEPART(mm, b.Completed_Date) AS Mon, DATEPART(yy, b.Completed_Date) AS Yr, COUNT(*) AS AnsCount, ");
		

		return sb.toString();
	}

	private String getCommonFromSQL(
			PatientSatSurveyReportParameters reportParams) {
		StringBuilder sb = new StringBuilder();
		sb.append("FROM Surveyed_Activity AS d ");
		sb
				.append("INNER JOIN Surveyed_Activity_Response AS b ON d.Surveyed_Activity_ID = b.Surveyed_Activity_ID ");
		sb
				.append("INNER JOIN Survey_Response AS a ON b.Surveyed_Activity_Response_ID = a.Surveyed_Activity_Response_ID ");
		sb
				.append("INNER JOIN Survey_Question AS c ON a.Survey_Question_ID = c.Survey_Question_ID ");
		sb.append("INNER JOIN Device ON d.Device_ID = Device.Device_ID ");

		if (reportParams.getVendor() != null) {
			sb.append(" INNER JOIN Vendor as v ON d.Vendor_ID = v.Vendor_ID ");
		}

		return sb.toString();
	}

	private String getCommonWhereSQL(
			PatientSatSurveyReportParameters reportParams) {
		StringBuilder sb = new StringBuilder();

		sb.append("WHERE ");

		// If both dates are not null append dates to the query
		if (reportParams.getSubmittedFromDate() != null
				&& reportParams.getSubmittedToDate() != null) {
			SimpleDateFormat dateFormat = new SimpleDateFormat(
					DateUtils.MMDDYYYY);
			sb.append("(b.Completed_Date between :fromDate AND :toDate) ");

		}

		sb.append(PSSReportUtils.getSurveyIdClause(reportParams));

		if (reportParams.getVendor() != null
				&& reportParams.getVendor().getValue() != null) {

			if (reportParams.getVendor().getLabel().equalsIgnoreCase(
					"All Vendors")) {

			} else {
				sb.append("AND v.Vendor_Number = :vendorNumber ");

			}
		}

		sb.append("AND c.question_number in (");
		sb.append(":selectedQuestionList");
		sb.append(") ");

		return sb.toString();
	}
	
	private String getAverageClause(PatientSatSurveyReportParameters reportParams)
	{
		StringBuilder qString = new StringBuilder();
		switch(reportParams.getPatientSatisfactionVersion())
		{
		case 2:
			qString
			.append(" AVG((CASE a.answer WHEN 6 THEN NULL ELSE a.answer END)/5.0) AS AvgAnswer, ");
			break;
		case 3:
			qString
			.append(" AVG((CASE a.answer WHEN 6 THEN NULL ELSE a.answer END)/4.0) AS AvgAnswer, ");
			break;
		}
		
		return qString.toString();
	}

	public PSSQuestionManager getQuestionsForReportVersion(Integer version)
			throws DAOException {
		PSSQuestionManager questionManager = new PSSQuestionManager();
		List<PSSQuestion> returnQuestions = new ArrayList<PSSQuestion>();
		QueryAndParams query = new QueryAndParams();

		StringBuilder qString = new StringBuilder();

		qString
				.append("select sq.Question_Number, sq.Question_Text, sq.Question_Text_Short, sc.Choice_Number, sc.Choice_Text ");
		qString.append("from Survey_Question as sq ");
		qString
				.append("inner join Survey_Choice as sc on sc.Survey_Question_ID = sq.Survey_Question_ID ");
		qString.append("where Survey_ID = :surveyId ");
		qString.append("order by sq.Question_Number");

		query.append(qString.toString());
		query.addParam("surveyId", version);

		try {
			List<?> report = executeSQLQuery(query);
			Map<Integer, PSSQuestion> questions = new HashMap<Integer, PSSQuestion>();
			for (Object object : report) {
				Object[] record = (Object[]) object;

				Integer questionNumber = (Integer) record[0];
				String questionText = (String) record[1];
				String questionTextShort = (String) record[2];
				Integer choiceNumber = (Integer) record[3];
				String choiceText = (String) record[4];

				if (questions.containsKey(questionNumber)) {
					PSSQuestion currentQuestion = questions
							.remove(questionNumber);
					PSSQuestionChoice choice = new PSSQuestionChoice();
					choice.setChoiceNumber(choiceNumber);
					choice.setChoiceText(choiceText);
					currentQuestion.addChoice(choice);
					questions.put(questionNumber, currentQuestion);
				} else {
					PSSQuestion newPSSQuestion = new PSSQuestion();
					newPSSQuestion.setQuestionNumber(questionNumber);

					if (questionTextShort == null
							|| questionTextShort.length() < 1) {
						newPSSQuestion.setQuestionText(questionText);
					} else {
						newPSSQuestion.setQuestionText(questionTextShort);
					}

					PSSQuestionChoice choice = new PSSQuestionChoice();
					choice.setChoiceNumber(choiceNumber);
					choice.setChoiceText(choiceText);
					newPSSQuestion.addChoice(choice);

					questions.put(questionNumber, newPSSQuestion);
				}
			}

			returnQuestions.addAll(questions.values());
		} catch (Exception e) {
			throw new DAOException("getQuestionsForReportVersion failed " + e);
		}
		questionManager.setVersion(version);
		questionManager.setQuestionList(returnQuestions);

		return questionManager;
	}

}
