/********************************************************************
 * Copyright � 2010 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.ccht.persistent.hibernate;

import gov.va.med.ccht.model.report.CompletedReport;
import gov.va.med.ccht.model.report.CompletedReportFieldType;
import gov.va.med.ccht.model.report.ScheduledReport;
import gov.va.med.ccht.model.report.SimpleCompletedReport;
import gov.va.med.ccht.model.terminology.StandardReport;
import gov.va.med.ccht.persistent.ReportDAO;
import gov.va.med.ccht.service.report.CompletedReportsSearchQueryInfo;
import gov.va.med.ccht.service.report.ReportFilterSearchQueryInfo;
import gov.va.med.fw.model.EntityKey;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.fw.persistent.MaxRecordsExceededException;
import gov.va.med.fw.persistent.hibernate.AbstractDAOAction;
import gov.va.med.fw.persistent.hibernate.GenericDAOImpl;
import gov.va.med.fw.report.ReportExportedType;
import gov.va.med.fw.security.UserPrincipal;
import gov.va.med.fw.util.StringUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.persistence.EntityManager;
import javax.persistence.Query;

import org.apache.commons.lang.Validate;

/**
 * 
 * 
 * IHTA IHTA_common Aug 8, 2010
 * 
 * @author vhaisakatikm
 */
public class ReportDAOImpl extends GenericDAOImpl implements ReportDAO {

	private static final long serialVersionUID = 5155804876579593142L;

	// These static final values match the names in hibernate mapping files.
	private static final String QUERY_GET_DISTINCT_STANDARD_REPORTS_FROM_COMPLETED_REPORTS = "standardReportQuery_GetDistinctStandardReportsFromCompletedReports";
	private static final String QUERY_GET_DISTINCT_EXPORT_TYPES_FROM_COMPLETED_REPORTS = "reportExportTypeQuery_GetDistinctExportTypesFromCompletedReports";
	private static final String USER_PARAM_NAME = "user";
	private static final String REPORT_IDS = "reportIds";
	private static final String REPORT_EXPIRING_DAYS = "expiringDays";
	private static final String REPORT_WARNIG_DAYS = "warningDays";

	/**
	 * A default constructor
	 */
	public ReportDAOImpl() {
		super();
	}

	public ScheduledReport getScheduledReport(EntityKey<ScheduledReport> identifier) throws DAOException {
		Validate.notNull(identifier, "ScheduledReport identifier can not be null.");
		Validate.notNull(identifier.getKeyValue(), "ScheduledReport identifier key can not be null.");

		try {
			ScheduledReport scheduledReport = super.getByKey(identifier);
			return scheduledReport;
		} catch (Exception e) {
			throw new DAOException("Failed to get ScheduledReport by id "
					+ identifier.getKeyValueAsString(), e);
		}
	}

	public ScheduledReport saveScheduledReport(ScheduledReport scheduledReport) throws DAOException {
		try {
			// CCR 407.  Avoid modifying CISS framework so changes are made 
			// in ReportDAOImpl to set an ID in the new entity
			
			Validate.notNull(scheduledReport, "ScheduledReport identifier can not be null.");
			ScheduledReport persisted = (ScheduledReport)super.getJpaTemplate().merge( scheduledReport );
			super.getJpaTemplate().flush();
			return persisted;
		}
		catch( Exception e ) {
			throw new DAOException("Failed to schedule a report", e);
		}
	}

	public void deleteScheduledReport(EntityKey<ScheduledReport> identifier) throws DAOException {
		Validate.notNull(identifier, "ScheduledReport identifier can not be null.");
		Validate.notNull(identifier.getKeyValue(), "ScheduledReport identifier key can not be null.");
		super.removeObject(identifier);
	}

	public CompletedReport getCompletedReport(EntityKey<CompletedReport> identifier)
			throws DAOException {
		Validate.notNull(identifier, "CompletedReport identifier can not be null.");
		Validate.notNull(identifier.getKeyValue(),
				"CompletedReport identifier key can not be null.");

		try {
			CompletedReport completedReport = super.getByKey(identifier);
			return completedReport;
		} catch (Exception e) {
			throw new DAOException("Failed to get CompletedReport by id "
					+ identifier.getKeyValueAsString(), e);
		}
	}

	/*
	 * @see
	 * gov.va.med.esr.common.persistent.report.ReportDAO#getSimpleCompletedReport
	 * (gov.va.med.fw.model.EntityKey)
	 */
	public SimpleCompletedReport getSimpleCompletedReport(
			EntityKey<SimpleCompletedReport> identifier) throws DAOException {
		Validate.notNull(identifier, "SimpleCompletedReport identifier can not be null.");
		Validate.notNull(identifier.getKeyValue(),
				"SimpleCompletedReport identifier key can not be null.");

		try {
			SimpleCompletedReport simpleCompletedReport = super.getByKey(identifier);
			return simpleCompletedReport;
		} catch (Exception e) {
			throw new DAOException("Failed to get SimpleCompletedReport by id "
					+ identifier.getKeyValueAsString(), e);
		}
	}

	public void saveCompletedReport(CompletedReport completedReport) throws DAOException {
		super.persist(completedReport);
	}

	public void saveSimpleCompletedReport(SimpleCompletedReport completedReport)
			throws DAOException {
		super.persist(completedReport);
	}

	/*
	 * @see
	 * gov.va.med.esr.common.persistent.report.ReportDAO#searchCompletedReports
	 * (gov.va.med.esr.service.CompletedReportsSearchQueryInfo)
	 */
	@SuppressWarnings("unchecked")
	public List<SimpleCompletedReport> searchCompletedReports(
			CompletedReportsSearchQueryInfo searchCriteriaInfo) throws DAOException,
			MaxRecordsExceededException {
		UserPrincipal user = searchCriteriaInfo.getUser();
		List<String> reportIds = searchCriteriaInfo.getReportIds();
		Map<String, Object> parms = new HashMap<String, Object>();
		if (user != null && StringUtils.isNotEmpty(user.getName())) {
			parms.put(USER_PARAM_NAME, user.getName());
			return getJpaTemplate().findByNamedQueryAndNamedParams("getCompletedReportsByUser",
					parms);
		} else if (reportIds != null && reportIds.size() > 0) {
			List<Long> ids = new ArrayList<Long>();
			for (String reportId : reportIds) {
				ids.add(new Long(reportId));
			}
			parms.put(REPORT_IDS, ids);
			return getJpaTemplate().findByNamedQueryAndNamedParams("getCompletedReportsByIds",
					parms);
		} else {
			return getJpaTemplate().findByNamedQuery("getCompletedReports");
		}
	}

	/*
	 * @see
	 * gov.va.med.esr.common.persistent.report.ReportDAO#searchScheduledReports
	 * (gov.va.med.esr.service.ScheduledReportsSearchQueryInfo)
	 */
	@SuppressWarnings("unchecked")
	public List<ScheduledReport> searchScheduledReports(ReportFilterSearchQueryInfo searchCriteriaInfo)
			throws DAOException, MaxRecordsExceededException {
		UserPrincipal user = searchCriteriaInfo.getUser();
		Map<String, Object> parms = new HashMap<String, Object>();
		if (user != null) {
			parms.put(USER_PARAM_NAME, user.getName());
			return getJpaTemplate().findByNamedQueryAndNamedParams("getScheduledReportsByUser",
					parms);
		} else {
			return getJpaTemplate().findByNamedQuery("getScheduledReports");
		}
	}

	/*
	 * @see
	 * gov.va.med.esr.common.persistent.report.ReportDAO#deleteCompletedReport
	 * (gov.va.med.fw.model.EntityKey)
	 */
	public void deleteCompletedReport(EntityKey<CompletedReport> identifier) throws DAOException {
		Validate.notNull(identifier, "CompletedReport identifier can not be null.");
		Validate.notNull(identifier.getKeyValue(),
				"CompletedReport identifier key can not be null.");
		super.removeObject(identifier);
	}

	/*
	 * @see
	 * gov.va.med.esr.common.persistent.report.ReportDAO#getDistinctStandardReports
	 * ()
	 */
	@SuppressWarnings("unchecked")
	public List<StandardReport> getDistinctStandardReports() throws DAOException {
		try {
			return getJpaTemplate().findByNamedQuery(
					QUERY_GET_DISTINCT_STANDARD_REPORTS_FROM_COMPLETED_REPORTS);
		} catch (Exception e) {
			throw new DAOException(
					"Failed to get distinct standard reports from complated reports", e);
		}
	}

	/*
	 * @seegov.va.med.esr.common.persistent.report.ReportDAO#
	 * getDistinctReportExportTypes()
	 */
	@SuppressWarnings("unchecked")
	public List<ReportExportedType> getDistinctReportExportTypes() throws DAOException {
		try {
			return getJpaTemplate().findByNamedQuery(
					QUERY_GET_DISTINCT_EXPORT_TYPES_FROM_COMPLETED_REPORTS);
		} catch (Exception e) {
			throw new DAOException(
					"Failed to get distinct report export types from complated reports", e);
		}
	}

	/**
	 * Purge alerts and completed reports which are older than ?(365) days
	 */
	public Integer purgeCompletedReports() throws DAOException {
		// get named query and execute
		try {
			AbstractDAOAction callback = new AbstractDAOAction() {
				@Override
				public Object execute(EntityManager entityManager) {

					// delete reports
					Query sqlQuery = entityManager.createNamedQuery("purgeCompletedReports");
					return new Integer(sqlQuery.executeUpdate());
				}
			};
			return (Integer) execute(callback);
		} catch (Exception e) {
			throw new DAOException("purgeCompletedReports failed", e);
		}
	}

	/**
	 * Get expiring reportid, user map to create alerts
	 */
	@SuppressWarnings("unchecked")
	public Map<String, String> getExpiringReportIdUserMap(Integer expiringDays, Integer warningDays)
			throws DAOException {
		try {
			Map<String, String> map = new HashMap<String, String>();
			// execute and get the results
			Map<String, Object> paramMap = new HashMap<String, Object>();
			paramMap.put(REPORT_EXPIRING_DAYS, expiringDays);
			paramMap.put(REPORT_WARNIG_DAYS, warningDays);
			List<Object[]> results = getJpaTemplate().findByNamedQueryAndNamedParams(
					"expiringReports", paramMap);
			for (Object[] result : results) {
				map.put((String) result[0], (String) result[1]);
			}
			return map;
		} catch (Exception e) {
			throw new DAOException("getExpiringReportIds failed " + e.getMessage(), e);
		}
	}

	@SuppressWarnings("unchecked")
	public LinkedHashMap<String, String> getUniqueFilterValues(CompletedReportFieldType fieldType) {
		LinkedHashMap<String, String> uniqueValMap = new LinkedHashMap<String, String>();
		List<Object[]> results = getJpaTemplate().findByNamedQuery(fieldType.getNamedQuery());
		for (Object[] item : results)
			uniqueValMap.put((String) item[0], (String) item[1]);
		return uniqueValMap;
	}
}
