package gov.va.med.ccht.persistent.hibernate;

import edu.emory.mathcs.backport.java.util.Collections;
import gov.va.med.ccht.model.dmp.PatientSatSurveyReportParameters;
import gov.va.med.ccht.model.dmpreport.FacilityNameComparator;
import gov.va.med.ccht.model.dmpreport.PSSQuestion;
import gov.va.med.ccht.model.dmpreport.PSSQuestionChoice;
import gov.va.med.ccht.model.dmpreport.PSSQuestionManager;
import gov.va.med.ccht.model.dmpreport.SatisfactionSurveyQuestion;
import gov.va.med.ccht.model.dmpreport.SatisfactionSurveyQuestionResult;
import gov.va.med.ccht.model.dmpreport.SatisfactionSurveyTrendEntry;
import gov.va.med.ccht.model.dmpreport.SurveyDistributionEntry;
import gov.va.med.ccht.model.dmpreport.SurveyDistributionSearchResult;
import gov.va.med.ccht.model.dmpreport.VRTwelveReportEntry;
import gov.va.med.ccht.model.dmpreport.VRTwelveReportResults;
import gov.va.med.ccht.model.dmpreport.VendorSubmissionReportEntry;
import gov.va.med.ccht.model.dmpreport.VisnIdComparator;
import gov.va.med.ccht.persistent.PSSReportsDAO;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.fw.persistent.hibernate.GenericDAOImpl;
import gov.va.med.fw.persistent.hibernate.QueryAndParams;
import gov.va.med.fw.util.DateUtils;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class PSSReportsDAOImpl extends GenericDAOImpl implements PSSReportsDAO
{

	@Override
	public List<VRTwelveReportResults> getVRTwelveReport(
			PatientSatSurveyReportParameters reportParams) throws DAOException
	{

		List<VRTwelveReportResults> returnResults = new ArrayList<VRTwelveReportResults>();
		try
		{
			switch(reportParams.getSelectedReportType())
			{
			case 0:
				// Record that comprises the overall data for the report
				VRTwelveReportResults vrTwelveReportResults = getOverAllVRTwelveAverage(reportParams);

				// This list is each individual month in the selected range
				List<VRTwelveReportEntry> returnRecords = getVRTwelveAveragesByMonth(reportParams);

				vrTwelveReportResults.setIndividualVRTwelveRecords(returnRecords);
				returnResults.add(vrTwelveReportResults);
				break;
			case 1:
				
				
				String visnValue = reportParams.getVisn().getValue();
				
				if(visnValue.equalsIgnoreCase("All VISNs"))
				{
					returnResults = getVRTwelveForAllVisns(reportParams);
				}
				else
				{
					// Record that comprises the overall data for the report
					VRTwelveReportResults visnReportResults = getOverAllVRTwelveAverage(reportParams);

					// This list is each individual month in the selected range
					List<VRTwelveReportEntry> visnReturnRecords = getVRTwelveAveragesByMonth(reportParams);

					visnReportResults.setIndividualVRTwelveRecords(visnReturnRecords);
					returnResults.add(visnReportResults);				
				}
				
				break;
			case 2:
				// Record that comprises the overall data for the report
				VRTwelveReportResults facilityReportResults = getOverAllVRTwelveAverage(reportParams);

				// This list is each individual month in the selected range
				List<VRTwelveReportEntry> facilityReturnRecords = getVRTwelveAveragesByMonth(reportParams);

				facilityReportResults.setIndividualVRTwelveRecords(facilityReturnRecords);
				returnResults.add(facilityReportResults);

				break;
			}
			return returnResults;
		} catch (Exception e)
		{
			throw new DAOException("getVRTwelveReport failed", e);
		}

	}
		
	private List<VRTwelveReportResults> getVRTwelveForAllVisns(PatientSatSurveyReportParameters reportParams) throws DAOException
	{
		String query = "select b.VISN_ID, Convert(decimal(38,3),IsNull(avg(a.Physical_Score),0)) as PhysicalScore, " +  
		 "Convert(decimal(38,3),IsNull(avg(a.Mental_Score),0)) as MentalScore, " + 
		"count(*) as NumSurveys " +
		"from Surveyed_Activity_Response a, Surveyed_Activity b " + 
		"where a.Surveyed_Activity_ID=b.Surveyed_Activity_ID  " +
		"and a.survey_id=1  " +
		"and a.physical_score is not null " + 
		"and a.mental_score is not null  " +
		"and a.Completed_Date >= :fromDate " + 
		"and a.Completed_Date <= :toDate  " +
		"and b.VISN_ID > 0 " +
		"group by b.VISN_ID ";
		
		QueryAndParams queryAndParams = new QueryAndParams();
		SimpleDateFormat dateFormat = new SimpleDateFormat(DateUtils.MMDDYYYY);

		if (reportParams.getSubmittedFromDate() != null
				&& reportParams.getSubmittedToDate() != null)
		{
		
			queryAndParams.addParam("fromDate", dateFormat.format(reportParams
					.getSubmittedFromDate()));
			queryAndParams.addParam("toDate", dateFormat.format(reportParams
					.getSubmittedToDate()));
		}
		
		queryAndParams.append(query);
		List<?> report = executeSQLQuery(queryAndParams);

		// This list is all the records returned by the query

		List<VRTwelveReportResults> returnResults = new ArrayList<VRTwelveReportResults>();
		VRTwelveReportResults reportResults = new VRTwelveReportResults();
		int numRecords = report.size();
		for (int j = 0; j < numRecords; j++)
		{
			Object[] record = (Object[]) report.get(j);
			if (record != null)
			{

				Integer visnId = (Integer)record[0];
				String visnName = "VISN " + visnId; 
				// Gather data for the current record
				Double currentRecordsPhysicalScore = ((BigDecimal) record[1])
						.doubleValue();
				Double currentRecordsMentalScore = ((BigDecimal) record[2])
						.doubleValue();
				Integer currentRecordsNumberOfSurveys = (Integer) record[3];
				reportResults = new VRTwelveReportResults();
				reportResults.setVisnId(visnId);
				reportResults.setVisnName(visnName);
				reportResults.setAvgPhysicalScore(currentRecordsPhysicalScore);
				reportResults.setAvgMentalScore(currentRecordsMentalScore);
				reportResults
						.setTotalNumberOfSurveys(currentRecordsNumberOfSurveys);
				
				
				List<VRTwelveReportEntry> visnReturnRecords = getVRTwelveAveragesByMonthAndVisn(reportParams, visnId);

								
				
				
				reportResults.setIndividualVRTwelveRecords(visnReturnRecords);
				returnResults.add(reportResults);
			}
		}

		return returnResults;

	}
		
	private List<VRTwelveReportEntry> getVRTwelveAveragesByMonth(
			PatientSatSurveyReportParameters reportParams) throws DAOException
	{
		String query = "select Convert(decimal(38,3),IsNull(avg(a.Physical_Score),0)) as PhysicalScore, "
				+ "Convert(decimal(38,3),IsNull(avg(a.Mental_Score),0)) as MentalScore, "
				+ "DATEPART(mm, a.Completed_Date) AS Mon, "
				+ "DATEPART(yy, a.Completed_Date) AS Yr, "
				+ "count(*) as NumSurveys "
				+ "from Surveyed_Activity_Response a, Surveyed_Activity b "
				+ "where a.Surveyed_Activity_ID = b.Surveyed_Activity_ID "
				+ "and a.survey_id = 1 "
				+ "and a.physical_score is not null "
				+ "and a.mental_score is not null ";

		QueryAndParams queryAndParams = new QueryAndParams();
		SimpleDateFormat dateFormat = new SimpleDateFormat(DateUtils.MMDDYYYY);

		if (reportParams.getSubmittedFromDate() != null
				&& reportParams.getSubmittedToDate() != null)
		{
			query += " AND a.Completed_Date >= :fromDate "
					+ " AND a.Completed_Date <= :toDate ";

			queryAndParams.addParam("fromDate", dateFormat.format(reportParams
					.getSubmittedFromDate()));
			queryAndParams.addParam("toDate", dateFormat.format(reportParams
					.getSubmittedToDate()));
		}

		if (reportParams.getSelectedReportType() == 1
				&& reportParams.getVisn() != null)
		{
			if (reportParams.getVisn().getLabel().equalsIgnoreCase("All Visns"))
			{
				query += " AND b.VISN_ID >= 0";
			} else
			{
				query += " AND b.VISN_ID = :visn";
				queryAndParams.addParam("visn", reportParams.getVisn()
						.getValue());
			}
		} else if (reportParams.getSelectedReportType() == 2
				&& reportParams.getFacility() != null
				&& reportParams.getFacility().getValue() != null)
		{

			if (reportParams.getFacility().getLabel().equalsIgnoreCase(
					"All Facilities"))
			{
				query += " AND b.FACILITY_ID >= 0";
			} else
			{
				query += " AND b.FACILITY_ID = :facility";
				queryAndParams.addParam("facility", reportParams.getFacility()
						.getValue());
			}

		}

		query += " GROUP BY DATEPART(yy, a.Completed_Date), DATEPART(mm, a.Completed_Date) ORDER BY Mon, Yr";

		queryAndParams.append(query);
		List<?> report = executeSQLQuery(queryAndParams);

		// This list is all the records returned by the query
		List<VRTwelveReportEntry> returnRecords = new ArrayList();

		// The totals of each column
		Double physicalScoreTotal = new Double(0);
		Double mentalScoreTotal = new Double(0);
		Integer totalNumberOfSurveys = new Integer(0);

		int numRecords = report.size();
		for (int j = 0; j < numRecords; j++)
		{
			Object[] record = (Object[]) report.get(j);
			if (record != null)
			{
				VRTwelveReportEntry currentRecord = new VRTwelveReportEntry();

				// Gather data for the current record
				Double currentRecordsPhysicalScore = ((BigDecimal) record[0])
						.doubleValue();
				Double currentRecordsMentalScore = ((BigDecimal) record[1])
						.doubleValue();
				Integer currentRecordsMonth = (Integer) record[2];
				Integer currentRecordsYear = (Integer) record[3];
				Integer currentRecordsNumberOfSurveys = (Integer) record[4];

				physicalScoreTotal += currentRecordsPhysicalScore;
				mentalScoreTotal += currentRecordsMentalScore;
				totalNumberOfSurveys += currentRecordsNumberOfSurveys;

				currentRecord.setPhysicalScore(currentRecordsPhysicalScore);
				currentRecord.setMentalScore(currentRecordsMentalScore);
				currentRecord.setTheDatesMonth(currentRecordsMonth);
				currentRecord.setTheDatesYear(currentRecordsYear);
				currentRecord.setNumberOfSurveys(currentRecordsNumberOfSurveys);

				returnRecords.add(currentRecord);
			}
		}

		return returnRecords;

	}
		
	private VRTwelveReportResults getOverAllVRTwelveAverage(
			PatientSatSurveyReportParameters reportParams) throws DAOException
	{
		String query = "select Convert(decimal(38,3),IsNull(avg(a.Physical_Score),0)) as PhysicalScore, "
				+ "Convert(decimal(38,3),IsNull(avg(a.Mental_Score),0)) as MentalScore, "
				+ "count(*) as NumSurveys "
				+ "from Surveyed_Activity_Response a, Surveyed_Activity b "
				+ "where a.Surveyed_Activity_ID = b.Surveyed_Activity_ID "
				+ "and a.survey_id = 1 "
				+ "and a.physical_score is not null "
				+ "and a.mental_score is not null ";

		QueryAndParams queryAndParams = new QueryAndParams();
		SimpleDateFormat dateFormat = new SimpleDateFormat(DateUtils.MMDDYYYY);

		if (reportParams.getSubmittedFromDate() != null
				&& reportParams.getSubmittedToDate() != null)
		{
			query += " AND a.Completed_Date >= :fromDate "
					+ " AND a.Completed_Date <= :toDate ";

			queryAndParams.addParam("fromDate", dateFormat.format(reportParams
					.getSubmittedFromDate()));
			queryAndParams.addParam("toDate", dateFormat.format(reportParams
					.getSubmittedToDate()));
		}

		if (reportParams.getSelectedReportType() == 1
				&& reportParams.getVisn() != null)
		{
			if (reportParams.getVisn().getLabel().equalsIgnoreCase("All Visns"))
			{
				query += " AND b.VISN_ID >= 0";
			} else
			{
				query += " AND b.VISN_ID = :visn";
				queryAndParams.addParam("visn", reportParams.getVisn()
						.getValue());
			}
		} else if (reportParams.getSelectedReportType() == 2
				&& reportParams.getFacility() != null
				&& reportParams.getFacility().getValue() != null)
		{

			if (reportParams.getFacility().getLabel().equalsIgnoreCase(
					"All Facilities"))
			{
				query += " AND b.FACILITY_ID >= 0";
			} else
			{
				query += " AND b.FACILITY_ID = :facility";
				queryAndParams.addParam("facility", reportParams.getFacility()
						.getValue());
			}

		}
		queryAndParams.append(query);
		List<?> report = executeSQLQuery(queryAndParams);

		// This list is all the records returned by the query

		// The totals of each column
		Double physicalScoreTotal = new Double(0);
		Double mentalScoreTotal = new Double(0);
		Integer totalNumberOfSurveys = new Integer(0);
		VRTwelveReportResults reportResults = new VRTwelveReportResults();
		int numRecords = report.size();
		for (int j = 0; j < numRecords; j++)
		{
			Object[] record = (Object[]) report.get(j);
			if (record != null)
			{

				// Gather data for the current record
				Double currentRecordsPhysicalScore = ((BigDecimal) record[0])
						.doubleValue();
				Double currentRecordsMentalScore = ((BigDecimal) record[1])
						.doubleValue();
				// Integer currentRecordsMonth = (Integer) record[2];
				// Integer currentRecordsYear = (Integer) record[3];
				Integer currentRecordsNumberOfSurveys = (Integer) record[2];
				reportResults.setAvgPhysicalScore(currentRecordsPhysicalScore);
				reportResults.setAvgMentalScore(currentRecordsMentalScore);
				reportResults
						.setTotalNumberOfSurveys(currentRecordsNumberOfSurveys);

			}
		}

		return reportResults;

	}

	private List<VRTwelveReportEntry> getVRTwelveAveragesByMonthAndVisn(
			PatientSatSurveyReportParameters reportParams, Integer visnId) throws DAOException
	{
		String query = "select Convert(decimal(38,3),IsNull(avg(a.Physical_Score),0)) as PhysicalScore, "
				+ "Convert(decimal(38,3),IsNull(avg(a.Mental_Score),0)) as MentalScore, "
				+ "DATEPART(mm, a.Completed_Date) AS Mon, "
				+ "DATEPART(yy, a.Completed_Date) AS Yr, "
				+ "count(*) as NumSurveys "
				+ "from Surveyed_Activity_Response a, Surveyed_Activity b "
				+ "where a.Surveyed_Activity_ID = b.Surveyed_Activity_ID "
				+ "and a.survey_id = 1 "
				+ "and a.physical_score is not null "
				+ "and a.mental_score is not null ";

		QueryAndParams queryAndParams = new QueryAndParams();
		SimpleDateFormat dateFormat = new SimpleDateFormat(DateUtils.MMDDYYYY);

		if (reportParams.getSubmittedFromDate() != null
				&& reportParams.getSubmittedToDate() != null)
		{
			query += " AND a.Completed_Date >= :fromDate "
					+ " AND a.Completed_Date <= :toDate ";

			queryAndParams.addParam("fromDate", dateFormat.format(reportParams
					.getSubmittedFromDate()));
			queryAndParams.addParam("toDate", dateFormat.format(reportParams
					.getSubmittedToDate()));
		}

		query += " AND b.VISN_ID = :visnId";
		queryAndParams.addParam("visnId", visnId);	
		query += " GROUP BY DATEPART(yy, a.Completed_Date), DATEPART(mm, a.Completed_Date) ORDER BY Mon, Yr";

		queryAndParams.append(query);
		List<?> report = executeSQLQuery(queryAndParams);

		// This list is all the records returned by the query
		List<VRTwelveReportEntry> returnRecords = new ArrayList();

		// The totals of each column
		Double physicalScoreTotal = new Double(0);
		Double mentalScoreTotal = new Double(0);
		Integer totalNumberOfSurveys = new Integer(0);

		int numRecords = report.size();
		for (int j = 0; j < numRecords; j++)
		{
			Object[] record = (Object[]) report.get(j);
			if (record != null)
			{
				VRTwelveReportEntry currentRecord = new VRTwelveReportEntry();

				// Gather data for the current record
				Double currentRecordsPhysicalScore = ((BigDecimal) record[0])
						.doubleValue();
				Double currentRecordsMentalScore = ((BigDecimal) record[1])
						.doubleValue();
				Integer currentRecordsMonth = (Integer) record[2];
				Integer currentRecordsYear = (Integer) record[3];
				Integer currentRecordsNumberOfSurveys = (Integer) record[4];

				physicalScoreTotal += currentRecordsPhysicalScore;
				mentalScoreTotal += currentRecordsMentalScore;
				totalNumberOfSurveys += currentRecordsNumberOfSurveys;

				currentRecord.setPhysicalScore(currentRecordsPhysicalScore);
				currentRecord.setMentalScore(currentRecordsMentalScore);
				currentRecord.setTheDatesMonth(currentRecordsMonth);
				currentRecord.setTheDatesYear(currentRecordsYear);
				currentRecord.setNumberOfSurveys(currentRecordsNumberOfSurveys);

				returnRecords.add(currentRecord);
			}
		}

		return returnRecords;

	}

	/*

	@Override
	public List<SatisfactionSurveyQuestionResult> getSatisfactionSurveyReport(
			PatientSatSurveyReportParameters reportParams) throws DAOException
	{

		List<SatisfactionSurveyQuestionResult> satisfactionSurveyQuestionResults = null;
		QueryAndParams queryAndParams = null;

		switch (reportParams.getSelectedReportType())
		{
		case 0:
			queryAndParams = getNationalSatisfactionSurveyReport(reportParams);
			break;
		case 1:
			queryAndParams = getSatisfactionSurveyReportByVisn(reportParams);
			break;
		case 2:
			queryAndParams = getSatisfactionSurveyReportByFacilities(reportParams);
			break;
		}

		List<?> report = executeSQLQuery(queryAndParams);

		int numIterations = report.size();

		satisfactionSurveyQuestionResults = new ArrayList<SatisfactionSurveyQuestionResult>();

		Map<Integer, SatisfactionSurveyQuestionResult> resultsByQuestion = new HashMap<Integer, SatisfactionSurveyQuestionResult>();
		for (int j = 0; j < numIterations; j++)
		{
			Object[] record = (Object[]) report.get(j);
			if (record != null)
			{
				Integer questionNumber = (Integer) record[0];
				String questionText = (String) record[1];
				String modality = (String) record[2];
				int month = (Integer) record[4];
				SatisfactionSurveyQuestion ssq = new SatisfactionSurveyQuestion();
				ssq.setQuestionNumber(questionNumber);
				ssq.setQuestionText(questionText);
				ssq.setModality(modality);
				ssq.setAvgAnswer(((BigDecimal) record[3]).doubleValue());
				ssq.setMonthAsInt(month);
				ssq.setYear((Integer) record[5]);
				ssq.setAnswerCount((Integer) record[6]);
				ssq.setMonthAsString();
				ssq.setHeaderString(false);
				ssq.setTheDate();
				if (resultsByQuestion.containsKey(questionNumber))
				{
					SatisfactionSurveyQuestionResult tempResult = resultsByQuestion
							.remove(questionNumber);
					tempResult.addToResultsKeyedByMonthMap(ssq);
					resultsByQuestion.put(questionNumber, tempResult);
				} else
				{
					SatisfactionSurveyQuestionResult newResult = new SatisfactionSurveyQuestionResult();
					newResult.setQuestionNumber(questionNumber);
					newResult.setQuestionText(questionText);
					newResult.setModality(reportParams.getModality());
					
					newResult.addToResultsKeyedByMonthMap(ssq);
					resultsByQuestion.put(questionNumber, newResult);
				}
			}

		}

		satisfactionSurveyQuestionResults.addAll(resultsByQuestion.values());

		return satisfactionSurveyQuestionResults;

	}

	private QueryAndParams getNationalSatisfactionSurveyReport(
			PatientSatSurveyReportParameters reportParams) throws DAOException
	{
		StringBuilder qString = new StringBuilder();
		List<SatisfactionSurveyQuestionResult> satisfactionSurveyQuestionResults = null;
		QueryAndParams queryAndParams = new QueryAndParams(qString.toString());
		SimpleDateFormat dateFormat = new SimpleDateFormat(DateUtils.MMDDYYYY);

		// BUILD SELECT SECTION OF SQL
		qString.append(getCommonSelectSQL());
		
		qString.append(getModalityString(reportParams));
		
		
		// BUILD FROM SECTION OF SQL
		qString.append(getCommonFromSQL(reportParams));		
		qString.append(getModalityInnerJoinString(reportParams));		
		
		// BUILD WHERE SECTION OF SQL
		qString.append(getCommonWhereSQL(reportParams));
	
		queryAndParams.addParam("fromDate", 
				dateFormat.format(reportParams.getSubmittedFromDate()));
		queryAndParams.addParam("toDate", 
				dateFormat.format(reportParams.getSubmittedToDate()));
		
		if(reportParams.getVendor() !=  null)
		{
			queryAndParams.addParam("vendorNumber", 
				reportParams.getVendor().getValue());
		}
		List<Integer> selectedQuestions = reportParams.getSelectedQuestions();

		queryAndParams.addParam("selectedQuestionList", selectedQuestions);

		String modalityInString = getModalityInString(reportParams);
		qString.append(modalityInString);
		
		if(modalityInString != null && modalityInString.length() > 0)
		{
			queryAndParams.addParam("modalityIdList", selectedQuestions);
		}
		
		qString.append("GROUP BY DATEPART(yy, b.Completed_Date), DATEPART(mm, b.Completed_Date), ");
		qString.append(" c.Question_Number, c.Question_Text_Short ");	
		qString.append(getModalityStringForGroupByClause(reportParams));
		
		qString.append("ORDER BY Yr, Mon, c.Question_Number, Modality");

		queryAndParams.append(qString.toString());

		return queryAndParams;
	}

	private QueryAndParams getSatisfactionSurveyReportByVisn(
			PatientSatSurveyReportParameters reportParams) throws DAOException
	{
		StringBuilder qString = new StringBuilder();
		List<SatisfactionSurveyQuestionResult> satisfactionSurveyQuestionResults = null;
		QueryAndParams queryAndParams = new QueryAndParams(qString.toString());
		SimpleDateFormat dateFormat = new SimpleDateFormat(DateUtils.MMDDYYYY);
		
		// BUILD SELECT SECTION OF SQL
		qString.append(getCommonSelectSQL());
		
		qString.append(getModalityString(reportParams));
		
		
		// BUILD FROM SECTION OF SQL
		qString.append(getCommonFromSQL(reportParams));		
		qString.append(getModalityInnerJoinString(reportParams));		
		
		// BUILD WHERE SECTION OF SQL
		qString.append(getCommonWhereSQL(reportParams));
	
		queryAndParams.addParam("fromDate", 
				dateFormat.format(reportParams.getSubmittedFromDate()));
		queryAndParams.addParam("toDate", 
				dateFormat.format(reportParams.getSubmittedToDate()));
		
		if(reportParams.getVendor() !=  null)
		{
			queryAndParams.addParam("vendorNumber", 
				reportParams.getVendor().getValue());
		}
		

		if (reportParams.getVisn().getLabel().equalsIgnoreCase("All Visns"))
		{
			qString.append(" AND d.VISN_ID >= 0");
		} else
		{
			qString.append("AND d.visn_id = :visnId ");
			queryAndParams
					.addParam("visnId", reportParams.getVisn().getValue());
		}

		List<Integer> selectedQuestions = reportParams.getSelectedQuestions();
		queryAndParams.addParam("selectedQuestionList", selectedQuestions);
		
		String modalityInString = getModalityInString(reportParams);
		qString.append(modalityInString);
		
		if(modalityInString != null && modalityInString.length() > 0)
		{
			queryAndParams.addParam("modalityIdList", selectedQuestions);
		}
		
		qString.append("GROUP BY DATEPART(yy, b.Completed_Date), DATEPART(mm, b.Completed_Date), ");
		qString.append(" c.Question_Number, c.Question_Text_Short ");	
		qString.append(getModalityStringForGroupByClause(reportParams));
		
		qString.append("ORDER BY Yr, Mon, c.Question_Number, Modality");

		queryAndParams.append(qString.toString());

		return queryAndParams;

	}

	private QueryAndParams getSatisfactionSurveyReportByFacilities(
			PatientSatSurveyReportParameters reportParams) throws DAOException
	{
		StringBuilder qString = new StringBuilder();

		QueryAndParams queryAndParams = new QueryAndParams(qString.toString());
		SimpleDateFormat dateFormat = new SimpleDateFormat(DateUtils.MMDDYYYY);
		
		// BUILD SELECT SECTION OF SQL
		qString.append(getCommonSelectSQL());
		
		qString.append(getModalityString(reportParams));
		
		
		// BUILD FROM SECTION OF SQL
		qString.append(getCommonFromSQL(reportParams));		
		qString.append(getModalityInnerJoinString(reportParams));		
		
		// BUILD WHERE SECTION OF SQL
		qString.append(getCommonWhereSQL(reportParams));
	
		queryAndParams.addParam("fromDate", 
				dateFormat.format(reportParams.getSubmittedFromDate()));
		queryAndParams.addParam("toDate", 
				dateFormat.format(reportParams.getSubmittedToDate()));
		
		if(reportParams.getVendor() !=  null)
		{
			queryAndParams.addParam("vendorNumber", 
				reportParams.getVendor().getValue());
		}
		

		if (reportParams.getFacility().getLabel().equalsIgnoreCase(
				"All Facilities"))
		{
			qString.append(" AND d.facility_id >= 0 ");
		} else
		{
			qString.append(" AND d.facility_id = :facilityId ");
			queryAndParams.addParam("facilityId", reportParams.getFacility()
					.getValue());
		}

		List<Integer> selectedQuestions = reportParams.getSelectedQuestions();
		queryAndParams.addParam("selectedQuestionList", selectedQuestions);
		
		String modalityInString = getModalityInString(reportParams);
		qString.append(modalityInString);
		
		if(modalityInString != null && modalityInString.length() > 0)
		{
			queryAndParams.addParam("modalityIdList", selectedQuestions);
		}
		
		qString.append("GROUP BY DATEPART(yy, b.Completed_Date), DATEPART(mm, b.Completed_Date), ");
		qString.append(" c.Question_Number, c.Question_Text_Short ");	
		qString.append(getModalityStringForGroupByClause(reportParams));
		
		qString.append("ORDER BY Yr, Mon, c.Question_Number, Modality");

		queryAndParams.append(qString.toString());

		return queryAndParams;

	}

*/

	private String getSurveyIdClause(
			PatientSatSurveyReportParameters reportParams)
	{
		StringBuilder qString = new StringBuilder();

		switch (reportParams.getPatientSatisfactionVersion())
		{
		case 1:
			qString.append("AND (b.Survey_ID = 1) ");
			break;
		case 2:
			qString.append("AND (b.Survey_ID = 2) ");
			break;
		case 3:
			qString.append("AND (b.Survey_ID = 3) ");
			break;
		}

		return qString.toString();
	}

	@Override
	public List<SatisfactionSurveyTrendEntry> getSatisfactionSurveyTrendReport(
			PatientSatSurveyReportParameters reportParams) throws DAOException
	{

		List<SatisfactionSurveyTrendEntry> returnRecords = new ArrayList<SatisfactionSurveyTrendEntry>();
		QueryAndParams queryAndParams = new QueryAndParams();

		switch (reportParams.getSelectedReportType())
		{
		case 0:
			queryAndParams = getSurveyTrendsQueryForNational(reportParams);
			break;
		case 1:
			queryAndParams = getSurveyTrendsQueryForVisn(reportParams);
			break;
		case 2:
			queryAndParams = getSurveyTrendsQueryForFacility(reportParams);
			break;
		}

		try
		{

			List<?> report = executeSQLQuery(queryAndParams);

			int numIterations = report.size();

			for (int j = 0; j < numIterations; j++)
			{
				Object[] record = (Object[]) report.get(j);
				if (record != null)
				{
					SatisfactionSurveyTrendEntry sste = new SatisfactionSurveyTrendEntry();
					sste.setMonthAsString((String) record[0]);
					sste.setMonthAsInt((Integer) record[1]);
					sste.setCount((Integer) record[2]);
					returnRecords.add(sste);
				}
			}

		} catch (Exception e)
		{
			throw new DAOException("getSatisfactionSurveyTrendReport failed "
					+ e);
		}

		return returnRecords;

	}

	private QueryAndParams getSurveyTrendsQueryForNational(
			PatientSatSurveyReportParameters reportParams)
	{
		QueryAndParams retVal = new QueryAndParams();
		StringBuilder query = new StringBuilder();
		String fromDate = null;
		String toDate = null;

		query.append("SELECT convert(varchar(3), received_date) as [Month], ");
		query.append("datepart(month, received_date) as [MonthNum], ");
		query.append("count(*) as [Count]  ");
		query.append("from Surveyed_Activity_Response a, surveyed_activity b ");
		query.append("where a.surveyed_activity_id= b.surveyed_activity_id ");
		query.append("and (a.survey_id in (1,2,3)) ");
		query.append("and a.survey_accepted_status=1 ");
		query.append("and received_date BETWEEN :fromDate AND :toDate ");
		query
				.append("group by convert(varchar(3), received_date),datepart(month, received_date) order by 2 ");

		if (reportParams.getSubmittedFromDate() != null
				&& reportParams.getSubmittedToDate() != null)
		{
			SimpleDateFormat dateFormat = new SimpleDateFormat(
					DateUtils.MMDDYYYY);

			fromDate = dateFormat.format(reportParams.getSubmittedFromDate())
					+ " 12:00:00 AM";
			toDate = dateFormat.format(reportParams.getSubmittedToDate())
					+ " 12:00:00 AM";
		}

		retVal.append(query.toString());
		retVal.addParam("fromDate", fromDate);
		retVal.addParam("toDate", toDate);

		return retVal;
	}

	private QueryAndParams getSurveyTrendsQueryForVisn(
			PatientSatSurveyReportParameters reportParams)
	{
		QueryAndParams retVal = new QueryAndParams();
		StringBuilder query = new StringBuilder();
		String fromDate = null;
		String toDate = null;

		query.append("SELECT convert(varchar(3), received_date) as [Month], ");
		query.append("datepart(month, received_date) as [MonthNum], ");
		query.append("count(*) as [Count]  ");
		query.append("from Surveyed_Activity_Response a, surveyed_activity b ");
		query.append("where a.surveyed_activity_id= b.surveyed_activity_id ");
		query.append("and (a.survey_id in (1,2,3)) ");
		query.append("and a.survey_accepted_status=1 ");
		query.append("and received_date BETWEEN :fromDate AND :toDate ");

		if (reportParams.getVisn().getLabel().equalsIgnoreCase("All VISNs"))
		{
			query.append(" AND b.VISN_ID >= '0'");
		} 
		else
		{
			query.append(" AND b.VISN_ID = :visnId ");
			retVal.addParam("visnId", reportParams.getVisn().getValue());
		}

		query
				.append("group by convert(varchar(3), received_date),datepart(month, received_date) order by 2");

		if (reportParams.getSubmittedFromDate() != null
				&& reportParams.getSubmittedToDate() != null)
		{
			SimpleDateFormat dateFormat = new SimpleDateFormat(
					DateUtils.MMDDYYYY);

			fromDate = dateFormat.format(reportParams.getSubmittedFromDate())
					+ " 12:00:00 AM";
			toDate = dateFormat.format(reportParams.getSubmittedToDate())
					+ " 12:00:00 AM";
		}

		retVal.append(query.toString());
		retVal.addParam("fromDate", fromDate);
		retVal.addParam("toDate", toDate);

		return retVal;
	}

	private QueryAndParams getSurveyTrendsQueryForFacility(
			PatientSatSurveyReportParameters reportParams)
	{
		QueryAndParams retVal = new QueryAndParams();
		StringBuilder query = new StringBuilder();
		String fromDate = null;
		String toDate = null;

		query.append("SELECT convert(varchar(3), received_date) as [Month], ");
		query.append("datepart(month, received_date) as [MonthNum], ");
		query.append("count(*) as [Count]  ");
		query
				.append("from Surveyed_Activity_Response a, surveyed_activity b, facilities c ");
		query.append("where a.surveyed_activity_id= b.surveyed_activity_id ");
		query.append(" and c.id = b.Facility_id ");

		if (reportParams.getFacility().getValue().equalsIgnoreCase(
				"All Facilities")
				|| reportParams.getFacility().getValue()
						.equalsIgnoreCase("All"))
		{
			query.append(" AND b.facility_ID >= 0");
			query.append("and (a.survey_id in (1,2,3)) ");
			query.append("and a.survey_accepted_status= 1 ");
			query.append("and received_date BETWEEN :fromDate AND :toDate ");
			query
					.append(" GROUP BY CONVERT(varchar(3), a.Received_Date) ,DATEPART(month, a.Received_Date) ORDER BY 2");
		} else
		{
			query.append(" AND b.facility_ID = :facilityId ");
			query.append("and (a.survey_id in (1,2,3)) ");
			query.append("and a.survey_accepted_status= 1 ");
			query.append("and received_date BETWEEN :fromDate AND :toDate ");
			query
					.append(" GROUP BY CONVERT(varchar(3), a.Received_Date) ,DATEPART(month, a.Received_Date), c.Facility_Name ORDER BY 2");

			retVal
					.addParam("facilityId", reportParams.getFacility()
							.getValue());
		}

		if (reportParams.getSubmittedFromDate() != null
				&& reportParams.getSubmittedToDate() != null)
		{
			SimpleDateFormat dateFormat = new SimpleDateFormat(
					DateUtils.MMDDYYYY);

			fromDate = dateFormat.format(reportParams.getSubmittedFromDate())
					+ " 12:00:00 AM";
			toDate = dateFormat.format(reportParams.getSubmittedToDate())
					+ " 12:00:00 AM";
		}

		retVal.append(query.toString());
		retVal.addParam("fromDate", fromDate);
		retVal.addParam("toDate", toDate);

		return retVal;
	}



	@Override
	public List<VendorSubmissionReportEntry> getVendorSubmissionReport(
			PatientSatSurveyReportParameters reportParams) throws DAOException
	{
		StringBuilder qString = new StringBuilder();
		qString.append("SELECT DISTINCT Vendor.Vendor_Name as [Vendor], ");
		qString
				.append("MAX(Surveyed_Activity_Response.Received_Date) AS [Last_Date_Received], ");
		qString
				.append("DATEDIFF(dd, CONVERT(varchar, MAX(Surveyed_Activity_Response.Received_Date), 101), ");
		qString.append("GETDATE()) AS [NumDays], ");
		qString.append("COUNT(*) AS SurveyCount, ");
		qString.append("Survey.Survey_ID, ");
		qString
				.append("CASE Survey.Survey_ID WHEN 1 THEN 'VR-12' WHEN 2 THEN 'PS Ver. 1.0' WHEN 3 THEN 'PS Ver. 2.0' ELSE '' END as [SurveyName] ");
		qString.append("FROM Vendor ");
		qString.append("INNER JOIN Surveyed_Activity ");
		qString
				.append("INNER JOIN Surveyed_Activity_Response ON Surveyed_Activity.Surveyed_Activity_ID = Surveyed_Activity_Response.Surveyed_Activity_ID ");
		qString.append("ON Vendor.Vendor_ID = Surveyed_Activity.Vendor_ID ");
		qString
				.append("INNER JOIN Survey ON Surveyed_Activity_Response.Survey_ID = Survey.Survey_ID ");
		qString.append("WHERE survey_accepted_status = 1 ");
		qString.append("GROUP BY Vendor.Vendor_Name, Survey.Survey_ID ");
		qString.append("ORDER BY Vendor.Vendor_Name, Survey.Survey_ID");

		List<VendorSubmissionReportEntry> retVal = new ArrayList<VendorSubmissionReportEntry>();
		Map<String, List<VendorSubmissionReportEntry>> temp = new HashMap<String, List<VendorSubmissionReportEntry>>();
		try
		{
			QueryAndParams queryAndParams = new QueryAndParams(qString
					.toString());
			List<?> report = executeSQLQuery(queryAndParams);

			int iteration = 0;

			for (Object object : report)
			{
				Object[] record = (Object[]) object;
				VendorSubmissionReportEntry currentRecord = new VendorSubmissionReportEntry();

				String vendorName = (String) record[0];
				currentRecord.setVendorName(vendorName);
				currentRecord.setDateOfLastSubmission((Date) record[1]);
				currentRecord
						.setNumDaysSinceLastSubmission((Integer) record[2]);

				int surveyId = (Integer) record[4];
				currentRecord.setSurveyId(surveyId);
				currentRecord.setSurveyName((String) record[5]);

				switch (surveyId)
				{
				case 1:
					currentRecord.setNumVrTwelveSurveys((Integer) record[3]);
					break;
				case 2:
					currentRecord.setNumVersionOneSurveys((Integer) record[3]);
					break;
				case 3:
					currentRecord.setNumVersionTwoSurveys((Integer) record[3]);
					break;
				}

				List<VendorSubmissionReportEntry> currentVendor = temp
						.remove(vendorName);

				if (currentVendor == null)
				{
					currentVendor = new ArrayList<VendorSubmissionReportEntry>();
				}

				currentVendor.add(currentRecord);

				temp.put(vendorName, currentVendor);
			}

			for (String key : temp.keySet())
			{
				List<VendorSubmissionReportEntry> currentEntries = temp
						.get(key);

				VendorSubmissionReportEntry newEntry = new VendorSubmissionReportEntry();
				int days = -1;
				Date lastDate = null; 
				
				for (VendorSubmissionReportEntry currentEntry : currentEntries)
				{
					int surId = currentEntry.getSurveyId();
										
					newEntry.setVendorName(currentEntry.getVendorName());
					//Find the last submission date for each vendor.
					if (days == -1
							|| days > currentEntry
									.getNumDaysSinceLastSubmission()
							|| (currentEntry
									.getDateOfLastSubmission().after(lastDate))) {
						newEntry.setDateOfLastSubmission(currentEntry
								.getDateOfLastSubmission());
						newEntry.setNumDaysSinceLastSubmission(currentEntry
								.getNumDaysSinceLastSubmission());
						
						days = currentEntry
								.getNumDaysSinceLastSubmission();
						lastDate = currentEntry.getDateOfLastSubmission();
					}
					
					
					switch (surId)
					{
					case 1:
						newEntry.setNumVrTwelveSurveys(currentEntry
								.getNumVrTwelveSurveys());
						break;
					case 2:
						newEntry.setNumVersionOneSurveys(currentEntry
								.getNumVersionOneSurveys());
						break;
					case 3:
						newEntry.setNumVersionTwoSurveys(currentEntry
								.getNumVersionTwoSurveys());
						break;
					}

				}

				retVal.add(newEntry);

			}
		} catch (Exception e)
		{
			throw new DAOException("getVendorSubmissionReport failed " + e);
		}

		return retVal;

	}

	public PSSQuestionManager getQuestionsForReportVersion(Integer version)
			throws DAOException
	{
		PSSQuestionManager questionManager = new PSSQuestionManager();
		List<PSSQuestion> returnQuestions = new ArrayList<PSSQuestion>();
		QueryAndParams query = new QueryAndParams();

		StringBuilder qString = new StringBuilder();

		qString
				.append("select sq.Question_Number, sq.Question_Text, sq.Question_Text_Short, sc.Choice_Number, sc.Choice_Text ");
		qString.append("from Survey_Question as sq ");
		qString
				.append("inner join Survey_Choice as sc on sc.Survey_Question_ID = sq.Survey_Question_ID ");
		qString.append("where Survey_ID = :surveyId ");
		qString.append("order by sq.Question_Number");

		query.append(qString.toString());
		query.addParam("surveyId", version);

		try
		{
			List<?> report = executeSQLQuery(query);
			Map<Integer, PSSQuestion> questions = new HashMap<Integer, PSSQuestion>();
			for (Object object : report)
			{
				Object[] record = (Object[]) object;

				Integer questionNumber = (Integer) record[0];
				String questionText = (String) record[1];
				String questionTextShort = (String) record[2];
				Integer choiceNumber = (Integer) record[3];
				String choiceText = (String) record[4];

				if (questions.containsKey(questionNumber))
				{
					PSSQuestion currentQuestion = questions
							.remove(questionNumber);
					PSSQuestionChoice choice = new PSSQuestionChoice();
					choice.setChoiceNumber(choiceNumber);
					choice.setChoiceText(choiceText);
					currentQuestion.addChoice(choice);
					questions.put(questionNumber, currentQuestion);
				} else
				{
					PSSQuestion newPSSQuestion = new PSSQuestion();
					newPSSQuestion.setQuestionNumber(questionNumber);

					if (questionTextShort == null
							|| questionTextShort.length() < 1)
					{
						newPSSQuestion.setQuestionText(questionText);
					} else
					{
						newPSSQuestion.setQuestionText(questionTextShort);
					}

					PSSQuestionChoice choice = new PSSQuestionChoice();
					choice.setChoiceNumber(choiceNumber);
					choice.setChoiceText(choiceText);
					newPSSQuestion.addChoice(choice);

					questions.put(questionNumber, newPSSQuestion);
				}
			}

			returnQuestions.addAll(questions.values());
		} catch (Exception e)
		{
			throw new DAOException("getQuestionsForReportVersion failed " + e);
		}
		questionManager.setVersion(version);
		questionManager.setQuestionList(returnQuestions);

		return questionManager;
	}

	private String getCommonSelectSQL()
	{
		StringBuilder sb = new StringBuilder();
		sb.append("SELECT c.Question_Number, c.Question_Text_Short, ");
		sb.append("DATEPART(mm, b.Completed_Date) AS Mon, DATEPART(yy, b.Completed_Date) AS Yr, COUNT(*) AS AnsCount, ");
		sb.append("AVG((CASE a.answer WHEN 6 THEN NULL ELSE a.answer END)/4.0) AS AvgAnswer, ");
				
		return sb.toString();
	}
	
	private String getCommonFromSQL(PatientSatSurveyReportParameters reportParams)
	{
		StringBuilder sb = new StringBuilder();
		sb.append("FROM Surveyed_Activity AS d ");
		sb.append("INNER JOIN Surveyed_Activity_Response AS b ON d.Surveyed_Activity_ID = b.Surveyed_Activity_ID ");
		sb.append("INNER JOIN Survey_Response AS a ON b.Surveyed_Activity_Response_ID = a.Surveyed_Activity_Response_ID ");
		sb.append("INNER JOIN Survey_Question AS c ON a.Survey_Question_ID = c.Survey_Question_ID ");
		sb.append("INNER JOIN Device ON d.Device_ID = Device.Device_ID ");
		
		if(reportParams.getVendor() != null)
		{
			sb.append(" INNER JOIN Vendor as v ON d.Vendor_ID = v.Vendor_ID ");
		}
		
		return sb.toString();
	}
	
	
	private String getCommonWhereSQL(PatientSatSurveyReportParameters reportParams)
	{
		StringBuilder sb = new StringBuilder();
		
		sb.append("WHERE ");

		// If both dates are not null append dates to the query
		if (reportParams.getSubmittedFromDate() != null
				&& reportParams.getSubmittedToDate() != null)
		{
			SimpleDateFormat dateFormat = new SimpleDateFormat(
					DateUtils.MMDDYYYY);
			sb.append("(b.Completed_Date between :fromDate AND :toDate) ");
			


		}
		
		sb.append(getSurveyIdClause(reportParams));

		if (reportParams.getVendor() != null
				&& reportParams.getVendor().getValue() != null)
		{

			if (reportParams.getVendor().getLabel().equalsIgnoreCase(
					"All Vendors"))
			{
				
			} else
			{
				sb.append("AND v.Vendor_Number = :vendorNumber ");
			
			}
		}
		
		sb.append("AND c.question_number in (");
		sb.append(":selectedQuestionList");
		sb.append(") ");
		
		return sb.toString();
	}

}
