package gov.va.med.ccht.persistent.hibernate;

import gov.va.med.ccht.model.inventory.DIReportParameters;
import gov.va.med.ccht.model.inventory.Device;
import gov.va.med.ccht.model.inventory.DeviceDetail;
import gov.va.med.ccht.model.inventory.DeviceRequirement;
import gov.va.med.ccht.model.inventory.DeviceSearchParameters;
import gov.va.med.ccht.model.inventory.DeviceStatusType;
import gov.va.med.ccht.model.inventory.DeviceTrackerLog;
import gov.va.med.ccht.model.inventory.DeviceType;
import gov.va.med.ccht.model.inventory.Facility;
import gov.va.med.ccht.model.inventory.FacilityActivity;
import gov.va.med.ccht.model.inventory.SimpleFacility;
import gov.va.med.ccht.model.inventory.SimpleVisn;
import gov.va.med.ccht.model.inventory.Vendor;
import gov.va.med.ccht.model.inventory.Visn;
import gov.va.med.ccht.model.report.ReportSetup;
import gov.va.med.ccht.persistent.InventoryDAO;
import gov.va.med.ccht.service.report.ReportConstants;
import gov.va.med.fw.model.lookup.AbstractLookup;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.fw.persistent.hibernate.GenericDAOImpl;
import gov.va.med.fw.persistent.hibernate.QueryAndParams;
import gov.va.med.fw.ui.model.TermType;
import gov.va.med.fw.util.DateUtils;
import gov.va.med.fw.util.StringUtils;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.springframework.dao.DataAccessException;

public class InventoryDAOImpl extends GenericDAOImpl implements InventoryDAO {

	private static final long serialVersionUID = -4423741640122878164L;

	@SuppressWarnings("unchecked")
	public List<Device> getDevices() throws DAOException {
		try {
			String query = "select l from " + Device.class.getName() + 
				" l where l.deviceName is not null and l.deviceName != '' order by l.deviceName asc";
			return getJpaTemplate().find(query);			 
		} catch (DataAccessException e) {
			throw new DAOException(e.getMessage(), e);
		}
	}

	@SuppressWarnings("unchecked")
	public List<Facility> getFacilities() throws DAOException {
		try {
			String query = "select l from " + Facility.class.getName() + " l order by l.name asc";
			return getJpaTemplate().find(query);			 
		} catch (DataAccessException e) {
			throw new DAOException(e.getMessage(), e);
		}
	}
	
	@SuppressWarnings("unchecked")
	public List<Facility> getFacilities(String visnName) throws DAOException {
		try {
			String query = "select l from " + Facility.class.getName() + " l where l.visn.name = :visnName order by l.name asc";
			Map<String,Object> params = new HashMap<String,Object>();
			params.put("visnName", visnName);			
			return (List<Facility>)getJpaTemplate().findByNamedParams(query, params);				 
		} catch (DataAccessException e) {
			throw new DAOException(e.getMessage(), e);
		}
	}

	@SuppressWarnings("unchecked")
	public List<Vendor> getVendors() throws DAOException {
		try {
			// HTCE_CodeCR471
			String query = "select l from " + Vendor.class.getName() + " l where l.number is not null and l.dateInactive is null and l.dateActive is not null order by l.name asc";
			return getJpaTemplate().find(query);			 
		} catch (DataAccessException e) {
			throw new DAOException(e.getMessage(), e);
		}
	}

	@SuppressWarnings("unchecked")
	public List<Visn> getVisns() throws DAOException {
		try {
			String query = "select l from " + Visn.class.getName() + " l order by l.id asc";
			return getJpaTemplate().find(query);			 
		} catch (DataAccessException e) {
			throw new DAOException(e.getMessage(), e);
		}
	}

	public Device getDevice(String name) throws DAOException {
		try {
			String query = "select l from " + Device.class.getName() + " l where l.deviceName =:name";
			Map<String,Object> params = new HashMap<String,Object>();
			params.put("name", name);			
			List<?> list = getJpaTemplate().findByNamedParams(query, params);
			return (Device) (list.size() > 0 ? list.get(0) : null);
		} catch (DataAccessException e) {
			throw new DAOException(e.getMessage(), e);
		}
	}

	public DeviceType getDeviceType(String code) throws DAOException {
		try {
			String query = "select l from " + DeviceType.class.getName() + " l where l.code =:code";
			Map<String,Object> params = new HashMap<String,Object>();
			params.put("code", code);			
			List<?> list = getJpaTemplate().findByNamedParams(query, params);	
			return (DeviceType) (list.size() > 0 ? list.get(0) : null);
		} catch (DataAccessException e) {
			throw new DAOException(e.getMessage(), e);
		}
	}
	
	public Facility getFacility(String stationNumber) throws DAOException {
		try {
			String query = "select l from " + Facility.class.getName() + " l where l.stationNumber = :stationNumber";
			Map<String,Object> params = new HashMap<String,Object>();
			params.put("stationNumber", stationNumber);			
			List<?> list = getJpaTemplate().findByNamedParams(query, params);
			return (Facility) (list.size() > 0 ? list.get(0) : null);
		} catch (DataAccessException e) {
			throw new DAOException(e.getMessage(), e);
		}
	}

	public Vendor getVendor(String name) throws DAOException {
		try {
			String query = "select l from " + Vendor.class.getName() + " l where l.name =:name";
			Map<String,Object> params = new HashMap<String,Object>();
			params.put("name", name);			
			List<?> list = getJpaTemplate().findByNamedParams(query, params);		
			return (Vendor) (list.size() > 0 ? list.get(0) : null);
		} catch (DataAccessException e) {
			throw new DAOException(e.getMessage(), e);
		}
	}

	public Visn getVisn(String name) throws DAOException {
		try {
			String query = "select l from " + Visn.class.getName() + " l where l.name =:name";
			Map<String,Object> params = new HashMap<String,Object>();
			params.put("name", name);			
			List<?> list = getJpaTemplate().findByNamedParams(query, params);
			return (Visn) (list.size() > 0 ? list.get(0) : null);
		} catch (DataAccessException e) {
			throw new DAOException(e.getMessage(), e);
		}
	}

	public SimpleVisn getSimpleVisn(String name) throws DAOException {
		try {
			String query = "select l from " + SimpleVisn.class.getName() + " l where l.name = :name";
			Map<String,Object> params = new HashMap<String,Object>();
			params.put("name", name);			
			List<?> list = getJpaTemplate().findByNamedParams(query, params);		
			return (SimpleVisn) (list.size() > 0 ? list.get(0) : null);
		} catch (DataAccessException e) {
			throw new DAOException(e.getMessage(), e);
		}
	}	
	
	public SimpleFacility getSimpleFacility(String stationNumber) throws DAOException {
		try {
			String query = "select l from " + SimpleFacility.class.getName() + " l where l.stationNumber =:stationNumber";
			Map<String,Object> params = new HashMap<String,Object>();
			params.put("stationNumber", stationNumber);			
			List<?> list = getJpaTemplate().findByNamedParams(query, params);
			return (SimpleFacility) (list.size() > 0 ? list.get(0) : null);
		} catch (DataAccessException e) {
			throw new DAOException(e.getMessage(), e);
		}
	}
	
	@SuppressWarnings("unchecked")
	public List<SimpleVisn> getSimpleVisns() throws DAOException {
		try {
			String query = "select l from " + SimpleVisn.class.getName() + " l order by l.id asc";
			return getJpaTemplate().find(query);			 
		} catch (DataAccessException e) {
			throw new DAOException(e.getMessage(), e);
		}
	}
	
	@SuppressWarnings("unchecked")
	public List<SimpleFacility> getSimpleFacilities() throws DAOException {
		try {
			String query = "select l from " + SimpleFacility.class.getName() + " l order by l.name asc";
			return getJpaTemplate().find(query);			 
		} catch (DataAccessException e) {
			throw new DAOException(e.getMessage(), e);
		}
	}
	
	@SuppressWarnings("unchecked")
	public List<SimpleFacility> getSimpleFacilities(SimpleVisn visn) throws DAOException
	{
		try {
			String query = "select l from " + SimpleFacility.class.getName() + 
				" l where l.visn.name = '" + visn.getName() + "' order by l.name asc";
			return getJpaTemplate().find(query);			 
		} catch (DataAccessException e) {
			throw new DAOException(e.getMessage(), e);
		}
	}
	public List<Object[]> generateInventoryReports(ReportSetup reportSetup) throws DAOException {
		DIReportParameters reportParameters = (DIReportParameters) reportSetup.getReportParameters();
		if (reportSetup.getStandardReport().getCode().equals("DI1")) {
			return generateInventoryReports(reportParameters);
		}else if (reportSetup.getStandardReport().getCode().equals("DI2")) {
			return generateDeviceStatusReports(reportParameters);
		}
		else if (reportSetup.getStandardReport().getCode().equals("DI3")) {
			return generateDeviceFailureReports(reportParameters);
		}
		else if (reportSetup.getStandardReport().getCode().equals("DI4")) {
			return generateDeviceShortageReports(reportParameters);
		}
		return null;
	}	
	
	@SuppressWarnings("unchecked")
	public DeviceDetail getDeviceDetail(DeviceSearchParameters deviceSearchParameters) throws DAOException
	{
		try {
			Map<String,Object> map = new HashMap<String,Object>();
			map.put("deviceType", deviceSearchParameters.getDeviceType().getValue());
			map.put("serialNumber", deviceSearchParameters.getSerialNumber());
			String query = "select d from " + DeviceDetail.class.getName() + 
				" d where d.deviceType.code = :deviceType and d.serialNumber = :serialNumber";
			List<DeviceDetail> dds = getJpaTemplate().findByNamedParams(query,map);
			if (dds != null && dds.size() > 0) {
				return dds.iterator().next();
			}else {
				return null;
			}
		} catch (Exception e) {
			throw new DAOException(e.getMessage(), e);
		}
	}
	
	/**
	 * 
	 * @param reportParameters
	 * @return
	 * @throws DAOException
	 */
	@SuppressWarnings("unchecked")
	private List<Object[]> generateDeviceShortageReports(DIReportParameters reportParameters) throws DAOException {
		try {
				//get the named query and execute with parameters
			Map<String,Object> parameters = new HashMap<String,Object>();
			String queryName = null;
			if (reportParameters.getReportSummaryType().equalsIgnoreCase(ReportConstants.REPORT_SUMMARY_TYPE_NATIONAL)) {
				//no parameters
				queryName = "di4_summary_national";
			} else if (reportParameters.getReportSummaryType().equalsIgnoreCase(ReportConstants.REPORT_SUMMARY_TYPE_VISN)) {
				if (reportParameters.getVisn() == null) {
					parameters.put("allVisn", "AllVisn");
					parameters.put("visnId", 0);
				}else {
					parameters.put("allVisn", "Visn");
					parameters.put("visnId", reportParameters.getVisn().getId());
				}
				queryName = "di4_summary_visn";
			} else if (reportParameters.getReportSummaryType().equalsIgnoreCase(ReportConstants.REPORT_SUMMARY_TYPE_FACILITY)) {
				if (reportParameters.getFacility() == null) {
					parameters.put("allFacilities", "AllFacilities");
					parameters.put("facilityId", 0);
				}else {
					parameters.put("allFacilities", "Facility");
					parameters.put("facilityId", reportParameters.getFacility().getId());
				}				
				queryName = "di4_summary_facility";
			}
			
			if (queryName != null) {
				return getJpaTemplate().findByNamedQueryAndNamedParams(queryName, parameters);
			}else {
				throw new DAOException("Unsupported DI4 Report configuration summary type " + reportParameters.getReportSummaryType());
			}
		} catch (DataAccessException e) {
			throw new DAOException(e.getMessage(), e);
		}
	}
	
	/**
	 * 
	 * @param reportParameters
	 * @return
	 * @throws DAOException
	 */
	@SuppressWarnings("unchecked")
	private List<Object[]> generateDeviceFailureReports(DIReportParameters reportParameters) throws DAOException {
		try {
				//get the named query and execute with parameters
			Map<String,Object> parameters = new HashMap<String,Object>();
			//Date Range (set default dates of not specified
			if ( reportParameters.getFromDate() != null) { 
				parameters.put("fromDate", reportParameters.getFromDate());
			}else {
				Calendar cal = Calendar.getInstance();
				cal.set(2000,0,1,0,0,0); //Olddest date for IHTA database
				parameters.put("fromDate", cal.getTime());
			}
			if (reportParameters.getToDate() != null) {
				parameters.put("toDate", reportParameters.getToDate());
			} else {
				parameters.put("toDate", DateUtils.getTomorrowDate());
			}
			
			String queryName = null;
			if (reportParameters.getReportSummaryType().equalsIgnoreCase(ReportConstants.REPORT_SUMMARY_TYPE_NATIONAL)) {
				//no parameters
				queryName = "di3_summary_national";
			} else if (reportParameters.getReportSummaryType().equalsIgnoreCase(ReportConstants.REPORT_SUMMARY_TYPE_VISN)) {
				if (reportParameters.getVisn() == null) {
					parameters.put("allVisn", "AllVisn");
					parameters.put("visnId", 0);
				}else {
					parameters.put("allVisn", "Visn");
					parameters.put("visnId", reportParameters.getVisn().getId());
				}
				queryName = "di3_summary_visn";
			} else if (reportParameters.getReportSummaryType().equalsIgnoreCase(ReportConstants.REPORT_SUMMARY_TYPE_FACILITY)) {
				if (reportParameters.getFacility() == null) {
					parameters.put("allFacilities", "AllFacilities");
					parameters.put("facilityId", 0);
				}else {
					parameters.put("allFacilities", "Facility");
					parameters.put("facilityId", reportParameters.getFacility().getId());
				}				
				queryName = "di3_summary_facility";
			}
			
			if (queryName != null) {
				return getJpaTemplate().findByNamedQueryAndNamedParams(queryName, parameters);
			}else {
				throw new DAOException("Unsupported DI3 Report configuration summary type " + reportParameters.getReportSummaryType());
			}
		} catch (DataAccessException e) {
			throw new DAOException(e.getMessage(), e);
		}
	}
	
	/**
	 * Device Status reports data
	 * @param reportParameters
	 * @return
	 * @throws DAOException
	 */
	@SuppressWarnings("unchecked")	
	private List<Object[]> generateDeviceStatusReports(DIReportParameters reportParameters) throws DAOException {
		try {
				//get the named query and execute with parameters
			Map<String,Object> parameters = new HashMap<String,Object>();
			Set<DeviceStatusType> deviceStatuses = reportParameters.getDeviceStatuses();
			if (deviceStatuses != null && deviceStatuses.size() > 0){
				parameters.put("deviceStatuses", getLookupIds(deviceStatuses));
			}
			String queryName = null;
			if (reportParameters.getReportSummaryType().equalsIgnoreCase(ReportConstants.REPORT_SUMMARY_TYPE_NATIONAL)) {
				//no parameters
				queryName = "di2_summary_national";
			} else if (reportParameters.getReportSummaryType().equalsIgnoreCase(ReportConstants.REPORT_SUMMARY_TYPE_VISN)) {
				if (reportParameters.getVisn() == null) {
					parameters.put("allVisn", "AllVisn");
					parameters.put("visnId", 0);
				}else {
					parameters.put("allVisn", "Visn");
					parameters.put("visnId", reportParameters.getVisn().getId());
				}
				queryName = "di2_summary_visn";
			} else if (reportParameters.getReportSummaryType().equalsIgnoreCase(ReportConstants.REPORT_SUMMARY_TYPE_FACILITY)) {
				if (reportParameters.getFacility() == null) {
					parameters.put("allFacilities", "AllFacilities");
					parameters.put("facilityId", 0);
				}else {
					parameters.put("allFacilities", "Facility");
					parameters.put("facilityId", reportParameters.getFacility().getId());
				}				
				queryName = "di2_summary_facility";
			}
			//sub total type
			if (reportParameters.getSubTotalType().equalsIgnoreCase(ReportConstants.REPORT_SUB_TOTAL_TYPE_VENDOR)) {
				queryName += "_vendor";
			}else {
				queryName += "_device";
			}
			
			if (queryName != null) {
				return getJpaTemplate().findByNamedQueryAndNamedParams(queryName, parameters);
			}else {
				throw new DAOException("Unsupported DI2 Report Summary Type " + reportParameters.getReportSummaryType());
			}
		} catch (DataAccessException e) {
			throw new DAOException(e.getMessage(), e);
		}
	}
	
	@SuppressWarnings("unchecked")
	private List<Object[]> generateInventoryReports(DIReportParameters reportParameters) throws DAOException {
		try {
				//get the named query and execute with parameters
			Map<String,Object> parameters = new HashMap<String,Object>();
			parameters.put("reportStartDate", reportParameters.getReportWeek().getStartDate());
			parameters.put("reportEndDate", reportParameters.getReportWeek().getEndDate());
			if (reportParameters.getReportSummaryType().equalsIgnoreCase(ReportConstants.REPORT_SUMMARY_TYPE_NATIONAL)) {
				if (reportParameters.getSubTotalType().equalsIgnoreCase(ReportConstants.REPORT_SUB_TOTAL_TYPE_VENDOR)) {
					return getJpaTemplate().findByNamedQueryAndNamedParams("di1_summary_national_vendor", parameters);
				}else {
					return getJpaTemplate().findByNamedQueryAndNamedParams("di1_summary_national_device", parameters);
				}					 
			}
			if (reportParameters.getReportSummaryType().equalsIgnoreCase(ReportConstants.REPORT_SUMMARY_TYPE_VISN)) {
				if (reportParameters.getVisn() == null) {
					parameters.put("allVisn", "allVisn");
					parameters.put("visnId", 0);
				}else {
					parameters.put("allVisn", "Visn");
					parameters.put("visnId", reportParameters.getVisn().getId());
				}
				if (reportParameters.getSubTotalType().equalsIgnoreCase(ReportConstants.REPORT_SUB_TOTAL_TYPE_VENDOR)) {
					return getJpaTemplate().findByNamedQueryAndNamedParams("di1_summary_visn_vendor", parameters);
				}else {
					return getJpaTemplate().findByNamedQueryAndNamedParams("di1_summary_visn_device", parameters);
				}					 
			}
			if (reportParameters.getReportSummaryType().equalsIgnoreCase(ReportConstants.REPORT_SUMMARY_TYPE_FACILITY)) {
				if (reportParameters.getFacility() == null) {
					parameters.put("allFacilities", "allFacilities");
					parameters.put("facilityId", 0);
				}else {
					parameters.put("allFacilities", "Facility");
					parameters.put("facilityId", reportParameters.getFacility().getId());
				}				
				if (reportParameters.getSubTotalType().equalsIgnoreCase(ReportConstants.REPORT_SUB_TOTAL_TYPE_VENDOR)) {
					return getJpaTemplate().findByNamedQueryAndNamedParams("di1_summary_facility_vendor", parameters);
				}else {
					return getJpaTemplate().findByNamedQueryAndNamedParams("di1_summary_facility_device", parameters);
				}					 
			}
			throw new DAOException("Unsupported Report Types");
		} catch (DataAccessException e) {
			throw new DAOException(e.getMessage(), e);
		}
	}
	
	@SuppressWarnings("unchecked")
	public List<Object[]> searchDevice(DeviceSearchParameters deviceSearchParameters) throws DAOException {
		try {
			String searchType = deviceSearchParameters.getSearchType().getValue();
			QueryAndParams queryAndParams = null;
			if (DeviceSearchParameters.DEVICE_SEARCH_TYPE_SERIAL_NUMBER.equals(searchType)){
				queryAndParams = buildQuerySearchBySerialNumber(deviceSearchParameters);			
			}
			else if (DeviceSearchParameters.DEVICE_SEARCH_TYPE_PATIENT.equals(searchType)){
				queryAndParams = buildQuerySearchByPatient(deviceSearchParameters);
			}
			else if (DeviceSearchParameters.DEVICE_SEARCH_TYPE_ACTIVATION_DATE.equals(searchType)){
				queryAndParams = buildQuerySearchByActivationDate(deviceSearchParameters);
			}
			else if (DeviceSearchParameters.DEVICE_SEARCH_TYPE_STATUS.equals(searchType)){
				queryAndParams = buildQuerySearchByStatus(deviceSearchParameters);
			}
			else if (DeviceSearchParameters.DEVICE_SEARCH_TYPE_GENERIC.equals(searchType)){
				queryAndParams = buildQuerySearchByGeneric(deviceSearchParameters);
			}
			else {
				throw new DAOException("Invalid Device search type" + searchType);
			}
			List<Object[]> results = (List<Object[]>) executeSQLQuery(queryAndParams);
			return results;
			} catch (DataAccessException e) {
				throw new DAOException(e.getMessage(), e);
			}
	}
	
	@SuppressWarnings("unchecked")
	public List<Object[]> generateVCCurrentReport(
			DeviceSearchParameters deviceSearchParameters) throws DAOException {
		// vendorComplianceCurrent
		Map<String,Object> parameters = new HashMap<String,Object>();
		parameters.put("reportStartDate", deviceSearchParameters.getReportWeek().getStartDate());
		parameters.put("reportEndDate", deviceSearchParameters.getReportWeek().getEndDate());
		return getJpaTemplate().findByNamedQueryAndNamedParams("vendorComplianceCurrent", parameters);
	}

	@SuppressWarnings("unchecked")
	public List<Object[]> generateVCHistoryReport(
			DeviceSearchParameters deviceSearchParameters) throws DAOException {
		Map<String,Object> parameters = new HashMap<String,Object>();
		parameters.put("reportStartDate", deviceSearchParameters.getReportWeek().getStartDate());
		parameters.put("reportEndDate", deviceSearchParameters.getReportWeek().getEndDate());
		return getJpaTemplate().findByNamedQueryAndNamedParams("vendorComplianceHistory", parameters);
	}

	@SuppressWarnings("unchecked")
	public List<Object[]> generateVCReportCharts(
			DeviceSearchParameters deviceSearchParameters) throws DAOException {
		Map<String,Object> parameters = new HashMap<String,Object>();
		parameters.put("reportStartDate", deviceSearchParameters.getReportWeek().getStartDate());
		parameters.put("reportEndDate", deviceSearchParameters.getReportWeek().getEndDate());
		return getJpaTemplate().findByNamedQueryAndNamedParams("vendorComplianceHistory", parameters);
	}
	
	@SuppressWarnings("unchecked")
	public List<DeviceRequirement> getDeviceRequirements(SimpleFacility simpleFacility) throws DAOException {
		try {
			Map<String, Object> params = new HashMap<String, Object>();
			params.put("facilityId", simpleFacility.getId());	    
			String query = "select dr from gov.va.med.ccht.model.inventory.DeviceRequirement dr where dr.facility.id = :facilityId";
			
			//get device requirements from shortage table
			List<DeviceRequirement> deviceRequirements =  getJpaTemplate().findByNamedParams(query, params);
			
			//get device inventory from device details table		
			Map<String,Object> parameters = new HashMap<String,Object>();
			parameters.put("facilityId", simpleFacility.getId());
			List<Object[]> data =  getJpaTemplate().findByNamedQueryAndNamedParams("facilityDeviceInventory", parameters);
			
			//merge
			for (Object[] objs:data) {
				
				Boolean deviceNotFound = true;
				for (DeviceRequirement dr:deviceRequirements) {
					if (dr.getDeviceType().getCode().equals((String)objs[0])) {
						dr.setDevicesOnHand((Integer) objs[1]);
						deviceNotFound = false;
						break;
					}
				}
				
				//device requirement not found add a new entry
				if (deviceNotFound) {
					DeviceRequirement newDevReq = new DeviceRequirement();
					newDevReq.setFacility(simpleFacility);
					DeviceType dt = getDeviceType((String)objs[0]);
					newDevReq.setDeviceType(dt);
					newDevReq.setDevicesOnHand((Integer) objs[1]);
					newDevReq.setDevicesNeeded(0);
					deviceRequirements.add(newDevReq);
				}
			}				
			return deviceRequirements;
		}catch (Exception e){
			throw new DAOException(e.getMessage(),e);
		}
	}
			
	//query builders
	private QueryAndParams buildQuerySearchByActivationDate(DeviceSearchParameters deviceSearchParameters) 
	throws DAOException {
		
		Map<String, Object> parameters = new HashMap<String, Object>();
		parameters.put("reportStartDate", deviceSearchParameters.getReportWeek().getStartDate());
		parameters.put("reportEndDate", deviceSearchParameters.getReportWeek().getEndDate());
		String query = 
			"SELECT	Vendor.Vendor_Name AS [vendor_name], " +
					"isnull(CensusPatient.Home_Dev_Model,'{none}') AS [model_name], " +
					"isnull(CensusPatient.Home_Dev_Serial,'{none}') AS [serial_number], " +
					"CensusPatient.Activation_Date AS [activation_date] " +
			"FROM	Census INNER JOIN " +
					"CensusPatient ON Census.Census_ID = CensusPatient.Census_ID INNER JOIN " +
					"Vendor ON Census.Vendor_ID = Vendor.Vendor_ID " +
			"WHERE   (Census.Report_Start_Date >= :reportStartDate) " +
					"AND (Census.Report_End_Date <= :reportEndDate) " ;
			if (Device.NULL_DEVICE_NAME.equals(deviceSearchParameters.getDeviceType().getLabel())){
				query = query + "AND CensusPatient.Home_Dev_Model is null ";
			}else {
				parameters.put("deviceModelName", deviceSearchParameters.getDeviceType().getLabel());
				query = query + "AND CensusPatient.Home_Dev_Model is not null and CensusPatient.Home_Dev_Model = :deviceModelName ";
			}
			if (deviceSearchParameters.getFromActivationDate() != null){
				parameters.put("fromActivationDate", deviceSearchParameters.getFromActivationDate());
				query = query + "AND CensusPatient.Activation_Date is not null and CensusPatient.Activation_Date >= :fromActivationDate ";
			}
			if (deviceSearchParameters.getToActivationDate() != null){
				parameters.put("toActivationDate", deviceSearchParameters.getToActivationDate());
				query = query + "AND CensusPatient.Activation_Date is not null and CensusPatient.Activation_Date <= :toActivationDate	";
			}	
			query = query + 										
				"ORDER BY [vendor_name], CensusPatient.Home_Dev_Model, CensusPatient.Home_Dev_Serial, CensusPatient.Activation_Date ";

		QueryAndParams queryAndParams = new QueryAndParams(query,parameters);
		return queryAndParams;
	}
	
	private QueryAndParams buildQuerySearchByGeneric(DeviceSearchParameters deviceSearchParameters) 
	throws DAOException {
		
		Map<String, Object> parameters = new HashMap<String, Object>();
		parameters.put("facilityCode", deviceSearchParameters.getFacility().getValue());
		
		String query = 		
			"select mdd.MED_DVC_DETAIL_ID id, mdmt.name device_name, mdd.serial_number serial_number, mult_patient_usage_flag," +
			"mds.status_change_date status_change_date, mdst.name device_status, p.name_last patient_name, u.last_name coordinator from " +
			"dbo.medical_device_detail mdd " +
			"inner join dbo.medical_device_status mds ON mdd.MED_DVC_DETAIL_ID = mds.MED_DVC_DETAIL_ID " +
			"inner join dbo.MEDICAL_DEVICE_STATUS_TYPE mdst	on mds.MED_DVC_STATUS_TYPE_ID = mdst.ID " +
			"left join dbo.MEDICAL_DEVICE_MODEL_TYPE mdmt on mdd.MED_DVC_MODEL_TYPE_ID = mdmt.id " +			
			"left join dbo.patient_device pd on mdd.MED_DVC_DETAIL_ID = pd.MED_DVC_DETAIL_ID " +
			"left join dbo.patient p on pd.patient_id = p.patient_id " +
			"left join ht.app_user u on pd.care_coordinator_id = u.app_user_id ";
		
			//facility is always provided required field			
			query = query + " where mdd.facilities_id in (select id from dbo.facilities where facility_id = :facilityCode) ";
			//model name
			if (deviceSearchParameters.getDeviceType() != null){
				if (Device.NULL_DEVICE_NAME.equals(deviceSearchParameters.getDeviceType().getLabel())){
					query = query + "AND mdd.MED_DVC_MODEL_TYPE_ID is null ";
				} else {	
					parameters.put("device_type", deviceSearchParameters.getDeviceType().getValue());
					query = query + " AND mdmt.code = :device_type";
				}
			}
			//serial number
			String sn = deviceSearchParameters.getSerialNumber();
			if (StringUtils.isNotEmpty(sn)){
				if (sn.endsWith("*")) {
					parameters.put("serial_number", sn.replace('*', '%'));
					query = query + " AND mdd.serial_number like :serial_number";
				}else {
					parameters.put("serial_number", sn);				
					query = query + " AND mdd.serial_number = :serial_number";
				}				
			}			
			//device statuses
			List<TermType> deviceStatuses = deviceSearchParameters.getDeviceStatuses ();
			if (deviceStatuses != null && deviceStatuses.size() > 0) {
				List<String> selectedDevStatuses = new ArrayList<String>();
				for (TermType tt:deviceStatuses) {
					selectedDevStatuses.add(tt.getValue()); 					
				}
				parameters.put("deviceStatuses", selectedDevStatuses);
				query = query + " AND mdst.code in (:deviceStatuses)";
			}
			//patient last name
			if (StringUtils.isNotEmpty(deviceSearchParameters.getLastName())){
				parameters.put("lastName", deviceSearchParameters.getLastName());
				query = query + " AND p.name_last = :lastName";
			}
			
			//care coordinator
			if (deviceSearchParameters.getCareCoordinator() != null) {
				parameters.put("careCoordinator", deviceSearchParameters.getCareCoordinator().getValue());
				query = query + " AND u.user_name = :careCoordinator";
			}
			
			//order by
			query = query + 										
				" ORDER BY p.name_last, u.last_name, mdmt.name, mdd.serial_number";

		QueryAndParams queryAndParams = new QueryAndParams(query,parameters);
		return queryAndParams;
	}
	
	private QueryAndParams buildQuerySearchByStatus(DeviceSearchParameters deviceSearchParameters) 
	throws DAOException {
		
		Map<String, Object> parameters = new HashMap<String, Object>();
		parameters.put("deviceModelName", deviceSearchParameters.getDeviceType().getLabel());
		
		String query = 				
			"SELECT	v.Vendor_Name AS vendor_name,  mdm.name AS model_name,  " +
			"isnull(di.SERIAL_NUMBER,'{none}') AS serial_number, mdst.name AS device_status " +
			"FROM	dbo.MEDICAL_DEVICE_DETAIL di INNER JOIN  " +
			"dbo.MEDICAL_DEVICE_MODEL_TYPE mdm on di.MED_DVC_MODEL_TYPE_ID = mdm.id INNER JOIN  " +
			"dbo.Vendor V ON mdm.Vendor_ID = v.Vendor_ID INNER JOIN  " +
			"dbo.medical_device_status mds ON di.MED_DVC_DETAIL_ID = mds.MED_DVC_DETAIL_ID INNER JOIN " +
			"dbo.MEDICAL_DEVICE_STATUS_TYPE mdst	on mds.MED_DVC_STATUS_TYPE_ID = mdst.ID " +
			"WHERE   mdm.name = :deviceModelName " ;		
			List<TermType> deviceStatuses = deviceSearchParameters.getDeviceStatuses ();
			if (deviceStatuses != null && deviceStatuses.size() > 0) {
				List<String> selectedDevStatuses = new ArrayList<String>();
				for (TermType tt:deviceStatuses) {
					selectedDevStatuses.add(tt.getValue()); 					
				}
				parameters.put("deviceStatuses", selectedDevStatuses);
				query = query + "AND mdst.code in (:deviceStatuses)";
			}
			//order by
			query = query + 										
			" ORDER BY vendor_name, mdm.name, di.SERIAL_NUMBER";

		QueryAndParams queryAndParams = new QueryAndParams(query,parameters);
		return queryAndParams;
	}	
	
	private QueryAndParams buildQuerySearchBySerialNumber(DeviceSearchParameters deviceSearchParameters) 
	throws DAOException {
		
		Map<String, Object> parameters = new HashMap<String, Object>();
		parameters.put("reportStartDate", deviceSearchParameters.getReportWeek().getStartDate());
		parameters.put("reportEndDate", deviceSearchParameters.getReportWeek().getEndDate());
		String query = 
			"SELECT	isnull(CensusPatient.Home_Dev_Serial,'{none}') AS [serial_number], " +
					"Facilities.Facility_Name as [facility_name], " +
					"Vendor.Vendor_Name AS [vendor_name], " +	 
					"count(*) AS [device_count]	" +			
			"FROM	Census INNER JOIN " +
					"CensusPatient ON Census.Census_ID = CensusPatient.Census_ID INNER JOIN " +
					"Facilities ON CensusPatient.Facility_ID = Facilities.ID INNER JOIN " +
					"VISN ON Facilities.VISN_ID = VISN.VISN_ID INNER JOIN " +
					"Vendor ON Census.Vendor_ID = Vendor.Vendor_ID " +
			"WHERE   (Census.Report_Start_Date >= :reportStartDate) " +
					"AND (Census.Report_End_Date <= :reportEndDate) " ;
			if (Device.NULL_DEVICE_NAME.equals(deviceSearchParameters.getDeviceType().getLabel())){
				query = query + "AND CensusPatient.Home_Dev_Model is null ";
			}else {
				parameters.put("deviceModelName", deviceSearchParameters.getDeviceType().getLabel());
				query = query + "AND CensusPatient.Home_Dev_Model is not null and CensusPatient.Home_Dev_Model = :deviceModelName ";
			}
			if (StringUtils.isNotEmpty(deviceSearchParameters.getFromSerialNumber())){
				parameters.put("fromserialNumber", deviceSearchParameters.getFromSerialNumber());
				query = query + "AND CensusPatient.Home_Dev_Serial is not null and CensusPatient.Home_Dev_Serial >= :fromserialNumber ";
			}
			if (StringUtils.isNotEmpty(deviceSearchParameters.getToSerialNumber())){
				parameters.put("toSerialNumber", deviceSearchParameters.getToSerialNumber());
				query = query + "AND CensusPatient.Home_Dev_Serial is not null and CensusPatient.Home_Dev_Serial <= :toSerialNumber	";
			}	
			query = query + 										
				"GROUP BY CensusPatient.Home_Dev_Serial, Facilities.Facility_Name, Vendor.Vendor_Name " +
				"ORDER BY CensusPatient.Home_Dev_Serial, Facilities.Facility_Name, [vendor_name]";

		QueryAndParams queryAndParams = new QueryAndParams(query,parameters);
		return queryAndParams;
	}
	
	private QueryAndParams buildQuerySearchByPatient(DeviceSearchParameters deviceSearchParameters) 
	throws DAOException {
		
		Map<String, Object> parameters = new HashMap<String, Object>();
		parameters.put("reportStartDate", deviceSearchParameters.getReportWeek().getStartDate());
		parameters.put("reportEndDate", deviceSearchParameters.getReportWeek().getEndDate());
		String query = 
			"SELECT	Vendor.Vendor_Name AS vendor_name, " +
					"isnull(CensusPatient.Home_Dev_Model,'{none}') AS model_name, " +
					"isnull(CensusPatient.Home_Dev_Serial,'{none}') AS serial_number, " +
					"CensusPatient.Activation_Date AS activation_date, " +
					"CensusPatient.Name_Last as last_name, " +
					"CensusPatient.Name_First as first_name, " +
					"CensusPatient.Name_MI as middle_name, " +
					"CensusPatient.Patient_ssn as ssn, " +
					"cast(CensusPatient.Patient_icn As VARCHAR) as icn " +
			"FROM	Census INNER JOIN " +
					"CensusPatient ON Census.Census_ID = CensusPatient.Census_ID INNER JOIN " +
					"Vendor ON Census.Vendor_ID = Vendor.Vendor_ID " +
			"WHERE   (Census.Report_Start_Date >= :reportStartDate) " +
					"AND (Census.Report_End_Date <= :reportEndDate) " ;
		
			if (StringUtils.isNotEmpty(deviceSearchParameters.getLastName())){
				parameters.put("lastName", deviceSearchParameters.getLastName());
				query = query + "AND CensusPatient.Name_Last = :lastName ";
			}
			if (StringUtils.isNotEmpty(deviceSearchParameters.getFirstName())){
				parameters.put("firstName", deviceSearchParameters.getFirstName());
				query = query + "AND CensusPatient.Name_First = :firstName ";
			}
			if (StringUtils.isNotEmpty(deviceSearchParameters.getSsn())){
				parameters.put("ssn", deviceSearchParameters.getSsn());
				query = query + "AND CensusPatient.Patient_ssn = :ssn ";
			}
			if (StringUtils.isNotEmpty(deviceSearchParameters.getIcn())){
				parameters.put("icn", deviceSearchParameters.getIcn());
				query = query + "AND CensusPatient.Patient_icn = :icn ";
			}			
			query = query + 										
				"ORDER BY [vendor_name], CensusPatient.Home_Dev_Model, CensusPatient.Home_Dev_Serial, CensusPatient.Activation_Date ";

		QueryAndParams queryAndParams = new QueryAndParams(query,parameters);
		return queryAndParams;
	}	
	@SuppressWarnings("unchecked")
	public List<DeviceTrackerLog> getUnprocessedDeviceTrackerLogs() throws DAOException {
		try {
			String query = "select l from " + DeviceTrackerLog.class.getName() + " l where l.processed is null";
			return getJpaTemplate().find(query);			 
		} catch (DataAccessException e) {
			throw new DAOException(e.getMessage(), e);
		}
	}

	@SuppressWarnings("unchecked")
	public List<Vendor> getActiveVendorsForComplianceReport() throws DAOException {
		try {
			String query = "select l from " + Vendor.class.getName() + " l where l.dateActive is not null and l.dateInactive is null and " +
					"l.id in (select m.vendor.id from "+ DeviceType.class.getName()+ " m)";
			logger.debug("getActiveVendorsForComplianceReport query:  "+query);
			List<Vendor> results = getJpaTemplate().find(query);
			return results;			 
		} catch (DataAccessException e) {
			throw new DAOException(e.getMessage(), e);
		}
	}
	
	public List<TermType> getModalityType(int type) throws DAOException
	{
		try
		{
			List<TermType> modalityTypes = new ArrayList<TermType>();
			String tableName = null;
			
			switch(type)
			{
			case 0:
				tableName = "ht.SURVEY_DVC_MODALITY_TYPE ";				
				break;
				
			case 1:
				tableName = "ht.CENSUS_DVC_MODALITY_TYPE ";
				break;
			}
			
			
			StringBuilder sb = new StringBuilder();
			sb.append("SELECT NAME ");
			sb.append("FROM ");
			sb.append(tableName);
			sb.append("WHERE INACTIVE_DATE IS Null");
			
			QueryAndParams queryAndParams = new QueryAndParams(sb.toString());
			
			List<?> data = executeSQLQuery(queryAndParams);
			int reportSize = data.size();
			for(int x = 0; x < reportSize; x++)
			{
				
				String name = (String) data.get(x);
				
				TermType termType = new TermType(name, name);
				modalityTypes.add(termType);
			}
			
			return modalityTypes;
		}
		catch(DataAccessException e)
		{
			throw new DAOException(e.getMessage(), e);
		}
	}

   public void setVendorComplianceReportCompletion(List<DeviceTrackerLog> deviceTrackerLogs) throws DAOException
   {
	   for(DeviceTrackerLog deviceTrackerLog: deviceTrackerLogs)
	   {
		   deviceTrackerLog.setProcessed(true);
		   try
		   {
		     super.update(deviceTrackerLog);
		   }
		   catch (Exception e) {
				throw new DAOException("setVendorComplianceReportCompletion failed", e);
			}
	   }
   }
	
	protected List<Long> getLookupIds(Set<? extends AbstractLookup> lookupSet) {
		List<Long> list = new ArrayList<Long>();
		if (lookupSet != null && lookupSet.size() > 0) {
			for (AbstractLookup lookup : lookupSet)
				list.add(lookup.getIdentifier());
		}
		return list;
	}
	
	@SuppressWarnings("unchecked")
	public List<FacilityActivity> getFacilityActivitiesReportWeeks() throws DAOException {
		try {
			String query = "select DISTINCT l from " + FacilityActivity.class.getName() + 
				" order by l.reportWeek desc";
			return getJpaTemplate().find(query);			 
		} catch (DataAccessException e) {
			throw new DAOException(e.getMessage(), e);
		}
	}
	
}
