package gov.va.med.ccht.model.report.scheduled;

import gov.va.med.fw.model.lookup.Lookup;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;

/**
 * In a scheduled report, these type represent the various options applicable to
 * the "From Date" and "To Date" of the report data range.
 * 
 * @author vhaisdbarryc
 */
public enum DateRangeBoundaryType implements Lookup {
	CURRENT_DAY(new RequiredMethods() {
		public Date calculateDate(Date startDate, DateRangeBoundary drb) {
			return startDate;
		}

		public String getFriendlyDescription(DateRangeBoundary dateRangeBoundary) {
			return "The current day";
		}
	}),
	/* For example, "January 19, 2010" */
	SPECIFIC_FIXED_DATE(new RequiredMethods() {
		public Date calculateDate(Date startDate, DateRangeBoundary drb) {
			return drb.getSpecificFixedDate();
		}

		public String getFriendlyDescription(DateRangeBoundary dateRangeBoundary) {
			DateFormat df = SimpleDateFormat.getDateInstance(DateFormat.MEDIUM, Locale.US);
			return df.format(dateRangeBoundary.getSpecificFixedDate());
		}
	}),
	/* For example, "Exactly 2 Weeks prior to current day" */
	EXACT_UNITS_PRIOR_TO_CURRENT_DAY(new RequiredMethods() {
		public Date calculateDate(Date startDate, DateRangeBoundary drb) {
			return drb.getUnitPriorToCurrentDay().calculateRelativeDate(startDate,
					drb.getNumUnitsPriorToCurrentDay() * -1);
		}

		public String getFriendlyDescription(DateRangeBoundary drb) {
			Integer numUnitsPriorToCurrentDay = drb.getNumUnitsPriorToCurrentDay();
			SimpleTimeUnitType unitPriorToCurrentDay = drb.getUnitPriorToCurrentDay();
			return "Exactly " + numUnitsPriorToCurrentDay + " "
					+ unitPriorToCurrentDay.getName().toLowerCase()
					+ (numUnitsPriorToCurrentDay != 1 ? "s" : "") + " prior to the current day";
		}
	}),
	/* For example, "Second of Previous Month" */
	FIXED_DAY_OF_PREVIOUS_OR_CURRENT_UNIT(new RequiredMethods() {
		public Date calculateDate(Date startDate, DateRangeBoundary drb) {
			return drb.getMultiDayTimeUnitType().calculateDate(startDate, drb.getFixedDateType(),
					drb.getUsePreviousDatePeriod());
		}

		public String getFriendlyDescription(DateRangeBoundary drb) {
			return "The " + drb.getFixedDateType().getName().toLowerCase() + " of the "
					+ (drb.getUsePreviousDatePeriod() ? "previous" : "current") + " "
					+ drb.getMultiDayTimeUnitType().getName().toLowerCase();
		}
	});

	private RequiredMethods dateCalculator;

	private DateRangeBoundaryType(RequiredMethods dateCalculator) {
		this.dateCalculator = dateCalculator;
	}

	public String getName() {
		return name();
	}

	public String getCode() {
		return name();
	}

	public String getDescription() {
		return getName();
	}

	public String getFriendlyDescription(DateRangeBoundary dateRangeBoundary) {
		return dateCalculator.getFriendlyDescription(dateRangeBoundary);
	}

	/**
	 * Returns a new date given a starting date and a set of parameters
	 * describing how to change that date (with variable length and type based
	 * on the DateRangeBoundaryType used).
	 * 
	 * @param startDate
	 *            The starting date
	 * @param params
	 *            The array of parameters as required by the type of
	 *            DateRangeBoundaryType used; see each specific value's
	 *            requirement above.
	 * @return The new calculated date
	 */
	public Date calculateDate(Date startDate, DateRangeBoundary drb) {
		return dateCalculator.calculateDate(startDate, drb);
	}

	private interface RequiredMethods {
		Date calculateDate(Date startDate, DateRangeBoundary drb);

		String getFriendlyDescription(DateRangeBoundary dateRangeBoundary);
	}

}
