/********************************************************************
 * Copyright � 2010 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.ccht.model;

import gov.va.med.ccht.model.terminology.DayOfMonth;
import gov.va.med.ccht.model.terminology.DayOfQuarter;
import gov.va.med.ccht.model.terminology.DayOfWeek;
import gov.va.med.ccht.model.terminology.Month;
import gov.va.med.ccht.model.terminology.RunFrequency;
import gov.va.med.ccht.model.terminology.ScheduleType;
import gov.va.med.ccht.model.terminology.WeekOfMonth;
import gov.va.med.ccht.util.SchedulerUtils;
import gov.va.med.fw.model.AbstractEntity;
import gov.va.med.fw.util.date.DateWithTimeZone;
import gov.va.med.fw.util.date.TimeZoneUtils;

import java.text.DateFormat;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import java.util.Set;
import java.util.SortedSet;
import java.util.TimeZone;
import java.util.TreeSet;

import org.apache.commons.lang.builder.ToStringBuilder;
import org.quartz.CronTrigger;

/**
 * 
 * @author vhaisakatikm
 */
public class Schedule extends AbstractEntity {

	private static final long serialVersionUID = -3780350573837478688L;

	// ---------------------------------- Fields
	private RunFrequency runFrequency;
	private ScheduleType scheduleType;
	private DayOfMonth dayOfMonth;
	private WeekOfMonth weekOfMonth;
	private DayOfWeek dayOfWeek;
	private DayOfQuarter dayOfQuarter;
	private Month month;
	private Integer numericDayOfMonth;
	private Date internalDateToGenerate;
	private TimeZone internalDateToGenerateTimeZone = TimeZone.getDefault();
	private Integer hour;
	private Integer minute;
	private Set<DayOfWeek> daysOfWeek;

	// ---------------------------------- Business Methods

	public String getFriendlyDescription() {
		if (scheduleType == null)
			return "Unknown";
		StringBuilder sb = new StringBuilder();

		if (ScheduleType.DAY_OF_WEEK.getCode().equals(scheduleType.getCode())) {
			sb.append("Every week on ");
			SortedSet<DayOfWeek> sortedDays = new TreeSet<DayOfWeek>(daysOfWeek);

			int i = 0;
			for (DayOfWeek dow : sortedDays) {
				i++;
				sb.append(dow.getName());
				if (i < daysOfWeek.size() - 1) {
					sb.append(", ");
				} else if (i == daysOfWeek.size() - 1) {
					sb.append(" and ");
				}
			}
		} else if (ScheduleType.DAY_OF_MONTH.getCode().equals(scheduleType.getCode())) {
			sb.append("The ").append(dayOfMonth.getName().toLowerCase()).append(
					" day of every month");
		} else if (ScheduleType.WEEK_DAY.getCode().equals(scheduleType.getCode())) {
			sb.append("The ").append(weekOfMonth.getName().toLowerCase()).append(" ").append(
					dayOfWeek.getName()).append(" of every month");
		} else if (ScheduleType.DAY_OF_QUARTER.getCode().equals(scheduleType.getCode())) {
			sb.append("The ").append(dayOfQuarter.getName().toLowerCase()).append(
					" day of every quarter");
		} else if (ScheduleType.WEEK_DAY_MONTH.getCode().equals(scheduleType.getCode())) {
			sb.append("Yearly on the ").append(weekOfMonth.getName().toLowerCase()).append(" ")
					.append(dayOfWeek.getName()).append(" in ").append(month.getName());
		} else if (ScheduleType.MONTH_DAY.getCode().equals(scheduleType.getCode())) {
			sb.append("Every ").append(month.getName()).append(" on day ")
					.append(numericDayOfMonth).append(" of the month");
		} else if (ScheduleType.DATE.getCode().equals(scheduleType.getCode())) {
			DateFormat df = SimpleDateFormat.getDateInstance(DateFormat.MEDIUM, Locale.US);
			df.setTimeZone(internalDateToGenerateTimeZone);
			sb.append("Only on ").append(df.format(internalDateToGenerate));
		}

		NumberFormat nf = DecimalFormat.getIntegerInstance();
		nf.setMinimumIntegerDigits(2);
		sb.append(" at ");
		if (hour == 0 && minute == 0) {
			sb.append("midnight");
		} else {
			sb.append(nf.format(hour)).append(":").append(nf.format(minute));
		}

		return sb.toString();
	}

	protected void buildToString(ToStringBuilder builder) {	}

	protected void finalize() throws Throwable {
		super.finalize();
		daysOfWeek = null;
	}

	public RunFrequency getRunFrequency() {
		return runFrequency;
	}

	public void setRunFrequency(RunFrequency runFrequency) {
		this.runFrequency = runFrequency;
	}

	public ScheduleType getScheduleType() {
		return scheduleType;
	}

	public void setScheduleType(ScheduleType scheduleType) {
		this.scheduleType = scheduleType;
	}

	public DayOfMonth getDayOfMonth() {
		return dayOfMonth;
	}

	public void setDayOfMonth(DayOfMonth dayOfMonth) {
		this.dayOfMonth = dayOfMonth;
	}

	public WeekOfMonth getWeekOfMonth() {
		return weekOfMonth;
	}

	public void setWeekOfMonth(WeekOfMonth weekOfMonth) {
		this.weekOfMonth = weekOfMonth;
	}

	public DayOfWeek getDayOfWeek() {
		return dayOfWeek;
	}

	public void setDayOfWeek(DayOfWeek dayOfWeek) {
		this.dayOfWeek = dayOfWeek;
	}

	public DayOfQuarter getDayOfQuarter() {
		return dayOfQuarter;
	}

	public void setDayOfQuarter(DayOfQuarter dayOfQuarter) {
		this.dayOfQuarter = dayOfQuarter;
	}

	public Month getMonth() {
		return month;
	}

	public void setMonth(Month month) {
		this.month = month;
	}

	public Integer getNumericDayOfMonth() {
		return numericDayOfMonth;
	}

	public void setNumericDayOfMonth(Integer numericDayOfMonth) {
		this.numericDayOfMonth = numericDayOfMonth;
	}

	public Integer getHour() {
		return hour;
	}

	public void setHour(Integer hour) {
		this.hour = hour;
	}

	public Integer getMinute() {
		return minute;
	}

	public void setMinute(Integer minute) {
		this.minute = minute;
	}

	public Set<DayOfWeek> getDaysOfWeek() {
		return daysOfWeek;
	}

	public void setDaysOfWeek(Set<DayOfWeek> daysOfWeek) {
		this.daysOfWeek = daysOfWeek;
	}

	public DateWithTimeZone getDateToGenerate() {
		return internalDateToGenerate != null ? new DateWithTimeZone(getInternalDateToGenerate(),
				getInternalDateToGenerateTimeZone()) : null;
	}

	public void setDateToGenerate(DateWithTimeZone dateToGenerate) {
		setInternalDateToGenerate(dateToGenerate == null ? null : dateToGenerate.getDate());
		setInternalDateToGenerateTimeZone(dateToGenerate == null ? null : dateToGenerate
				.getTimeZone());
	}

	public Date getInternalDateToGenerate() {
		return internalDateToGenerate;
	}

	public void setInternalDateToGenerate(Date internalDateToGenerate) {
		this.internalDateToGenerate = internalDateToGenerate;
	}

	public TimeZone getInternalDateToGenerateTimeZone() {
		return internalDateToGenerateTimeZone;
	}

	public void setInternalDateToGenerateTimeZone(
			TimeZone internalDateToGenerateTimeZone) {
		this.internalDateToGenerateTimeZone = internalDateToGenerateTimeZone;
	}
	
	public String getCronExpression() {
		return SchedulerUtils.getCronExpression(this);
	}
	
	public Date getNextScheduledDate() {
		String cron = getCronExpression();
		TimeZone timezone = TimeZoneUtils.getTimeZone();
		CronTrigger trigger = new CronTrigger();
		trigger.setName("unimportant");
		trigger.setGroup("also-unimportant");
		trigger.setJobName("still-not-important");
		trigger.setJobGroup("nope-not-important");
		trigger.setTimeZone(timezone);
		try {
			trigger.setCronExpression(cron);
		} catch (ParseException e) {
			return null;
		}
		Date now = new Date();
		trigger.setStartTime(now);
		Date next = trigger.getFireTimeAfter(now);
		return next;
	}
}
