/**
 * 
 */


package gov.va.med.cds.socket.server.handler;


import static org.easymock.EasyMock.createMock;
import static org.easymock.EasyMock.createNiceMock;
import static org.easymock.EasyMock.eq;
import static org.easymock.EasyMock.expect;
import static org.easymock.EasyMock.isA;
import static org.easymock.EasyMock.replay;
import static org.junit.Assert.assertTrue;

import gov.va.med.cds.exception.ErrorCodeEnum;
import gov.va.med.cds.hapi.HL7SupportException;
import gov.va.med.cds.hapi.HL7SupportInterface;
import gov.va.med.cds.junit.runners.Suite;
import gov.va.med.cds.junit.runners.SuiteAwareSpringRunner;
import gov.va.med.cds.testharness.AbstractBaseTest;
import gov.va.med.cds.testharness.logger.AssertApplicationLogger;
import gov.va.med.cds.util.MllpUtil;
import gov.va.med.cds.util.StreamUtil;

import org.easymock.EasyMock;
import org.easymock.IAnswer;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.FileSystemResource;
import org.springframework.jdbc.core.JdbcOperations;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.ContextConfiguration;
import org.xsocket.connection.INonBlockingConnection;

import java.nio.ByteBuffer;
import java.nio.charset.Charset;

import javax.annotation.Resource;


/**
 * @author susarlan
 * 
 */
@RunWith( SuiteAwareSpringRunner.class )
@ContextConfiguration( locations = { "file:src/test/resources/mllpAccessValidatorSpringContext.xml" } )
public class MllpMessageMediatorDataHandlerTest
    extends
        AbstractBaseTest
{
    private MllpAccessValidator mllpAccessValidator;
    
    @Autowired
    private JdbcOperations jdbcTemplateMock; 


    @Resource
    public void setMllpAccessValidator( MllpAccessValidator mllpAccessValidator )
    {
        this.mllpAccessValidator = mllpAccessValidator;
    }


    /***
     * Test validateMessage on MllpMessageValidator
     */
    @DirtiesContext
    @Test
    @Suite( groups = "checkintest", order = 1 )
    public void testOnData( )
        throws Exception
    {
    	Exception ex = null;
        // Set up mock object for non blocking connection
        INonBlockingConnection nbcMock = ( INonBlockingConnection )createMock( INonBlockingConnection.class );
        ResponseGeneratorInterface responseGenerator = createMock( ResponseGeneratorInterface.class );
        MessageMediatorDispatcherInterface messageMediatorDispatcher = createMock( MessageMediatorDispatcherInterface.class );

        // Create request
        String hl7Request = StreamUtil.resourceToString( new FileSystemResource( "./src/test/resources/chdr-allergies.rpc" ) );
        final ByteBuffer reqBuffer = MllpUtil.encode( ByteBuffer.wrap( hl7Request.getBytes() ), Charset.defaultCharset(),
                        Charset.forName( "ISO-8859-1" ) );

        // Set expectations
        expect( nbcMock.available()).andReturn(reqBuffer.limit());
        nbcMock.markReadPosition();
        expect( nbcMock.read( isA( ByteBuffer.class ) ) ).andAnswer( new IAnswer<Integer>()
        {
            public Integer answer( )
                throws Throwable
            {
        		ByteBuffer buffer = (ByteBuffer)EasyMock.getCurrentArguments()[0];
        		buffer.put(reqBuffer);
        		return reqBuffer.limit();
        	}
        });
        nbcMock.removeReadMark();
        expect(nbcMock.getId()).andReturn("1");
        
        expect( messageMediatorDispatcher.dispatch( isA( HL7SupportInterface.class ) ) ).andReturn( "responseString" );
        expect(
                        responseGenerator.createResponse( isA( HL7SupportInterface.class ), eq( "responseString" ), eq( "00000000000000000000" ),
                                        EasyMock.eq( ex ) ) ).andReturn( "generated message" );
        //send request
        expect( nbcMock.write( isA( ByteBuffer.class ) ) ).andReturn( 0 );
        nbcMock.flush();
        
        // Set mocks on handler
        MllpMessageMediatorDataHandler dataHandler = new MllpMessageMediatorDataHandler();
        dataHandler.setAccessValidator( mllpAccessValidator );
        dataHandler.setMessageMediatorDispatcher( messageMediatorDispatcher );
        dataHandler.setResponseGenerator( responseGenerator );

        replay( nbcMock, messageMediatorDispatcher, responseGenerator, jdbcTemplateMock );

        dataHandler.onData( nbcMock );
        EasyMock.verify( nbcMock, messageMediatorDispatcher, responseGenerator );

    }


    @DirtiesContext
	@Test
    @Suite( groups = "checkintest", order = 2 )
    public void testOnDataWithDispatchException( )
        throws Exception
    {

		DispatchException distpatchException = new DispatchException(ErrorCodeEnum.DISPATCH_EXCEPTION);
		String response = null;
		
        // Set up mock object for non blocking connection
        INonBlockingConnection nbcMock = ( INonBlockingConnection )createMock( INonBlockingConnection.class );
        MessageMediatorDispatcherInterface messageMediatorDispatcher = createMock( MessageMediatorDispatcherInterface.class );
        ResponseGeneratorInterface responseGenerator = createMock( ResponseGeneratorInterface.class );

        // Create request
        String hl7Request = StreamUtil.resourceToString( new FileSystemResource( "./src/test/resources/chdr-allergies.rpc" ) );
        final ByteBuffer reqBuffer = MllpUtil.encode( ByteBuffer.wrap( hl7Request.getBytes() ), Charset.defaultCharset(),
                        Charset.forName( "ISO-8859-1" ) );
        
        // Set mocks on handle
        MllpMessageMediatorDataHandler dataHandler = new MllpMessageMediatorDataHandler();
        dataHandler.setAppLogger( new AssertApplicationLogger( "Error occured processing message.", new DispatchException(
                        ErrorCodeEnum.DISPATCH_EXCEPTION ) ) );
        dataHandler.setAccessValidator( mllpAccessValidator );
        dataHandler.setMessageMediatorDispatcher( messageMediatorDispatcher );
        dataHandler.setResponseGenerator( responseGenerator );
        
        // Set expectations
        expect( nbcMock.available()).andReturn(reqBuffer.limit());
        nbcMock.markReadPosition();
        expect( nbcMock.read( isA( ByteBuffer.class ) ) ).andAnswer( new IAnswer<Integer>()
        {
            public Integer answer( )
                throws Throwable
            {
        		ByteBuffer buffer = (ByteBuffer)EasyMock.getCurrentArguments()[0];
        		buffer.put(reqBuffer);
        		return reqBuffer.limit();
        	}
        });
        nbcMock.removeReadMark();
        expect(nbcMock.getId()).andReturn("1");
 
        expect( messageMediatorDispatcher.dispatch( isA( HL7SupportInterface.class ) ) ).andThrow( distpatchException );

        expect(
                        responseGenerator.createResponse( isA( HL7SupportInterface.class ), eq( response ), eq( "00000000000000000000" ),
                                        eq( distpatchException ) ) ).andReturn( "Dispatch Exception" );
        // send response
        expect( nbcMock.write( isA( ByteBuffer.class ) ) ).andReturn( 0 );
        nbcMock.flush();

        replay( nbcMock, messageMediatorDispatcher, responseGenerator );
        dataHandler.onData( nbcMock );
        EasyMock.verify( nbcMock, messageMediatorDispatcher, responseGenerator );
    }
    

	@Test
    @Suite( groups = "checkintest", order = 3 )
    public void testOnDataWithValidatonException( )
        throws Exception
    {
		AccessValidatorInterface mllpAccessValidatorMock = createMock(AccessValidatorInterface.class);
    	AccessValidatorException validationException = new AccessValidatorException(ErrorCodeEnum.HL7_SUPPORT_HAPI_PARSE_EXCEPTION);
		String response = null;
		
        // Set up mock object for non blocking connection
        INonBlockingConnection nbcMock = ( INonBlockingConnection )createMock( INonBlockingConnection.class );
        MessageMediatorDispatcherInterface messageMediatorDispatcher = createMock( MessageMediatorDispatcherInterface.class );
        ResponseGeneratorInterface responseGenerator = createMock( ResponseGeneratorInterface.class );

        // Create request
        String hl7Request = StreamUtil.resourceToString( new FileSystemResource( "./src/test/resources/chdr-allergies.rpc" ) );
        final ByteBuffer reqBuffer = MllpUtil.encode( ByteBuffer.wrap( hl7Request.getBytes() ), Charset.defaultCharset(),
                        Charset.forName( "ISO-8859-1" ) );
        
        // Set mocks on handle
        MllpMessageMediatorDataHandler dataHandler = new MllpMessageMediatorDataHandler();
        dataHandler.setAppLogger( new AssertApplicationLogger( "Error occured validating message. Access to message mediator denied.",
                        validationException ) );
        dataHandler.setAccessValidator( mllpAccessValidatorMock );
        dataHandler.setMessageMediatorDispatcher( messageMediatorDispatcher );
        dataHandler.setResponseGenerator( responseGenerator );
        
        // Set expectations
        expect( nbcMock.available()).andReturn(reqBuffer.limit());
        nbcMock.markReadPosition();
        expect( nbcMock.read( isA( ByteBuffer.class ) ) ).andAnswer( new IAnswer<Integer>()
        {
            public Integer answer( )
                throws Throwable
            {
        		ByteBuffer buffer = (ByteBuffer)EasyMock.getCurrentArguments()[0];
        		buffer.put(reqBuffer);
        		return reqBuffer.limit();
        	}
        });
        nbcMock.removeReadMark();
        expect(nbcMock.getId()).andReturn("1");
 
        mllpAccessValidatorMock.validateAccess( isA( HL7SupportInterface.class ) ) ;
        EasyMock.expectLastCall().andThrow( validationException );

        expect(
                        responseGenerator.createResponse( isA( HL7SupportInterface.class ), eq( response ), eq( "00000000000000000000" ),
                                        eq( validationException ) ) ).andReturn( "Dispatch Exception" );
        // send response
        expect( nbcMock.write( isA( ByteBuffer.class ) ) ).andReturn( 0 );
        nbcMock.flush();

        replay( nbcMock, messageMediatorDispatcher, responseGenerator, mllpAccessValidatorMock );
        dataHandler.onData( nbcMock );
        EasyMock.verify( nbcMock, messageMediatorDispatcher, responseGenerator, mllpAccessValidatorMock );
    }
    

    /***
     * Test validateMessage on MllpMessageValidator
     */
    @Test
    @Suite( groups = "checkintest", order = 4 )
    public void testOnDataWithErrorRequest( )
        throws Exception
    {
        // Set up mock object for non blocking connection
        INonBlockingConnection nbcMock = ( INonBlockingConnection )createNiceMock( INonBlockingConnection.class );

        MllpMessageMediatorDataHandler dataHandler = new MllpMessageMediatorDataHandler();

        dataHandler.setAppLogger( new AssertApplicationLogger( "Error occured creating hl7 traslation support.", new HL7SupportException(
                        ErrorCodeEnum.HL7_SUPPORT_HAPI_PARSE_EXCEPTION ) ) );

        dataHandler.setAccessValidator( mllpAccessValidator );

        // dispatcher never called, exception happened parsing message
        dataHandler.setMessageMediatorDispatcher( null );

        ResponseGeneratorInterface responseGenerator = createMock( ResponseGeneratorInterface.class );
        dataHandler.setResponseGenerator( responseGenerator );

        // Create request
        String hl7Request = "MSH";
        final ByteBuffer reqBuffer = MllpUtil.encode( ByteBuffer.wrap( hl7Request.getBytes() ), Charset.defaultCharset(),
                        Charset.forName( "ISO-8859-1" ) );

        expect( nbcMock.available()).andReturn(reqBuffer.limit());
        expect( nbcMock.read( isA( ByteBuffer.class ) ) ).andAnswer( new IAnswer<Integer>()
        {
            public Integer answer( )
                throws Throwable
            {
        		ByteBuffer buffer = (ByteBuffer)EasyMock.getCurrentArguments()[0];
        		buffer.put(reqBuffer);
        		return reqBuffer.limit();
        	}
        });
        expect(nbcMock.getId()).andReturn("1");

        String response = null;
        // Response is null and hl7support is not null after we tried to use NonHAPI.
        expect(
                        responseGenerator.createResponse( ( HL7SupportInterface )EasyMock.isNull(), EasyMock.eq( response ), isA( String.class ),
                                        isA( HL7SupportException.class ) ) ).andReturn( "Failed to parse message." );

        expect( nbcMock.write( isA( ByteBuffer.class ) ) ).andReturn( 0 );
        nbcMock.flush();

        replay( nbcMock, responseGenerator );

        dataHandler.onData( nbcMock );
    }

    
    /**
     * Test on disconnect
     */
    @Test
    @Suite( groups = "checkintest", order = 5 )
    public void testOnDisconnect( )
        throws Exception
    {
        // Set up mock object for non blocking connection
        INonBlockingConnection nbcMock = ( INonBlockingConnection )createNiceMock( INonBlockingConnection.class );

        MllpMessageMediatorDataHandler dataHandler = new MllpMessageMediatorDataHandler();
        dataHandler.setAccessValidator( mllpAccessValidator );

        replay( nbcMock );

        boolean val = dataHandler.onDisconnect( nbcMock );
        assertTrue( val );
    }


    @Test
    @Suite( groups = "checkintest", order = 6 )
    public void testOnDisconnect2( )
        throws Exception
    {

        MllpMessageMediatorDataHandler mllpMessageMediatorDataHandler = new MllpMessageMediatorDataHandler();

        INonBlockingConnection nbcMock = ( INonBlockingConnection )createNiceMock( INonBlockingConnection.class );
        replay( nbcMock );

        boolean val = mllpMessageMediatorDataHandler.onDisconnect( nbcMock );
        assertTrue( val );

    }

}
