

package gov.va.med.cds.socket.server.handler;


import gov.va.med.cds.exception.ErrorCodeEnum;

import gov.va.med.cds.exception.MllpIllegalEncodingException;

import gov.va.med.cds.hapi.HL7SupportException;

import gov.va.med.cds.hapi.HL7SupportInterface;

import gov.va.med.cds.hapi.HL7Utility;

import gov.va.med.cds.hapi.Hl7SupportHdrLegacy;

import gov.va.med.cds.logger.ApplicationLoggerInterface;

import gov.va.med.cds.logger.LogApplicationLogger;

import gov.va.med.cds.util.MllpUtil;

import java.io.IOException;

import java.nio.BufferUnderflowException;

import java.nio.ByteBuffer;

import java.nio.channels.ClosedChannelException;

import java.nio.charset.Charset;

import org.apache.commons.logging.Log;

import org.apache.commons.logging.LogFactory;

import org.xsocket.MaxReadSizeExceededException;

import org.xsocket.connection.IDataHandler;

import org.xsocket.connection.IDisconnectHandler;

import org.xsocket.connection.INonBlockingConnection;


public class MllpMessageMediatorDataHandler
    implements
        IDataHandler,
        IDisconnectHandler
{
    /** Logger to be used by the data handler. */
    private static Log logger = LogFactory.getLog( MllpMessageMediatorDataHandler.class );
    /** A handle to the access validator that will validate the access parameters in the request. */
    private AccessValidatorInterface accessValidator = new NoOpAccessValidator();
    /** Responsible for dispatching messages to the appropriate message mediator instance */
    private MessageMediatorDispatcherInterface messageMediatorDispatcher = null;
    /** Responsible for generating client specific response messages */
    private ResponseGeneratorInterface responseGenerator = null;
    public ApplicationLoggerInterface appLogger = new LogApplicationLogger( logger );


    /**

     * Reads hl7 message off the wire, processes it and sends response back to a client

     * @param connection

     * @return 

     */
    public boolean onData( INonBlockingConnection connection )
        throws IOException,
            BufferUnderflowException,
            ClosedChannelException,
            MaxReadSizeExceededException
    {
        String mediatorResponse = null;
        Exception exception = null;
        HL7SupportInterface hl7Support = null;
        ByteBuffer decodedBytes = null;
        String responseMessageControlId = null;
        try
        {
            int available = connection.available();
            if ( available <= 0 )
            {
                return true;
            }
            connection.markReadPosition();
            ByteBuffer buffer = ByteBuffer.allocate( available );
            int read = connection.read( buffer );
            assert ( available == read );
            decodedBytes = MllpUtil.decode( new ByteBuffer[] { buffer }, Charset.forName( "ISO-8859-1" ), Charset.defaultCharset() );
            if ( decodedBytes == null )
            {
                // If the decoded bytes are null, then that means that the data available to be read 
                // from the connection was not a complete message (eg. SOB+MESSAGE+EOB). By throwing the 
                // BufferUnderflowException, things will be reset and the onData method will be called
                // again when more data is available to be read.
                connection.resetToReadMark();
                throw new BufferUnderflowException();
            }
            connection.removeReadMark();
            /** Use the connection id as the default value for the response message control id. **/
            responseMessageControlId = connection.getId();
            // If this fails, then try the NonHapiHL7Support try/catch HL7SupportException
            hl7Support = new Hl7SupportHdrLegacy( new String( decodedBytes.array() ) );
            /** Call to hdr2 database required to create message controlId this will override the default value.

             * Note: This logic will prevent sending messages to queues if data base is down.  

             **/
            responseMessageControlId = createMessageControlId();
            // Validate access throws exception if access denied
            accessValidator.validateAccess( hl7Support );
            /** Process message using message specific processor instance. **/
            mediatorResponse = messageMediatorDispatcher.dispatch( hl7Support );
        }
        catch ( MllpIllegalEncodingException mle )
        {
            /** Log exceptions caused by decode **/
            exception = mle;
            appLogger.logMessage( "Error occured decoding message.", mle );
        }
        catch ( HL7SupportException hse )
        {
            exception = hse;
            appLogger.logMessage( "Error occured creating hl7 traslation support.", hse );
        }
        catch ( AccessValidatorException ave )
        {
            /** Error validating message. */
            exception = ave;
            appLogger.logMessage( "Error occured validating message. Access to message mediator denied.", ave );
        }
        catch ( DataBaseAccessException dae )
        {
            /** Error creating message control id **/
            exception = dae;
            appLogger.logMessage( "Error connecting to database", dae );
        }
        catch ( DispatchException mde )
        {
            /** Log exception caused by dispatch **/
            exception = mde;
            appLogger.logMessage( "Error occured processing message.", mde );
        } // Only catch our exceptions let framework handle runtime and unchecked exceptions
        /** Create response to client and indicate what errors occurred processing the message **/
        String response = responseGenerator.createResponse( hl7Support, mediatorResponse, responseMessageControlId, exception );
        sendResponse( response, connection );
        return true;
    }


    /**

     * Wrap any database exception into checked exception

     * @param hl7Support

     * @return

     * @throws DataBaseAccessException

     */
    private String createMessageControlId( )
        throws DataBaseAccessException
    {
        String responseMessageControlId = null;
        try
        {
            responseMessageControlId = HL7Utility.createMessageControlId();
        }
        catch ( Throwable e )
        {
            throw new DataBaseAccessException( ErrorCodeEnum.DATABASE_ACCESS_EXCEPTION, e );
        }
        return responseMessageControlId;
    }


    public boolean onDisconnect( INonBlockingConnection arg0 )
        throws IOException
    {
        return true;
    }


    /**

     * Sends response back to a client

     * @param response

     * @param connection

     * @throws IOException

     */
    private void sendResponse( String response, INonBlockingConnection connection )
        throws IOException
    {
        /** Convert Response to ISO-8859-1 - Happens in the MLLP Utility encode method. **/
        /** Send response **/
        connection.write( MllpUtil.encode( ByteBuffer.wrap( response.getBytes() ), Charset.defaultCharset(), Charset.forName( "ISO-8859-1" ) ) );
        connection.flush();
    }


    public void setAppLogger( ApplicationLoggerInterface appLogger )
    {
        this.appLogger = appLogger;
    }


    public void setAccessValidator( AccessValidatorInterface accessValidator )
    {
        this.accessValidator = accessValidator;
    }


    public void setMessageMediatorDispatcher( MessageMediatorDispatcherInterface messageMediatorDispatcher )
    {
        this.messageMediatorDispatcher = messageMediatorDispatcher;
    }


    public void setResponseGenerator( ResponseGeneratorInterface responseGenerator )
    {
        this.responseGenerator = responseGenerator;
    }
}
