

package gov.va.med.cds.socket.server;


import gov.va.med.cds.properties.PropertiesUtil;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.log4j.xml.DOMConfigurator;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.FileSystemXmlApplicationContext;
import org.xsocket.connection.ConnectionUtils;
import org.xsocket.connection.IServer;
import java.io.File;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.rmi.UnmarshalException;
import java.util.ArrayList;
import java.util.List;
import javax.management.JMException;
import javax.management.MBeanServerConnection;
import javax.management.ObjectName;


public class ServerManager
{
    private static Log logger = LogFactory.getLog( ServerManager.class );
    /** The application context used by this application. */
    private static ApplicationContext applicationContext = null;
    /** List of servers running under the instance of the framework. */
    private List<IServer> servers = new ArrayList<IServer>();


    /**
     * Default constructor for class made private so that creation is controlled.
     */
    private ServerManager( )
    { /* no-op */

    }


    public void startServers( )
        throws Exception
    {
        for ( IServer server : servers )
        {
            try
            {
                ConnectionUtils.registerMBean( server );
                ConnectionUtils.start( server );
            }
            catch ( Exception e )
            {
                logger.error( String.format( "Error starting server listening on %s:%d", server.getLocalAddress(), server.getLocalPort() ), e );
                throw e;
            }
        }
    }
    private static final String DEFUALT_DOMAIN = "org.xsocket.connection";


    private static void unregisterMBean( IServer server )
        throws JMException
    {
        String address = server.getLocalAddress().getCanonicalHostName() + ":" + server.getLocalPort();
        address = address.replace( ":", "_" );
        // unregister handler
        ObjectName hdlObjectName = new ObjectName( DEFUALT_DOMAIN + ".server." + address + ":type=" + server.getHandler().getClass().getSimpleName() );
        ManagementFactory.getPlatformMBeanServer().unregisterMBean( hdlObjectName );
        // unregister worker pool
        ObjectName workerpoolObjectName = new ObjectName( DEFUALT_DOMAIN + ".server." + address + ":type=Workerpool" );
        ManagementFactory.getPlatformMBeanServer().unregisterMBean( workerpoolObjectName );
    }


    public void stopServers( )
        throws Exception
    {
        for ( IServer server : this.servers )
        {
            try
            {
                unregisterMBean( server );
                logger.info( String.format( "Shutting down server listening on %s:%d", server.getLocalAddress(), server.getLocalPort() ) );
                server.close();
            }
            catch ( IOException e )
            {
                logger.error( "Error shutting down server." );
            }
        }
    }


    /*
      * (non-Javadoc)
      * 
      * @see gov.va.med.hds.hdr.nio.server.ServerManagerInterface#shutdown()
      */
    public void shutdown( )
        throws Exception
    {
        stopServers();
        System.exit( 0 );
    }


    /**
      * Entry point for Server Manager application.
      * 
      * @param args
      */
    public static void main( String[] args )
    {
        allMethodsFromMain( args );
    }


    /** convenience Method for unit testing **/
    public static void allMethodsFromMain( String[] args )
    {
        configureLoggingAndProperties();
        if ( isShutdown( args ) )
        {
            doShutdown();
        }
    }


    private static boolean isShutdown( String[] args )
    {
        return args.length != 0 && "-shutdown".equals( args[0] ) ? true : false;
    }


    private static void doShutdown( )
    {
        try
        {
            MBeanServerConnection mbsc = ( MBeanServerConnection )applicationContext.getBean( "clientConnector" );
            // MBeanServer's domain
            String domain = "gov.va.med.cds.adapter.socket";
            // Create MigrationManager MBean
            ObjectName stdMBeanName = new ObjectName( domain + ":name=ServerManager" );
            mbsc.getObjectInstance( stdMBeanName );
            mbsc.invoke( stdMBeanName, "shutdown", null, null );
        }
        catch ( UnmarshalException ue )
        {
            // this is expected here.
        }
        catch ( Exception e )
        {
            logger.error( "Error invoking ServerManager.shutdown MBean operation.", e );
        }
        finally
        {
            System.exit( 0 );
        }
    }


    private static void configureLoggingAndProperties( )
    {
        // configure the runtime environment
        try
        {
            // load up the system properties
            new PropertiesUtil().loadSystemRunProperties();
            // load up the log4j configuration
            File logFile = new File( System.getProperty( "log4j.configuration" ) );
            DOMConfigurator.configure( logFile.getAbsolutePath() );
            logger.info( "using log file: " + logFile.getAbsolutePath() );
            // load up the Spring application context
            applicationContext = new FileSystemXmlApplicationContext( new String[] { System.getProperty( "socket.adapter.context" ) } );
        }
        catch ( Throwable t )
        {
            // if any errors are encountered while starting up the server manager, shut it down.
            t.printStackTrace();
            System.err.println( "Error configuring application logging and properties. Exiting CDS socket adapter application." );
            System.exit( -1 );
        }
    }


    /**
      * Sets the list of servers that this server manager manages.
      * 
      * @param servers The servers that this server manager manages.
      */
    public void setServers( List<IServer> servers )
        throws Exception
    {
        this.servers = servers;
        startServers();
    }


    public static ServerManager createServerManagerInstance( )
    {
        ServerManager serverManager = new ServerManager();
        return serverManager;
    }

}