

package gov.va.med.cds.client.proxy;

import static org.easymock.EasyMock.createMock;
import static org.easymock.EasyMock.createNiceMock;
import static org.easymock.EasyMock.expect;
import static org.easymock.EasyMock.getCurrentArguments;
import static org.easymock.EasyMock.isA;
import static org.easymock.EasyMock.replay;
import gov.va.med.cds.junit.runners.Suite;
import gov.va.med.cds.junit.runners.SuiteAwareRunner;
import gov.va.med.cds.testharness.AbstractLoggingTest;
import gov.va.med.cds.util.MllpUtil;

import java.nio.ByteBuffer;
import java.nio.charset.Charset;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.easymock.IAnswer;
import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.xsocket.connection.IBlockingConnection;


/**
 * This test will ensure socket based client proxy is operational.
 * 
 * 
 */

@RunWith( SuiteAwareRunner.class )
public class MllpSocketBasedClientProxyTest
    extends
        AbstractLoggingTest
{
    private static final Log logger = LogFactory.getLog( MllpSocketBasedClientProxyTest.class );


    /**
     * Test sendMessage on the MllpSocketBasedClientProxy
     * 
     */
    @Test
    @Suite( groups = { "checkintest" } )
    public void testSendMessage( )
    {
        String request = "MllpSocketBasedClientProxyTest ACK Message";
        String response = "MllpSocketBasedClientProxyTest ACK Response Message";
        ByteBuffer requestBuffer = MllpUtil.encode(ByteBuffer.wrap(request.getBytes()), Charset.defaultCharset(), Charset.forName( "ISO-8859-1" ) );
        final ByteBuffer responseBuffer = MllpUtil.encode(ByteBuffer.wrap(response.getBytes()), Charset.defaultCharset(), Charset.forName( "ISO-8859-1" ));
        
        String host = "localhost";
        int port = 5001;

        MllpSocketBasedClientProxy mllpSocketProxy = null;

        try
        {
            IBlockingConnection mockCon = createNiceMock( IBlockingConnection.class );
            IBlockingConnectionFactory mockFact = createMock( IBlockingConnectionFactory.class );

            expect( mockFact.getConnection( host, port ) ).andReturn( mockCon );

            mockCon.setAutoflush( false );
            expect( mockCon.write( requestBuffer ) ).andReturn( requestBuffer.limit() );
            mockCon.flush();
            expect( mockCon.read( isA( ByteBuffer.class ) ) ).andAnswer(new IAnswer<Integer>() {
            	public Integer answer() throws Throwable {
            		ByteBuffer buffer = (ByteBuffer)getCurrentArguments()[0];
            		buffer.put(responseBuffer);
            		buffer.limit(responseBuffer.limit());
            		return buffer.limit();
            	}
            });
            mockCon.close();

            replay( mockFact, mockCon );

            mllpSocketProxy = new MllpSocketBasedClientProxy();
            mllpSocketProxy.setBlockingConnectionFactory( mockFact );
            mllpSocketProxy.setHost( host );
            mllpSocketProxy.setPort( port );
        }
        catch ( Exception e )
        {
            logger.error( "Blocking connection mock object failure " + e.getMessage() + " " + e );
            Assert.fail( "Unexpected Blocking connection mock object failure  " + e.getMessage() );
        }

        String r = mllpSocketProxy.sendMessage( new String(requestBuffer.array()), null ).trim();
        Assert.assertEquals( response, r );
    }


    /**
     * Test sendMessage on the MllpSocketBasedClientProxy
     * without specifying the properties
     * 
     */
    @Test( expected = IllegalArgumentException.class )
    @Suite( groups = { "checkintest" } )
    public void testSendMessageException( )
    {
        String msg = "MllpSocketBasedClientProxyTest ACK Message";
        String encMsg = ( char )MllpUtil.SOB + msg + ( char )MllpUtil.EOB + ( char )MllpUtil.CR;
        //       Properties properties = new Properties();

        MllpSocketBasedClientProxy mllpSocketProxy = null;
        mllpSocketProxy = new MllpSocketBasedClientProxy();
        mllpSocketProxy.sendMessage( encMsg, null );
    }

}
