

package gov.va.med.cds.client.proxy;


import gov.va.med.cds.exception.ErrorCodeEnum;
import gov.va.med.cds.exception.SocketClientException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.xsocket.connection.BlockingConnection;
import org.xsocket.connection.IBlockingConnection;

import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.BufferOverflowException;


/**
 * Concrete connection factory class used to create a
 * new blocking connection each time the getConnection
 * interface is called 
 * 
 * @author susarlan
 *
 */
public class BlockingConnectionFactory
    implements
        IBlockingConnectionFactory
{
 	private static Log logger = LogFactory.getLog( BlockingConnectionFactory.class );
 	
    private int connectionTimeoutMillis;
    private int readTimeoutMillis;
    private int idleTimeoutMillis;


    /***
     * Returns a new blocking connection to the the specified host and port.
     *  
     * TODO Modify signature to throw app friendly exception
     * 
     * @param : String host - host name of the server to which to connect to
     * @param : int port - port number on the host server to which to connect to
     * @throws : ConnectionException
     * @return : IBlockingConnection - a new blocking connection to the specified host and port
     */
    public IBlockingConnection getConnection( String host, int port )
    {
        IBlockingConnection blockingConnection = null;
        InetAddress inetAddress = null;
        try
        {
            inetAddress = InetAddress.getByName( host.trim() );
        }
        catch ( UnknownHostException e1 )
        {
            logger.error( "Unknown host specified", e1 );
            throw new SocketClientException( ErrorCodeEnum.SOCKET_CLIENT_UNKNOWN_HOST_EXCEPTION, e1 );
        }
        try
        {
            blockingConnection = new BlockingConnection( inetAddress, port,connectionTimeoutMillis);
            blockingConnection.setReadTimeoutMillis(readTimeoutMillis);
            blockingConnection.setIdleTimeoutMillis(idleTimeoutMillis);
            
        }
        catch ( BufferOverflowException boe )
        {
            // logger.error( "Buffer Overflow Exception:", boe );
            // Eat this one
        }
        catch ( IOException e )
        {
            logger.error( "Socket timeout exception occurred", e );
            throw new SocketClientException( ErrorCodeEnum.SOCKET_CLIENT_IO_EXCEPTION, e );
        }
        return blockingConnection;
    }


	public void setConnectionTimeoutMillis(int connectionTimeoutMillis) {
		this.connectionTimeoutMillis = connectionTimeoutMillis;
	}


	public void setReadTimeoutMillis(int readTimeoutMillis) {
		this.readTimeoutMillis = readTimeoutMillis;
	}


	public void setIdleTimeoutMillis(int idleTimeoutMillis) {
		this.idleTimeoutMillis = idleTimeoutMillis;
	}
    
}
