/**
 * 
 */


package gov.va.med.mediator.client.messagemediator;


import gov.va.med.cds.util.DateTimeUtil;

import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.util.Date;
import java.util.Map;

import javax.jms.BytesMessage;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.Session;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.jms.core.JmsTemplate;
import org.springframework.jms.core.MessageCreator;


/**
 * @author vhaislegberb
 *
 */
public class JmsMessageMediatorDelegate
    implements
        MessageMediatorInterface
{

    public static final String SUCCESS = "SUCCESS";
    public static final String ERROR_SECTION = "errorSection";

    private enum SendingOrganization
    {
        VHA, DOD, IHS
    };

    private static Log logger = LogFactory.getLog( JmsMessageMediatorDelegate.class );

    private Map<String, JmsTemplate> sendingApplicationToJmsTemplateMap;

    private SendingOrganization sendingOrganization;


    /*
     * (non-Javadoc)
     * @see gov.va.med.cds.client.messagemediator.MessageMediatorInterface#isAlive()
     */
    public boolean isAlive( )
    {
        return true;
    }


    /*
     * (non-Javadoc)
     * @see gov.va.med.cds.client.messagemediator.MessageMediatorInterface#processMessage(java.lang.String)
     */
    public String processMessage( String hl7MessageRequest )
    {
        return processMessage( hl7MessageRequest, false );
    }


    /*
     * (non-Javadoc)
     * @see gov.va.med.cds.client.messagemediator.MessageMediatorInterface#processMessage(java.lang.String, java.lang.Boolean)
     */
    public String processMessage( final String hl7MessageRequest, final Boolean persistOriginalMessage )
    {
        try
        {
            JmsTemplate jmsTemplate = null;

            switch ( sendingOrganization )
            {
            case VHA:
                String sendingApplication = getMessageSendingApplication( hl7MessageRequest );
                jmsTemplate = sendingApplicationToJmsTemplateMap.get( sendingApplication );

                if ( jmsTemplate == null )
                {
                    throw new IllegalArgumentException( String.format( "No JMS Template defined for '%s'.", sendingApplication ) );
                }

                break;
            case DOD:
                jmsTemplate = sendingApplicationToJmsTemplateMap.get( "CHDR_QUEUE" );
                break;
            case IHS:
            default:
                throw new IllegalArgumentException( "Invalid sending orgainzation." );
            }

            sendFromJMSTemplate( jmsTemplate, hl7MessageRequest, persistOriginalMessage );

        }
        catch ( Exception e )
        {
            logger.debug( "Error processing message.", e );

            return createErrorMessage( e.getMessage() );
        }

        return SUCCESS;
    }


    protected String createErrorMessage( String message )
    {
        //get VDEFVIE4 and date
        String vdef = getVDEFString( message );
        String date = createMessageDateTime();

        String errorMessage = "<errorSection>" + "CDS-Socket-Facade: Error sending message to queue. " + date + " " + vdef + "</errorSection>";
        return errorMessage;
    }


    protected String getVDEFString( String message )
    {
        if ( message == null )
            return "";
        int beginIndex = message.indexOf( "VDEF" );
        if ( beginIndex < 0 )
        {
            return "";
        }
        String vdefString = message.substring( beginIndex );
        return ( vdefString.indexOf( " " ) < 0 ) ? vdefString : vdefString.substring( 0, vdefString.indexOf( " " ) );
    }

    // MSH|^~\&|LA7LAB|...  - Nothing else beyond this point matters.
    public String getMessageSendingApplication( String hl7MessageRequest )
        throws Exception
    {
        String messageSendingApplication = "";
        String[] parsedMSH = null;
        
        if ( ( hl7MessageRequest != null ) && hl7MessageRequest.startsWith( "MSH" ) )
        {
            char fieldSeparator = hl7MessageRequest.charAt( 3 );
            parsedMSH = hl7MessageRequest.split( String.valueOf( "\\" + fieldSeparator ) );            
            messageSendingApplication = parsedMSH[2];
        }
        
        if ( "HDRVTLS".equals( messageSendingApplication ) )
        {
            if ( "HTH HDR".equals( parsedMSH[4] ) )
            {
                messageSendingApplication += "-HTH";
            }
        }
        
        return messageSendingApplication;
    }


    public void setSendingOrganization( String sendingOrganization )
    {
        this.sendingOrganization = SendingOrganization.valueOf( sendingOrganization );
    }


    public void setSendingApplicationToJmsTemplateMap( Map<String, JmsTemplate> sendingApplicationToJmsTemplateMap )
    {
        this.sendingApplicationToJmsTemplateMap = sendingApplicationToJmsTemplateMap;
    }


    public void sendFromJMSTemplate( JmsTemplate jmsTemplate, final String hl7MessageRequest, final Boolean persistOriginalMessage )
    {
        jmsTemplate.send( new MessageCreator()
        {
            public Message createMessage( Session session )
                throws JMSException
            {
                BytesMessage message = session.createBytesMessage();
                message.setBooleanProperty( MessageMediatorInterface.PROP_PERSIST_ORIGINAL, persistOriginalMessage );

                // encode the HL7 message
                Charset charset = Charset.forName( "ISO-8859-1" );
                ByteBuffer buffer = charset.encode( hl7MessageRequest );

                message.writeBytes( buffer.array() );
                return message;
            }
        } );

    }


    protected String createMessageDateTime( )
    {
        return DateTimeUtil.getDateTimeDotSeparated( new Date().getTime(), null );
    }
}
