/**
 * 
 */
package gov.va.med.cds.hapi;

import gov.va.med.cds.junit.runners.Suite;
import gov.va.med.cds.junit.runners.SuiteAwareRunner;

import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;

/**
 * @author vhaislegberb
 *
 */
@RunWith( SuiteAwareRunner.class )
public class Hl7SupportHdrLegacyTest
{
    
    private static final String TEST_MESSAGE_GOOD = "MSH^~|\\&^HDRVTLS^200TX~[LOCALHOST]~DNS^HTH HDR^200HD~SERVER.server123.DOMAIN.EXT~DNS^20070215144357-0500^^ORU~R01^320070215144357^T^2.4^^^AL^SU^USA^^^^\rPID^";
    
    private static final String TEST_MESSAGE_SPQ_Q04 = "MSH^~|\\&^HDRVTLS^200TX~[LOCALHOST]~DNS^HTH HDR^200HD~SERVER.server123.DOMAIN.EXT~DNS^20070215144357-0500^^SPQ~Q04^320070215144357^T^2.4^^^AL^SU^USA^^^^\rPID^\rSPR^SPR-1^SPR-2^SPR-3^SPR-4-1~SPR-4-2-1&SPR-4-2-2&SPR-4-2-3&SPR-4-2-4^SPR-5";
    
    private static final String TEST_MESSAGE_QRY_Q01 = "MSH^~|\\&^HDRVTLS^200TX~[LOCALHOST]~DNS^HTH HDR^200HD~SERVER.server123.DOMAIN.EXT~DNS^20070215144357-0500^^SPQ~Q04^320070215144357^T^2.4^^^AL^SU^USA^^^^\rPID^\rQPD^QPD-1^QPD-2^QPD-3^QPD-4\r";
    
    private static final String TEST_MESSAGE_ACK = "MSH^~|\\&^HTH HDR^200HD~SERVER.server123.DOMAIN.EXT~DNS^HDRVTLS-HTH^200TX~server_1217.vha.domain.ext~DNS^20100723064632.321-0600^^ACK^00000000000000017436^P^2.4^^^AL^NE\rMSA^AA^320070215144357";
    
    /***
     * Test for exception missing MSH segment from validate message on
     * MllpMessageValidator
     */
    @Test ( expected = HL7SupportException.class )
    @Suite( groups = "checkintest" )
    public void testCreateHdrLegacyHl7SupportParseException()
    {
        HL7SupportInterface hl7Support = new Hl7SupportHdrLegacy("MSH");
    }
    
    
    @Test 
    @Suite( groups = "checkintest" )
    public void testCreateHdrLegacyHl7SupportHappyPath()
    {
        HL7SupportInterface hl7Support = new Hl7SupportHdrLegacy(TEST_MESSAGE_GOOD);
        Assert.assertNotNull( hl7Support );
        
        Assert.assertEquals( "SU", hl7Support.getApplicationAcknowledgementType() );
        Assert.assertEquals( "320070215144357", hl7Support.getMessageControlId() );
        Assert.assertEquals( "HDRVTLS", hl7Support.getMessageData( "/MSH-3" ) );
        Assert.assertEquals( "20070215144357-0500", hl7Support.getMessageDateTime() );
        Assert.assertEquals( "ORU", hl7Support.getMessageType() );
        Assert.assertEquals( "HTH HDR", hl7Support.getReceivingApplication());
        Assert.assertEquals( "SERVER.server123.DOMAIN.EXT", hl7Support.getReceivingFacility());
        Assert.assertEquals( "HDRVTLS-HTH", hl7Support.getSendingApplication());
        Assert.assertEquals( "[LOCALHOST]", hl7Support.getSendingFacility());
        Assert.assertEquals( "200TX", hl7Support.getStationNumber());
        Assert.assertEquals( "T", hl7Support.getMessageData( "/MSH-11" ));
        Assert.assertEquals( "2.4", hl7Support.getMessageData("/MSH-12"));
        Assert.assertEquals( TEST_MESSAGE_GOOD, hl7Support.toEr7() );
    }
    
    
    @Test ( expected = UnsupportedOperationException.class )
    @Suite( groups = "checkintest" )
    public void testHdrLegacyHl7SupportToXml()
    {
        HL7SupportInterface hl7Support = new Hl7SupportHdrLegacy(TEST_MESSAGE_GOOD);
        Assert.assertNotNull( hl7Support );
        hl7Support.toXml();
    }

    @Test ( expected = HL7SupportException.class )
    @Suite( groups = "checkintest" )
    public void testHdrLegacyHl7SupportGetMessageDataNullResult()
    {
        HL7SupportInterface hl7Support = new Hl7SupportHdrLegacy(TEST_MESSAGE_GOOD);
        Assert.assertNotNull( hl7Support );
        
        hl7Support.getMessageData( "/PID-8" );
    }
    
    
    @Test 
    @Suite( groups = "checkintest" )
    public void testHdrLegacyHl7SupportGetAcknoledgmentCode()
    {
        HL7SupportInterface hl7Support = new Hl7SupportHdrLegacy(TEST_MESSAGE_ACK);
        Assert.assertNotNull( hl7Support );
        
        Assert.assertEquals( "AA", hl7Support.getAcknowledgementCode() );
    }
    
    
    @Test
    @Suite( groups = "checkintest" )
    public void testHdrLegacyHl7SupportGetSPQQ04UserId()
    {
        HL7SupportInterface hl7Support = new Hl7SupportHdrLegacy(TEST_MESSAGE_SPQ_Q04);
        Assert.assertNotNull( hl7Support );
        
        Assert.assertEquals( "SPR-4-2-4", hl7Support.getMessageUserId());
    }
    
    @Test
    @Suite( groups = "checkintest" )
    public void testHdrLegacyHl7SupportGetQRYQ01WhatCode()
    {
        HL7SupportInterface hl7Support = new Hl7SupportHdrLegacy(TEST_MESSAGE_QRY_Q01);
        Assert.assertNotNull( hl7Support );
        
        Assert.assertEquals( "QPD-4", hl7Support.getRequestWhatCode());
    }


    @Test
    @Suite( groups = "checkintest" )
    public void testHL7PSWriteParsing( )
    {
        String hl7Request = "MSH^~|\\&^HTAPPL^200TX~SENDING_DNS~DNS^HDR HTPS^200HD~VAHDRSVWLS01.domain.ext~DNS^20130605-0600^^MDM~T02^10000000006^T^2.4^^^AL^AL^US\rPID^^^7890007890V666666~~~USVHA~PI~&540^^tPatient~HDRIIT~TT~Jr~Dr~MD~L|555555555~~~USVHA~PI~VA FACILITY&552&L|555555555~~~USSSA~SS~VA FACILITY ID&STATION&L^^TPatient~HDRIIT~TT~~~~L^Government~~~~~~M^00000512^m^^w^^^^^^M^C^^^^^American^^^^^veteran^\rPD1^^^^66666~DRHDR~SKY~M~~~~~USVHA~L~~~NI~facility&facility number&L\rTXA^^PR^TEXT^20130214214040-0500^^20130214214040-0500^^^33251~HDRDoctor~PHARM~~~~~~~~~~~facility&station&L^33251~HDRDoctor~PHARM~~~~~~~~~~~facility&station&L^^12345~USVHA^2.0~Patient Satisfaction^^320130214214545~HT^Patient Satisfaction^PA^U^Periodic Note Status is AV ^AC^end of collection period^\rOBX^1^TX^^^<hthsurvey:hthSurvey xsi:schemaLocation=\"HTHSurvey HTHSurveys.xsd\" xmlns:hthsurvey=\"HTHSurvey\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"><psSurveyResponse><patient><icn>8888888888</icn><ssn>123123125</ssn><name><family>TTPATIENT</family><middle>P</middle><given>TRINITYPT</given></name><birthdate>1967-08-13</birthdate></patient><enrollmentDate>1967-08-13</enrollmentDate><disenrollmentDate>2012-08-13</disenrollmentDate><activationDate>1967-08-13</activationDate><inactivationDate>1967-08-13</inactivationDate><facilityNumber>402</facilityNumber><vendorNumber>200TX</vendorNumber><vendorName>Viterion</vendorName><vendorMRN>String</vendorMRN><careCoordinator><id>12345</id><name><family>HDRIITT</family><middle>T</middle><given>TRINITY</given></name></careCoordinator><scheduledDate>2012-08-13</scheduledDate><completedDate>2014-06-10T08:41:47-07:00</completedDate><currentDialogueName>String</currentDialogueName><programName>String</programName><deviceName>String</deviceName><surveyAcceptedStatus>DISENROLLED</surveyAcceptedStatus><surveyTitle>Patient Satisfaction</surveyTitle><surveyVersion>2.0</surveyVersion><surveyQuestionsAndAnswers><surveyQuestionAndAnswer><questionNumber>1</questionNumber><answer>2</answer></surveyQuestionAndAnswer><surveyQuestionAndAnswer><questionNumber>2</questionNumber><answer>1</answer></surveyQuestionAndAnswer><surveyQuestionAndAnswer><questionNumber>3</questionNumber><answer>2</answer></surveyQuestionAndAnswer><surveyQuestionAndAnswer><questionNumber>4</questionNumber><answer>2</answer></surveyQuestionAndAnswer><surveyQuestionAndAnswer><questionNumber>5</questionNumber><answer>2</answer></surveyQuestionAndAnswer><surveyQuestionAndAnswer><questionNumber>6</questionNumber><answer>2</answer></surveyQuestionAndAnswer><surveyQuestionAndAnswer><questionNumber>7</questionNumber><answer>2</answer></surveyQuestionAndAnswer><surveyQuestionAndAnswer><questionNumber>8</questionNumber><answer>2</answer></surveyQuestionAndAnswer></surveyQuestionsAndAnswers></psSurveyResponse></hthsurvey:hthSurvey>^^^^^^^^^^^^^VISN20_89004542.2~605_120.5~VISN11_90850576.2~VISN15_36032759.2\r";
        HL7SupportInterface hl7Support = new Hl7SupportHdrLegacy( hl7Request );
        Assert.assertNotNull( hl7Support );
        String sendingApplication = hl7Support.getSendingApplication();
        Assert.assertEquals( "", "HTAPPL-PS2.0", sendingApplication );
    }
}
