

package gov.va.med.repositories.fpds.webservice;


import gov.va.med.cds.client.ClinicalDataServiceInterface;
import gov.va.med.repositories.fpds.filter.FilterBuilder;
import gov.va.med.repositories.fpds.template.TemplateBuilder;
import gov.va.med.repositories.fpds.validator.Validator;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.util.List;
import java.util.Map;

import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.ValidationError;
import javax.ws.rs.ValidationException;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.UriInfo;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.json.JSONObject;
import org.json.XML;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.Resource;
import org.springframework.util.StringUtils;


@Path( "/" )
public class FederatingPatientRecordRestService
{
    private static final String DEFAULT_RESPONSE_TYPE = "json";
    private static final String XML_RESPONSE_TYPE = "xml";
    private static final String REQUEST_ID = "requestId";
    private static final String XML_REQUEST = "<clinicaldata:ClinicalData";
    private static final String TYPE = "_type";

    private Log logger = LogFactory.getLog( FederatingPatientRecordRestService.class );

    private ClinicalDataServiceInterface clinicalDataService;

    private Validator requestValidator;

    private FilterBuilder filterBuilder;
    
    private Map<String,FilterBuilder> filterBuilderMap;

    private TemplateBuilder templateBuilder;

    private String applicationName;

    private Document errorTemplateDocument;
    
    public static final String PATIENT_KEY="PGD_SINGLE_PATIENT_FILTER";
    public static final String REC_IDENTIFIER_KEY="PGD_UNIQUE_ID_FILTER";
    


    @GET
    @Path( "isAlive" )
    @Produces( { "application/json", "application/xml" } )
    public boolean isAlive( )
    {
        if ( logger.isDebugEnabled() )
            logger.debug( "GenericRestService - isAlive()" );
        return clinicalDataService.isAlive();
    }


    @GET
    @Path( "{templateId}/{filterId}/{patientId}/{domain}" )
    @Produces( { "application/json", "application/xml" } )
    public String readData( @PathParam( "templateId" )
    String templateId, @PathParam( "filterId" )
    String filterId, @PathParam( "patientId" )
    String patientId, @PathParam( "domain" )
    String domain, @Context
    UriInfo uriInfo )
    {
        String result = null;
        String requestId = null;
        MultivaluedMap<String, String> queryParameters = null;

        try
        {
            queryParameters = uriInfo.getQueryParameters();
            requestId = trimTrailingWhitespace(queryParameters.getFirst( REQUEST_ID ));

            List<ValidationError> validationErrors = requestValidator.validateRequestParameters( templateId, filterId, domain, queryParameters );

            if ( validationErrors != null && validationErrors.size() > 0 )
            {
                throw new ValidationException( Status.BAD_REQUEST, validationErrors ); // indicates that the request being made was bad
            }

            String filterRequest = this.filterBuilder.createFilter( filterId, patientId, domain, queryParameters );
            result = clinicalDataService.readClinicalData( templateId, filterRequest, filterId, requestId );

        }
        catch ( Exception e )
        {
            requestId = StringUtils.hasText( requestId ) ? requestId : "requestId";
            String msg = "Exception occured while processing read request in FPDS. ";
            StringBuilder validationErrorMessage = new StringBuilder();

            if ( e instanceof ValidationException )
            {
                ValidationException ve = ( ValidationException )e;
                List<? extends ValidationError> verrs = ve.getViolations();

                for ( ValidationError verr : verrs )
                {
                    validationErrorMessage.append( verr.getMessage() );
                }

                msg = msg.concat( validationErrorMessage.toString() );

            }

            logger.fatal( gov.va.med.cds.util.LogMessageUtil.buildMessage( domain, requestId, this.applicationName, msg ), e );
            result = buildErrorResponse( e, templateId, requestId, patientId, queryParameters, true );
        }

        return result;
    }
    
    @GET
    @Path( "{templateId}/{filterId}/{identifier}" )
    @Produces( { "application/json", "application/xml" } )
    public String readData( @PathParam( "templateId" )
    String templateId, @PathParam( "filterId" )
    String filterId, @PathParam( "identifier" )
    String identifier, @Context
    UriInfo uriInfo )
    {
        String result = null;
        String requestId = null;
        MultivaluedMap<String, String> queryParameters = null;
        final String DOMAIN="PatientGeneratedDataEvent";

        try
        {
            queryParameters = uriInfo.getQueryParameters();
            requestId = trimTrailingWhitespace(queryParameters.getFirst( REQUEST_ID ));
          
            List<ValidationError> validationErrors = requestValidator.validateRequestParameters( templateId, filterId, DOMAIN, queryParameters );

            if ( validationErrors != null && validationErrors.size() > 0 )
            {
                throw new ValidationException( Status.BAD_REQUEST, validationErrors ); // indicates that the request being made was bad
            }
                   
            FilterBuilder filtrBuilder = filterBuilderMap.get(filterId);
            if(filtrBuilder==null){
            	throw new ValidationException(Status.BAD_REQUEST,  requestValidator.constructValidationError( filterId, "Invalid Filter Id: " + filterId) );
            }
            
            String filterRequest = filtrBuilder.createFilter( filterId, identifier, DOMAIN, queryParameters );
            result = clinicalDataService.readClinicalData( templateId, filterRequest, filterId, requestId );

        }
        catch ( Exception e )
        {
            requestId = StringUtils.hasText( requestId ) ? requestId : "requestId";
            String msg = "Exception occured while processing read request in FPDS. ";
            StringBuilder validationErrorMessage = new StringBuilder();

            if ( e instanceof ValidationException )
            {
                ValidationException ve = ( ValidationException )e;
                List<? extends ValidationError> verrs = ve.getViolations();

                for ( ValidationError verr : verrs )
                {
                    validationErrorMessage.append( verr.getMessage() );
                }

                msg = msg.concat( validationErrorMessage.toString() );

            }

            logger.fatal( gov.va.med.cds.util.LogMessageUtil.buildMessage( DOMAIN, requestId, this.applicationName, msg ), e );
           
            
            result = buildErrorResponse( e, templateId, requestId, identifier, queryParameters, requestValidator.isPatientCentric(filterId) );
   
        }
  System.out.println(result);
        return result;
    }



	protected String trimTrailingWhitespace(String str) {
		
		return StringUtils.trimTrailingWhitespace(str);
	}


	@SuppressWarnings( { "rawtypes" } )
    @POST
    @Path( "{templateId}/{request}/{requestId}" )
    @Produces( { "application/json", "application/xml" } )
    @Consumes( { "application/json", "application/xml" } )
    public String writeData( @PathParam( "templateId" )
    String templateId, @PathParam( "request" )
    String request, @PathParam( "requestId" )
    String requestId, @Context
    UriInfo uriInfo )
    {
        String result = null;
        String reqst = null;
        String responseType = null;

        try
        {
            MultivaluedMap<String, String> queryParameters = uriInfo.getQueryParameters();
           
            List<ValidationError> validationErrors = requestValidator.validateRequestParameters( templateId, request, queryParameters );

            if ( validationErrors != null && validationErrors.size() > 0 )
            {
                throw new ValidationException( Status.BAD_REQUEST, validationErrors ); // indicates that the request being made was bad
            }
            
            List<String> typeList = ( List<String> )queryParameters.get( "_type" );

            if ( typeList != null )
            {
                responseType = ( String )typeList.get( 0 );
            }
            else
            {
                if ( request.contains( XML_REQUEST ) )
                {
                    responseType = XML_RESPONSE_TYPE;
                }
                else
                {
                    responseType = DEFAULT_RESPONSE_TYPE;
                }

            }
            if ( responseType.equals( DEFAULT_RESPONSE_TYPE ) )
            {
                JSONObject json = new JSONObject( request );
                String xml = XML.toString( json );
                reqst = templateBuilder.createTemplate( xml, templateId );

            }
            else
            {
                reqst = request;
            }

            //result = clinicalDataService.createClinicalData( reqst, templateId, requestId );

        }
        catch ( Exception e )
        {
            requestId = StringUtils.hasText( requestId ) ? requestId : REQUEST_ID;
            String msg = "Exception occured while processing write request in FPDS. ";

            logger.fatal( gov.va.med.cds.util.LogMessageUtil.buildMessage( templateId, requestId, this.applicationName, msg ), e );
            result = buildWriteErrorResponse( e, templateId, requestId );
        }

        return result;
    }


    @SuppressWarnings( { "rawtypes" } )
    @DELETE
    @Path( "{templateId}/{request}/{requestId}" )
    @Produces( { "application/json", "application/xml" } )
    @Consumes( { "application/json", "application/xml" } )
    public String deleteData( @PathParam( "templateId" )
    String templateId, @PathParam( "request" )
    String request, @PathParam( "requestId" )
    String requestId, @Context
    UriInfo uriInfo )
    {
        String result = null;
        String reqst = null;
        String responseType = null;

        try
        {
            MultivaluedMap<String, String> queryParameters = uriInfo.getQueryParameters();
            List<ValidationError> validationErrors = requestValidator.validateRequestParameters( templateId, request, queryParameters );

            if ( validationErrors != null && validationErrors.size() > 0 )
            {
                throw new ValidationException( Status.BAD_REQUEST, validationErrors ); // indicates that the request being made was bad
            }

            List<String> typeList = ( List<String> )queryParameters.get( "_type" );

            if ( typeList != null )
            {
                responseType = ( String )typeList.get( 0 );
            }
            else
            {
                if ( request.contains( XML_REQUEST ) )
                {
                    responseType = XML_RESPONSE_TYPE;
                }
                else
                {
                    responseType = DEFAULT_RESPONSE_TYPE;
                }
                queryParameters.add( TYPE, responseType );
            }

            if ( responseType.equals( DEFAULT_RESPONSE_TYPE ) )
            {
                JSONObject json = new JSONObject( request );
                String xml = XML.toString( json );
                reqst = templateBuilder.createTemplate( xml, templateId );

            }
            else
            {
                reqst = request;
            }

            //result = clinicalDataService.deleteClinicalData( reqst, templateId, requestId );

        }
        catch ( Exception e )
        {
            requestId = StringUtils.hasText( requestId ) ? requestId : REQUEST_ID;
            String msg = "Exception occured while processing write request in FPDS. ";

            logger.fatal( gov.va.med.cds.util.LogMessageUtil.buildMessage( templateId, requestId, this.applicationName, msg ), e );
            result = buildWriteErrorResponse( e, templateId, requestId );
        }

        return result;
    }


    private String buildErrorResponse( Exception e, String templateId, String requestId, String patientOrRecordIdentifier,
                    MultivaluedMap<String, String> queryParameters, boolean patientCentric )
    {

        /*
         * String.format( "{\"errorSection\": [{\"fatalErrors\": [{\"fatalError\": {\"errorId\": \"%s\",\"exception\": \"%s\",\"exceptionMessage\": \"CDS_GENERIC_WEB_SERVICE_ERROR\",\"errorCode\": \"CDS_GENERIC_WEB_SERVICE_ERROR\",\"displayMessage\": \"%s\"}}]}]}",
         *                   reqId, e.getClass().getName(), e.getMessage() );
         */
        Document errorDocument = ( Document )this.errorTemplateDocument.clone();
        Element errorsElement = errorDocument.getRootElement().element( "errorSection" ).element( "fatalErrors" );
        errorDocument.getRootElement().element( "templateId" ).setText( templateId );
        errorDocument.getRootElement().element( "requestId" ).setText( requestId );
        if(patientCentric){
        	errorDocument.getRootElement().element( "patients" ).element( "patient" ).element( "resultantIdentifiers" ).element( "resultantIdentifier" )
                        .element( "identity" ).setText( patientOrRecordIdentifier );
        }
        else{
        	errorDocument.getRootElement().element( "uniqueId" ).setText( patientOrRecordIdentifier );
        }

        if ( e instanceof ValidationException )
        {
            ValidationException ve = ( ValidationException )e;
            List<? extends ValidationError> verrs = ve.getViolations();

            for ( ValidationError verr : verrs )
            {
                // add a fatal exception to the document.
                Element error = DocumentHelper.createElement( "fatalError" );
                error.add( DocumentHelper.createElement( "errorId" ).addText( requestId.trim() ) );
                error.add( DocumentHelper.createElement( "exception" ).addText( ( String )verr.getClass().getName() ) );
                error.add( DocumentHelper.createElement( "exceptionMessage" ).addText( "FPDS_ERROR" ) );
                error.add( DocumentHelper.createElement( "errorCode" ).addText( "FPDS_ERROR" ) );
                error.add( DocumentHelper.createElement( "displayMessage" ).addText( ( String )verr.getMessage() ) );
                errorsElement.add( error );
            }
        }
        else
        {
            // add a fatal exception to the document.
            Element error = DocumentHelper.createElement( "fatalError" );
            error.add( DocumentHelper.createElement( "errorId" ).addText( requestId.trim() ) );
            error.add( DocumentHelper.createElement( "exception" ).addText( ( String )e.getClass().getName() ) );
            error.add( DocumentHelper.createElement( "exceptionMessage" ).addText( "FPDS_ERROR" ) );
            error.add( DocumentHelper.createElement( "errorCode" ).addText( "FPDS_ERROR" ) );
            String detailedMessage = e.getMessage();
            if(detailedMessage == null)
            {
            	/*sometimes a throwable message is null - we can't set text to null*/
            	detailedMessage="null";
            			
            }
            error.add( DocumentHelper.createElement( "displayMessage" ).addText( detailedMessage ) );

            errorsElement.add( error );
        }

        // If the type is not XML or JSON than default the error message format to JSON
        /*if ( !queryParameters.get( "_type" ).equals( "xml" ) && !queryParameters.get( "_type" ).equals( "json" ) )
        {
            return XML.toJSONObject( errorDocument.asXML() ).toString();
        }*/

        return errorDocument.asXML();
    }
    

    private String buildWriteErrorResponse( Exception e, String templateId, String requestId )
    {
        /*
         * String.format( "{\"errorSection\": [{\"fatalErrors\": [{\"fatalError\": {\"errorId\": \"%s\",\"exception\": \"%s\",\"exceptionMessage\": \"CDS_GENERIC_WEB_SERVICE_ERROR\",\"errorCode\": \"CDS_GENERIC_WEB_SERVICE_ERROR\",\"displayMessage\": \"%s\"}}]}]}",
         *                   reqId, e.getClass().getName(), e.getMessage() );
         */
        Document errorDocument = ( Document )this.errorTemplateDocument.clone();
        Element errorsElement = errorDocument.getRootElement().element( "errorSection" ).element( "fatalErrors" );
        errorDocument.getRootElement().element( "templateId" ).setText( templateId );
        errorDocument.getRootElement().element( "requestId" ).setText( requestId );

        if ( e instanceof ValidationException )
        {
            ValidationException ve = ( ValidationException )e;
            List<? extends ValidationError> verrs = ve.getViolations();

            for ( ValidationError verr : verrs )
            {
                // add a fatal exception to the document.
                Element error = DocumentHelper.createElement( "fatalError" );
                error.add( DocumentHelper.createElement( "errorId" ).addText( requestId ) );
                error.add( DocumentHelper.createElement( "exception" ).addText( ( String )verr.getClass().getName() ) );
                error.add( DocumentHelper.createElement( "exceptionMessage" ).addText( "FPDS_ERROR" ) );
                error.add( DocumentHelper.createElement( "errorCode" ).addText( "FPDS_ERROR" ) );
                error.add( DocumentHelper.createElement( "displayMessage" ).addText( ( String )verr.getMessage() ) );
                errorsElement.add( error );
            }
        }
        else
        {
            // add a fatal exception to the document.
            Element error = DocumentHelper.createElement( "fatalError" );
            error.add( DocumentHelper.createElement( "errorId" ).addText( requestId ) );
            error.add( DocumentHelper.createElement( "exception" ).addText( ( String )e.getClass().getName() ) );
            error.add( DocumentHelper.createElement( "exceptionMessage" ).addText( "FPDS_ERROR" ) );
            error.add( DocumentHelper.createElement( "errorCode" ).addText( "FPDS_ERROR" ) );
            String detailedMessage = e.getMessage();
            if(detailedMessage == null)
            {
            	/*sometimes a throwable message is null - we can't set text to null*/
            	detailedMessage="null";
            			
            }
            error.add( DocumentHelper.createElement( "displayMessage" ).addText(detailedMessage ) );

            errorsElement.add( error );
        }

        return errorDocument.asXML();
    }


    public void setApplicationName( String applicationName )
    {
        this.applicationName = applicationName;
    }


    public void setClinicalDataService( ClinicalDataServiceInterface clinicalDataService )
    {
        this.clinicalDataService = clinicalDataService;
    }


    public void setFilterBuilder( FilterBuilder filterBuilder )
    {
        this.filterBuilder = filterBuilder;
    }
    

	public void setFilterBuilderMap(Map<String,FilterBuilder> filterBuilderMap) 
	{
		this.filterBuilderMap = filterBuilderMap;
	}


    public void setRequestValidator( Validator requestValidator )
    {
        this.requestValidator = requestValidator;
    }


    public void setErrorTemplateDocument( String errorTemplateDocument )
        throws Exception
    {
        this.errorTemplateDocument = loadDocumentFromResouce( new ClassPathResource( errorTemplateDocument ) );
    }


    public void setTemplateBuilder( TemplateBuilder templateBuilder )
    {
        this.templateBuilder = templateBuilder;
    }


    private Document loadDocumentFromResouce( Resource resource )
        throws Exception
    {
        BufferedReader br = null;

        try
        {
            br = new BufferedReader( new InputStreamReader( resource.getInputStream() ) );
            StringBuffer sbResouce = new StringBuffer();
            String line = null;

            while ( ( line = br.readLine() ) != null )
            {
                sbResouce.append( line.trim() ).append( '\n' );
            }

            return DocumentHelper.parseText( sbResouce.toString() );
        }
        finally
        {
            if ( br != null )
            {
                br.close();
            }
        }
    }
}


