

package gov.va.med.cds.util;


import gov.va.med.cds.exception.EncryptionException;
import gov.va.med.cds.exception.ErrorCodeEnum;

import org.apache.commons.codec.binary.Base64;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableEntryException;
import java.security.cert.CertificateException;
import java.util.Enumeration;
import java.util.Properties;

import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.KeyGenerator;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKey;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;


public class CipherUtilities
{
    public static final String ENCRYPTED_PREFIX = "{AES}";

    private static final String ENCRYPTION_ALGORITHM = "AES";

    
    public static SecretKey generateKey( )
    {

        KeyGenerator kgen = null;
        try
        {
            kgen = KeyGenerator.getInstance( ENCRYPTION_ALGORITHM );
            kgen.init( 128 );
        }
        catch ( NoSuchAlgorithmException e )
        {
            throw new EncryptionException( ErrorCodeEnum.ENCRYPTION_MISSING_KEYSTORE_FILEPATH_EXCEPTION );
        }

        return kgen.generateKey();
    }


    public static SecretKeySpec generateKeySpec( )
    {
        SecretKey skey = generateKey();
        byte[] raw = skey.getEncoded();

        return new SecretKeySpec( raw, ENCRYPTION_ALGORITHM );
    }


    public static SecretKeySpec generateKeySpec( SecretKey skey )

    {
        byte[] raw = skey.getEncoded();
        return new SecretKeySpec( raw, ENCRYPTION_ALGORITHM );
    }


    public static SecretKeySpec configureKeyStore( String keyStoreFileLocation, String keyStorePassword )
        throws IOException
    {
        java.io.FileInputStream fileInStream = null;

        KeyStore kstore = null;

        if ( keyStorePassword == null )
        {
            throw new EncryptionException( ErrorCodeEnum.ENCRYPTION_MISSING_KEYSTORE_PASSWORD_EXCEPTION );
        }
        if ( keyStoreFileLocation == null )
        {
            throw new EncryptionException( ErrorCodeEnum.ENCRYPTION_MISSING_KEYSTORE_FILEPATH_EXCEPTION );
        }

        try
        {

            kstore = KeyStore.getInstance( "JCEKS" );

            // get user password and file input stream
            char[] password = keyStorePassword.toCharArray();

            fileInStream = new java.io.FileInputStream( keyStoreFileLocation );

            kstore.load( fileInStream, password );

        }
        catch ( KeyStoreException e )
        {
            throw new EncryptionException( ErrorCodeEnum.ENCRYPTION_EXCEPTION, e );
        }
        catch ( NoSuchAlgorithmException e )
        {
            throw new EncryptionException( ErrorCodeEnum.ENCRYPTION_EXCEPTION, e );
        }
        catch ( CertificateException e )
        {
            throw new EncryptionException( ErrorCodeEnum.ENCRYPTION_EXCEPTION, e );
        }
        finally
        {
            if ( fileInStream != null )
            {
                fileInStream.close();
            }
        }

        return retrieveKeyEntry( keyStoreFileLocation, keyStorePassword, kstore );

    }


    private static SecretKeySpec retrieveKeyEntry( String keyStoreFileLocation, String keyStorePassword, KeyStore kstore )
        throws IOException
    {

        KeyStore.PasswordProtection kp = new KeyStore.PasswordProtection( keyStorePassword.toCharArray() );

        // get secret key
        SecretKey secretKey = null;
        KeyStore.SecretKeyEntry secretEntry = null;
        try
        {
            secretEntry = ( KeyStore.SecretKeyEntry )kstore.getEntry( "secretKeyAlias", kp );

            if ( secretEntry == null )
            {
                // save secret key
                secretKey = generateKey();
                KeyStore.SecretKeyEntry skEntry = new KeyStore.SecretKeyEntry( secretKey );
                kstore.setEntry( "secretKeyAlias", skEntry, kp );

                // store away the keystore
                java.io.FileOutputStream fos = new java.io.FileOutputStream( keyStoreFileLocation );
                kstore.store( fos, keyStorePassword.toCharArray() );
                fos.close();
            }

            else
            {
                secretKey = secretEntry.getSecretKey();
            }
        }
        catch ( NoSuchAlgorithmException e )
        {
            throw new EncryptionException( ErrorCodeEnum.ENCRYPTION_EXCEPTION, e );
        }
        catch ( UnrecoverableEntryException e )
        {
            throw new EncryptionException( ErrorCodeEnum.ENCRYPTION_EXCEPTION, e );
        }
        catch ( KeyStoreException e )
        {
            throw new EncryptionException( ErrorCodeEnum.ENCRYPTION_EXCEPTION, e );
        }
        catch ( CertificateException e )
        {
            throw new EncryptionException( ErrorCodeEnum.ENCRYPTION_EXCEPTION, e );
        }

        return generateKeySpec( secretKey );
    }

    public static String encrypt( SecretKeySpec key, String toEncrypt )
    {
    	return encrypt( key, toEncrypt, true );
    }

    /**
     * 
     * @param key
     * @param toEncrypt
     * @return
     */
    public static String encrypt( SecretKeySpec key, String toEncrypt, boolean addPrefix )
    {
        
        byte[] encrypted = null;
        try
        {
            Cipher cipher = Cipher.getInstance( ENCRYPTION_ALGORITHM );
            cipher.init( Cipher.ENCRYPT_MODE, key );
            
            byte[] in =  toEncrypt.getBytes();
            encrypted = cipher.doFinal( in );
        }
        catch ( InvalidKeyException e )
        {
            throw new EncryptionException( ErrorCodeEnum.ENCRYPTION_EXCEPTION, e );
        }
        catch ( NoSuchAlgorithmException e )
        {
            throw new EncryptionException( ErrorCodeEnum.ENCRYPTION_EXCEPTION, e );
        }
        catch ( NoSuchPaddingException e )
        {
            throw new EncryptionException( ErrorCodeEnum.ENCRYPTION_EXCEPTION, e );
        }
        catch ( IllegalBlockSizeException e )
        {
            throw new EncryptionException( ErrorCodeEnum.ENCRYPTION_EXCEPTION, e );
        }
        catch ( BadPaddingException e )
        {
            throw new EncryptionException( ErrorCodeEnum.ENCRYPTION_EXCEPTION, e );
        }
        
        //use 64 base encoding to preserve padding, when convert bytes to string
        String encryptedString = Base64.encodeBase64String( encrypted );
        
        if (addPrefix)
        {
        	encryptedString = ENCRYPTED_PREFIX + encryptedString;
    }
        return encryptedString;
    }



    public static void encryptAndSaveProperties( Properties props, SecretKeySpec key, String propertyFilePath )
        throws IOException
    {
        Properties propsToBeSaved = ( Properties )props.clone();

        Enumeration<?> e = propsToBeSaved.propertyNames();
        String propKey = ( String )e.nextElement();
        if ( propKey.endsWith( ".password" ) )
        {
            String propValue = propsToBeSaved.getProperty( propKey );
            propsToBeSaved.setProperty( propKey, encrypt( key, propValue ) );
        }

        propsToBeSaved.store( new FileOutputStream( new File( propertyFilePath ) ), "File encrypted" );
    }

    /**
     * Used for Cache calls
     * @param data
     * @param key
     * @return
     * @throws NoSuchAlgorithmException
     * @throws NoSuchPaddingException
     * @throws IllegalBlockSizeException
     * @throws BadPaddingException
     * @throws InvalidKeyException
     * @throws InvalidAlgorithmParameterException
     */
	public static String encrypt(String data, byte[] key)
			throws NoSuchAlgorithmException, NoSuchPaddingException,
			IllegalBlockSizeException, BadPaddingException,
			InvalidKeyException, InvalidAlgorithmParameterException 
	{

		String encodedResult = null;

		SecretKeySpec skeySpec = new SecretKeySpec(key, "AES");

		// build the initialization vector. This example is all zeros, but it could be any value or generated using a random number generator.
		byte[] iv = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
		IvParameterSpec ivspec = new IvParameterSpec(iv);
		
		// initialize the cipher for encrypt mode
		Cipher cipher = Cipher.getInstance("AES/CBC/PKCS5Padding");
		cipher.init(Cipher.ENCRYPT_MODE, skeySpec, ivspec);

		// encrypt the message
		byte[] encrypted = cipher.doFinal(data.getBytes());
		encodedResult = Base64.encodeBase64String(encrypted);

		return encodedResult;
	}


    /**
     * 
     * @param key
     * @param encryptedString
     * @return
     */
    public static String decrypt( SecretKeySpec key, String encrypted )
    {
        String decryptedString = null;
        if ( isEncrypted( encrypted ) )
        {
            encrypted = removePrefix( encrypted );

            try
            {
                Cipher cipher = Cipher.getInstance( ENCRYPTION_ALGORITHM );

                cipher.init( Cipher.DECRYPT_MODE, key );
                
                //encrypted string was encoded with padded cipher, get bytes back using the same method
                byte[] in = Base64.decodeBase64( encrypted );
                byte[] bitesOut = cipher.doFinal( in );
                decryptedString = new String(bitesOut);
            }
            catch ( NoSuchAlgorithmException e )
            {
                throw new EncryptionException( ErrorCodeEnum.ENCRYPTION_EXCEPTION, e );
            }
            catch ( NoSuchPaddingException e )
            {
                throw new EncryptionException( ErrorCodeEnum.ENCRYPTION_EXCEPTION, e );
            }
            catch ( InvalidKeyException e )
            {
                throw new EncryptionException( ErrorCodeEnum.ENCRYPTION_EXCEPTION, e );
            }
            catch ( IllegalBlockSizeException e )
            {
                throw new EncryptionException( ErrorCodeEnum.ENCRYPTION_EXCEPTION, e );
            }
            catch ( BadPaddingException e )
            {
                throw new EncryptionException( ErrorCodeEnum.ENCRYPTION_EXCEPTION, e );
            }

            return  decryptedString ;
        }
        else
        {
            return encrypted;
        }
    }

    /**
     * 
     * @param key
     * @param encryptedString
     * @return
     */
    public static String decrypt( byte[] enncrptKey, String encrypted )
    {
        String decryptedString = null;
        
        try
        {
          SecretKeySpec keySpec = new SecretKeySpec( enncrptKey, "AES" );
          Cipher cipher = Cipher.getInstance( ENCRYPTION_ALGORITHM );

          cipher.init( Cipher.DECRYPT_MODE, keySpec );
                
          //encrypted string was encoded with padded cipher, get bytes back using the same method
          byte[] in = Base64.decodeBase64( encrypted );
          byte[] bitesOut = cipher.doFinal( in );
          decryptedString = new String(bitesOut);
        }
        catch ( NoSuchAlgorithmException e )
        {
           throw new EncryptionException( ErrorCodeEnum.ENCRYPTION_EXCEPTION, e );
        }
        catch ( NoSuchPaddingException e )
        {
           throw new EncryptionException( ErrorCodeEnum.ENCRYPTION_EXCEPTION, e );
        }
        catch ( InvalidKeyException e )
        {
            throw new EncryptionException( ErrorCodeEnum.ENCRYPTION_EXCEPTION, e );
        }
        catch ( IllegalBlockSizeException e )
        {
            throw new EncryptionException( ErrorCodeEnum.ENCRYPTION_EXCEPTION, e );
        }
        catch ( BadPaddingException e )
        {
            throw new EncryptionException( ErrorCodeEnum.ENCRYPTION_EXCEPTION, e );
        }

        return  decryptedString ;
      
    }
        
    public static boolean isEncrypted( String encrypted )
    {
        return encrypted.startsWith( ENCRYPTED_PREFIX );
    }

    private static String removePrefix( String encrypted )
    {
        return encrypted.replace( ENCRYPTED_PREFIX, "" );
    }

}
