

package gov.va.med.cds.logger;


import gov.va.med.cds.audit.AbstractAuditLog;
import gov.va.med.cds.audit.persistence.AuditLogPersistenceManagerInterface;
import gov.va.med.cds.logger.config.LoggerBeanConstants;

import org.springframework.context.access.ContextSingletonBeanFactoryLocator;
import org.springframework.ejb.support.AbstractJmsMessageDrivenBean;
import org.springframework.jndi.JndiTemplate;

import javax.ejb.MessageDrivenContext;
import javax.jms.Message;
import javax.jms.ObjectMessage;
import javax.naming.NamingException;


/**
 * ReadAuditMessageDrivenBean receives JMS Messages on ReadAuditQueue, and stores those messages
 * in the database. The asynchronous handling of READ Requests audit information is expected to improve overall performance of the system. 
 */
public class ReadAuditMessageDrivenBean
    extends
        AbstractJmsMessageDrivenBean
{
    private static final long serialVersionUID = -4681966553773847069L;

    private static final String DEFAULT_AUDIT_LOG_PM_BEAN_ID = "cdsHibernateAuditLogPersistenceManager";

    private JndiTemplate jndiTemplate = new JndiTemplate();

    private AuditLogPersistenceManagerInterface auditLogPersistenceManager;


    protected void onEjbCreate( )
    {
        String auditPmBeanId = ( String )safeGetEnvironmentEntry( "java:comp/env/ejb/AuditLogPersistenceManagerBeanId", String.class,
                        DEFAULT_AUDIT_LOG_PM_BEAN_ID );
        this.auditLogPersistenceManager = ( AuditLogPersistenceManagerInterface )getBeanFactory().getBean( auditPmBeanId );
    }


    public void setMessageDrivenContext( MessageDrivenContext messageDrivenContext )
    {
        super.setMessageDrivenContext( messageDrivenContext );
        setBeanFactoryLocator( ContextSingletonBeanFactoryLocator.getInstance() );
        setBeanFactoryLocatorKey( LoggerBeanConstants.PRIMARY_CONTEXT_ID );
    }


    /***
     * onMessage receives JMS Messages from ReadAuditQueue, and writes audit information to database 
     * @param readRequestAuditInformation is the audit information created from a READ Request
     */
    public void onMessage( Message readRequestAuditInformation )
    {
        try
        {
            ObjectMessage objectMessage = ( ( ObjectMessage )readRequestAuditInformation );
            auditLogPersistenceManager.storeAuditLog( ( AbstractAuditLog )objectMessage.getObject() );
        }
        catch ( Exception e )
        {
            logger.error( gov.va.med.cds.util.LogMessageUtil.buildMessage( null, null, auditLogPersistenceManager.getAppName(),
                            "Error auditing read request message. " ), e );
        }
    }


    @SuppressWarnings( "unchecked" )
    protected Object safeGetEnvironmentEntry( String name, @SuppressWarnings( "rawtypes" )
    Class type, Object defaultValue )
    {
        Object environmentEntry = defaultValue;
        try
        {
            environmentEntry = this.jndiTemplate.lookup( name, type );
        }
        catch ( NamingException ne )
        {
            logger.warn( String.format( "Error initializing CDS VIM MDB. Error Reason: %s", ne.getMessage() ), ne );
        }

        return environmentEntry;
    }
}