/**
 * 
 */
package gov.va.med.cds.persistence.hibernate.common;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import gov.va.med.cds.junit.runners.PowerMockSuiteAwareRunner;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import oracle.jdbc.OracleConnection;
import oracle.sql.OPAQUE;
import oracle.xdb.XMLType;

import org.apache.commons.dbcp.DelegatingConnection;
import org.dom4j.CDATA;
import org.dom4j.DocumentFactory;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.dom4j.Node;
import org.easymock.EasyMock;
import org.hibernate.HibernateException;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.powermock.api.easymock.PowerMock;
import org.powermock.core.classloader.annotations.PrepareForTest;

/**
 * @author vhaislegberb
 *
 */
@RunWith( PowerMockSuiteAwareRunner.class )
@PrepareForTest( { XMLType.class} )
public class Dom4jXmlTypeUserTypeTest {
	
	private Dom4jXmlTypeUserType userType = null; 
	
	/**
	 * @throws java.lang.Exception
	 */
	@Before
	public void setUp() throws Exception {
		
		Properties parameters = new Properties();
		parameters.setProperty("containerElementName", "detailXML");
		//parameters.setProperty("nodeType", "CDATA");
		
		userType = new Dom4jXmlTypeUserType();
		userType.setParameterValues(parameters);
	}

	/**
	 * @throws java.lang.Exception
	 */
	@After
	public void tearDown() throws Exception {
	}

	/**
	 * Test method for {@link gov.va.med.cds.persistence.hibernate.common.Dom4jXmlTypeUserType#sqlTypes()}.
	 */
	@Test
	public void testSqlTypes() {
		int[] sqlTypes = userType.sqlTypes();
		assertTrue(sqlTypes.length == 1);
		assertTrue(sqlTypes[0] == XMLType._SQL_TYPECODE);
	}

	/**
	 * Test method for {@link gov.va.med.cds.persistence.hibernate.common.Dom4jXmlTypeUserType#returnedClass()}.
	 */
	@Test
	public void testReturnedClass() {
		assertEquals(userType.returnedClass(), Element.class);
	}

	/**
	 * Test method for {@link gov.va.med.cds.persistence.hibernate.common.Dom4jXmlTypeUserType#equals(java.lang.Object, java.lang.Object)}.
	 */
	@Test
	public void testEqualsAllNull() {
		assertFalse(userType.equals(null, null));
	}
	
	/**
	 * Test method for {@link gov.va.med.cds.persistence.hibernate.common.Dom4jXmlTypeUserType#equals(java.lang.Object, java.lang.Object)}.
	 */
	@Test
	public void testEqualsNonNullNotEqual() {
		assertFalse(userType.equals(DocumentHelper.createElement("test1"), DocumentHelper.createElement("test2")));
	}
	
	/**
	 * Test method for {@link gov.va.med.cds.persistence.hibernate.common.Dom4jXmlTypeUserType#equals(java.lang.Object, java.lang.Object)}.
	 */
	@Test
	public void testEqualsNonNullEqual() {
		Element e = DocumentHelper.createElement("test1");
		assertTrue(userType.equals(e, e));
	}

	/**
	 * Test method for {@link gov.va.med.cds.persistence.hibernate.common.Dom4jXmlTypeUserType#hashCode(java.lang.Object)}.
	 */
	@Test
	public void testHashCodeObject() {
		Element e = DocumentHelper.createElement("test1");
		assertTrue(e.hashCode() == userType.hashCode(e));
	}

	/**
	 * Test method for {@link gov.va.med.cds.persistence.hibernate.common.Dom4jXmlTypeUserType#nullSafeGet(java.sql.ResultSet, java.lang.String[], java.lang.Object)}.
	 */
	@Test
	public void testNullSafeGetResultSetStringArrayObject() throws SQLException {
		String names[] = new String[]{"name1"};
		ResultSet resultSetMock = EasyMock.createMock(ResultSet.class);
		XMLType xmlTypeMock = EasyMock.createMock(XMLType.class);
		
		EasyMock.expect(resultSetMock.getObject(names[0])).andReturn(xmlTypeMock);
		EasyMock.expect(resultSetMock.wasNull()).andReturn(Boolean.FALSE);
		EasyMock.expect(xmlTypeMock.getStringVal()).andReturn("<document><data1>1</data1><data2>2</data2></document>");
		
		EasyMock.replay(resultSetMock, xmlTypeMock);
		
		Object result = userType.nullSafeGet(resultSetMock, names, null);
		assertTrue(result != null);
		assertTrue(result instanceof Element);
		assertTrue("<detailXML><![CDATA[<document><data1>1</data1><data2>2</data2></document>]]></detailXML>".equals(((Element)result).asXML()));
	}
	
	
	/**
	 * Test method for {@link gov.va.med.cds.persistence.hibernate.common.Dom4jXmlTypeUserType#nullSafeGet(java.sql.ResultSet, java.lang.String[], java.lang.Object)}.
	 */
	@Test
	public void testNullSafeGetResultSetStringArrayObjectOpaque() throws SQLException {
		String names[] = new String[]{"name1"};
		ResultSet resultSetMock = EasyMock.createMock(ResultSet.class);
		XMLType xmlTypeMock = EasyMock.createMock(XMLType.class);
		OPAQUE opaqueMock = EasyMock.createMock(OPAQUE.class);
		PowerMock.mockStaticPartial( XMLType.class, "createXML", OPAQUE.class );
		
		EasyMock.expect(resultSetMock.getObject(names[0])).andReturn(opaqueMock);
		EasyMock.expect(resultSetMock.wasNull()).andReturn(Boolean.FALSE);
		EasyMock.expect(XMLType.createXML(opaqueMock)).andReturn(xmlTypeMock);
		EasyMock.expect(xmlTypeMock.getStringVal()).andReturn("<document><data1>1</data1><data2>2</data2></document>");
		
		PowerMock.replay(resultSetMock, xmlTypeMock, opaqueMock, XMLType.class);
		
		Object result = userType.nullSafeGet(resultSetMock, names, null);
		assertTrue(result != null);
		assertTrue(result instanceof Element);
		assertTrue("<detailXML><![CDATA[<document><data1>1</data1><data2>2</data2></document>]]></detailXML>".equals(((Element)result).asXML()));
	}
	
	
	
	/**
	 * Test method for {@link gov.va.med.cds.persistence.hibernate.common.Dom4jXmlTypeUserType#nullSafeGet(java.sql.ResultSet, java.lang.String[], java.lang.Object)}.
	 */
	@Test
	public void testNullSafeGetResultSetStringArrayObjectNullXML() throws SQLException {
		String names[] = new String[]{"name1"};
		ResultSet resultSetMock = EasyMock.createMock(ResultSet.class);
		
		EasyMock.expect(resultSetMock.getObject(names[0])).andReturn(null);
		EasyMock.expect(resultSetMock.wasNull()).andReturn(Boolean.TRUE);
		
		EasyMock.replay(resultSetMock);
		
		Object result = userType.nullSafeGet(resultSetMock, names, null);
		assertTrue(result != null);
		assertTrue(result instanceof Element);
		assertTrue("<detailXML><![CDATA[]]></detailXML>".equals(((Element)result).asXML()));
	}
	

	/**
	 * Test method for {@link gov.va.med.cds.persistence.hibernate.common.Dom4jXmlTypeUserType#nullSafeSet(java.sql.PreparedStatement, java.lang.Object, int)}.
	 */
	@Test
	public void testNullSafeSetDbcpConnection() throws SQLException {
		String value = "<document><data1>1</data1><data2>2</data2></document>";
		
		PreparedStatement statementMock = EasyMock.createMock(PreparedStatement.class);
		DelegatingConnection dbcpConnectionMock = EasyMock.createMock(DelegatingConnection.class);
		OracleConnection oracleConnectionMock = EasyMock.createMock(OracleConnection.class);
		PowerMock.mockStaticPartial( XMLType.class, "createXML", OracleConnection.class, String.class );
		XMLType xmlTypeMock = EasyMock.createMock(XMLType.class);
		
		EasyMock.expect(statementMock.getConnection()).andReturn(dbcpConnectionMock);
		EasyMock.expect(dbcpConnectionMock.getInnermostDelegate()).andReturn(oracleConnectionMock);
		EasyMock.expect(XMLType.createXML(oracleConnectionMock, value)).andReturn(xmlTypeMock);
		statementMock.setObject(0, xmlTypeMock);
		
		PowerMock.replay(statementMock, dbcpConnectionMock, oracleConnectionMock, XMLType.class, xmlTypeMock );
		
		userType.nullSafeSet(statementMock, value, 0);
	}
	
	/**
	 * Test method for {@link gov.va.med.cds.persistence.hibernate.common.Dom4jXmlTypeUserType#nullSafeSet(java.sql.PreparedStatement, java.lang.Object, int)}.
	 */
	@Test
	public void testNullSafeSetOracleConnection() throws SQLException {
		String value = "<document><data1>1</data1><data2>2</data2></document>";
		
		PreparedStatement statementMock = EasyMock.createMock(PreparedStatement.class);
		OracleConnection oracleConnectionMock = EasyMock.createMock(OracleConnection.class);
		PowerMock.mockStaticPartial( XMLType.class, "createXML", OracleConnection.class, String.class );
		XMLType xmlTypeMock = EasyMock.createMock(XMLType.class);
		
		EasyMock.expect(statementMock.getConnection()).andReturn(oracleConnectionMock);
		EasyMock.expect(XMLType.createXML(oracleConnectionMock, value)).andReturn(xmlTypeMock);
		statementMock.setObject(0, xmlTypeMock);
		
		PowerMock.replay(statementMock, oracleConnectionMock, XMLType.class, xmlTypeMock );
		
		userType.nullSafeSet(statementMock, value, 0);
	}
	
	/**
	 * Test method for {@link gov.va.med.cds.persistence.hibernate.common.Dom4jXmlTypeUserType#nullSafeSet(java.sql.PreparedStatement, java.lang.Object, int)}.
	 */
	@Test (expected=SQLException.class)
	public void testNullSafeSetNullConnection() throws SQLException {
		String value = "<document><data1>1</data1><data2>2</data2></document>";
		
		PreparedStatement statementMock = EasyMock.createMock(PreparedStatement.class);
		
		EasyMock.expect(statementMock.getConnection()).andReturn(null);
		
		EasyMock.replay(statementMock );
		
		userType.nullSafeSet(statementMock, value, 0);
	}

	/**
	 * Test method for {@link gov.va.med.cds.persistence.hibernate.common.Dom4jXmlTypeUserType#deepCopy(java.lang.Object)}.
	 */
	@Test
	public void testDeepCopyNode() {
		Node nodeMock = EasyMock.createMock(Node.class);
		
		EasyMock.expect(nodeMock.clone()).andReturn(DocumentHelper.createElement("test"));
		
		EasyMock.replay(nodeMock);
		
		Node node = (Node)userType.deepCopy(nodeMock);
		assertNotNull(node);
		assertEquals("test", node.getName());
	}
	
	/**
	 * Test method for {@link gov.va.med.cds.persistence.hibernate.common.Dom4jXmlTypeUserType#deepCopy(java.lang.Object)}.
	 */
	@Test
	public void testDeepCopyString() {
		String deepCopy = "Deep Copy";
		
		String copy = (String)userType.deepCopy(deepCopy);
		assertNotNull(copy);
		assertFalse(deepCopy == copy);
		assertEquals(deepCopy, copy);
	}
	
	/**
	 * Test method for {@link gov.va.med.cds.persistence.hibernate.common.Dom4jXmlTypeUserType#deepCopy(java.lang.Object)}.
	 */
	@Test(expected=HibernateException.class)
	public void testDeepCopyInteger() {
		Integer original = new Integer(1);
		userType.deepCopy(original);
	}

	/**
	 * Test method for {@link gov.va.med.cds.persistence.hibernate.common.Dom4jXmlTypeUserType#isMutable()}.
	 */
	@Test
	public void testIsMutable() {
		assertTrue(userType.isMutable());
	}

	/**
	 * Test method for {@link gov.va.med.cds.persistence.hibernate.common.Dom4jXmlTypeUserType#disassemble(java.lang.Object)}.
	 */
	@Test
	public void testDisassembleNode() {
		Node nodeMock = EasyMock.createMock(Node.class);
		
		EasyMock.expect(nodeMock.asXML()).andReturn("<test>disassemble</test>");
		
		EasyMock.replay(nodeMock);
		
		String s = (String)userType.disassemble(nodeMock);
		assertNotNull(s);
		assertEquals(s, "<test>disassemble</test>");
	}
	
	/**
	 * Test method for {@link gov.va.med.cds.persistence.hibernate.common.Dom4jXmlTypeUserType#disassemble(java.lang.Object)}.
	 */
	@Test(expected=HibernateException.class)
	public void testDisassembleOther() {
		
		userType.disassemble("<test>disassemble</test>");
	}

	/**
	 * Test method for {@link gov.va.med.cds.persistence.hibernate.common.Dom4jXmlTypeUserType#assemble(java.io.Serializable, java.lang.Object)}.
	 */
	@Test
	public void testAssemble() {

		String cached = "<test>1</test>\n<test>2</test>";
		Element ownerMock = EasyMock.createMock(Element.class);
		ownerMock.add(EasyMock.anyObject(Element.class));
		EasyMock.replay(ownerMock);
		
		Element container = (Element)userType.assemble(cached, ownerMock);
		
		assertNotNull(container);
		assertTrue("detailXML".equals(container.getName()));
		assertTrue(container.asXML().equals("<detailXML><![CDATA[<test>1</test>\n<test>2</test>]]></detailXML>"));
		
		
	}

	/**
	 * Test method for {@link gov.va.med.cds.persistence.hibernate.common.Dom4jXmlTypeUserType#objectToSQLString(java.lang.Object)}.
	 */
	@Test
	public void testObjectToSQLStringNode() {
		Element nodeMock = EasyMock.createMock(Element.class);
		CDATA cdataMock = EasyMock.createMock(CDATA.class);
		
		List<Node> contentList = new ArrayList<Node>();
		contentList.add(cdataMock);
		
		EasyMock.expect(nodeMock.content()).andReturn(contentList);
		EasyMock.expect(cdataMock.getNodeType()).andReturn(Node.CDATA_SECTION_NODE);
		EasyMock.expect(cdataMock.getText()).andReturn("<test>1</test>\n<test>2</test>");
		
		EasyMock.replay(nodeMock, cdataMock);
		
		String sqlString = userType.objectToSQLString(nodeMock); 
		
		assertNotNull(sqlString);
		assertEquals(sqlString, "<test>1</test>\n<test>2</test>");
	}
	
	/**
	 * Test method for {@link gov.va.med.cds.persistence.hibernate.common.Dom4jXmlTypeUserType#objectToSQLString(java.lang.Object)}.
	 */
	@Test
	public void testObjectToSQLStringNonNode() {
		String sqlString = "sqlString";
		
		assertEquals(userType.objectToSQLString(sqlString), sqlString);
	}
	
	
	@Test
	public void testFromXmlStringElement() {
		
		Properties parameters = new Properties();
		parameters.setProperty("containerElementName", "detailXML");
		parameters.setProperty("nodeType", "ELEMENT");
		this.userType.setParameterValues(parameters);
		
		Object obj = this.userType.fromXMLString("<test>1</test>\n<test>2</test>");
		assertTrue(obj instanceof Element);
		
		Element element = (Element)obj;
		assert(element.getName().equals("detailXML"));
		assert(element.elements().size() == 2);
	}
	
	@Test
	public void testObjectToSQLStringElement() {
		Properties parameters = new Properties();
		parameters.setProperty("containerElementName", "detailXML");
		parameters.setProperty("nodeType", "ELEMENT");
		this.userType.setParameterValues(parameters);
		
		Element parent = DocumentHelper.createElement("detailXML");
		Element child1 = DocumentHelper.createElement("test");
		Element child2 = DocumentHelper.createElement("test");
		
		child1.setText("1");
		child2.setText("2");
		
		parent.add(child1);
		parent.add(child2);
		
		String value = this.userType.objectToSQLString(parent);
		assertTrue(value != null && value.equals("<test>1</test><test>2</test>"));
	}

}
