

package gov.va.med.cds.persistence.hibernate.common;


import static org.easymock.EasyMock.expect;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import gov.va.med.cds.junit.runners.Suite;

import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.easymock.EasyMock;
import org.hibernate.Hibernate;
import org.hibernate.HibernateException;
import org.hibernate.engine.SessionImplementor;
import org.hibernate.type.StandardBasicTypes;
import org.junit.BeforeClass;
import org.junit.Test;

import java.io.Serializable;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Properties;


public class Dom4jPointInTimeUserTypeTest
{

    private static Dom4jPointInTimeUserType dom4jPointInTimeUserType = null;


    @BeforeClass
    @Suite( groups = { "checkintest" } )
    public static void beforeDom4jPointInTimeUserTypeTestClassSetUp( )
        throws Exception
    {
        dom4jPointInTimeUserType = new Dom4jPointInTimeUserType();
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testNullSafeGet( )
        throws HibernateException,
            SQLException
    {
        ResultSet rs = EasyMock.createMock( ResultSet.class );
        String[] names = { "literal" };
        SessionImplementor sessionImplementor = EasyMock.createMock( SessionImplementor.class );

        String timestamp = "20101231235950.050-0600";
        expect( rs.getString( names[0] ) ).andReturn( timestamp );
        expect( rs.wasNull() ).andReturn( false );

        EasyMock.replay( rs );
        Element result = ( Element )dom4jPointInTimeUserType.nullSafeGet( rs, names, sessionImplementor, null );
        assertEquals( result.getText(), timestamp );
        EasyMock.verify( rs );

    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testNullSafeGetRsWasNull( )
        throws HibernateException,
            SQLException
    {
        ResultSet rs = EasyMock.createMock( ResultSet.class );
        String[] names = { "literal" };
        SessionImplementor sessionImplementor = EasyMock.createMock( SessionImplementor.class );

        String timestamp = "20101231235950.050-0600";
        expect( rs.getString( names[0] ) ).andReturn( timestamp );
        expect( rs.wasNull() ).andReturn( true );

        EasyMock.replay( rs );
        Element result = ( Element )dom4jPointInTimeUserType.nullSafeGet( rs, names, sessionImplementor, null );
        assertNull( result );
        EasyMock.verify( rs );

    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testNullSafeSetWithNullLiteral( )
        throws HibernateException,
            SQLException
    {
        PreparedStatement statement = EasyMock.createMock( PreparedStatement.class );
        SessionImplementor sessionImplementor = EasyMock.createMock( SessionImplementor.class );

        //setup for literal
        setValuesToLiteral();
        statement.setNull( 1, StandardBasicTypes.STRING.sqlType() );
        EasyMock.replay( statement );
        dom4jPointInTimeUserType.nullSafeSet( statement, null, 1, sessionImplementor );
        EasyMock.verify( statement );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testNullSafeSetWithNullNumeric( )
        throws HibernateException,
            SQLException
    {
        PreparedStatement statement = EasyMock.createMock( PreparedStatement.class );
        SessionImplementor sessionImplementor = EasyMock.createMock( SessionImplementor.class );

        //next setValues numeric
        setValuesToNumeric();
        statement.setNull( 1, StandardBasicTypes.DOUBLE.sqlType() );
        EasyMock.replay( statement );
        dom4jPointInTimeUserType.nullSafeSet( statement, null, 1, sessionImplementor );
        EasyMock.verify( statement );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testNullSafeSetElementLiteral( )
        throws HibernateException,
            SQLException
    {
        int index = 1;
        PreparedStatement statement = EasyMock.createMock( PreparedStatement.class );
        SessionImplementor sessionImplementor = EasyMock.createMock( SessionImplementor.class );

        String timest = "20101231235950.050-0600";
        Element timestamp = DocumentHelper.createElement( "literal" );
        timestamp.setText( timest );

        setValuesToLiteral();
        statement.setString( index, timest );

        EasyMock.replay( statement );
        dom4jPointInTimeUserType.nullSafeSet( statement, timestamp, 1, sessionImplementor );
        EasyMock.verify( statement );

    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testNullSafeSetElementNumericWithTimestamp( )
        throws HibernateException,
            SQLException
    {
        int index = 1;
        PreparedStatement statement = EasyMock.createMock( PreparedStatement.class );
        SessionImplementor sessionImplementor = EasyMock.createMock( SessionImplementor.class );

        String timest = "20101231235950.050-0600";
        Element timestamp = DocumentHelper.createElement( "literal" );
        timestamp.setText( timest );
        //Double timedb = new Double(3101231.235959);
        Double timedb = dom4jPointInTimeUserType.isoTime2Double( timest );

        setValuesToNumeric();
        statement.setDouble( index, timedb );

        EasyMock.replay( statement );
        dom4jPointInTimeUserType.nullSafeSet( statement, timestamp, 1, sessionImplementor );
        EasyMock.verify( statement );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testNullSafeSetElementNumericEmptyTimestamp( )
        throws HibernateException,
            SQLException
    {

        int index = 1;
        PreparedStatement statement = EasyMock.createMock( PreparedStatement.class );
        SessionImplementor sessionImplementor = EasyMock.createMock( SessionImplementor.class );

        Element timestamp = DocumentHelper.createElement( "literal" );

        setValuesToNumeric();
        statement.setNull( index, StandardBasicTypes.DOUBLE.sqlType() );

        EasyMock.replay( statement );
        dom4jPointInTimeUserType.nullSafeSet( statement, timestamp, 1, sessionImplementor );
        EasyMock.verify( statement );

    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testNullSafeSetPointInTimeLiteral( )
        throws HibernateException,
            SQLException
    {

        int index = 1;
        PreparedStatement statement = EasyMock.createMock( PreparedStatement.class );
        SessionImplementor sessionImplementor = EasyMock.createMock( SessionImplementor.class );

        String timest = "20101231235950.050-0600";
        PointInTime timestamp = new PointInTime();
        timestamp.setTimestamp( timest );

        setValuesToLiteral();
        statement.setString( index, timest );

        EasyMock.replay( statement );
        dom4jPointInTimeUserType.nullSafeSet( statement, timestamp, 1, sessionImplementor );
        EasyMock.verify( statement );

    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testNullSafeSetPointInTimeNumeric( )
        throws HibernateException,
            SQLException
    {

        int index = 1;
        PreparedStatement statement = EasyMock.createMock( PreparedStatement.class );
        SessionImplementor sessionImplementor = EasyMock.createMock( SessionImplementor.class );

        String timest = "20101231235950.050-0600";
        PointInTime timestamp = new PointInTime();
        timestamp.setTimestamp( timest );

        setValuesToNumeric();
        statement.setDouble( index, dom4jPointInTimeUserType.isoTime2Double( timest ) );

        EasyMock.replay( statement );
        dom4jPointInTimeUserType.nullSafeSet( statement, timestamp, 1, sessionImplementor );
        EasyMock.verify( statement );

    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testNullSafeSetPointInTimeNumericNull( )
        throws HibernateException,
            SQLException
    {

        int index = 1;
        PreparedStatement statement = EasyMock.createMock( PreparedStatement.class );
        SessionImplementor sessionImplementor = EasyMock.createMock( SessionImplementor.class );

        PointInTime timestamp = new PointInTime();
        timestamp.setTimestamp( null );

        setValuesToNumeric();
        statement.setNull( index, StandardBasicTypes.DOUBLE.sqlType() );

        EasyMock.replay( statement );
        dom4jPointInTimeUserType.nullSafeSet( statement, timestamp, 1, sessionImplementor );
        EasyMock.verify( statement );

    }


    @Test( expected = HibernateException.class )
    @Suite( groups = { "checkintest" } )
    public void testNullSafeSetNonElementValue( )
        throws HibernateException,
            SQLException
    {
        int index = 1;
        PreparedStatement statement = EasyMock.createMock( PreparedStatement.class );
        SessionImplementor sessionImplementor = EasyMock.createMock( SessionImplementor.class );
        Object object = new String( "NotAnElementInstance" );
        //this should throw exception ...
        dom4jPointInTimeUserType.nullSafeSet( statement, object, index, sessionImplementor );
    }


    private void resetSetValues( )
    {
        //this will nullify setValues - various test methods need it null or need to set it to a different value
        dom4jPointInTimeUserType.setParameterValues( new Properties() );
    }


    private void setValuesToLiteral( )
    {
        resetSetValues();
        Properties props = new Properties();
        props.put( dom4jPointInTimeUserType.SET_VALUES, dom4jPointInTimeUserType.SV_LITERAL_ONLY );
        dom4jPointInTimeUserType.setParameterValues( props );
    }


    private void setValuesToNumeric( )
    {
        resetSetValues();
        Properties props = new Properties();
        props.put( dom4jPointInTimeUserType.SET_VALUES, dom4jPointInTimeUserType.SV_NUMERIC_ONLY );
        dom4jPointInTimeUserType.setParameterValues( props );
    }


    private void setValuesToAll( )
    {
        Properties props = new Properties();
        props.put( dom4jPointInTimeUserType.SET_VALUES, dom4jPointInTimeUserType.SV_ALL );
        dom4jPointInTimeUserType.setParameterValues( props );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testBasicGetValuesTest( )
    {

        assertEquals( dom4jPointInTimeUserType.getPropertyNames()[0], "literal" );
        assertEquals( dom4jPointInTimeUserType.getPropertyNames()[1], "numeric" );
        //calling getPropertyTypes with setValues == null - below will call getPropertyTypes with setValues != null
        resetSetValues();
        assertEquals( dom4jPointInTimeUserType.getPropertyTypes()[0], StandardBasicTypes.STRING );
        assertEquals( dom4jPointInTimeUserType.getPropertyTypes()[1], StandardBasicTypes.DOUBLE );

        Element cached = DocumentHelper.createElement( "original" );
        Object result = dom4jPointInTimeUserType.assemble( ( Serializable )cached, null, null );
        assertTrue( result instanceof Element );
        result = dom4jPointInTimeUserType.disassemble( ( Serializable )cached, null );
        assertTrue( result instanceof Element );
        Object originalObj = new String( "original" );
        assertEquals( dom4jPointInTimeUserType.replace( originalObj, null, null, null ), originalObj );

        Element x = DocumentHelper.createElement( "x" );
        x.setText( new String( "20071231235959.999-0600" ) );
        Element y = DocumentHelper.createElement( "y" );
        y.setText( new String( "20071231235959.999-0600" ) );
        assertTrue( dom4jPointInTimeUserType.equals( x, y ) );
        y.setText( "20101231235950.050-0600" );
        assertFalse( dom4jPointInTimeUserType.equals( x, y ) );

        assertEquals( ( ( Element )dom4jPointInTimeUserType.deepCopy( x ) ).getText(), x.getText() );

        assertEquals( dom4jPointInTimeUserType.returnedClass(), Element.class );
        assertFalse( dom4jPointInTimeUserType.isMutable() );

        String object = "object";
        assertEquals( dom4jPointInTimeUserType.hashCode( x ), x.hashCode() );
        assertEquals( dom4jPointInTimeUserType.hashCode( null ), 0 );

        //12 is the int value for Hibernate.STRING.sqlType() and 8 is the in value for Hibernate.DOUBLE.sqlType()
        assertEquals( dom4jPointInTimeUserType.sqlTypes()[0], 12 );
        assertEquals( dom4jPointInTimeUserType.sqlTypes()[1], 8 );

        assertEquals( dom4jPointInTimeUserType.getPropertyValue( x, 1 ), "20071231235959.999-0600" );
        assertNull( dom4jPointInTimeUserType.getPropertyValue( "someObject", 1 ) );

        setValuesToAll();
        assertEquals( dom4jPointInTimeUserType.setValues, dom4jPointInTimeUserType.SV_ALL );

        setValuesToLiteral();
        assertEquals( dom4jPointInTimeUserType.getPropertyTypes()[0], StandardBasicTypes.STRING );

        setValuesToNumeric();
        assertEquals( dom4jPointInTimeUserType.getPropertyTypes()[0], StandardBasicTypes.DOUBLE );

    }

}
