/**
 * 
 */


package gov.va.med.cds.persistence.hibernate;


import static org.easymock.EasyMock.createNiceMock;
import static org.easymock.EasyMock.expect;
import static org.easymock.EasyMock.expectLastCall;
import gov.va.med.cds.clinicaldata.DomainEntryPoint;
import gov.va.med.cds.clinicaldata.vhim400.ClinicalDataResponseInterface;
import gov.va.med.cds.common.person.correlation.PersonIdentifierInterface;
import gov.va.med.cds.exceptionframework.ExceptionHandler;
import gov.va.med.cds.exceptionframework.ExceptionLoggerInterface;
import gov.va.med.cds.filter.EntryFilterInterface;
import gov.va.med.cds.junit.runners.Suite;
import gov.va.med.cds.persistence.QueryStrategyInterface;
import gov.va.med.cds.persistence.QueryWorkInterface;
import gov.va.med.cds.persistence.hibernate.common.VistaPointInTimeUserType;
import gov.va.med.cds.request.ErrorSectionHelperInterface;
import gov.va.med.cds.rules.BooleanRuleInterface;
import gov.va.med.cds.template.TemplateHelperInterface;
import gov.va.med.cds.transaction.ReadWork;
import gov.va.med.cds.transaction.WorkManagerInterface;
import gov.va.med.cds.util.TimeoutUtil;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.dom4j.Document;
import org.dom4j.DocumentHelper;
import org.easymock.EasyMock;
import org.hibernate.SessionFactory;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import commonj.work.WorkItem;


/**
 * @author vhaislegberb
 *
 */
public class SiteSpecificHibernateReadPersistenceManagerTest
{

    private ErrorSectionHelperInterface errorSectionHelper;
    private ExceptionLoggerInterface guaranteedLogger;
    private ClinicalDataResponseInterface clinicalDataResponse;
    private TemplateHelperInterface templateHelper;


    @Suite( groups = { "checkintest" } )
    private void initMocks( )
        throws Exception
    {
        templateHelper = EasyMock.createNiceMock( TemplateHelperInterface.class );
        guaranteedLogger = createNiceMock( ExceptionLoggerInterface.class );
        errorSectionHelper = createNiceMock( ErrorSectionHelperInterface.class );
        clinicalDataResponse = createNiceMock( ClinicalDataResponseInterface.class );
        ExceptionHandler.setGuaranteedLogger( guaranteedLogger );
        ExceptionHandler.setTemplateHelper( templateHelper );
        ExceptionHandler.setClinicalDataResponse( clinicalDataResponse );
    }


    /**   * Called before each test.   */
    @Before
    @Suite( groups = { "checkintest" } )
    public void setUp( )
        throws Exception
    {
        initMocks();
        EasyMock.reset( templateHelper );
        EasyMock.reset( guaranteedLogger );
        EasyMock.reset( errorSectionHelper );
        EasyMock.reset( clinicalDataResponse );

    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testIsApplicableIsTrue( )
    {
        AbstractReadHibernatePersistenceManager persistenceManager = new SiteSpecificHibernateReadPersistenceManager();

        EntryFilterInterface entryFilter = EasyMock.createMock( EntryFilterInterface.class );
        BooleanRuleInterface rule1 = EasyMock.createMock( BooleanRuleInterface.class );
        BooleanRuleInterface rule2 = EasyMock.createMock( BooleanRuleInterface.class );
        PersonIdentifierInterface pid1 = EasyMock.createMock( PersonIdentifierInterface.class );
        PersonIdentifierInterface pid2 = EasyMock.createMock( PersonIdentifierInterface.class );

        EasyMock.expect( entryFilter.getDomainEntryPoint() ).andReturn( "AllergyAssessment" );
        EasyMock.expect( entryFilter.isPatientCentricFilter() ).andReturn( true );
        EasyMock.expect( rule1.evaluate( DomainEntryPoint.AllergyAssessment ) ).andReturn( Boolean.FALSE );
        EasyMock.expect( pid1.getAssigningFacility() ).andReturn( "444" );
        EasyMock.expect( pid2.getAssigningFacility() ).andReturn( "555" );
        EasyMock.expect( rule2.evaluate( DomainEntryPoint.AllergyAssessment ) ).andReturn( Boolean.TRUE );
        EasyMock.expect( pid1.getAssigningFacility() ).andReturn( "444" );
        EasyMock.expect( pid2.getAssigningFacility() ).andReturn( "555" );

        EasyMock.replay( entryFilter, rule1, rule2, pid1, pid2 );

        List<BooleanRuleInterface> rules = new ArrayList<BooleanRuleInterface>();
        rules.add( rule1 );
        rules.add( rule2 );

        List<PersonIdentifierInterface> personIdentifiers = new ArrayList<PersonIdentifierInterface>();
        personIdentifiers.add( pid1 );
        personIdentifiers.add( pid2 );

        persistenceManager.setRules( rules );
        persistenceManager.setSiteIdentifier( "555" );

        Assert.assertTrue( persistenceManager.isApplicable( entryFilter, personIdentifiers ) );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testIsApplicableIsFalse( )
    {
        AbstractReadHibernatePersistenceManager persistenceManager = new SiteSpecificHibernateReadPersistenceManager();

        EntryFilterInterface entryFilter = EasyMock.createMock( EntryFilterInterface.class );
        BooleanRuleInterface rule1 = EasyMock.createMock( BooleanRuleInterface.class );
        BooleanRuleInterface rule2 = EasyMock.createMock( BooleanRuleInterface.class );
        PersonIdentifierInterface pid1 = EasyMock.createMock( PersonIdentifierInterface.class );
        PersonIdentifierInterface pid2 = EasyMock.createMock( PersonIdentifierInterface.class );

        EasyMock.expect( entryFilter.getDomainEntryPoint() ).andReturn( "AllergyAssessment" );
        EasyMock.expect( rule1.evaluate( DomainEntryPoint.AllergyAssessment ) ).andReturn( Boolean.FALSE );
        EasyMock.expect( pid1.getAssigningFacility() ).andReturn( "444" );
        EasyMock.expect( pid2.getAssigningFacility() ).andReturn( "555" );
        EasyMock.expect( rule2.evaluate( DomainEntryPoint.AllergyAssessment ) ).andReturn( Boolean.FALSE );
        EasyMock.expect( pid1.getAssigningFacility() ).andReturn( "444" );
        EasyMock.expect( pid2.getAssigningFacility() ).andReturn( "555" );

        EasyMock.replay( entryFilter, rule1, rule2, pid1, pid2 );

        List<BooleanRuleInterface> rules = new ArrayList<BooleanRuleInterface>();
        rules.add( rule1 );
        rules.add( rule2 );

        List<PersonIdentifierInterface> personIdentifiers = new ArrayList<PersonIdentifierInterface>();
        personIdentifiers.add( pid1 );
        personIdentifiers.add( pid2 );

        persistenceManager.setRules( rules );
        persistenceManager.setSiteIdentifier( "555" );

        Assert.assertFalse( persistenceManager.isApplicable( entryFilter, personIdentifiers ) );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testIsApplicableIsNoMatchingSiteIdFalse( )
    {
        AbstractReadHibernatePersistenceManager persistenceManager = new SiteSpecificHibernateReadPersistenceManager();

        EntryFilterInterface entryFilter = EasyMock.createMock( EntryFilterInterface.class );
        BooleanRuleInterface rule1 = EasyMock.createMock( BooleanRuleInterface.class );
        BooleanRuleInterface rule2 = EasyMock.createMock( BooleanRuleInterface.class );
        PersonIdentifierInterface pid1 = EasyMock.createMock( PersonIdentifierInterface.class );
        PersonIdentifierInterface pid2 = EasyMock.createMock( PersonIdentifierInterface.class );

        EasyMock.expect( entryFilter.getDomainEntryPoint() ).andReturn( "AllergyAssessment" );
        EasyMock.expect( entryFilter.isPatientCentricFilter() ).andReturn( true );
        EasyMock.expect( rule1.evaluate( DomainEntryPoint.AllergyAssessment ) ).andReturn( Boolean.FALSE );
        EasyMock.expect( pid1.getAssigningFacility() ).andReturn( "444" );
        EasyMock.expect( pid2.getAssigningFacility() ).andReturn( "666" );
        EasyMock.expect( rule2.evaluate( DomainEntryPoint.AllergyAssessment ) ).andReturn( Boolean.TRUE );
        EasyMock.expect( pid1.getAssigningFacility() ).andReturn( "444" );
        EasyMock.expect( pid2.getAssigningFacility() ).andReturn( "666" );

        EasyMock.replay( entryFilter, rule1, rule2, pid1, pid2 );

        List<BooleanRuleInterface> rules = new ArrayList<BooleanRuleInterface>();
        rules.add( rule1 );
        rules.add( rule2 );

        List<PersonIdentifierInterface> personIdentifiers = new ArrayList<PersonIdentifierInterface>();
        personIdentifiers.add( pid1 );
        personIdentifiers.add( pid2 );

        persistenceManager.setRules( rules );
        persistenceManager.setSiteIdentifier( "555" );

        Assert.assertFalse( persistenceManager.isApplicable( entryFilter, personIdentifiers ) );
    }


    @Test
    @Suite( groups = { "checkintest" } )
    public void testIsApplicableIsNoPatientsFalse( )
    {
        AbstractReadHibernatePersistenceManager persistenceManager = new SiteSpecificHibernateReadPersistenceManager();

        EntryFilterInterface entryFilter = EasyMock.createMock( EntryFilterInterface.class );
        BooleanRuleInterface rule1 = EasyMock.createMock( BooleanRuleInterface.class );
        BooleanRuleInterface rule2 = EasyMock.createMock( BooleanRuleInterface.class );

        EasyMock.expect( entryFilter.getDomainEntryPoint() ).andReturn( "AllergyAssessment" );
        EasyMock.expect( entryFilter.isPatientCentricFilter() ).andReturn( true );
        EasyMock.expect( rule1.evaluate( DomainEntryPoint.AllergyAssessment ) ).andReturn( Boolean.FALSE );
        EasyMock.expect( rule2.evaluate( DomainEntryPoint.AllergyAssessment ) ).andReturn( Boolean.TRUE );

        EasyMock.replay( entryFilter, rule1, rule2 );

        List<BooleanRuleInterface> rules = new ArrayList<BooleanRuleInterface>();
        rules.add( rule1 );
        rules.add( rule2 );

        List<PersonIdentifierInterface> personIdentifiers = new ArrayList<PersonIdentifierInterface>();

        persistenceManager.setRules( rules );
        persistenceManager.setSiteIdentifier( "555" );

        Assert.assertFalse( persistenceManager.isApplicable( entryFilter, personIdentifiers ) );
    }


    @SuppressWarnings( "unchecked" )
    @Test
    @Suite( groups = { "checkintest" } )
    public void testPerformReadOnClinicalData( )
        throws Exception
    {
        String templateId = "TestTemplateId";
        String entryPoint = "EntryPoint1";
        String requestId = "REQUEST-99999";
        String applicationName = "CDS";
        String siteId = "999";

        AbstractReadHibernatePersistenceManager persistenceManager = new SiteSpecificHibernateReadPersistenceManager();
        persistenceManager.setSiteIdentifier(siteId);
        persistenceManager.setPointInTimeUserType( VistaPointInTimeUserType.class );
        List<PersonIdentifierInterface> personIdentifiers = new ArrayList<PersonIdentifierInterface>();
        Map<String, QueryStrategyInterface> queryStrategiesMap = new HashMap<String, QueryStrategyInterface>();
        List<QueryWorkInterface> queryWork = new ArrayList<QueryWorkInterface>();
        QueryWorkInterface work = EasyMock.createMock( QueryWorkInterface.class );
        expect( work.getExceptions() ).andReturn( new ArrayList<Exception>() );
        //TODO: we will need to remove the two lines below once the transaction fix is on place..SD
        work.release();
        expectLastCall();
        queryWork.add( work );

        EntryFilterInterface entryFilter = EasyMock.createMock( EntryFilterInterface.class );
        expect( entryFilter.getTemplateId() ).andReturn( templateId ).anyTimes();
        expect( entryFilter.getDomainEntryPoint() ).andReturn( entryPoint ).anyTimes();
        expect( entryFilter.getRequestId() ).andReturn( requestId ).anyTimes();
        expect( entryFilter.getTimeout() ).andReturn( null ).anyTimes();
        expect( entryFilter.getQueryName() ).andReturn( null ).anyTimes();

        PersonIdentifierInterface pid1 = EasyMock.createMock( PersonIdentifierInterface.class );
        PersonIdentifierInterface pid2 = EasyMock.createMock( PersonIdentifierInterface.class );
        personIdentifiers.add( pid1 );
        personIdentifiers.add( pid2 );

        QueryStrategyInterface queryStrategy = EasyMock.createMock( QueryStrategyInterface.class );
        SessionFactory sessionFactory = EasyMock.createMock( SessionFactory.class );
        queryStrategiesMap.put( String.format( "%s-%s", templateId, entryPoint ), queryStrategy );

        expect( pid1.getAssigningFacility() ).andReturn( "111" ).times( 2 );
        expect( pid2.getAssigningFacility() ).andReturn( "999" ).times( 2 );
        expect( pid1.getIdentity() ).andReturn( "111" ).times( 2 );
        expect( pid2.getIdentity() ).andReturn( "999" ).times( 2 );
        expect(
                        queryStrategy.createQueryWork( EasyMock.eq( sessionFactory ), EasyMock.eq( entryFilter ), EasyMock.isA( List.class ),
                                        EasyMock.eq( VistaPointInTimeUserType.class ),EasyMock.eq(siteId) ) ).andReturn( queryWork );

        ModelAssemblerInterface modelAssembler = EasyMock.createMock( ModelAssemblerInterface.class );
        expect( modelAssembler.assembleModel( entryFilter, queryWork ) ).andReturn( DocumentHelper.createDocument() );
        expect( queryStrategy.getModelAssembler() ).andReturn( modelAssembler );

        expect( templateHelper.getErrorSectionHelper( templateId ) ).andReturn( errorSectionHelper );
        expect( errorSectionHelper.getErrorCount( ErrorSectionHelperInterface.ErrorType.ALL, DocumentHelper.createDocument() ) ).andReturn( 0 );

        Map<String, String> templateTimeoutMap = new HashMap<String, String>();
        templateTimeoutMap.put( TimeoutUtil.UPPER_LIMIT_TIMEOUT, "300" );
        
        WorkManagerInterface workManager = EasyMock.createMock( WorkManagerInterface.class );
        WorkItem workItem1 = EasyMock.createMock( WorkItem.class );
        EasyMock.expect( workManager.schedule( ( ( ReadWork )EasyMock.anyObject() ) ) ).andReturn( workItem1 );
        EasyMock.expect( workManager.waitForAll( ((List<WorkItem>)EasyMock.anyObject()), EasyMock.eq(300000L) ) ).andReturn( true ).anyTimes();
        
        persistenceManager.setTemplateTimeoutMap( templateTimeoutMap );
        persistenceManager.setQueryThreadWorkManager( workManager );

        EasyMock.replay( entryFilter, queryStrategy, pid1, pid2, modelAssembler, work, workManager, templateHelper, errorSectionHelper );

        persistenceManager.setSiteIdentifier( siteId );
        persistenceManager.setTemplateQueryStrategyMap( queryStrategiesMap );
        Map<String, SessionFactory> sessionFactoryMap = new HashMap<String, SessionFactory>();
        sessionFactoryMap.put( templateId, sessionFactory );
        persistenceManager.setSessionFactoryMap( sessionFactoryMap );
        Document result = persistenceManager.performReadOnClinicalData( entryFilter, personIdentifiers, applicationName );

        Assert.assertNotNull( result );
    }

    //    @Test
    //    @Suite( groups = { "brokentest" } )
    //    public void testPerformReadOnClinicalData_PartialReads_DBConnFailure() throws Exception
    //    {
    //        String templateId = "TestTemplateId";
    //        String requestId = "REQUEST-99999";
    //        
    //        AbstractReadHibernatePersistenceManager persistenceManager = new SiteSpecificHibernateReadPersistenceManager();
    //        persistenceManager.setPointInTimeUserType(  Dom4jPointInTimeUserType.class.getName() );
    //        List<PersonIdentifierInterface> personIdentifiers = new ArrayList<PersonIdentifierInterface>();
    //        Map<String, QueryStrategyInterface> queryStrategiesMap = new HashMap<String, QueryStrategyInterface>();
    //        List<QueryWorkInterface> queryWork = new ArrayList<QueryWorkInterface>();
    //        QueryWorkInterface work = EasyMock.createMock( QueryWorkInterface.class );
    //        expect(work.getExceptions()).andReturn( new ArrayList<Exception>()) ;
    //        queryWork.add( work );
    //        
    //        EntryFilterInterface entryFilter = EasyMock.createMock( EntryFilterInterface.class );
    //        expect(entryFilter.getTemplateId() ).andReturn(templateId).anyTimes();
    //        expect(entryFilter.getRequestId() ).andReturn(requestId).anyTimes();
    //        
    //        PersonIdentifierInterface pid1 = EasyMock.createMock( PersonIdentifierInterface.class );
    //        PersonIdentifierInterface pid2 = EasyMock.createMock( PersonIdentifierInterface.class );
    //        personIdentifiers.add( pid1 );
    //        personIdentifiers.add( pid2 );
    //        
    //        QueryStrategyInterface queryStrategy = EasyMock.createMock( QueryStrategyInterface.class );
    //        SessionFactory sessionFactory = EasyMock.createMock( SessionFactory.class );
    //        queryStrategiesMap.put( templateId, queryStrategy );
    //       
    //        expect( pid1.getAssigningFacility() ).andReturn( "111" );
    //        expect( pid2.getAssigningFacility() ).andReturn( "999" );
    //        expect( pid1.getIdentity() ).andReturn( "111" );
    //        expect( pid2.getIdentity() ).andReturn( "999" );
    //        expect( queryStrategy.createQueryWork( EasyMock.eq( sessionFactory), EasyMock.eq( entryFilter), EasyMock.isA( List.class ), EasyMock.eq( Dom4jPointInTimeUserType.class ))).andThrow( new HibernateException("DB Failure" ) );
    //        persistenceManager.setTemplateQueryStrategyMap( queryStrategiesMap );
    //
    //        expect(templateHelper.getErrorSectionHelper( templateId) ).andReturn( errorSectionHelper );
    //
    //        
    //        ClinicalDataResponseInterface cdr = new ClinicalDataBuilder();
    //        Document emptyClinicalDoc = cdr.buildEmptyClinicalDocument( templateId, requestId );
    //        ExceptionInfo exceptionInfo = new ExceptionInfo();
    //        exceptionInfo.setClientMessage( "DB Conn Failed" );
    //        exceptionInfo.setErrorCode( ErrorCodeEnum.READ_REQUEST_DATA_SOURCE_FAILURE );
    //        exceptionInfo.setException( new HibernateException( "DB Conn Failed" ) );
    //        exceptionInfo.setExceptionMessage( "DB Conn Failed" );
    //        exceptionInfo.setLoggingSeverity( LoggingSeverity.WARNING );
    //        
    //        ErrorSectionHelperInterface esh = new FlatErrorSectionHelper();
    //        Document responseWithErrorSectionDocument = esh.buildErrorSection( emptyClinicalDoc, exceptionInfo, requestId );
    //        
    //        
    //        expect(errorSectionHelper.buildErrorSection(EasyMock.isA(Document.class), EasyMock.isA(ExceptionInfo.class), EasyMock.eq(requestId))).andReturn(responseWithErrorSectionDocument);
    //        
    //        expect(clinicalDataResponse.buildEmptyClinicalDocument(templateId, requestId)).andReturn(emptyClinicalDoc);
    //        
    //        expect(pid1.getAssigningFacility()).andReturn("111");
    //        expect(pid1.getIdentity()).andReturn("111");
    //        expect(entryFilter.getDomainEntryPoint()).andReturn("TIU");
    //        
    //        EasyMock.replay( entryFilter, queryStrategy, pid1, pid2, work, templateHelper, errorSectionHelper, clinicalDataResponse );
    //        
    //        persistenceManager.setSiteIdentifier( "111" );
    //        persistenceManager.setTemplateQueryStrategyMap( queryStrategiesMap );
    //        Map<String, SessionFactory> sessionFactoryMap = new HashMap<String, SessionFactory>();
    //        sessionFactoryMap.put( templateId, sessionFactory );
    //        persistenceManager.setSessionFactoryMap( sessionFactoryMap );        
    //        
    //        
    //        Document result = persistenceManager.performReadOnClinicalData( entryFilter, personIdentifiers );
    //        
    //        Assert.assertNotNull( result ); 
    //        Assert.assertTrue( result.getRootElement().element( "errorSection" ).element( "warnings" ).element( "errorCode" ).getText().equals( "READ_REQUEST_DATA_SOURCE_FAILURE" ));
    //        
    ////        System.out.print( result.asXML() + "\n");      
    //    }    

    //    @Test
    //    @Suite( groups = { "brokentest" } )
    //    public void testPerformReadOnClinicalData_PartialReads_UnknownException() throws Exception
    //    {
    //        String templateId = "TestTemplateId";
    //        String requestId = "REQUEST-99999";
    //        
    //        AbstractReadHibernatePersistenceManager persistenceManager = new SiteSpecificHibernateReadPersistenceManager();
    //        persistenceManager.setPointInTimeUserType(  Dom4jPointInTimeUserType.class.getName() );
    //        List<PersonIdentifierInterface> personIdentifiers = new ArrayList<PersonIdentifierInterface>();
    //        Map<String, QueryStrategyInterface> queryStrategiesMap = new HashMap<String, QueryStrategyInterface>();
    //        List<QueryWorkInterface> queryWork = new ArrayList<QueryWorkInterface>();
    //        QueryWorkInterface work = EasyMock.createMock( QueryWorkInterface.class );
    //        expect(work.getExceptions()).andReturn( new ArrayList<Exception>()) ;
    //        queryWork.add( work );
    //        
    //        EntryFilterInterface entryFilter = EasyMock.createMock( EntryFilterInterface.class );
    //        expect(entryFilter.getTemplateId() ).andReturn(templateId).anyTimes();
    //        expect(entryFilter.getRequestId() ).andReturn(requestId).anyTimes();
    //        
    //        PersonIdentifierInterface pid1 = EasyMock.createMock( PersonIdentifierInterface.class );
    //        personIdentifiers.add( pid1 );
    //        
    //        QueryStrategyInterface queryStrategy = EasyMock.createMock( QueryStrategyInterface.class );
    //        SessionFactory sessionFactory = EasyMock.createMock( SessionFactory.class );
    //        queryStrategiesMap.put( templateId, queryStrategy );
    //       
    //        expect( pid1.getAssigningFacility() ).andReturn( "111" );
    //        expect( pid1.getIdentity() ).andReturn( "111" );
    //        expect( queryStrategy.createQueryWork( EasyMock.eq( sessionFactory), EasyMock.eq( entryFilter), EasyMock.isA( List.class ), EasyMock.eq( Dom4jPointInTimeUserType.class ))).andThrow( new RuntimeException( "Unknown error occurred" ) );
    //        persistenceManager.setTemplateQueryStrategyMap( queryStrategiesMap );
    //
    //        expect(templateHelper.getErrorSectionHelper( templateId) ).andReturn( errorSectionHelper );
    //
    //        
    //        ClinicalDataResponseInterface cdr = new ClinicalDataBuilder();
    //        Document emptyClinicalDoc = cdr.buildEmptyClinicalDocument( templateId, requestId );
    //        ExceptionInfo exceptionInfo = new ExceptionInfo();
    //        exceptionInfo.setClientMessage( "Unknown error occurred" );
    //        exceptionInfo.setErrorCode( ErrorCodeEnum.UNEXPECTED_READ_EXCEPTION );
    //        exceptionInfo.setException( new RuntimeException( "Something was null" ) );
    //        exceptionInfo.setExceptionMessage( "Unknown error occurred" );
    //        
    //        ErrorSectionHelperInterface esh = new FlatErrorSectionHelper();
    //        Document responseWithErrorSectionDocument = esh.buildErrorSection( emptyClinicalDoc, exceptionInfo, requestId );
    //        
    //        
    //        expect(errorSectionHelper.buildErrorSection(EasyMock.isA(Document.class), EasyMock.isA(ExceptionInfo.class), EasyMock.eq(requestId))).andReturn(responseWithErrorSectionDocument);
    //        
    //        expect(clinicalDataResponse.buildEmptyClinicalDocument(templateId, requestId)).andReturn(emptyClinicalDoc);
    //        
    //        expect(pid1.getAssigningFacility()).andReturn("111");
    //        expect(pid1.getIdentity()).andReturn("111");
    //        expect(entryFilter.getDomainEntryPoint()).andReturn("TIU");
    //        
    //        EasyMock.replay( entryFilter, queryStrategy, pid1, work, templateHelper, errorSectionHelper, clinicalDataResponse );
    //        
    //        persistenceManager.setSiteIdentifier( "111" );
    //        persistenceManager.setTemplateQueryStrategyMap( queryStrategiesMap );
    //        Map<String, SessionFactory> sessionFactoryMap = new HashMap<String, SessionFactory>();
    //        sessionFactoryMap.put( templateId, sessionFactory );
    //        persistenceManager.setSessionFactoryMap( sessionFactoryMap );        
    //        
    //        
    //        Document result = persistenceManager.performReadOnClinicalData( entryFilter, personIdentifiers );
    //        
    //        Assert.assertNotNull( result );  
    //        
    ////        Assert.assertTrue( result.getRootElement().element( "errorSection" ).element( "fatalErrors" ).element( "errorCode" ).getText().equals( "UNEXPECTED_READ_EXCEPTION" ));
    ////        System.out.print( result.asXML() + "\n");      
    //    }       

    //    @Test
    //    @Suite( groups = { "brokentest" } )
    //    public void testPerformReadOnClinicalData_PartialReads_DBConnFailure() throws Exception
    //    {
    //        String templateId = "TestTemplateId";
    //        String requestId = "REQUEST-99999";
    //        
    //        AbstractReadHibernatePersistenceManager persistenceManager = new SiteSpecificHibernateReadPersistenceManager();
    //        persistenceManager.setPointInTimeUserType(  Dom4jPointInTimeUserType.class.getName() );
    //        List<PersonIdentifierInterface> personIdentifiers = new ArrayList<PersonIdentifierInterface>();
    //        Map<String, QueryStrategyInterface> queryStrategiesMap = new HashMap<String, QueryStrategyInterface>();
    //        List<QueryWorkInterface> queryWork = new ArrayList<QueryWorkInterface>();
    //        QueryWorkInterface work = EasyMock.createMock( QueryWorkInterface.class );
    //        expect(work.getExceptions()).andReturn( new ArrayList<Exception>()) ;
    //        queryWork.add( work );
    //        
    //        EntryFilterInterface entryFilter = EasyMock.createMock( EntryFilterInterface.class );
    //        expect(entryFilter.getTemplateId() ).andReturn(templateId).anyTimes();
    //        expect(entryFilter.getRequestId() ).andReturn(requestId).anyTimes();
    //        
    //        PersonIdentifierInterface pid1 = EasyMock.createMock( PersonIdentifierInterface.class );
    //        PersonIdentifierInterface pid2 = EasyMock.createMock( PersonIdentifierInterface.class );
    //        personIdentifiers.add( pid1 );
    //        personIdentifiers.add( pid2 );
    //        
    //        QueryStrategyInterface queryStrategy = EasyMock.createMock( QueryStrategyInterface.class );
    //        SessionFactory sessionFactory = EasyMock.createMock( SessionFactory.class );
    //        queryStrategiesMap.put( templateId, queryStrategy );
    //       
    //        expect( pid1.getAssigningFacility() ).andReturn( "111" );
    //        expect( pid2.getAssigningFacility() ).andReturn( "999" );
    //        expect( pid1.getIdentity() ).andReturn( "111" );
    //        expect( pid2.getIdentity() ).andReturn( "999" );
    //        expect( queryStrategy.createQueryWork( EasyMock.eq( sessionFactory), EasyMock.eq( entryFilter), EasyMock.isA( List.class ), EasyMock.eq( Dom4jPointInTimeUserType.class ))).andThrow( new HibernateException("DB Failure" ) );
    //        persistenceManager.setTemplateQueryStrategyMap( queryStrategiesMap );
    //
    //        expect(templateHelper.getErrorSectionHelper( templateId) ).andReturn( errorSectionHelper );
    //
    //        
    //        ClinicalDataResponseInterface cdr = new ClinicalDataBuilder();
    //        Document emptyClinicalDoc = cdr.buildEmptyClinicalDocument( templateId, requestId );
    //        ExceptionInfo exceptionInfo = new ExceptionInfo();
    //        exceptionInfo.setClientMessage( "DB Conn Failed" );
    //        exceptionInfo.setErrorCode( ErrorCodeEnum.READ_REQUEST_DATA_SOURCE_FAILURE );
    //        exceptionInfo.setException( new HibernateException( "DB Conn Failed" ) );
    //        exceptionInfo.setExceptionMessage( "DB Conn Failed" );
    //        exceptionInfo.setLoggingSeverity( LoggingSeverity.WARNING );
    //        
    //        ErrorSectionHelperInterface esh = new FlatErrorSectionHelper();
    //        Document responseWithErrorSectionDocument = esh.buildErrorSection( emptyClinicalDoc, exceptionInfo, requestId );
    //        
    //        
    //        expect(errorSectionHelper.buildErrorSection(EasyMock.isA(Document.class), EasyMock.isA(ExceptionInfo.class), EasyMock.eq(requestId))).andReturn(responseWithErrorSectionDocument);
    //        
    //        expect(clinicalDataResponse.buildEmptyClinicalDocument(templateId, requestId)).andReturn(emptyClinicalDoc);
    //        
    //        expect(pid1.getAssigningFacility()).andReturn("111");
    //        expect(pid1.getIdentity()).andReturn("111");
    //        expect(entryFilter.getDomainEntryPoint()).andReturn("TIU");
    //        
    //        EasyMock.replay( entryFilter, queryStrategy, pid1, pid2, work, templateHelper, errorSectionHelper, clinicalDataResponse );
    //        
    //        persistenceManager.setSiteIdentifier( "111" );
    //        persistenceManager.setTemplateQueryStrategyMap( queryStrategiesMap );
    //        Map<String, SessionFactory> sessionFactoryMap = new HashMap<String, SessionFactory>();
    //        sessionFactoryMap.put( templateId, sessionFactory );
    //        persistenceManager.setSessionFactoryMap( sessionFactoryMap );        
    //        
    //        
    //        Document result = persistenceManager.performReadOnClinicalData( entryFilter, personIdentifiers );
    //        
    //        Assert.assertNotNull( result ); 
    //        Assert.assertTrue( result.getRootElement().element( "errorSection" ).element( "warnings" ).element( "errorCode" ).getText().equals( "READ_REQUEST_DATA_SOURCE_FAILURE" ));
    //        
    ////        System.out.print( result.asXML() + "\n");      
    //    }    

    //    @Test
    //    @Suite( groups = { "brokentest" } )
    //    public void testPerformReadOnClinicalData_PartialReads_UnknownException() throws Exception
    //    {
    //        String templateId = "TestTemplateId";
    //        String requestId = "REQUEST-99999";
    //        
    //        AbstractReadHibernatePersistenceManager persistenceManager = new SiteSpecificHibernateReadPersistenceManager();
    //        persistenceManager.setPointInTimeUserType(  Dom4jPointInTimeUserType.class.getName() );
    //        List<PersonIdentifierInterface> personIdentifiers = new ArrayList<PersonIdentifierInterface>();
    //        Map<String, QueryStrategyInterface> queryStrategiesMap = new HashMap<String, QueryStrategyInterface>();
    //        List<QueryWorkInterface> queryWork = new ArrayList<QueryWorkInterface>();
    //        QueryWorkInterface work = EasyMock.createMock( QueryWorkInterface.class );
    //        expect(work.getExceptions()).andReturn( new ArrayList<Exception>()) ;
    //        queryWork.add( work );
    //        
    //        EntryFilterInterface entryFilter = EasyMock.createMock( EntryFilterInterface.class );
    //        expect(entryFilter.getTemplateId() ).andReturn(templateId).anyTimes();
    //        expect(entryFilter.getRequestId() ).andReturn(requestId).anyTimes();
    //        
    //        PersonIdentifierInterface pid1 = EasyMock.createMock( PersonIdentifierInterface.class );
    //        personIdentifiers.add( pid1 );
    //        
    //        QueryStrategyInterface queryStrategy = EasyMock.createMock( QueryStrategyInterface.class );
    //        SessionFactory sessionFactory = EasyMock.createMock( SessionFactory.class );
    //        queryStrategiesMap.put( templateId, queryStrategy );
    //       
    //        expect( pid1.getAssigningFacility() ).andReturn( "111" );
    //        expect( pid1.getIdentity() ).andReturn( "111" );
    //        expect( queryStrategy.createQueryWork( EasyMock.eq( sessionFactory), EasyMock.eq( entryFilter), EasyMock.isA( List.class ), EasyMock.eq( Dom4jPointInTimeUserType.class ))).andThrow( new RuntimeException( "Unknown error occurred" ) );
    //        persistenceManager.setTemplateQueryStrategyMap( queryStrategiesMap );
    //
    //        expect(templateHelper.getErrorSectionHelper( templateId) ).andReturn( errorSectionHelper );
    //
    //        
    //        ClinicalDataResponseInterface cdr = new ClinicalDataBuilder();
    //        Document emptyClinicalDoc = cdr.buildEmptyClinicalDocument( templateId, requestId );
    //        ExceptionInfo exceptionInfo = new ExceptionInfo();
    //        exceptionInfo.setClientMessage( "Unknown error occurred" );
    //        exceptionInfo.setErrorCode( ErrorCodeEnum.UNEXPECTED_READ_EXCEPTION );
    //        exceptionInfo.setException( new RuntimeException( "Something was null" ) );
    //        exceptionInfo.setExceptionMessage( "Unknown error occurred" );
    //        
    //        ErrorSectionHelperInterface esh = new FlatErrorSectionHelper();
    //        Document responseWithErrorSectionDocument = esh.buildErrorSection( emptyClinicalDoc, exceptionInfo, requestId );
    //        
    //        
    //        expect(errorSectionHelper.buildErrorSection(EasyMock.isA(Document.class), EasyMock.isA(ExceptionInfo.class), EasyMock.eq(requestId))).andReturn(responseWithErrorSectionDocument);
    //        
    //        expect(clinicalDataResponse.buildEmptyClinicalDocument(templateId, requestId)).andReturn(emptyClinicalDoc);
    //        
    //        expect(pid1.getAssigningFacility()).andReturn("111");
    //        expect(pid1.getIdentity()).andReturn("111");
    //        expect(entryFilter.getDomainEntryPoint()).andReturn("TIU");
    //        
    //        EasyMock.replay( entryFilter, queryStrategy, pid1, work, templateHelper, errorSectionHelper, clinicalDataResponse );
    //        
    //        persistenceManager.setSiteIdentifier( "111" );
    //        persistenceManager.setTemplateQueryStrategyMap( queryStrategiesMap );
    //        Map<String, SessionFactory> sessionFactoryMap = new HashMap<String, SessionFactory>();
    //        sessionFactoryMap.put( templateId, sessionFactory );
    //        persistenceManager.setSessionFactoryMap( sessionFactoryMap );        
    //        
    //        
    //        Document result = persistenceManager.performReadOnClinicalData( entryFilter, personIdentifiers );
    //        
    //        Assert.assertNotNull( result );  
    //        
    ////        Assert.assertTrue( result.getRootElement().element( "errorSection" ).element( "fatalErrors" ).element( "errorCode" ).getText().equals( "UNEXPECTED_READ_EXCEPTION" ));
    ////        System.out.print( result.asXML() + "\n");      
    //    }       

}
