

package gov.va.med.cds.request;


import gov.va.med.cds.exception.ErrorCodeEnum;
import gov.va.med.cds.exceptionframework.ExceptionHandler;
import gov.va.med.cds.filter.FilterValidatorInterface;
import gov.va.med.cds.internal.ClinicalDataServiceSynchronousInternalInterface;
import gov.va.med.cds.template.TemplateHelperInterface;

import java.util.List;

import org.dom4j.Document;
import org.dom4j.DocumentHelper;
import org.springframework.util.StringUtils;


/**
 * 
 * Performs the CRUAD operations based on the client request. See gov.va.med.cds.config.requestProcessorContext.xml for
 * Spring DI details.
 * 
 */
public class TemplateRequestProcessor
    implements
        RequestProcessorInterface,
        ResponseValidatorMBeanInterface
{
    private ClinicalDataServiceSynchronousInternalInterface delegatingClinicalDataServiceSynchronousInternal = null;

    private TemplateHelperInterface templateHelper;
    private FilterValidatorInterface filterManager;
    private boolean validateReadResponseXML = false;

    /**
     * 
     * @see gov.va.med.cds.request.RequestProcessorInterface#readClinicalData(java.lang.String, java.lang.String,
     *      java.lang.String, java.lang.String)
     */
    /**
     * Read data
     */
    public Document readClinicalData( String templateId, String filterXmlRequest, String filterId, String requestId )
    {
        Document responseDocument = null;
        try
        {
            checkForNullReadParameters( templateId, filterId, filterXmlRequest, requestId );

            responseDocument = DocumentHelper.createDocument();

            filterManager.validateFilterXml( filterId, filterXmlRequest );

            List<Document> readResponses = delegatingClinicalDataServiceSynchronousInternal.readClinicalData( filterXmlRequest, filterId, templateId,
                            requestId );
            responseDocument = templateHelper.getResponseAggregator( templateId ).aggregateResponses( readResponses );
            responseDocument = templateHelper.getResponseSequencer( templateId ).sequence( templateId, responseDocument, validateReadResponseXML );
            responseDocument = ( templateHelper.getResponseOrganizer( templateId ) != null ) ? templateHelper.getResponseOrganizer( templateId )
                            .organize( filterXmlRequest, responseDocument ) : responseDocument;
        }
        catch ( Exception e )
        {
            responseDocument = processException( e, templateId, filterXmlRequest, filterId, requestId, this.templateHelper.getApplicationName(templateId) );
        }

        return responseDocument;
    }


    private void checkForNullReadParameters( String templateId, String filterId, String filterRequest, String requestId )
    {
        StringBuffer nullParams = new StringBuffer();

        if ( !StringUtils.hasText( templateId ) )
        {
            nullParams.append( "TemplateId " );
        }

        if ( !StringUtils.hasText( filterId ) )
        {
            nullParams.append( "FilterId " );
        }

        if ( !StringUtils.hasText( filterRequest ) )
        {
            nullParams.append( "FilterRequestXML " );
        }

        if ( !StringUtils.hasText( requestId ) )
        {
            nullParams.append( "RequestId" );
        }

        if ( nullParams.length() > 0 )
        {
            throw new ValidationException( ErrorCodeEnum.READ_REQUEST_INPUT_PARAMETERS_NULL, nullParams.toString() );
        }

    }


    private void checkForNullCUADParameters( String writeRequest, String templateId, String requestId )
    {
        StringBuffer nullParams = new StringBuffer();

        if ( !StringUtils.hasText( writeRequest ) )
        {
            nullParams.append( "RequestXML " );
        }

        if ( !StringUtils.hasText( templateId ) )
        {
            nullParams.append( "TemplateId " );
        }

        if ( !StringUtils.hasText( requestId ) )
        {
            nullParams.append( "RequestId " );
        }

        if ( nullParams.length() > 0 )
        {
            throw new ValidationException( ErrorCodeEnum.WRITE_REQUEST_INPUT_PARAMETERS_NULL, nullParams.toString() );
        }
    }


    private Document processException( Exception e, String templateId, String requestId, String appName )
    {
        Document errorResponseDocument = null;
        try
        {
            errorResponseDocument = ExceptionHandler.handleException( e, templateId, requestId, appName );
        }
        catch ( Exception ex )
        {
            // ignore this top level exception since it was already logged and we want to send info back to the
            // client
        }
        return errorResponseDocument;
    }


    private Document processException( Exception e, String templateId, String filterRequest, String filterId, String requestId, String appName )
    {
        Document errorResponseDocument = null;
        try
        {
            errorResponseDocument = ExceptionHandler.handleException( e, templateId, filterRequest, filterId, requestId, appName );
        }
        catch ( Exception ex )
        {
            // ignore this top level exception since it was already logged and we want to send info back to the
            // client
        }
        return errorResponseDocument;
    }


    /**
     * 
     * @see gov.va.med.cds.request.RequestProcessorInterface#createClinicalData(java.lang.String, java.lang.String,
     *      java.lang.String)
     */
    public Document createClinicalData( String request, String templateId, String requestId )
    {
        Document writeResponse = null;
        try
        {
            checkForNullCUADParameters( request, templateId, requestId );

            writeResponse = delegatingClinicalDataServiceSynchronousInternal.createClinicalData( request, templateId, requestId );
        }
        catch ( Exception e )
        {
            writeResponse = processException( e, templateId, requestId, this.templateHelper.getApplicationName(templateId) );
        }
        return writeResponse;
    }


    /**
     * Create or Update. This method is a place holder method for work that could be done to handle a create or update
     * where we don't know the order of records received. This method would need to resolve if the call was a create or
     * an update and if the data was the most recent data. Currently this logic is being done in the database by
     * triggers.
     * 
     * @param request request xml
     * @param templateId template id
     * @param requestId request id
     * @return cds response object
     */
    public Document createOrUpdateClinicalData( String request, String templateId, String requestId )
    {
        Document writeResponse = null;
        try
        {
            checkForNullCUADParameters( request, templateId, requestId );
            writeResponse = delegatingClinicalDataServiceSynchronousInternal.createClinicalData( request, templateId, requestId );
        }
        catch ( Exception e )
        {
            writeResponse = processException( e, templateId, requestId, this.templateHelper.getApplicationName(templateId) );
        }
        return writeResponse;
    }


    /**
     * 
     * @see gov.va.med.cds.request.RequestProcessorInterface#updateClinicalData(java.lang.String, java.lang.String,
     *      java.lang.String)
     */
    public Document updateClinicalData( String request, String templateId, String requestId )
    {
        Document writeResponse = null;
        try
        {
            checkForNullCUADParameters( request, templateId, requestId );
            writeResponse = delegatingClinicalDataServiceSynchronousInternal.updateClinicalData( request, templateId, requestId );
        }
        catch ( Exception e )
        {
            writeResponse = processException( e, templateId, requestId, this.templateHelper.getApplicationName(templateId) );
        }
        return writeResponse;
    }


    /**
     * 
     * @see gov.va.med.cds.request.RequestProcessorInterface#appendClinicalData(java.lang.String, java.lang.String,
     *      java.lang.String)
     */
    public Document appendClinicalData( String request, String templateId, String requestId )
    {
        Document writeResponse = null;
        try
        {
            checkForNullCUADParameters( request, templateId, requestId );
            writeResponse = delegatingClinicalDataServiceSynchronousInternal.appendClinicalData( request, templateId, requestId );
        }
        catch ( Exception e )
        {
            writeResponse = processException( e, templateId, requestId, this.templateHelper.getApplicationName(templateId) );
        }
        return writeResponse;
    }


    /**
     * 
     * @see gov.va.med.cds.request.RequestProcessorInterface#deleteClinicalData(java.lang.String, java.lang.String,
     *      java.lang.String)
     */
    public Document deleteClinicalData( String request, String templateId, String requestId )
    {
        Document writeResponse = null;
        try
        {
            checkForNullCUADParameters( request, templateId, requestId );
            writeResponse = delegatingClinicalDataServiceSynchronousInternal.deleteClinicalData( request, templateId, requestId );
        }
        catch ( Exception e )
        {
            writeResponse = processException( e, templateId, requestId, this.templateHelper.getApplicationName(templateId) );
        }
        return writeResponse;
    }


    public void setFilterManager( FilterValidatorInterface filterManager )
    {
        this.filterManager = filterManager;
    }


    public void setDelegatingClinicalDataServiceSynchronousInternal(
                    ClinicalDataServiceSynchronousInternalInterface delegatingClinicalDataServiceSynchronousInternal )
    {
        this.delegatingClinicalDataServiceSynchronousInternal = delegatingClinicalDataServiceSynchronousInternal;
    }


    public void setTemplateHelper( TemplateHelperInterface templateHelper )
    {
        this.templateHelper = templateHelper;
    }


    public boolean isValidateReadResponseXML( )
    {
        return validateReadResponseXML;
    }


    public void setValidateReadResponseXML( boolean validateReadResponseXML )
    {
        this.validateReadResponseXML = validateReadResponseXML;
    }

}
