/**
 * 
 */

package gov.va.med.cds.persistence.hibernate.common;

import java.io.Serializable;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Properties;

import oracle.sql.OPAQUE;
import oracle.xdb.XMLType;

import org.dom4j.DocumentException;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.dom4j.Node;
import org.hibernate.HibernateException;
import org.hibernate.engine.SessionImplementor;
import org.hibernate.usertype.EnhancedUserType;
import org.hibernate.usertype.ParameterizedType;

/**
 * @author vhaislegberb
 * 
 */
public class Dom4jXmlTypeUserType implements EnhancedUserType,
		ParameterizedType {

	private static final int[] SQL_TYPES = { XMLType._SQL_TYPECODE };
	private static final String CONTAINER_ELEMENT_NAME = "containerElementName";
    private static final String NT_PARAMETER_NAME = "nodeType";
    
    enum NodeType { CDATA, ELEMENT, TEXT };

	private Properties parameters;

	private NodeType nodeType = NodeType.CDATA;

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.hibernate.usertype.UserType#sqlTypes()
	 */
	@Override
	public int[] sqlTypes() {
		return SQL_TYPES;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.hibernate.usertype.UserType#returnedClass()
	 */
	@Override
	public Class<Element> returnedClass() {
		return Element.class;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.hibernate.usertype.UserType#equals(java.lang.Object,
	 * java.lang.Object)
	 */
	@Override
	public boolean equals(Object x, Object y) throws HibernateException {
		if (x == null || y == null)
			return false;
		if (x == y)
			return true;
		return x.equals(y);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.hibernate.usertype.UserType#hashCode(java.lang.Object)
	 */
	@Override
	public int hashCode(Object x) throws HibernateException {
		return x.hashCode();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.hibernate.usertype.UserType#nullSafeGet(java.sql.ResultSet,
	 * java.lang.String[], java.lang.Object)
	 */
	@Override
	public Object nullSafeGet(ResultSet rs, String[] names, Object owner)
			throws HibernateException, SQLException {
		return nullSafeGet(rs, names, null, owner);
	}

	public Object nullSafeGet(ResultSet rs, String[] names,
			SessionImplementor session, Object owner)
			throws HibernateException, SQLException {
		Element element = null;
		Object obj = rs.getObject(names[0]);
		XMLType xmlType = null;

		if (!rs.wasNull()) {
			if (obj instanceof XMLType) {
				xmlType = (XMLType) obj;
			} else if (obj instanceof OPAQUE) {
				xmlType = XMLType.createXML((OPAQUE) obj);
			}

			if (xmlType != null) {
				element = (Element) fromXMLString(xmlType.getStringVal());
			}
		} else {
			element = (Element) fromXMLString("");
		}
		return element;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.hibernate.usertype.UserType#nullSafeSet(java.sql.PreparedStatement,
	 * java.lang.Object, int)
	 */
	@Override
	public void nullSafeSet(PreparedStatement st, Object value, int index)
			throws HibernateException, SQLException {
		if (value instanceof String) {
			java.sql.Connection conn = st.getConnection();
			if (!(conn instanceof oracle.jdbc.OracleConnection)) {
				if (conn instanceof org.apache.commons.dbcp.DelegatingConnection) {
					conn = ((org.apache.commons.dbcp.DelegatingConnection) conn)
							.getInnermostDelegate();
				} else {
					conn = null;
				}
				if (conn == null) {
					throw new SQLException(
							"Could not locate an OracleConnection, required for XMLType");
				}
			}
			XMLType xmlTypeData = XMLType.createXML(conn, (String)value);
			st.setObject(index, xmlTypeData);
		} else {
			throw new SQLException(String.format(
					"Illegal argument. Expected java.lang.String got %s.",
					value.getClass().getName()));
		}

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.hibernate.usertype.UserType#deepCopy(java.lang.Object)
	 */
	@Override
	public Object deepCopy(Object value) throws HibernateException {
		if (value instanceof Node) {
			return ((Node) value).clone();
		} else if (value instanceof String ){
			return new String((String)value);
		} else {
			throw new HibernateException(String.format(
					"Illegal argument. %s got %s.", Node.class.getName(), value
							.getClass().getName()));
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.hibernate.usertype.UserType#isMutable()
	 */
	@Override
	public boolean isMutable() {
		return true;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.hibernate.usertype.UserType#disassemble(java.lang.Object)
	 */
	@Override
	public Serializable disassemble(Object value) throws HibernateException {
		if (value instanceof Node) {
			return (Serializable) ((Node) value).asXML();
		} else {
			throw new HibernateException(String.format(
					"Illegal argument. %s got %s.", Node.class.getName(), value
							.getClass().getName()));
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.hibernate.usertype.UserType#assemble(java.io.Serializable,
	 * java.lang.Object)
	 */
	@Override
	public Object assemble(Serializable cached, Object owner)
			throws HibernateException {
		if (cached instanceof String && owner instanceof Element) {
			Element parentElement = (Element) owner;

			String xmlData = (String) cached;

			Element element = DocumentHelper.createElement(this.parameters
					.getProperty(CONTAINER_ELEMENT_NAME));
			List<Node> nodes = createNode(xmlData);
			
			if( nodes != null && nodes.size() > 0 )
			{
				for( Node n : nodes )
				{
					element.add((Node)n.clone());
				}
			}
			

			parentElement.add(element);

			return element;
		} else {
			throw new HibernateException(String.format(
					"Illegal argument. %s got %s.", Node.class.getName(),
					cached.getClass().getName()));
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.hibernate.usertype.UserType#replace(java.lang.Object,
	 * java.lang.Object, java.lang.Object)
	 */
	@Override
	public Object replace(Object original, Object target, Object owner)
			throws HibernateException {
		// I'm not sure the implementation of this method is correct and we
		// don't do any
		// merges to test it out. vhaislegberb
		if (original instanceof Element && target instanceof Element
				&& owner instanceof Element) {
			Element originalElement = (Element) original;
			Element targetElement = (Element) target;
			Element ownerElement = (Element) owner;

			targetElement.detach();
			ownerElement.add(originalElement);
			return originalElement;
		} else {
			throw new HibernateException(String.format(
					"Illegal argument. EXpected %s.", Element.class.getName()));
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.hibernate.usertype.EnhancedUserType#objectToSQLString(java.lang.Object
	 * )
	 */
	@Override
	public String objectToSQLString(Object value) {
		if (value instanceof Node) {
			return nodeToXMLString(((Node) value));
		} else {
			return value.toString();
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.hibernate.usertype.EnhancedUserType#toXMLString(java.lang.Object)
	 */
	@Override
	public String toXMLString(Object value) {
		String stringXml = value.toString();

		if (value instanceof Element) {
			Element valueElement = (Element) value;
			for (Object o : valueElement.content()) {
				if (o instanceof Node) {
					stringXml = nodeToXMLString((Node) o);
				}
			}
		}
		return stringXml;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.hibernate.usertype.EnhancedUserType#fromXMLString(java.lang.String)
	 */
	@Override
	public Object fromXMLString(String xmlValue) {
		if (this.parameters != null
				&& this.parameters.getProperty(CONTAINER_ELEMENT_NAME) != null) {
			Element e = DocumentHelper.createElement(this.parameters
					.getProperty(CONTAINER_ELEMENT_NAME));
			
			List<Node> nodes = createNode(xmlValue);
			if( nodes != null && nodes.size() > 0)
			{
				for(Node n : nodes)
				{
					e.add((Node)n.clone());
				}
			}
			
			return e;
		} else {
			throw new HibernateException(String.format(
					"%s must be initialized with the parameter %s", getClass()
							.getName(), CONTAINER_ELEMENT_NAME));
		}
	}

	@Override
	public void setParameterValues(Properties parameters) {
		this.parameters = parameters;
		String nodeTypeParamValue = parameters.getProperty( NT_PARAMETER_NAME );
		
		if(nodeTypeParamValue == null || nodeTypeParamValue.length() == 0)
		{
			this.nodeType = NodeType.CDATA;
		}
		else 
		{
			try {
				this.nodeType = NodeType.valueOf(nodeTypeParamValue);
			} catch (Exception e) {
				throw new HibernateException(String.format("Error setting Node Type. '%s' parameter value ('%s') not set or set incorrectly.", NT_PARAMETER_NAME, nodeTypeParamValue));
			}
		}

	}

	@SuppressWarnings("unchecked")
	protected List<Node> createNode(String xmlData) {
		List<Node> nodes = new ArrayList<Node>();
		switch (this.nodeType) {
		case ELEMENT:
			try {
				// have to handle multiple values. wrap in a container element beofre adding them to the node list.
				xmlData = String.format("<container>%s</container>", xmlData);
				nodes.addAll((Collection<Node>)DocumentHelper.parseText(xmlData).getRootElement().elements());
				break;
			} catch (DocumentException e) {
				throw new HibernateException("Error parsing XML data.", e);
			}
		case CDATA:
		default:
			nodes.add(DocumentHelper.createCDATA(xmlData));
			break;
		}
		
		return nodes;

	}

	@SuppressWarnings("unchecked")
	protected String nodeToXMLString(Node nodeData) {
		
		StringBuffer xmlString = new StringBuffer();
		
		for(Node child :  (List<Node>)((Element)nodeData).content())
		{
			switch (this.nodeType) {
			case ELEMENT:
				if(child.getNodeType() == Node.ELEMENT_NODE )
				{
					xmlString.append(child.asXML());
				}
				break;
			case CDATA:
			default:
				if(child.getNodeType() == Node.CDATA_SECTION_NODE )
				{
					xmlString.append(child.getText());
				}
				break;
			}
			
		}
		
		return xmlString.toString();
	}

}
