package gov.va.med.cds.persistence.hibernate;

import static gov.va.med.cds.exception.ErrorCodeEnum.HDRII_OPERATION_FAILED;
import gov.va.med.cds.clinicaldata.Operation;
import gov.va.med.cds.exception.CensusException;
import gov.va.med.cds.exception.ErrorCodeEnum;
import gov.va.med.cds.exception.OperationNotSupportedException;
import gov.va.med.cds.persistence.PersistenceException;
import gov.va.med.cds.persistence.ReadException;
import gov.va.med.cds.persistence.WritePersistenceManagerInterface;
import gov.va.med.cds.rules.BooleanRuleInterface;
import gov.va.med.cds.valueobject.CensusMasterIngest;

import java.sql.Timestamp;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.lang.exception.ExceptionUtils;
import org.dom4j.Element;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.Example;
import org.hibernate.exception.ConstraintViolationException;
import gov.va.med.cds.clinicaldata.DomainEntryPoint;

public class WriteableSurveyPersistenceManager extends
		AbstractHibernatePersistenceManager implements
		WritePersistenceManagerInterface {

	private static final String SURVEY_QS_AND_AS = "surveyQuestionsAndAnswers";
	private static final String SURVEY = "SURVEY";
	private static final String SURVEY_TYPE = "type";
	private static final String SURVEY_TYPE_DMP = "DMP";
	private static final String SURVEY_PATIENT = "surveyPatient";
	private static final String PATIENT = "patient";
		
	private List<BooleanRuleInterface> rules;
	private String cdsAppName;

	public boolean isApplicable(Object critera) throws PersistenceException {
		boolean b = false;
		for (BooleanRuleInterface r : rules) {
			b = r.evaluate(critera);
			if (b) {
				break;
			}
		}

		return b;
	}

	public String performCUADOnClinicalData(String templateId,
			Operation operation, Object aObject, String domainEntryPoint)
			throws PersistenceException {
		Session session = null;
		String resultId = "";

		// we know this is a survey so we need to find values from lookup table
		// and use existing records.

		try {
			session = getSessionFactory(templateId).getCurrentSession();

			if (LOGGER.isDebugEnabled()) {
				LOGGER.debug(gov.va.med.cds.util.LogMessageUtil.buildMessage(
						templateId, null, cdsAppName, operation.name()
								+ aObject));
			}

			switch (operation) {
			case Create:
				resultId = createData(session, aObject);
				break;

			default:
				throw new UnsupportedOperationException(
						String.format(
								"The %s operation is not supported by the WriteableHibernatePersistenceManager.",
								operation.toString()));
			}

			session.flush();
			session.clear();
			return resultId;
		}
		// try to figure out what caused the violation
		catch (ConstraintViolationException e) {
			throw new HibernateConstraintViolationException(e);
		}
		// already handled and turned into one of ours so rethrow it
		catch (PersistenceException e) {
			throw e;
		}
		// any other problems are just big and will require someone to look at
		// so check log file
		catch (Exception e) {
			String rootCauseMessage = ExceptionUtils.getRootCause(e) == null ? e
					.getMessage() : ExceptionUtils.getRootCause(e).getMessage();
			throw new PersistenceException(HDRII_OPERATION_FAILED, e,
					operation.name(), rootCauseMessage, domainEntryPoint);
		}
	}

	/**
     * Handles the CUAD requests and applies the transaction to the data-source as directed by the request type.
     * 
     * @param templateId The identifier of the template being created, updated, appended, or deleted.
     * @param requestType - request type could be Create, Update, Append and Delete.
     * @param object - CDM objects to apply to the data-source.
     */
    //@Override
    public String performCUADOnPatientsData( String templateId, String aRequestId, Operation aOperation, Object aObject, DomainEntryPoint domainEntryPoint )
        throws PersistenceException
    {
        Session session = null;
        String resultId = "";
        List<Element> patientsList = null;
        Element patientsElement = (Element)aObject;
        patientsList = patientsElement.elements( "patient" );
        
        try 
        {
            session = getSessionFactory(templateId).getCurrentSession();

            if ( LOGGER.isDebugEnabled() ) 
            {
                LOGGER.debug(gov.va.med.cds.util.LogMessageUtil.buildMessage( templateId, aRequestId, cdsAppName, "Storing Census records for:" + patientsList.size() + " patients" ));
            }
            
            int batchCount = 0;

            double start= System.currentTimeMillis();

            Element censusRecord = null;
            for ( int i = 1; i <= patientsList.size(); i++ )
            {
            	batchCount++;
                Element patientElement = (Element) patientsList.get( i-1 ).clone();
                patientElement.detach();
                
                List<Element> clinicalRecords = domainEntryPoint.getClinicalRecords( patientElement );
                censusRecord = clinicalRecords.get( 0 );
                censusRecord.addElement( "requestId" ).addText( aRequestId );
                resultId = createData( session, censusRecord );
                if ( batchCount % 100 == 0 ) //100, same as the JDBC batch size
                { 
                    if ( LOGGER.isDebugEnabled() ) 
                    {
                        LOGGER.debug( "Writing batch of 100 census records: " + i );
                    }
                    //flush a batch of inserts and release memory:
                    session.flush();
                    session.clear();
                }
            }
  //          Timestamp aTimestamp = new Timestamp(System.currentTimeMillis());
  //          int aCurrentSegment = 1;
 //           performCUADOnCensusMetaData(session, aRequestId, templateId, aCurrentSegment, aTimestamp);
            
            double end = ( System.currentTimeMillis() - start );
            if ( LOGGER.isDebugEnabled() ) 
            {
            	LOGGER.debug( "===========================================================================\n");
                LOGGER.debug( "Total time storing census records: " + end + "\n");
                LOGGER.debug( "Total item or record count: " + batchCount + "\n" );
                LOGGER.debug( "===========================================================================\n");
            }
            //System.out.println("===========================================================================\n");
            //System.out.println("Total item or record count: " + batchCount + "\n");
            //System.out.println("Total time storing census records: " + end + "\n");
            //System.out.println("===========================================================================\n");
	
        }
        // try to figure out what caused the violation
        catch ( ConstraintViolationException e ) 
        {
            throw new HibernateConstraintViolationException(e);
        }
        // already handled and turned into one of ours so rethrow it
        catch ( PersistenceException e ) 
        {
            throw e;
        }
        // any other problems are just big and will require someone to look at
        // so check log file
        catch ( Exception e ) 
        {
            String rootCauseMessage = ExceptionUtils.getRootCause(e) == null ? e
                    .getMessage() : ExceptionUtils.getRootCause(e).getMessage();
            throw new PersistenceException( HDRII_OPERATION_FAILED, e, aOperation.name(), rootCauseMessage, domainEntryPoint.getName() );
        }
        
        return resultId;
    }

	/**
	 * Handles hibernate HDRII create processing.
	 * 
	 * @param session
	 *            - active hibernate session.
	 * @param aObject
	 *            - create request.
	 */
	protected String createData(Session session, Object aObject) {
		Element surveyResponse = (Element) aObject;

		validateAndRetrieveVendor(session, surveyResponse);
		validateAndRetrieveFacility(session, surveyResponse);
		retrieveOrStore(session, surveyResponse, "surveyPatient");
		retrieveOrStore(session, surveyResponse, "careCoordinator");
		storeSurveyAnswers(session, surveyResponse);
		return session.save(surveyResponse).toString();
	}

	/**
	 * RQ 279.8 HDR shall only persist HTH survey data to the HDR DB when the
	 * survey status is �ACCEPTED� and the data contains an answer for every
	 * question for the corresponding survey version.
	 * 
	 * @param session
	 * @param aSurveyResponse
	 */
	@SuppressWarnings("unchecked")
	private void storeSurveyAnswers(Session session, Element aSurveyResponse) {
		List<Element> results = null;
		Element incomingSurvey = aSurveyResponse.element("survey");
		Criteria criteria;

		try {
			criteria = session.createCriteria("survey");
			criteria.add(Example.create(incomingSurvey));
			results = criteria.list();

			if (results.size() != 1) {
				throw new IdentityException(
						ErrorCodeEnum.WRITE_REQUEST_SURVEY_SURVEYID_INVALID,
						incomingSurvey.elementText("surveyTitle"));
			}

			Element survey = results.get(0);
			// for saving survey id in response
			aSurveyResponse.remove(incomingSurvey);
			aSurveyResponse.add(survey);
			
			String surveyQandAElementName = getElementNameEndingWith(aSurveyResponse, SURVEY_QS_AND_AS);
			
			if (aSurveyResponse.elementText("responseStatus")
					.equalsIgnoreCase("ACCEPTED")) {
				
				// Because this is different depending upon the type of 
				// survey submitted, we have to use a matching algorithm
				// to get the name of the element.
				Element questions = survey.element("surveyQuestion");
				List<Element> questionList = (List<Element>) questions
						.elements();
				Element responseQAs = aSurveyResponse
						.element(surveyQandAElementName);
				
				// Schema will validate that there are no duplicate answers for
				// the same question.
				// DB will ensure the question/answer combination exists,
				// so check that there are the correct number of answers for
				// this survey.
				int numQuestions = (questions == null) ? -1 : questions.elements("question").size();
				int numSurveyResponses = (responseQAs == null) ? -1 : responseQAs.elements().size();
				
				if (numQuestions == -1 || numSurveyResponses == -1 || numQuestions != numSurveyResponses) {
					
					throw new PersistenceException(
							ErrorCodeEnum.WRITE_REQUEST_SURVEY_RESPONSES_INVALID,
							Integer.toString(numQuestions), Integer
									.toString(numSurveyResponses));
				}

				Iterator<Element> it = responseQAs.elementIterator();
				while (it.hasNext()) {
					Element answer = (Element) it.next();
					System.out.println(answer.asXML());

					String answerQuestionNumber = answer
							.elementText("questionNumber");
					String answerQuestionAnswer = answer.elementText("answer");

					// get question
					Element surveyQuestion = getQuestion(answerQuestionNumber,
							questionList);
					Element surveyChoice = getChoice(answerQuestionAnswer,
							surveyQuestion);
					surveyChoice.detach();
					answer.add(surveyChoice);
					answer.remove(answer.element("questionNumber"));
					answer.remove(answer.element("answer"));
				}
			} else { // only store the metadata, so don't store answers
				aSurveyResponse.remove(aSurveyResponse
						.element(surveyQandAElementName));
			}
		} catch (org.hibernate.exception.GenericJDBCException e) {
			throw new ReadException(ErrorCodeEnum.HIBERNATE_READ_FAILURE, e,
					e.getMessage());
		}
	}
	

	private String getElementNameEndingWith(Element parent, String endsWith) {
		
		@SuppressWarnings("rawtypes")
		Iterator childIterator = parent.elementIterator();
		while(childIterator.hasNext())
		{
			Element child = (Element)childIterator.next();
			if(child.getName().endsWith(endsWith))
			{
				return child.getName();
			}
		}
		
		throw new IllegalArgumentException("Survey response contains must contain questions and answers.");
	}

	// is there a better way to do this with xpath that won't require looping?
	private Element getQuestion(String questionNumber,
			List<Element> questionList) {
		// /surveyQuestion/question/questionNumber[text()="2"]
		for (Element question : questionList) {
			if (question.elementText("questionNumber").equals(questionNumber)) {
				return question;
			}
		}
		return null;
	}

	// is there a better way to do this with xpath that won't require looping?
	private Element getChoice(String answerNumber, Element question) {
		@SuppressWarnings("unchecked")
		List<Element> choices = (List<Element>) question
				.element("surveyChoice").elements();
		for (Element choice : choices) {
			if (choice.elementText("choiceNumber").equals(answerNumber)) {
				return choice;
			}
		}
		return null;
	}

	@SuppressWarnings("unchecked")
	protected void retrieveOrStore(Session session, Element response,
			String entityName) {
		List<Element> results = null;
		Element transitive = response.element(entityName);

		if( SURVEY.equalsIgnoreCase(entityName) ){
			transitive.addElement(SURVEY_TYPE).addText(SURVEY_TYPE_DMP);
		}
		Criteria criteria = session.createCriteria(entityName);
		criteria.add(Example.create(transitive));
		try {
			results = criteria.list();
		} catch (org.hibernate.exception.GenericJDBCException e) {
			throw new ReadException(ErrorCodeEnum.HIBERNATE_READ_FAILURE, e,
					e.getMessage());
		}

		if (results.size() < 1) {
			session.save(transitive);
		} else {
			if ( entityName.equals(SURVEY_PATIENT) ){
				Element elementPatient = (Element)transitive.element(PATIENT).clone();
				elementPatient.detach();
				results.get(0).add(elementPatient);
			}
			response.remove(transitive);
			response.add(results.get(0));
		}
	}

	/**
	 * HTH now only wants us to validate the vendor number, not the name. So
	 * make sure the vendor number exists, and if there is one or more, look for
	 * the one with the same name. If there isn't a match on name, create a new
	 * vendor entry.
	 * 
	 * @param session
	 *            hibernate session
	 * @param aSurvey
	 *            the survey to store
	 */
	@SuppressWarnings("unchecked")
	protected void validateAndRetrieveVendor(Session session,
			Element aSurveyResponse) {
		List<Element> results = null;
		boolean foundVendor = false;

		Element incomingVendor = aSurveyResponse.element("vendor");

		Element vendorName = incomingVendor.element("vendorName");
		String vendorNameToStore = vendorName.getText();

		incomingVendor.remove(vendorName);

		Criteria criteria = session.createCriteria("vendor");
		criteria.add(Example.create(incomingVendor));
		try {
			results = criteria.list();
		} catch (org.hibernate.exception.GenericJDBCException e) {
			throw new ReadException(ErrorCodeEnum.HIBERNATE_READ_FAILURE, e,
					e.getMessage());
		}

		if (results.size() < 1) {
			// vendor number doesn't exist
			throw new ReadException(
					ErrorCodeEnum.WRITE_REQUEST_SURVEY_VENDOR_NUMBER_INVALID,
					incomingVendor.element("vendorNumber").getText());
		}

		for (Element vendor : results) {
			if (vendor.element("vendorName").getText()
					.equals(vendorNameToStore)) {
				aSurveyResponse.remove(incomingVendor);
				aSurveyResponse.add(vendor);
				foundVendor = true;
				break;
			}
		}

		if (!foundVendor) {
			incomingVendor.add(vendorName);
			session.save(incomingVendor);
		}
	}

	@SuppressWarnings("unchecked")
	protected void validateAndRetrieveFacility(Session session,
			Element aSurveyResponse) {
		List<Element> results = null;
		Element incomingFacility = aSurveyResponse.element("facility");

		Criteria criteria = session.createCriteria("facility");
		criteria.add(Example.create(incomingFacility));
		try {
			results = criteria.list();
		} catch (org.hibernate.exception.GenericJDBCException e) {
			throw new ReadException(ErrorCodeEnum.HIBERNATE_READ_FAILURE, e,
					e.getMessage());
		}

		if (results.size() != 1) {
			throw new ReadException(
					ErrorCodeEnum.WRITE_REQUEST_SURVEY_FACILITY_INVALID,
					incomingFacility.elementText("facilityNumber"));
		}

		aSurveyResponse.remove(incomingFacility);
		aSurveyResponse.add(results.get(0));

	}

	/**
	 * Sets the rules to evaluate to determine if the persistence manager is
	 * applicable for the request.
	 * 
	 * @param rules
	 *            The boolean rules to be evaluated.
	 */
	public void setRules(List<BooleanRuleInterface> rules) {
		this.rules = rules;
	}

	public void setCdsAppName(String cdsAppName) {
		this.cdsAppName = cdsAppName;
	}

	@Override
	public List<Element> performReadOnCensusMetaData(
			CensusMasterIngest censusMasterIngest) throws CensusException {
		throw new OperationNotSupportedException(ErrorCodeEnum.METHOD_NOT_IMPLEMENTED);
	}

	@Override
	public String performCUADOnCensusMetaData(Operation aOperation,
			CensusMasterIngest censusMasterIngest) throws PersistenceException {
		throw new OperationNotSupportedException(ErrorCodeEnum.METHOD_NOT_IMPLEMENTED);
	}
	

}
