/**
 * 
 */


package gov.va.med.cds.persistence.hibernate;


import gov.va.med.cds.clinicaldata.DomainEntryPoint;
import gov.va.med.cds.common.person.correlation.PersonIdentifierInterface;
import gov.va.med.cds.exception.ErrorCodeEnum;
import gov.va.med.cds.filter.EntryFilterInterface;
import gov.va.med.cds.persistence.PersistenceException;
import gov.va.med.cds.persistence.QueryStrategyInterface;
import gov.va.med.cds.persistence.QueryWorkInterface;
import gov.va.med.cds.persistence.ReadException;
import gov.va.med.cds.persistence.hibernate.common.VistaPreQueryExecutionInterface;

import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.orm.hibernate3.SessionFactoryUtils;

import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.sql.DataSource;


/**
 * @author vhaislegberb
 *
 */
public class SiteSpecificHibernateReadPersistenceManager
    extends
        DefaultHibernateReadPersistenceManager
{

    private Map<String, VistaPreQueryExecutionInterface> vistaPreQueryExecutionMap;


    /* (non-Javadoc)
     * @see gov.va.med.cds.persistence.ReadPersistenceManagerInterface#isApplicable(gov.va.med.cds.filter.EntryFilterInterface, java.util.List)
     */
    @Override
    public boolean isApplicable( EntryFilterInterface entryFilter, List<PersonIdentifierInterface> personIdentifiers )
        throws PersistenceException
    {
        return ( super.isApplicable( entryFilter, personIdentifiers ) && containsSiteIdentifier( personIdentifiers ) );
    }


    /*
     * (non-Javadoc)
     * @see gov.va.med.cds.persistence.hibernate.DefaultHibernateReadPersistenceManager#createQueryWork(gov.va.med.cds.persistence.QueryStrategyInterface, gov.va.med.cds.filter.EntryFilterInterface, java.util.List, java.lang.Class)
     */
    @Override
    protected List<QueryWorkInterface> createQueryWork( QueryStrategyInterface queryStrategy, EntryFilterInterface entryFilter,
                    List<PersonIdentifierInterface> personIdentifiers, Class<?> pointInTimeUserType, String siteId )
        throws PersistenceException
    {
        SessionFactory sessionFactory = getSessionFactory( entryFilter.getTemplateId() );
        List<String> sitePatients = findSitePatients( personIdentifiers );
        List<String> patientIdentifiers = handlePatientIdentifiers( sessionFactory.openSession(), entryFilter, sitePatients );
        return queryStrategy.createQueryWork( sessionFactory, entryFilter, patientIdentifiers, pointInTimeUserType, siteId );
    }


    /**
     * Iterates over the list until it comes across the person identifier with the assigning facility 
     * that matches the site.
     * @param personIdentifiers This list of person identifiers to look through.
     * @return The person identifier with an assigning facility that matches the 
     * site identifier of the 
     */
    private List<String> findSitePatients( List<PersonIdentifierInterface> personIdentifiers )
    {
        List<String> sitePatients = new ArrayList<String>();

        if ( personIdentifiers == null || personIdentifiers.size() == 0 )
        {
            throw new ReadException( ErrorCodeEnum.READ_PERSISTENCE_MGR_NULL_PERSON_IDENTIFIERS_LIST, this.managerIdentifier );
        }

        for ( PersonIdentifierInterface personIdentifier : personIdentifiers )
        {
            if ( this.siteIdentifier.equals( personIdentifier.getAssigningFacility() ) )
            {
                sitePatients.add( personIdentifier.getIdentity() );
            }
        }

        if ( sitePatients.size() == 0 )
        {
            throw new ReadException( ErrorCodeEnum.READ_PERSISTENCE_MGR_NO_PERSON_WITH_SITE_ID, this.siteIdentifier );
        }

        return sitePatients;
    }


    /**
     * Determines whether the list of 
     * @param personIdentifiers
     * @return
     */
    private boolean containsSiteIdentifier( List<PersonIdentifierInterface> personIdentifiers )
    {
        for ( PersonIdentifierInterface personIdentifier : personIdentifiers )
        {
            if ( this.siteIdentifier.equals( personIdentifier.getAssigningFacility() ) )
            {
                return true;
            }
        }
        return false;
    }


    public String getSiteIdentifier( )
    {
        return this.siteIdentifier;
    }


    @SuppressWarnings( "rawtypes" )
    protected List handlePatientIdentifiers( Session session, EntryFilterInterface entryFilter, List patientIdentifiers )
        throws ReadException
    {
        if ( vistaPreQueryExecutionMap != null )
        {
            VistaPreQueryExecutionInterface vistaPreQueryExecutor = vistaPreQueryExecutionMap.get( entryFilter.getTemplateId() + "-" + entryFilter.getDomainEntryPoint() );
            if ( vistaPreQueryExecutor != null )
            {
                return vistaPreQueryExecutor.getRowIds( session, entryFilter, patientIdentifiers );
            }
        }

        return patientIdentifiers;
    }


    @Required
    public void setVistaPreQueryExecutionMap( Map<String, VistaPreQueryExecutionInterface> vistaPreQueryExecutionMap )
    {
        this.vistaPreQueryExecutionMap = vistaPreQueryExecutionMap;
    }

}
