package gov.va.med.edp.web.controller;

import gov.va.med.authentication.kernel.LoginUserInfoVO;
import gov.va.med.edp.dao.ServerPackageVersionDao;
import gov.va.med.edp.dao.TrackingDao;
import gov.va.med.edp.vo.BigBoardDebugInfoVO;
import gov.va.med.edp.web.view.XmlView;

import org.apache.log4j.Logger;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.servlet.mvc.AbstractController;
import org.springframework.web.servlet.support.WebContentGenerator;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Map;
import java.util.HashMap;
import java.util.Enumeration;
import java.util.Set;

/**
 * TrackingController implementation to handle requests for the Main Board.
 */
public class TrackingController extends AbstractController implements InitializingBean {

    private TrackingDao trackingDao;
    private ServerPackageVersionDao serverPackageVersionDao;
    protected static final String XML_VIEW = "xmlView";

    private static Logger log = Logger.getLogger(TrackingController.class);
    
	/**
	 *  Initialize the environment.
	 */
    public TrackingController() {
        setRequireSession(true);
        setSupportedMethods(new String[]{WebContentGenerator.METHOD_POST});
    }

	/**
	 *  Unit tests.
	 */
    public void afterPropertiesSet() throws Exception {
        Assert.notNull(trackingDao, "trackingDao is required");
    }

    /**
     * Initialize and set up the display parameters.
     * @param  request  The HTTP request.
	 * @param  response  The HTTP response.
	 * @return The ModelAndView of the data visualization.
     */
    protected ModelAndView handleRequestInternal(HttpServletRequest request, HttpServletResponse response) throws Exception {
       
    	debug("->handleRequestInternal request : " + getRequestParameterString(request) 
				+ " , response : " + response
				+ " , viewname : " + XML_VIEW);
    	
    	validateSwfID(request);
    	
        LoginUserInfoVO userInfo = getUserInfo(request);
        
        debug("->handleRequestInternal userInfo : " + userInfo.toString());
        
        String result = trackingDao.executeCommand(userInfo.getLoginStationNumber(), userInfo.getUserDuz(), buildParameterMap(request));
        
        if (isInitDisplayBoardCommand(request)) {
        	result = appendSiteAndVistaLinkConnectionInfo(result, request);
        }
        
        debug("->handleRequestInternal result " + result);
        
        return createModelAndView(result);
    }

    private void validateSwfID(HttpServletRequest request) throws UnrecognizedClientException {
        String swfID = request.getParameter("swfID");
        if (!StringUtils.hasLength(swfID)) return;

        String ids = (String) request.getSession().getAttribute(SessionConstants.RECOGNIZED_SWF_IDS_KEY);
        Set idSet = StringUtils.commaDelimitedListToSet(ids);
        if (idSet.isEmpty()) throw new UnrecognizedClientException(UnrecognizedClientException.UNRECOGNIZED_CLIENT_MESSAGE, swfID);
        if (!idSet.contains(swfID)) throw new UnrecognizedClientException(UnrecognizedClientException.UNRECOGNIZED_CLIENT_MESSAGE, swfID);
    }

	/**
	 * Set the tracking DAO.
	 * @param trackingDao The tracking DAO.
	 */
    public void setTrackingDao(TrackingDao trackingDao) {
        this.trackingDao = trackingDao;
    }

	/**
	 * Set the server package DAO.
	 * @param serverPackageVersionDao The server package DAO.
	 */
    public void setServerPackageVersionDao(ServerPackageVersionDao serverPackageVersionDao) {
        this.serverPackageVersionDao = serverPackageVersionDao;
    }    
    
	/**
	 *  Retrieve Login User information.
	 * @param request  The HTTP request.
	 * @return The Login User info.
	 */
    protected LoginUserInfoVO getUserInfo(HttpServletRequest request) throws NoUserInfoInSessionException {
        LoginUserInfoVO userInfo = (LoginUserInfoVO) request.getSession().getAttribute(LoginUserInfoVO.SESSION_KEY);
        if (userInfo == null) throw new NoUserInfoInSessionException();
        return userInfo;
    }

	/**
	 *  Create the ModelAndView.
	 * @param result The result from HTTP.
	 * @return The ModelAndView.
	 */
    protected ModelAndView createModelAndView(String result) {
        return new ModelAndView(XML_VIEW, XmlView.XML_FRAGMENT, result);
    }

	private boolean isInitDisplayBoardCommand(HttpServletRequest request) {
		if (request.getParameter("command").equals("initDisplayBoard")) return true;
		return false;
	}    
    
    private String appendSiteAndVistaLinkConnectionInfo(String result,HttpServletRequest request) throws Exception{
    	String stnNumber = getUserInfo(request).getLoginStationNumber();
    	BigBoardDebugInfoVO vistalinkInfo = serverPackageVersionDao.getVistaLinkConnectionInfo(stnNumber);
    	vistalinkInfo.setSiteId(stnNumber);
    	result = result.concat(vistalinkInfo.toString());
    	return result;
    }	
	
   /**
     * Initialize and set up the Site display parameters map.
     * @param  request  The HTTP request.
	 * @return The map of the data visualization.
     */
    protected Map buildParameterMap(HttpServletRequest request) {
        Map params = new HashMap();
        Enumeration e = request.getParameterNames();
        while (e.hasMoreElements()) {
            String paramName = (String) e.nextElement();
            params.put(paramName, request.getParameter(paramName));
        }
        return params;
    }
    
	/**
	 *  Parse the HTPP parameter string.
	 * @param request The HTTP request.
	 * @return The parsed data string.
	 */
    protected String getRequestParameterString(HttpServletRequest request) {

		StringBuilder builder = new StringBuilder("Request->(");

		Enumeration<String> keys = request.getParameterNames();
		
		while (keys.hasMoreElements()) {
			String paramName = (String) keys.nextElement();
			builder.append(paramName);
			builder.append("=");
			builder.append(request.getParameter(paramName));
			
			builder.append(",");
		}
		
		builder.setCharAt(builder.lastIndexOf(","), ')');

		return builder.toString();

	}
    
    private void debug(String s) {
    	if (log.isDebugEnabled()){
    		log.debug(s);
    	}
	}   
}
