package gov.va.med.edp.web.controller;

import gov.va.med.edp.dao.ServerPackageVersionDao;
import gov.va.med.edp.dao.SiteCodeLookUpDao;
import gov.va.med.edp.dao.TrackingDao;

import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;
import org.springframework.util.StringUtils;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.servlet.mvc.AbstractController;

/**
 * This controller handles request for the Site.
 */
public abstract class SiteCodeController extends AbstractController {

    public static final String MACHINE_NAME_PARAM = "machineName";
    public static final String SITE_CODE_PARAM = "sitecode";   
    public static final String BOARD_PARAM = "board";
    public static final String AREA_NAME_PARAM = "areaName";
    public static final String AREA_PARAM = "area";
    public static final String LAST_PARAM = "last";
    public static final String COMMAND_PARAM = "command";
    public static final String IS_FORM_PARAM = "isform";
    public static final String DEBUG_PARAM = "debug";
    public static final String MACHINE_NAME_HEADER = "EdisClientMachineName";
    public static final String VALID_SITECODE = "ValidSiteCode";
    
    private SiteCodeLookUpDao siteCodeDao;
	private TrackingDao trackingDao;
	protected ServerPackageVersionDao serverPackageVersionDao;
	
    private String errorViewName;
	private String defaultView;
	private String debugView;
	
	protected String defaultMsg;
	protected String errorMsg;
	
	protected boolean debugOn = false;
	
    private static Logger log = Logger.getLogger(SiteCodeController.class);
    

    /**
     * Is the application in debug mode.
     * @return True or False for debug.
     */
    public boolean isDebugOn() {
    	debug("->isDebugOn:" + this.debugOn);
    	return this.debugOn;
    }
 
    /**
     * Set the debug mode.
     * @param debugOn True or False for debug.
     */ 
    public void setDebugOn(String debugOn) {
    	debug("->setDebugOn:" + debugOn);
    	if(debugOn != null) {
    		if(debugOn.equals("true") || debugOn.equals("True")) {
    			this.debugOn = true;
    		}   		 
    	} else {
    		this.debugOn = false; 
    	}
    	debug("->setDebugOn out :" + this.debugOn);
    }
    
    /**
     * Create ModelAndView for the Site.
     * @param  errorMessage  The result of HTTP request processing.
	 * @return The modelAndView of the Site visualization.
     */
    protected ModelAndView createErrorModelAndView(String errorMessage) {
        ModelAndView mav = new ModelAndView(getErrorViewName());
        mav.addObject("errorMessage", errorMessage);
        return mav;
    }    
    
    /**
     * Create ModelAndView for the Site.
     * @param  deafaultMessage  The result of HTTP request processing.
	 * @return The modelAndView of the Site visualization.
     */
    protected ModelAndView createDefaultModelAndView(String deafaultMessage) {
        ModelAndView mav = new ModelAndView(getDefaultView());
        mav.addObject("defaultMessage", deafaultMessage);
        return mav;
    } 
        
	/**
	 *  Return site code DAO.
	 *  @return The site code DAO.
	 */
	public SiteCodeLookUpDao getSiteCodeDao() {
		return siteCodeDao;
	}

	/**
	 *  Set the site code DAO.
	 *  @param  siteCodeDao The site code DAO.
	 */
	public void setSiteCodeDao(SiteCodeLookUpDao siteCodeDao) {
		this.siteCodeDao = siteCodeDao;
	}

	/**
	 *  Return error view name.
	 *  @return The error view name.
	 */
	public String getErrorViewName() {
		return errorViewName;
	}

	/**
	 *  Set the error view name.
	 *  @param  errorViewName The error view name.
	 */
	public void setErrorViewName(String errorViewName) {
		this.errorViewName = errorViewName;
	}
	
	/**
	 * Get the default view page.
	 * @return the defaultView
	 */
	public String getDefaultView() {
		return defaultView;
	}

	/**
	 * Set the default view page.
	 * @param defaultView The defaultView to set
	 */
	public void setDefaultView(String defaultView) {
		this.defaultView = defaultView;
	}
	

	/**
	 * Get the debug view page.
	 * @return the debugView
	 */
	public String getDebugView() {
		return debugView;
	}

	/**
	 * Set the debug view page.
	 * @param debugView The debugView to set
	 */
	public void setDebugView(String debugView) {
		this.debugView = debugView;
	}
	
    /**
     * Initialize and set up the Site display parameters.
     * @param  request  The HTTP request.
	 * @param  machineName  The current machine name.
	 * @return The map of the Site visualization.
     */
    protected Map buildParameterMap(HttpServletRequest request, String machineName) {
        Map params = new HashMap();
        Enumeration e = request.getParameterNames();
        while (e.hasMoreElements()) {
            String paramName = (String) e.nextElement();
            params.put(paramName, request.getParameter(paramName));
        }
        //add the machine name that needs to be passed into M here
        params.put("machine", machineName.toUpperCase());
        debug("Parameters:" + params.toString());
        
        return params;
    }
    
	/**
	 * Build the parameter map.
	 * @param request
	 * @return The resulting map.
	 */
	@SuppressWarnings("rawtypes")
	protected Map<String, String> buildParameterMap(HttpServletRequest request) {
        Map<String, String> params = new HashMap<String, String>();
        Enumeration e = request.getParameterNames();
        
        while (e.hasMoreElements()) {
            String paramName = (String) e.nextElement();
            params.put(paramName, request.getParameter(paramName));         
        }
        
        debug("->buildParameterMap Parameters:" + params.toString());
        
        return params;
    }

	/**
	 *  Parse the CN values from the HTTP headers.
	 * @param machineName The current machine name.
	 * @return Returns the parsed machine name.
	 */
    protected String parseCNFromHttpHeaders(String machineName){
    	if (StringUtils.hasText(machineName) && machineName.startsWith("CN=")){
    		return machineName.substring(3);
    	}
        debug("Parsed Machine Name from Http Header is:" + machineName);
    	return machineName;
    }
    
	/**
	 * Retrieve the tracking DAO.
	 * @return The tracking DAO.
	 */
	public TrackingDao getTrackingDao() {
		return trackingDao;
	}

	/**
	 * Set the tracking DAO.
	 * @param trackingDao The tracking DAO.
	 */
	public void setTrackingDao(TrackingDao trackingDao) {
		this.trackingDao = trackingDao;
	}
	
    private void debug(String s) {
    	if (logger.isDebugEnabled()){
    		logger.debug(s);
    	}
	} 	
	
	/**
	 *  Retrieve the machine name from the Session.
	 * @param request The HTTP request.
	 * @return THe machine name.
	 */
	protected String getMachineNameFromUserPrincipalOrSessionOrRequest(HttpServletRequest request) throws UnrecognizedClientException {
		String machineName = "";
    	
		//first try user principal. This is for situations where we don't have a load balancer in the mix
		// this is where the client is talking directly to a weblogic server using 2-way ssl..
    	if (request.getUserPrincipal() != null){
    		machineName = request.getUserPrincipal().getName();
			debug("Machine name retrieved from user principal is:" + machineName);
    	}
		
		//then try request
		if (!StringUtils.hasText(machineName)){
			machineName = request.getParameter(MACHINE_NAME_PARAM);
			debug("Machine name retrieved from HTTP Request is:" + machineName);
		}
    	//then try the session..	
    	if (!StringUtils.hasText(machineName)){
    		machineName =(String)request.getSession().getAttribute(MACHINE_NAME_PARAM);
			debug("Machine name retrieved from HTTP Session is:" + machineName);
    	}		
    	
		if (StringUtils.hasText(machineName)){
			return machineName;
		} else {
    		throw new UnrecognizedClientException(UnrecognizedClientException.NO_MACHINE_NAME_PARAM_MESSAGE, MACHINE_NAME_PARAM);
		}
	}

	/**
	 * Set the server package DAO.
	 * @param serverPackageVersionDao The server package DAO.
	 */
	public void setServerPackageVersionDao(ServerPackageVersionDao serverPackageVersionDao) {
	    this.serverPackageVersionDao = serverPackageVersionDao;
	}	
    
	/**
	 * Retrieve the server package DAO.
	 * @return The server package DAO.
	 */
	public ServerPackageVersionDao getServerPackageVersionDao() {
	    return this.serverPackageVersionDao;
	}	
	
	/**
	 *  Resolve the site code.
	 * @param sitecode The current site code.
	 * @return The ModelAndView.
	 */
	protected ModelAndView evaluateSiteCode(String sitecode) {
		
		ModelAndView mav = null;
		
		if (!StringUtils.hasText(sitecode)) {
    		defaultMsg = "This URL does not resolve, please provide a sitecode. e.g. bigboard.html?sitecode=123ZZ&board=Main";
    		debug("->bigboard default " + defaultMsg + " , defaultView : " + getDefaultView());    		
    		mav = createDefaultModelAndView(defaultMsg);
    		return mav;
    	}
		
    	if (!isCiteCodeFormatValid(sitecode)){ 
    		errorMsg = "The siteCode parameter must be in the format 3 numbers and 2 letters or numbers (123ZZ).";
    		debug("->bigboard error " + errorMsg + " , errorViewName : " + getErrorViewName());   		
    		mav = createErrorModelAndView(errorMsg);
    		return mav;
    	}
    	
    	return mav;
	}
	
	
	// '636GF' 3 numbers, 2 chars
	private boolean isCiteCodeFormatValid(String sitecode) {
		boolean valid = false;
		
	
		if (sitecode != null && sitecode.length() == 5) {
			
			debug("->isCiteCodeFormatValid sitecode.length()  " 
			+ sitecode.length() + " sitecode.substring(0, 3) " 
			+ sitecode.substring(0, 3) + " sitecode.substring(3, 5) " + sitecode.substring(3, 5));
			
			valid = (sitecode.substring(0, 3).matches("[0-9]{3}") 
					&& sitecode.substring(3, 5).matches("[0-9a-zA-Z]{2}")) ;
			
		}
		
		if (sitecode != null && sitecode.length() == 3) {
			
			debug("->isCiteCodeFormatValid sitecode.length()  " 
			+ sitecode.length() + " sitecode " + sitecode);
			
			valid = sitecode.matches("[0-9]{3}");
			
		}
		
		return valid;
	}

}
