package gov.va.med.edp.rpc;

import gov.va.med.edp.vo.BigBoardDebugInfoVO;
import gov.va.med.exception.FoundationsException;
import gov.va.med.vistalink.adapter.cci.VistaLinkAppProxyConnectionSpec;
import gov.va.med.vistalink.adapter.cci.VistaLinkConnection;
import gov.va.med.vistalink.adapter.cci.VistaLinkConnectionSpec;
import gov.va.med.vistalink.adapter.cci.VistaLinkDuzConnectionSpec;
import gov.va.med.vistalink.adapter.record.LoginsDisabledFaultException;
import gov.va.med.vistalink.adapter.record.VistaLinkFaultException;
import gov.va.med.vistalink.adapter.spi.VistaLinkServerInfo;
import gov.va.med.vistalink.rpc.*;
import gov.va.med.vistalink.security.m.SecurityFaultException;
import gov.va.med.vistalink.security.m.SecurityDivisionDeterminationFaultException;
import org.apache.log4j.Logger;
import org.springframework.dao.*;
import org.springframework.jca.cci.core.CciTemplate;
import org.springframework.jca.cci.core.ConnectionCallback;
import org.springframework.util.Assert;

import javax.resource.ResourceException;
import javax.resource.cci.Connection;
import javax.resource.cci.ConnectionFactory;
import java.sql.SQLException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * This is the central class in the rpc package. It simplifies the use of VistaLink RPCs and helps to avoid common errors.
 * It executes core RPC workflow, leaving application code to provide RPC names and parameters and extract results. This
 * class executes RPCs with VistaLink, initiating iteration over RPC results and catching VistaLink exceptions and
 * translating them to the generic data access exception hierarchy defined in the org.springframework.dao package.
 */
public class VistaLinkRpcTemplate {

	private static final int USER_TYPE_DUZ = 1;
	private static final int USER_TYPE_APPLICATION_PROXY = 2;

	static final int DEFAULT_TIMEOUT = 10000;

	private static Logger log = Logger.getLogger(VistaLinkRpcTemplate.class);

	private int timeOut = DEFAULT_TIMEOUT;
	private String rpcContext;
	private ConnectionFactoryLocator connectionFactoryLocator;

	private static final String UNABLE_TO_EXECUTE_RPC = "Unable to execute RPC";
	/**
	 * Constructor.
	 * @param connectionFactoryLocator The connection location service.
	 */
	public VistaLinkRpcTemplate(ConnectionFactoryLocator connectionFactoryLocator) {
		this.connectionFactoryLocator = connectionFactoryLocator;
	}
	/**
	 * Execute a call as a user to Vista.
	 * 
	 * @param division The VAMC site number.
	 * @param duz The user duz.
	 * @param rpcContext The remote procedure call context.
	 * @param rpcName The remote procedure call name that maps to a vista call.
	 * @return The result from the vista call.
	 * @throws DataAccessException
	 */
	public String rpcAsUser(final String division,
			final String duz,
			final String rpcContext,
			final String rpcName) throws DataAccessException {
		
		debug("rpcAsUser(division = " + division 
    			+ ", duz = " + duz 
    			+ ", rpcContext = " + rpcContext 
    			+ ", rpcName = " + rpcName + ")");
		
		return rpcAsUser(division, duz, rpcContext, rpcName, null);
	}
	/**
	 * Execute a call as a user to Vista.
	 * 
	 * @param division The VAMC site number.
	 * @param duz The user duz.
	 * @param rpcContext The remote procedure call context.
	 * @param rpcName The remote procedure call name that maps to a vista call.
	 * @param params The mapping of parameters passed via the application.
	 * @return The result from the vista call.
	 * @throws DataAccessException
	 */
	public String rpcAsUser(final String division,
			final String duz,
			final String rpcContext,
			final String rpcName,
			final List params) throws DataAccessException {
		
		debug("rpcAsUser(division = " + division 
    			+ ", duz = " + duz 
    			+ ", rpcContext = " + rpcContext 
    			+ ", rpcName = " + rpcName 
    			+ ", params = " + params + ")");

		return rpc(division, duz, rpcContext, rpcName, params, USER_TYPE_DUZ);
	}
	/**
	 * Execute a call as an application to Vista.
	 * 
	 * @param division The VAMC site number.
	 * @param applicationProxyName The proxy user name.
	 * @param rpcContext The remote procedure call context.
	 * @param rpcName The remote procedure call name that maps to a vista call.
	 * @return The result from the vista call.
	 * @throws DataAccessException
	 */
	public String rpcAsApplication(final String division, final String applicationProxyName, final String rpcContext, final String rpcName) throws DataAccessException {
		
		debug("rpcAsApplication(division = " + division 
    			+ ", applicationProxyName = " + applicationProxyName 
    			+ ", rpcContext = " + rpcContext 
    			+ ", rpcName = " + rpcName + ")");
		
		return rpcAsApplication(division, applicationProxyName, rpcContext, rpcName, null);
	}
	/**
	 * Execute a call as an application to Vista.
	 * 
	 * @param division The VAMC site number.
	 * @param applicationProxyName The proxy user name.
	 * @param rpcContext The remote procedure call context.
	 * @param rpcName The remote procedure call name that maps to a vista call.
	 * @param params The mapping of parameters passed via the application.
	 * @return The result from the vista call.
	 * @throws DataAccessException
	 */
	public String rpcAsApplication(final String division, final String applicationProxyName, final String rpcContext, final String rpcName, final List params) throws DataAccessException {
		
		debug("rpcAsApplication(division = " + division 
    			+ ", applicationProxyName = " + applicationProxyName 
    			+ ", rpcContext = " + rpcContext 
    			+ ", rpcName = " + rpcName 
    			+ ", params = " + params + ")");
		
		return rpc(division, applicationProxyName, rpcContext, rpcName, params, USER_TYPE_APPLICATION_PROXY);
	}
	/**
	 * Retrieve the vista link connection information.
	 * @param division The VAMC site number.
	 * @param user The user value.
	 * @return The connection information.
	 * @throws DataAccessException
	 */
	public BigBoardDebugInfoVO getVistaLinkConnectionInfo(final String division, final String user) throws DataAccessException {
		
		debug("getVistaLinkConnectionInfo(division = " + division 
    			+ ", user = " + user + ")");
		
		return getConnectionInfo(division, user, USER_TYPE_APPLICATION_PROXY);
	}    
	/**
	 * Execute a call to Vista.
	 * 
	 * @param division The VAMC site number.
	 * @param user The user duz.
	 * @param rpcContext The remote procedure call context.
	 * @param rpcName The remote procedure call name that maps to a vista call.
	 * @param params The mapping of parameters passed via the application.
	 * @param userType The type of user , proxy or duz.
	 * @return The result of the RPC call.
	 * @throws DataAccessException
	 */
	public String rpc(final String division, final String user, final String rpcContext, final String rpcName, final List params, final int userType) throws DataAccessException {
		
		debug(MessageFormat.format("''{0}'' called in context ''{1}'' by ''{2}'' at facility ''{3}'' with params: {4}", new Object[]{rpcName, rpcContext, user, division, params}));

		ConnectionFactory connectionFactory = connectionFactoryLocator.getConnectionFactory(division);
		try {
			CciTemplate t = new CciTemplate(connectionFactory, createConnectionSpec(division, user, userType));
			RpcResponse response = (RpcResponse) t.execute(new ConnectionCallback() {
				public Object doInConnection(Connection connection, ConnectionFactory connectionFactory) throws ResourceException, SQLException, DataAccessException {
					VistaLinkConnection conn = null;
					try {
						Assert.isInstanceOf(VistaLinkConnection.class, connection);
						conn = (VistaLinkConnection) connection;
						conn.setTimeOut(getTimeOut());

						RpcRequest request = RpcRequestFactory.getRpcRequest(rpcContext, rpcName);
						request.setUseProprietaryMessageFormat(true);
						request.setXmlResponse(false);

						if (params != null) {

							if (params.contains("WORKSHEET")) {
								HashMap hmap = new HashMap();
								hmap = (HashMap)params.get(1);
								Set set = hmap.entrySet(); 
								Iterator i = set.iterator(); 
								int j = 0;
								HashMap worksheets = new HashMap();
								ArrayList ids = new ArrayList();
								while(i.hasNext()) { 
									Map.Entry me = (Map.Entry)i.next(); 
									if (((String)me.getKey()).startsWith("worksheet_")) {
										j++;
										worksheets.put(request.buildMultipleMSubscriptKey("\"worksheet\","+j),
												me.getValue());
										ids.add(me.getKey());
									}
								} 
								for (int k=0; k<ids.size(); k++) {
									hmap.remove(ids.get(k));
								}

								params.add(worksheets);
								params.remove("WORKSHEET");
							}
							 
							if (params.contains("ADHOC")) {
								HashMap hmap = new HashMap();
								hmap = (HashMap)params.get(1);
								Set set = hmap.entrySet(); 
								Iterator i = set.iterator(); 
								int j = 0;
								HashMap adhocs = new HashMap();
								ArrayList ids = new ArrayList();
								while(i.hasNext()) { 
									Map.Entry me = (Map.Entry)i.next(); 
									if (((String)me.getKey()).startsWith("element_")) {
										j++;
										adhocs.put(request.buildMultipleMSubscriptKey("\"element\","+j),
												me.getValue());
										ids.add(me.getKey());
									}
								} 
								for (int k=0; k<ids.size(); k++) {
									hmap.remove(ids.get(k));
								}

								params.add(adhocs);
								params.remove("ADHOC");
							}


							request.setParams(params);
						}

						return conn.executeRPC(request);
					} catch (IllegalArgumentException e) {
						throw new InvalidDataAccessApiUsageException(UNABLE_TO_EXECUTE_RPC, e);
					} catch (NoRpcContextFaultException e) {
						throw new VistaLinkRpcInvalidApiUsageException(e);
					} catch (RpcNotOkForProxyUseException e) {
						throw new VistaLinkRpcInvalidApiUsageException(e);
					} catch (RpcNotInContextFaultException e) {
						throw new VistaLinkRpcPermissionDeniedException(e);
					} catch (LoginsDisabledFaultException e) {
						throw new DataAccessResourceFailureException(UNABLE_TO_EXECUTE_RPC, e);
					} catch (SecurityDivisionDeterminationFaultException e) {
						throw new VistaLinkRpcInvalidApiUsageException(e);
					} catch (SecurityFaultException e) {
						throw new VistaLinkRpcPermissionDeniedException(e);
					} catch (VistaLinkFaultException e) {
						throw new DataRetrievalFailureException(UNABLE_TO_EXECUTE_RPC, e);
					} catch (FoundationsException e) {
						throw new DataRetrievalFailureException(UNABLE_TO_EXECUTE_RPC, e);
					} 
					finally {
						if (conn != null) {
							try {
								conn.close();
								debug("doInConnection closing VistaLinkConnection");
							} catch (ResourceException e) {
								throw new DataRetrievalFailureException(UNABLE_TO_EXECUTE_RPC, e);
							}
						}
					}

				}
			});
			String result = response.getResults();
			
			debug(MessageFormat.format("''{0}'' returned: {1}", new Object[]{rpcName, result}));
			
			return result;
		} catch (IllegalArgumentException e) {
			throw new InvalidDataAccessApiUsageException(UNABLE_TO_EXECUTE_RPC, e);
		}
	}

	private BigBoardDebugInfoVO getConnectionInfo(final String division,
			final String user,
			final int userType) throws DataAccessException {
		
		debug("getConnectionInfo(division = " + division 
    			+ ", user = " + user 
    			+ ", userType = " + userType + ")");
		
		VistaLinkConnection conn = null;
		BigBoardDebugInfoVO bbd = null;
		ConnectionFactory connectionFactory = connectionFactoryLocator.getConnectionFactory(division);
		try {
			conn = (VistaLinkConnection)connectionFactory.getConnection(createConnectionSpec(division, user, userType));
			VistaLinkServerInfo vistaLinkServerInfo = conn.getConnectionInfo();
			bbd = new BigBoardDebugInfoVO(vistaLinkServerInfo.getAddress().toString(), new Integer(vistaLinkServerInfo.getPort()).toString());
			
			debug("getConnectionInfo conn = " + conn 
	    			+ ", vistaLinkServerInfo = " + vistaLinkServerInfo 
	    			+ ", bbd = " + bbd);
			
		} catch (ResourceException e) {
			throw new DataAccessResourceFailureException("error getting connection info", e);
		}
		finally {
			if (conn != null) {
				try {
					conn.close();
					debug("getConnectionInfo closing VistaLinkConnection");
				} catch (ResourceException e) {
					throw new DataRetrievalFailureException(UNABLE_TO_EXECUTE_RPC, e);
				}
			}
		}

		return bbd;
	} 	   

	private VistaLinkConnectionSpec createConnectionSpec(String division, String user, int userType) {
		
		debug("createConnectionSpec(division = " + division + ", user = "
				+ user + ", userType = " + userType + ")");
		
		switch (userType) {
		case USER_TYPE_APPLICATION_PROXY:
			return new VistaLinkAppProxyConnectionSpec(division, user);
		case USER_TYPE_DUZ:
		default:
			return new VistaLinkDuzConnectionSpec(division, user);
		}
	}
	/**
	 * Retrieve the time out value.
	 * @return the time out number.
	 */
	public int getTimeOut() {
		return timeOut;
	}
	/**
	 * Set the time out value.
	 * @param timeOut The time in int format.
	 */
	public void setTimeOut(int timeOut) {
		this.timeOut = timeOut;
	}
	/**
	 * Retrieve the RpcContext.
	 * @return the RpcContext
	 */
	public String getRpcContext() {
		return rpcContext;
	}
	/**
	 * Set the RpcContext.
	 * @param rpcContext the RpcContext.
	 */
	public void setRpcContext(String rpcContext) {
		this.rpcContext = rpcContext;
	}
	/**
	 * Retrieve the ConnectionFactoryLocator
	 * @return the ConnectionFactoryLocator.
	 */
	public ConnectionFactoryLocator getConnectionFactoryLocator() {
		return connectionFactoryLocator;
	}
	
	private void debug(String s) {
    	if (log.isDebugEnabled()){
    		log.debug(s);
    	}
	} 
}
