/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package gov.va.med.nhin.adapter.datamanager;

import java.util.*;

import gov.va.med.nhin.adapter.datamanager.config.*;
import gov.va.med.nhin.adapter.utils.*;

/**
 * The DataAdapter class provides a common way to query for and retrieve data
 * from multiple sources of data. It does so by providing a common method to
 * query for the data as well as a common structure in which the data is
 * returned.
 *
 * To make data available via the DataManagerImpl class, one must implement the
 * DataAdapter interface specific to the source of data and register that
 * implementation with a DataManagerImpl instance. Registration can be done
 * via a configuration file or programatically. In the latter case, the
 * DataAdapter implementation must be annotated to describe how the data source
 * can be queried as well as how the data is transformed. The XML based
 * configuration file gives one all of the options that the annotations do.
 *
 * @author David Vazquez
 * @version 1.0
 */
@SuppressWarnings(value="unchecked")
public class DataManagerImpl implements DataManager
{
    private DataManagerType config;
    private HashMap<String, DataFilter> dataFilterCache = new HashMap<String, DataFilter>();
    private HashMap<String, DataTranslator> dataTranslatorCache = new HashMap<String, DataTranslator>();
    private HashMap<String, DataAdapter> dataAdapterCache = new HashMap<String, DataAdapter>();
    private HashMap<String, DataParser> dataParserCache = new HashMap<String, DataParser>();
    private HashMap<String, DataFieldGenerator> dataFieldGeneratorCache = new HashMap<String, DataFieldGenerator>();
    
    DataManagerImpl(String filename)
    {
        loadConfig(filename);
    }

    public DataQueryImpl getQuery(String queryName)
    {
        return new DataQueryImpl(this, getQueryType(queryName), queryName);
    }

    QueryType getQueryType(String name)
    {
        QueryType ret = null;

        for (QueryType qt : config.getQueries().getQuery()) {
            for (String n : qt.getName()) {
                if (n.equals(name)) {
                    ret = qt;
                    break;
                }
            }

            if (ret != null) {
                break;
            }
        }

        if (ret == null) {
            throw new QueryDoesNotExistException(name);
        }

        return ret;
    }

    public List invokeDataParser(ReferenceType parserType, List results, DataQuery dataQuery)
        throws Exception
    {
        Reference<DataParser> ref = getDataParser(parserType);
        return ref.getObj().parse(results, ref, dataQuery);
    }

    synchronized Reference<DataParser> getDataParser(ReferenceType parserType)
        throws Exception
    {
        DataParser dataParser = dataParserCache.get(parserType.getName());
        DeclarationType pdt = getParserDef(parserType.getName());

        if (dataParser == null) {
            Class cls = Thread.currentThread().getContextClassLoader().loadClass(pdt.getClassName());
            dataParser = (DataParser)cls.newInstance();
            dataParserCache.put(parserType.getName(), dataParser);
        }

        return new Reference<DataParser>(parserType, pdt, dataParser);
    }

    DeclarationType getParserDef(String name)
    {
        DeclarationType ret = getDeclaration(config.getParsers(), name);

        if (ret == null) {
            throw new ParserDoesNotExistException(name);
        }
        
        return ret;
    }

    DeclarationType getDeclaration(DeclarationsType declarations, String name)
    {
        DeclarationType ret = null;

        for (DeclarationType d : declarations.getDeclaration()) {
            if (d.getName().equals(name)) {
                ret = d;
                break;
            }
        }

        return ret;
    }

    synchronized DataAdapter getDataAdapter(ReferenceType adapterType)
        throws Exception
    {
        DataAdapter ret = dataAdapterCache.get(adapterType.getName());

        if (ret == null) {
            DeclarationType adt = getAdapterDef(adapterType.getName());
            Class cls = Thread.currentThread().getContextClassLoader().loadClass(adt.getClassName());
            ret = (DataAdapter)cls.newInstance();
        }

        return ret;
    }

    DeclarationType getAdapterDef(String name)
    {
        DeclarationType ret = getDeclaration(config.getAdapters(), name);

        if (ret == null) {
            throw new AdapterDoesNotExistException(name);
        }

        return ret;
    }

    public boolean invokeDataFilter(ReferenceType filterType, Object result, List results, DataQuery dataQuery)
        throws Exception
    {
        Reference<DataFilter> ref = getDataFilter(filterType);
        return ref.getObj().test(result, results, ref, dataQuery);
    }

    synchronized Reference<DataFilter> getDataFilter(ReferenceType filterType)
        throws Exception
    {
        DataFilter dataFilter = dataFilterCache.get(filterType.getName());
        DeclarationType fdt = getFilterDef(filterType.getName());

        if (dataFilter == null) {
            Class cls = Thread.currentThread().getContextClassLoader().loadClass(fdt.getClassName());
            dataFilter = (DataFilter)cls.newInstance();
            dataFilterCache.put(filterType.getName(), dataFilter);
        }

        return new Reference<DataFilter>(filterType, fdt, dataFilter);
    }

    DeclarationType getFilterDef(String name)
    {
        DeclarationType ret = getDeclaration(config.getFilters(), name);

        if (ret == null) {
            throw new FilterDoesNotExistException(name);
        }
        
        return ret;
    }

    public Object invokeDataTranslator(ReferenceType translationType, Object input, Object result, Object sourceResult, DataQuery dataQuery)
        throws Exception
    {
        Reference<DataTranslator> ref = getDataTranslator(translationType, result, sourceResult);
        return ref.getObj().translate(input, result, ref, dataQuery);
    }

    synchronized Reference<DataTranslator> getDataTranslator(ReferenceType translationType, Object result, Object sourceResult)
        throws Exception
    {
        DataTranslator dataTranslator = dataTranslatorCache.get(translationType.getName());
        DeclarationType ttd = getTranslationDef(translationType.getName());

        if (dataTranslator == null) {
            Class cls = Thread.currentThread().getContextClassLoader().loadClass(ttd.getClassName());
            dataTranslator = (DataTranslator)cls.newInstance();
            dataTranslatorCache.put(translationType.getName(), dataTranslator);
        }

        return new Reference<DataTranslator>(translationType, ttd, dataTranslator, result, sourceResult);
    }

    DeclarationType getTranslationDef(String name)
    {
        DeclarationType ret = getDeclaration(config.getTranslations(), name);
        
        if (ret == null) {
            throw new TranslatorDoesNotExistException(name);
        }

        return ret;
    }

    synchronized DataFieldGenerator getDataFieldGenerator(String fieldGeneratorName)
        throws Exception
    {
        DataFieldGenerator ret = dataFieldGeneratorCache.get(fieldGeneratorName);

        if (ret == null) {
            DeclarationType fgt = getFieldGeneratorDef(fieldGeneratorName);
            Class cls = Thread.currentThread().getContextClassLoader().loadClass(fgt.getClassName());
            ret = (DataFieldGenerator)cls.newInstance();
            dataFieldGeneratorCache.put(fieldGeneratorName, ret);
        }

        return ret;
    }

    DeclarationType getFieldGeneratorDef(String name)
    {
        DeclarationType ret = getDeclaration(config.getFieldGenerators(), name);

        if (ret == null) {
            throw new FieldGeneratorDoesNotExistException(name);
        }
        
        return ret;
    }

    /**
     * Loads a DataManagerImpl's configration from an XML based file.
     *
     * @param filename
     * @throws datamanager.DataManagerException
     */
    private void loadConfig(String filename)
    {
        try {
            config = XMLUtils.load(filename, DataManagerType.class);
        }
        catch (Throwable t) {
            throw new LoadConfigurationException(filename, t);
        }
    }
}
