/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package ext.domain.nhin.adapter.xmlutils;

import java.lang.reflect.*;
import java.util.logging.*;

import javax.xml.xpath.*;

import org.apache.commons.beanutils.*;
import org.w3c.dom.*;

import ext.domain.nhin.adapter.utils.*;
import ext.domain.nhin.adapter.xmlutils.config.*;

/**
 * Instances of XMLTemplate are used to create new XML documents from templates using a
 * mapping file to map properties of a POJO or HashMap to elements and attributes in the
 * template.  The mapping file, given at the time an XMLTemplate is constructed, is
 * an XML file which gives the name of the template, any namespace mappings which are
 * required to properly resolve elements in the template, and a collection of mappings
 * that bind properties to xpaths.
 *
 * @author David Vazquez
 */
public class XMLTemplate
{
    static private final Logger logger = Logger.getLogger(XMLTemplate.class.getName());
    
    private Document template;
    private TemplateType config;
    private XMLNamespaceContext namespaceContext = new XMLNamespaceContext();

    /**
     * Construct an XMLTemplate object.  An XMLUtilsException will be thrown
     * if the mapping file or template file cannot be parsed.
     *
     * @param mapFilename path of the file that contains the mapping configuration.
     * @throws xmltemplate.XMLUtilsException if the mapping or template file cannot be parsed.
     */
    XMLTemplate(String configFilename)
            throws XMLUtilsException
    {
        loadConfig(configFilename);
    }

    private void loadConfig(String configFilename)
            throws XMLUtilsException
    {
        try {
            config = XMLUtils.load(configFilename, TemplateType.class);
            template = XMLUtils.parse(config.getSourceFile().getPath());
            namespaceContext = new XMLNamespaceContext();
            if (!NullChecker.isNullOrEmpty(config.getSourceFile().getNamespaceMappings())) {
                for (NamespaceMappingType nsMapping : config.getSourceFile().getNamespaceMappings().getNamespaceMapping()) {
                    namespaceContext.addMapping(nsMapping.getPrefix(), nsMapping.getUri());
                }
            }
        }
        catch (Throwable t) {
            throw new XMLUtilsException("An error occurred when constructing XML template.", t);
        }
    }

    /**
     * Creates a new XML document using the file specified in the mapping file as a template.
     * Attribute and element values are filled in using values from the given bean based
     * on the mappings given in the mapping file.
     *
     * @param fromBean POJO or HashMap whose properties are set to the values which will be used to fill
     * in the template.
     * @return a new XML document.
     * @throws xmltemplate.XMLUtilsException if a required property is not present in the given bean or
     * and invalid xpath was specified in the mapping file.
     */
    public Document fillIn(Object fromBean)
    {
        logger.entering(getClass().getName(), "fillIn", fromBean);

        Document ret = null;
        
        try {
            XPath xpathTemplate = XPathFactory.newInstance(config.getXpathLibURI()).newXPath();
            XMLXPathEvaluationCache xpathEvaluationCache = new XMLXPathEvaluationCache();
            ret = (Document)template.cloneNode(true);
            xpathTemplate.setNamespaceContext(namespaceContext);
            fillIn(ret, fromBean, config.getMappings(), xpathTemplate, xpathEvaluationCache);
            return ret;
        }
        catch (XMLUtilsException xte) {
            throw xte;
        }
        catch (Throwable t) {
            throw new XMLUtilsException("An error occurred when filling in a template.", t);
        }
        finally {
            logger.exiting(getClass().getName(), "fillIn", ret);
        }
    }

    private void fillIn(Node root, Object fromBean, MappingsType mappings, XPath xpathTemplate, XMLXPathEvaluationCache xpathEvaluationCache)
    {
        fillIn(root, fromBean, mappings, xpathTemplate, xpathEvaluationCache, 0);
    }

    private void fillIn(Node root, Object fromBean, MappingsType mappings, XPath xpathTemplate, XMLXPathEvaluationCache xpathEvaluationCache, int index)
    {
        if (mappings != null) {
            for (MappingType mapping : mappings.getMapping()) {
                if (equalsProperty(mapping.getEqualsProperty(), mapping.getEqualsValue(), fromBean)) {
                    try {
                        String propertyString = mapping.getProperty().getValue();
                        String xpathString = mapping.getXpath().getValue();
                        XPathExpression xpathExpression = xpathEvaluationCache.getExpression(xpathTemplate, xpathString);
                        Node node = (Node) xpathExpression.evaluate(root, XPathConstants.NODE);

                        if (node != null) {
                            Object value = getPropertyFromBean(fromBean, propertyString);

                            if (mapping.getXpath().isCopy()) {
                                Node newNode = node.cloneNode(true);
                                node.getParentNode().insertBefore(newNode, node.getNextSibling());
                                node = newNode;
                            }

                            if (!NullChecker.isNullOrEmpty(value)) {
                                fillIn(root, node, fromBean, value, mapping, mapping.getIfNotNull(), xpathTemplate, xpathEvaluationCache, index);
                            }
                            else {
                                if (mapping.getProperty().isOptional()) {
                                    XMLUtilsException xue =
                                            new XMLUtilsException("Optional property is missing.",
                                                                  propertyString, xpathString);
                                    logger.finest(xue.toString());
                                }
                                else {
                                    XMLUtilsException xue =
                                            new XMLUtilsException("Required property is missing.",
                                                                  propertyString, xpathString);
                                    logger.warning(xue.toString());
                                }

                                if (!NullChecker.isNullOrEmpty(mapping.getIfNull())) {
                                    fillIn(root, node, fromBean, value, mapping, mapping.getIfNull(), xpathTemplate, xpathEvaluationCache, index);
                                }
                            }
                        }
                        else {
                            XMLUtilsException xue =
                                    new XMLUtilsException("Given xpath did not resolve to a node.",
                                                          propertyString, xpathString);
                            logger.warning(xue.toString());
                        }
                    }
                    catch (Throwable t) {
                        throw new XMLUtilsException("An error occurred when filling in a template.",
                                                    t, mapping.getProperty().getValue(), mapping.getXpath().getValue());
                    }
                }
            }
        }
    }

    private Object getPropertyFromBean(Object bean, String property)
    {
        Object value = null;

        try {
            if (!NullChecker.isNullOrEmpty(property)) {
                value = PropertyUtils.getProperty(bean, property);
            }
            else {
                value = bean;
            }
        }
        catch (Exception e) {
            value = null;
        }

        return value;
    }

    private void fillIn(Node root, Node node, Object fromBean, Object value, MappingType mapping, ActionsType actions, XPath xpathTemplate, XMLXPathEvaluationCache xpathEvaluationCache, int index)
            throws IllegalAccessException, InvocationTargetException, NoSuchMethodException,
                   XPathExpressionException
    {
        for (ActionType action : actions.getAction()) {
            if (equalsProperty(action.getEqualsProperty(), action.getEqualsValue(), fromBean)) {
                Node entryNode = getEntryNode(node, action, xpathTemplate, xpathEvaluationCache);
                Object entryValue = getValue(action, fromBean, value);

                if (entryNode != null) {
                    switch (action.getAction()) {
                        case SET_VALUE:
                        default:
                            if (!Utils.isCollection(entryValue)) {
                                entryNode.setTextContent(getValueAsString(action, entryValue, index));
                            }
                            else {
                                Node siblingNode = entryNode.getNextSibling();
                                for (int i = 0; i < Utils.getCollectionLength(entryValue); ++i) {
                                    Object o = PropertyUtils.getIndexedProperty(entryValue, "", i);
                                    if (!NullChecker.isNullOrEmpty(o)) {
                                        Node newNode = entryNode.cloneNode(true);
                                        entryNode.getParentNode().insertBefore(newNode, siblingNode);
                                        newNode.setTextContent(o.toString());
                                        if (!NullChecker.isNullOrEmpty(action.getAttribute())) {
                                            NamedNodeMap attributes = newNode.getAttributes();
                                            Node attributeNode = attributes.getNamedItem(action.getAttribute());
                                            if (attributeNode == null) {
                                                attributeNode = entryNode.getOwnerDocument().createAttribute(action.getAttribute());
                                                attributes.setNamedItem(attributeNode);
                                            }
                                            String textContent = attributeNode.getTextContent();
                                            if (!NullChecker.isNullOrEmpty(textContent)) {
                                                textContent += "-" + (i + 1);
                                            }
                                            else {
                                                textContent = Integer.toString(i);
                                            }
                                            attributeNode.setTextContent(textContent);
                                        }
                                    }
                                }
                                entryNode.getParentNode().removeChild(entryNode);
                            }
                            break;

                        case SET_ATTRIBUTE:
                            NamedNodeMap attributes = entryNode.getAttributes();
                            Node attributeNode = attributes.getNamedItem(action.getAttribute());
                            if (attributeNode == null) {
                                attributeNode = entryNode.getOwnerDocument().createAttribute(action.getAttribute());
                                attributes.setNamedItem(attributeNode);
                            }
                            attributeNode.setTextContent(getValueAsString(action, entryValue, index));
                            break;

                        case APPEND_ATTRIBUTE:
                            attributes = entryNode.getAttributes();
                            attributeNode = attributes.getNamedItem(action.getAttribute());
                            if (attributeNode == null) {
                                attributeNode = entryNode.getOwnerDocument().createAttribute(action.getAttribute());
                                attributes.setNamedItem(attributeNode);
                                attributeNode.setTextContent(getValueAsString(action, entryValue, index));
                            }
                            else {
                                attributeNode.setTextContent(attributeNode.getTextContent() + getValueAsString(action, entryValue, index));
                            }
                            break;

                        case APPEND_VALUE:
                            if (!Utils.isCollection(value)) {
                                String textContent = entryNode.getTextContent();
                                String appendText = getValueAsString(action, entryValue, index);
                                if (!NullChecker.isNullOrEmpty(textContent)) {
                                    textContent += appendText;
                                }
                                else {
                                    textContent = appendText;
                                }
                                entryNode.setTextContent(textContent);
                            }
                            else {
                                logger.warning("Cannot use appendValue on a collection.");
                            }
                            break;

                        case EVALUATE_MAPPINGS:
                            if (!Utils.isCollection(entryValue)) {
                                fillIn(entryNode, entryValue, mapping.getMappings(), xpathTemplate, xpathEvaluationCache);
                            }
                            else {
                                Node parentNode = entryNode.getParentNode();
                                Node siblingNode = entryNode.getNextSibling();
                                for (int i = 0; i < Utils.getCollectionLength(entryValue); ++i) {
                                    Object o = PropertyUtils.getIndexedProperty(entryValue, "", i);
                                    Node newNode = entryNode.cloneNode(true);
                                    if (siblingNode != null) {
                                        parentNode.insertBefore(newNode, siblingNode);
                                    }
                                    else {
                                        parentNode.appendChild(newNode);
                                    }
                                    fillIn(newNode, o, mapping.getMappings(), xpathTemplate, xpathEvaluationCache, i);
                                }
                                parentNode.removeChild(entryNode);
                            }
                            break;

                        case REMOVE:
                            entryNode.getParentNode().removeChild(entryNode);
                            break;

                        case REMOVE_CHILDREN:
                            NodeList childNodes = entryNode.getChildNodes();
                            while (childNodes.getLength() > 0) {
                                entryNode.removeChild(childNodes.item(0));
                            }
                            break;

                        case REMOVE_ATTRIBUTES:
                            NamedNodeMap attributeNodes = entryNode.getAttributes();
                            while (attributeNodes.getLength() > 0) {
                                Node attribute = attributeNodes.item(0);
                                attributeNodes.removeNamedItem(attribute.getNodeName());
                            }
                            break;
                    }
                }
                else {
                    XMLUtilsException xue = new XMLUtilsException("Relative xpath (" + action.getRelative() + ") did not resolve to a node.",
                                                                  mapping.getProperty().getValue(), mapping.getXpath().getValue());
                    logger.warning(xue.toString());
                }
            }
        }
    }

    private boolean equalsProperty(String equalsProperty, String equalsValue, Object fromBean)
    {
        boolean ret = true;

        if (!NullChecker.isNullOrEmpty(equalsProperty) && equalsValue != null) {
            Object value = getPropertyFromBean(fromBean, equalsProperty);
            ret = value != null && value.toString().equals(equalsValue);
        }

        return ret;
    }

    private Node getEntryNode(Node node, ActionType action, XPath xpathTemplate, XMLXPathEvaluationCache xpathEvaluationCache)
            throws XPathExpressionException
    {
        Node entryNode;

        if (!NullChecker.isNullOrEmpty(action.getRelative())) {
            XPathExpression xpathExpression = xpathEvaluationCache.getExpression(xpathTemplate, action.getRelative());
            entryNode = (Node) xpathExpression.evaluate(node, XPathConstants.NODE);
        }
        else {
            entryNode = node;
        }

        return entryNode;
    }

    private Object getValue(ActionType action, Object fromBean, Object value)
    {
        Object ret = action.getValue();

        if (NullChecker.isNullOrEmpty(ret)) {
            if (!NullChecker.isNullOrEmpty(action.getProperty())) {
                ret = getPropertyFromBean(fromBean, action.getProperty());
            }
            else {
                ret = value;
            }
        }

        return ret;
    }

    private String getValueAsString(ActionType action, Object value, int index)
    {
        String ret = value != null ? value.toString() : "";

        if (action.isIndexed()) {
            ret += (index + 1);
        }

        return ret;
    }

    TemplateType getConfig()
    {
        return config;
    }
}
