package ext.domain.nhin.adapter.adaptergateway.docretrieve;

import java.util.logging.*;
import java.util.concurrent.*;

import javax.ejb.*; 

import gov.hhs.fha.nhinc.adapterpolicyengine.*;
import gov.hhs.fha.nhinc.common.nhinccommonadapter.CheckPolicyRequestType;
import gov.hhs.fha.nhinc.common.nhinccommonadapter.CheckPolicyResponseType;
import ihe.iti.xds_b._2007.*;
import ihe.iti.xds_b._2007.RetrieveDocumentSetRequestType.*;
import ihe.iti.xds_b._2007.RetrieveDocumentSetResponseType.*;
import oasis.names.tc.ebxml_regrep.xsd.rs._3.*;
import oasis.names.tc.xacml._2_0.context.schema.os.*;

import ext.domain.nhin.adapter.facilitymanager.*;
import ext.domain.nhin.adapter.facilitymanager.Facility;
import ext.domain.nhin.adapter.policyengine.*;
import ext.domain.nhin.adapter.utils.*;
import ext.domain.nhin.adapter.adaptergateway.docquery.*;

import gov.hhs.fha.nhinc.nhinclib.NhincConstants;  
import gov.hhs.fha.nhinc.transform.policy.PolicyEngineTransformer;
import gov.hhs.fha.nhinc.common.eventcommon.DocRetrieveEventType;
import gov.hhs.fha.nhinc.common.eventcommon.DocRetrieveMessageType;
import gov.hhs.fha.nhinc.common.nhinccommon.AssertionType;
import gov.hhs.fha.nhinc.common.nhinccommon.HomeCommunityType;
import gov.hhs.fha.nhinc.common.nhinccommonentity.RespondingGatewayCrossGatewayRetrieveRequestType;

/**
 *
 * @author VHAISBWELCHR
 */

@Stateless(name = "AdapterGatewayDocRetrieve")
public class AdapterGatewayDocRetrieve implements EntityDocRetrievePortTypeLocal
{
    private class SendRequestResponse
    {
        public RetrieveDocumentSetResponseType retrieveDocumentSetResponse;
        public String remoteHomeCommunityId;
    }
    
    private class SendRequestException extends Exception
    {
        public String remoteHomeCommunityId;
    }

    private class SendRequestCallable implements Callable<SendRequestResponse>
    {
        public final RetrieveDocumentSetRequestType retrieveDocumentSetRequest;
        public final AssertionType assertion;
        public final String remoteHomeCommunityId;
        
        public SendRequestCallable(RetrieveDocumentSetRequestType retrieveDocumentSetRequest,
                                   AssertionType assertion,
                                   String remoteHomeCommunityId)
        {
            this.retrieveDocumentSetRequest = retrieveDocumentSetRequest;
            this.assertion = assertion;
            this.remoteHomeCommunityId = remoteHomeCommunityId;
        }

        public SendRequestResponse call() throws Exception
        {
            return sendRequest(retrieveDocumentSetRequest, assertion, remoteHomeCommunityId);
        }
    }
    
    static private final Logger logger = Logger.getLogger(AdapterGatewayDocRetrieve.class.getName());

    private FacilityManager facilityManager;
    private AdapterPolicyEnginePortType adapterPolicyEngine;
    private NHINDocRetrieveRequestSender nhinDocRetrieveRequestSender;    
    private ExecutorService executorService;

    @EJB(beanInterface = FacilityManagerLocal.class, beanName = "FacilityManager")
    public void setFacilityManager(FacilityManager facilityManager)
    {
        this.facilityManager = facilityManager;
    }

    @EJB(beanInterface = AdapterPolicyEnginePortTypeLocal.class, beanName = "AdapterPolicyEngine")
    public void setAdapterPolicyEngine(AdapterPolicyEnginePortType adapterPolicyEngine)
    {
        this.adapterPolicyEngine = adapterPolicyEngine;
    }   
                       
    @EJB(beanInterface = NHINDocRetrieveRequestSender.class, beanName = "NhincProxyNHINDocRetrieveRequestSender")
    public void setNhinDocQueryRequestSender(NHINDocRetrieveRequestSender nhinDocQueryRequestSender)
    {
        this.nhinDocRetrieveRequestSender = nhinDocQueryRequestSender;
    }

    @EJB(beanInterface = WorkManagerExecutorServiceLocal.class, beanName = "WorkManagerExecutorService")
    public void setExecutorService(ExecutorService executorService)
    {
        this.executorService = executorService;
    }

    public RetrieveDocumentSetResponseType respondingGatewayCrossGatewayRetrieve(RespondingGatewayCrossGatewayRetrieveRequestType respondingGatewayCrossGatewayRetrieveRequest)
    {    
        logger.entering(getClass().getName(), "respondingGatewayCrossGatewayRetrieve");

        RetrieveDocumentSetResponseType ret = new RetrieveDocumentSetResponseType();

        try {
            RetrieveDocumentSetRequestType retrieveDocumentSetRequest = respondingGatewayCrossGatewayRetrieveRequest.getRetrieveDocumentSetRequest();
            AssertionType assertion = respondingGatewayCrossGatewayRetrieveRequest.getAssertion();
            ExecutorCompletionService<SendRequestResponse> completionService = new ExecutorCompletionService<SendRequestResponse>(executorService);
            int numSubmitted = 0;

            for (DocumentRequest documentRequest : retrieveDocumentSetRequest.getDocumentRequest()) {
                RetrieveDocumentSetRequestType newRequest = createRetrieveDocumentSetRequest(documentRequest);
                if (checkPolicy(newRequest, assertion, documentRequest.getHomeCommunityId())) {
                    SendRequestCallable callable = new SendRequestCallable(newRequest, assertion, documentRequest.getHomeCommunityId());
                    completionService.submit(callable);
                    ++numSubmitted;
                }
            }

            for (int i = 0;  i < numSubmitted;  ++i) {
                try {
                    SendRequestResponse response = completionService.take().get();
                    addResultToResponse(ret, response.retrieveDocumentSetResponse);
                }
                catch (ExecutionException ee) {
                    SendRequestException sre = (SendRequestException)ee.getCause();
                    addErrorToResponse(ret, "Internal Registry/Repository Error - HCID = " + sre.remoteHomeCommunityId);
                    logger.log(Level.WARNING, "Error sending DocRetrieve to {0} - {1}", new Object[]{sre.remoteHomeCommunityId, sre.getCause().getMessage()});
                    logger.log(Level.WARNING, "Stack trace", sre.getCause());
                }
                catch (InterruptedException ie) {
                    break;
                }
            }
        }
        finally {
            logger.exiting(getClass().getName(), "respondingGatewayCrossGatewayRetrieve");
        }
        
        return ret;
    }
    
    private RetrieveDocumentSetRequestType createRetrieveDocumentSetRequest(DocumentRequest documentRequest)
    {
        RetrieveDocumentSetRequestType ret = new RetrieveDocumentSetRequestType();
        ret.getDocumentRequest().add(documentRequest);
        return ret;
    }
    
    private boolean checkPolicy(RetrieveDocumentSetRequestType retrieveDocumentSetRequest, AssertionType assertion, String remoteHomeCommunityId)
    {
        DocRetrieveMessageType docRetrieveMessage = new DocRetrieveMessageType();
        docRetrieveMessage.setRetrieveDocumentSetRequest(retrieveDocumentSetRequest);
        docRetrieveMessage.setAssertion(assertion);

        HomeCommunityType receivingHomeCommunity = new HomeCommunityType();
        receivingHomeCommunity.setHomeCommunityId(remoteHomeCommunityId);
        
        HomeCommunityType sendingHomeCommunity = new HomeCommunityType();
        sendingHomeCommunity.setHomeCommunityId(getHomeCommunityId());
        
        DocRetrieveEventType docRetrieveEvent = new DocRetrieveEventType();
        docRetrieveEvent.setMessage(docRetrieveMessage);
        docRetrieveEvent.setDirection(NhincConstants.POLICYENGINE_OUTBOUND_DIRECTION);
        docRetrieveEvent.setInterface(NhincConstants.AUDIT_LOG_ENTITY_INTERFACE);
        docRetrieveEvent.setReceivingHomeCommunity(receivingHomeCommunity);
        docRetrieveEvent.setSendingHomeCommunity(sendingHomeCommunity);
        
        PolicyEngineTransformer transformer = new PolicyEngineTransformer();
        CheckPolicyRequestType checkPolicyRequest = transformer.transformDocRetrieveToCheckPolicy(docRetrieveEvent);
        
        CheckPolicyResponseType checkPolicyResponse = adapterPolicyEngine.checkPolicy(checkPolicyRequest);
        
        return checkPolicyResponse != null
               && checkPolicyResponse.getResponse() != null
               && NullChecker.isNotNullOrEmpty(checkPolicyResponse.getResponse().getResult())
               && checkPolicyResponse.getResponse().getResult().get(0).getDecision() == DecisionType.PERMIT;
    }
    
    private void addResultToResponse(RetrieveDocumentSetResponseType retrieveDocumentSetResponse, RetrieveDocumentSetResponseType result)
    {
        if (result.getRegistryResponse().getRegistryErrorList() != null) {
            if (retrieveDocumentSetResponse.getRegistryResponse().getRegistryErrorList() == null) {
                retrieveDocumentSetResponse.getRegistryResponse().setRegistryErrorList(new RegistryErrorList());
            }
            retrieveDocumentSetResponse.getRegistryResponse().getRegistryErrorList().getRegistryError().addAll(result.getRegistryResponse().getRegistryErrorList().getRegistryError());
            retrieveDocumentSetResponse.getRegistryResponse().getRegistryErrorList().setHighestSeverity(result.getRegistryResponse().getRegistryErrorList().getHighestSeverity());
        }
        
        retrieveDocumentSetResponse.getDocumentResponse().addAll(result.getDocumentResponse());
    }
    
    private void addErrorToResponse(RetrieveDocumentSetResponseType retrieveDocumentSetResponse, String errorString)
    {
        if (retrieveDocumentSetResponse.getRegistryResponse().getRegistryErrorList() == null) {
            retrieveDocumentSetResponse.getRegistryResponse().setRegistryErrorList(new RegistryErrorList());
        }
        
        retrieveDocumentSetResponse.getRegistryResponse().getRegistryErrorList().setHighestSeverity("urn:oasis:names:tc:ebxml-regrep:ErrorSeverityType:Error");
        
        RegistryError registryError = new RegistryError();
        registryError.setErrorCode("XDSRegistryError");
        registryError.setCodeContext(errorString);
        registryError.setSeverity("urn:oasis:names:tc:ebxml-regrep:ErrorSeverityType:Error");
        retrieveDocumentSetResponse.getRegistryResponse().getRegistryErrorList().getRegistryError().add(registryError);
    }
    
    private SendRequestResponse sendRequest(RetrieveDocumentSetRequestType retrieveDocumentSetRequest, AssertionType assertion, String remoteHomeCommunityId)
        throws SendRequestException
    {
        SendRequestResponse ret = new SendRequestResponse();
        
        try {
            ret.retrieveDocumentSetResponse = nhinDocRetrieveRequestSender.sendRequest(retrieveDocumentSetRequest, assertion, remoteHomeCommunityId);
            ret.remoteHomeCommunityId = remoteHomeCommunityId;
            return ret;
        }
        catch (Exception e) {
            SendRequestException sre = new SendRequestException();
            sre.remoteHomeCommunityId = remoteHomeCommunityId;
            sre.initCause(e);
            throw sre;
        }
    }
    
    private RetrieveDocumentSetResponseType createErrorResponse(String codeContext)
    {
        RetrieveDocumentSetResponseType response = new RetrieveDocumentSetResponseType();
        RegistryResponseType responseType = new RegistryResponseType();
        response.setRegistryResponse(responseType);
        responseType.setStatus("urn:oasis:names:tc:ebxml-regrep:ResponseStatusType:Failure");
        RegistryErrorList regErrList = new RegistryErrorList();
        responseType.setRegistryErrorList(regErrList);
        RegistryError regErr = new RegistryError();
        regErrList.getRegistryError().add(regErr);
        regErr.setCodeContext(codeContext);
        regErr.setErrorCode("XDSRegistryError");
        regErr.setSeverity("Error");
        return response;
    }      

    private String getHomeCommunityId()
    {
        String ret = null;
        Facility facility = facilityManager.getFacilityByFacilityNumber("VA");
        if (facility != null) {
            ret = facility.getHomeCommunityId();
        }
        return ret;
    }   
  
}
