﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.Mvc;

using System.Threading.Tasks;
using log4net;

using OSEHRAuth.Core;
using OSEHRAuth.Core.Models;
using OSEHRAuth.Core.Models.Post;

using OSEHRAuth.Website.Models;

//ToDo
//- Add NotEnoughQuestionsException 
//- Persistence - Done using sessionData object
//- Validation 
//- Styling
//- Add SSL to site
//Test for 
//1. IA Response—Could Not Generate Enough Questions

//To prevent navigating directly to a action url either remove its GET method or use if(!ControllerContext.IsChildAction)
//Why user cannot directly to Authenticated etc in this app, is because only action URLs exposed are Verify [GET and POST]
// and Authenticate[POST only, GET will not work].
//MDWS Service Reference: http://mdws.vainnovation.us/mdws2/EmrSvc.asmx
//More than 3 subject not founds will result in finished 3 tries in the same session.

namespace OSEHRAuth.Website.Controllers
{
    /// <summary>
    /// Controller for person verify and authenticate.
    /// </summary>
    public class PersonController : AsyncController
    {
        private static readonly ILog log = LogManager.GetLogger(typeof(PersonController));
        private static readonly ILog mdwsLog = LogManager.GetLogger("mdws");

        private const int VERIFY_MAX_COUNT = 3;
        private const int VELOCITY_MAX_COUNT = 2;
        //
        // GET: /Person/Verify
        /// <summary>
        /// Verify action - Presents the form to get Person details such as Name, Street address, city etc 
        /// Verify [GET] url is accessible directly also.
        /// </summary>
        /// <returns></returns>
        public ActionResult Verify()
        {
            return View();
        }

        //
        // POST: /Person/Verify
        /// <summary>
        /// Receives the posted person details such as Name, Street Address etc. 
        /// Makes verify request. If the Verify passes then receives the authentication quiz. 
        /// Authenticate View is then returned along with the quiz data in view model.
        /// If verify fails, 2 more attempts are made. Allows max 3 verify attempts in the same session.
        /// </summary>
        /// <param name="verifyDetails"></param>
        /// <returns></returns>
        [HttpPost]
        public ActionResult VerifyAsync(VerifyViewModel verifyViewModel)
        {
            try
            {
                //increment verify count
                SessionData sessionData = HttpContext.GetSessionData();
                sessionData.VerifyCount++;
                sessionData.Person = verifyViewModel.Person;

                if ((sessionData.VerifyCount - sessionData.AuthenticateCount) > VERIFY_MAX_COUNT)
                {
                    log.Info(string.Format("Verify: {0} {1} Session Verify count {2} reached maximum - SessionId: {3} TransactionId: {4} ",
                        sessionData.Person.FirstName, sessionData.Person.LastName, (sessionData.VerifyCount - sessionData.AuthenticateCount), 
                        HttpContext.Session.SessionID));                   
                    SubjectNotFoundViewModel snfvm = new SubjectNotFoundViewModel();
                    snfvm.FirstName = sessionData.Person.FirstName;
                    snfvm.TxnId = sessionData.TxnId;
                    return View("SubjectNotFoundError", snfvm);
                }

                if (!ValidatePerson(verifyViewModel.Person))
                    throw new AuthenticationServiceException("Verify: Invalid Person Data..");

                //ToDo Add validation to personToVerify
                IdentityService identityService = new IdentityService(sessionData);
                Quiz authenticationQuiz =  identityService.Verify();

                log.Info(string.Format("Verify:  {0} {1} Passed Verify at Attempt#: {2} SessionId: {3} TransactionId: {4} ResponseUniqueId: {5} returned quiz successfully",
                    sessionData.Person.FirstName, sessionData.Person.LastName, sessionData.VerifyCount - sessionData.AuthenticateCount, HttpContext.Session.SessionID, 
                    sessionData.TxnId, sessionData.ResponseUniqueId));

                AuthenticateViewModel model = new AuthenticateViewModel();
                model.AuthenticationQuiz = authenticationQuiz;
                model.Person = sessionData.Person;
                return View("Authenticate", model);
            }
            catch (VelocityFailedException vfe)
            {
                SessionData sessionData = HttpContext.GetSessionData();
                sessionData.TxnId = vfe.TxnId;
                log.Error(string.Format("Verify: {0} {1} PersonId: {2} Txn Id: {3} session Id: {4} Source IP: {5} - Verification failed - Velocity (Authentication tries > 2) exceeded.",
                    sessionData.Person.FirstName, sessionData.Person.LastName, vfe.PersonId, vfe.TxnId, HttpContext.Session.SessionID, HttpContext.Request.UserHostAddress));
                UserErrorViewModel uevm = new UserErrorViewModel();
                uevm.FirstName = sessionData.Person.FirstName;
                uevm.TxnId = sessionData.TxnId;
                return View("UserLockedOut", uevm);
            }
            catch (AuthenticationServiceException ex)
            {
                SessionData sessionData = HttpContext.GetSessionData();
                sessionData.TxnId = ex.TxnId;
                log.Error(string.Format("Verify: PersonId: {0} Txn Id: {1} session Id: {2} Source IP: {3} Attempt count: {4} LastName: {5} FirstName: {6} Message: {7}",
                    ex.PersonId, ex.TxnId, HttpContext.Session.SessionID, HttpContext.Request.UserHostAddress,
                    sessionData.VerifyCount, sessionData.Person.LastName, sessionData.Person.FirstName, ex.Message));
                //verify < 3 and not authenticated yet in this session, can try verify again.
                if ((sessionData.VerifyCount < VERIFY_MAX_COUNT) && (sessionData.AuthenticateCount < 1))
                {
                    string remainingAttempts = string.Empty, remainingAttemptsDigit = string.Empty, remainingAttemptsOrdinal = string.Empty;
                    switch (sessionData.VerifyCount)
                    {
                        case 1:
                            remainingAttempts = "two";
                            remainingAttemptsDigit = "2";
                            remainingAttemptsOrdinal = "second";
                            break;
                        case 2:
                            remainingAttempts = "one";
                            remainingAttemptsDigit = "1";
                            remainingAttemptsOrdinal = "next";
                            break;
                    }
                    ModelState.AddModelError("", string.Format(Global.SUBJECT_NOT_FOUND_ERROR_MESSAGE, remainingAttempts,
                        remainingAttemptsDigit, remainingAttemptsOrdinal));

                    return View(verifyViewModel);
                }
                else
                {
                    SubjectNotFoundViewModel snfvm = new SubjectNotFoundViewModel();
                    snfvm.FirstName = sessionData.Person.FirstName;
                    snfvm.TxnId = sessionData.TxnId;
                    return View("SubjectNotFoundError", snfvm);
                }
            }
            catch (Exception ex)
            {
                SessionData sessionData = HttpContext.GetSessionData();
                log.Error(string.Format(@"Verify: Txn Id: {0} session Id: {1} Source IP: {2} Attempt count: {3} LastName: {4} FirstName: {5} 
                    Message: {6} Stack trace: {7}",
                    sessionData.TxnId, HttpContext.Session.SessionID, HttpContext.Request.UserHostAddress,
                    sessionData.VerifyCount, sessionData.Person.LastName, sessionData.Person.FirstName, ex.Message, ex.StackTrace));
                return View("Error");
            }
            finally
            {
                //Date shows model error - could not figure out why so clear it explisitely for next iteration
                if (ModelState["Person.DateOfBirth"] != null)
                    ModelState["Person.DateOfBirth"].Errors.Clear();
            }
        }

        //
        // POST: /Person/Authenticate
        /// <summary>
        /// Receives the posted quiz answers. 
        /// Submits the quiz answers for Scoring. 
        /// If passing score is received, presents the Authenticated View along with further instructions to complete the authentication process.
        /// If fails the quiz and velocity is not exceeded, then presents AuthenticateFailedView for re-taking the quiz.
        /// If fails the quiz and velocity is exceeded then presents UserErrorView with instructions to complete the IPA at the nearest VA Facility.
        /// </summary>
        /// <param name="verifyDetails"></param>
        /// <returns></returns>
        [HttpPost]
        public async Task<ActionResult> AuthenticateAsync(AuthenticateViewModel authenticateViewModel)
        {
            SessionData sessionData = HttpContext.GetSessionData();
            if (sessionData.VerifyCount == 0)
            {
                //this is an indication that session expired and a new session was created thus initializing the VerifyCount to 0
                // re-direct to person page after logging
                log.Warn("Authenticate: New session created due to session expiry, lost the earlier session data, need to start from verify");
                log.Warn(string.Format("Authenticate: New session details - SessionId: {0} {1}", HttpContext.Session.SessionID, sessionData));
                return RedirectToAction("Verify");
            }
            sessionData.AuthenticateCount++;

            if (authenticateViewModel.AuthenticationQuiz != null)
                authenticateViewModel.AuthenticationQuiz.LexId = sessionData.LexId;

            //MdwsEmrService mdwsEmrService = new MdwsEmrService();
            //mdwsEmrService.VistaSiteCode = Global.VISTA_SITECODE;
            //mdwsEmrService.VistaAccessCode = Global.VISTA_ACCESSCODE;
            //mdwsEmrService.VistaVerifyCode = Global.VISTA_VERIFYCODE;
            MdwsMhvService mdwsMhvService = new MdwsMhvService();
            IdentityService identityService = new IdentityService(sessionData);

            try
            {
                if (!ValidateQuizAnswers(authenticateViewModel.AuthenticationQuiz))
                {
                    AuthenticationFailedException afe = new AuthenticationFailedException("Authenticate: Incomplete Quiz Answers.");
                    afe.VelocityNotExceeded = (sessionData.AuthenticateCount < VELOCITY_MAX_COUNT);
                    afe.TxnId = sessionData.TxnId;
                    afe.PersonId = sessionData.LexId;
                    afe.SessionId = HttpContext.Session.SessionID;
                    throw afe;
                }

                QuizScoreDTO quizScore = identityService.Authenticate(authenticateViewModel.AuthenticationQuiz, sessionData);
                log.Info(string.Format("Authenticate: Authentication Success for {0} {1} - SessionId: {2} TxnId: {3} PersonId: {4} QuizId: {5} Status: {6} ",
                    sessionData.Person.FirstName, sessionData.Person.LastName, HttpContext.Session.SessionID, quizScore.TxnId, quizScore.LexId,
                    quizScore.QuizScore.QuizId, quizScore.IdentityResponseStatus));
                try
                {

                    //bool personExistsInVista = await mdwsEmrService.IsPatientInVistaAsync(sessionData.Person);
                    bool personExistsInVista = await mdwsMhvService.IsPatientInCdwAsync(sessionData.Person);
                    if (!personExistsInVista)
                    {
                        throw new MdwsServiceException("Authenticate: Person does not exist in vista although authentication is successful.");
                    }
                    string logMessage = string.Format("Authenticate:  +++ | {0} {1} | Exist in Vista: |{2}| PersonId: | {3}",
                            sessionData.Person.FirstName, sessionData.Person.LastName, personExistsInVista, sessionData.LexId);
                    mdwsLog.Info(logMessage); //also log to mdws for statistics gathering.
                }
                catch (MdwsServiceException mse)
                {
                    string logMessage = string.Format("Authenticate:  -+- | {0} {1} | Mdws Exception: | {2} | PersonId: | {3}",
                            sessionData.Person.FirstName, sessionData.Person.LastName, mse.Message, sessionData.LexId);
                    mdwsLog.Error(logMessage);
                    UserErrorViewModel uevm = new UserErrorViewModel();
                    uevm.FirstName = sessionData.Person.FirstName;
                    uevm.TxnId = sessionData.TxnId;
                    return View("UserLockedOut", uevm);
                }
                AuthenticatedViewModel model = new AuthenticatedViewModel();
                model.FirstName = sessionData.Person.FirstName;
                model.TxnId = sessionData.TxnId;
                return View("Authenticated", model);
            }
            catch (AuthenticationFailedException afe)
            {
                if (afe.VelocityNotExceeded)
                {
                    log.Error(string.Format("Authenticate: {0} {1} - PersonId: {2} Txn Id: {3} session Id: {4} Source IP: {5} - Authentication failed - Velocity (Authentication tries > 2) NOT exceeded.",
                        sessionData.Person.FirstName, sessionData.Person.LastName, afe.PersonId, afe.TxnId, HttpContext.Session.SessionID, HttpContext.Request.UserHostAddress));
                    AuthenticateFailedViewModel afvm = new AuthenticateFailedViewModel();
                    afvm.FirstName = sessionData.Person.FirstName;
                    afvm.TxnId = sessionData.TxnId;
                    return View("AuthenticateFailed", afvm);
                }
                else
                {
                    //show usererror 
                    log.Error(string.Format("Authenticate: {0} {1} - PersonId: {2} Txn Id: {3} session Id: {4} Source IP: {5} - Authentication failed - Velocity exceeded.",
                       sessionData.Person.FirstName, sessionData.Person.LastName, afe.PersonId, afe.TxnId, HttpContext.Session.SessionID, HttpContext.Request.UserHostAddress));
                    UserErrorViewModel uevm = new UserErrorViewModel();
                    uevm.FirstName = sessionData.Person.FirstName;
                    uevm.TxnId = sessionData.TxnId;
                    return View("UserLockedOut", uevm);
                }
            }
            catch (QuizExpiredException qee)
            {
                if (qee.VelocityNotExceeded)
                {
                    log.Error(string.Format("Authenticate: {0} {1} PersonId: {2} Txn Id: {3} session Id: {4} Source IP: {5} - Quiz Expired - Velocity NOT exceeded.",
                        sessionData.Person.FirstName, sessionData.Person.LastName, qee.PersonId, qee.TxnId, HttpContext.Session.SessionID, HttpContext.Request.UserHostAddress));
                    AuthenticateFailedViewModel afvm = new AuthenticateFailedViewModel();
                    afvm.FirstName = sessionData.Person.FirstName;
                    afvm.TxnId = sessionData.TxnId;
                    return View("AuthenticateFailed", afvm);
                }
                else
                {
                    log.Error(string.Format("Authenticate: {0} {1} PersonId: {2} Txn Id: {3} session Id: {4} Source IP: {5} - Quiz Expired - Velocity (Authentication tries > 2) exceeded.",
                    sessionData.Person.FirstName, sessionData.Person.LastName, qee.PersonId, qee.TxnId, HttpContext.Session.SessionID, HttpContext.Request.UserHostAddress));
                    UserErrorViewModel uevm = new UserErrorViewModel();
                    uevm.FirstName = sessionData.Person.FirstName;
                    uevm.TxnId = sessionData.TxnId;
                    return View("UserLockedOut", uevm);
                }
            }
            catch (ScoreQuizException sqe)
            { 
                log.Error(string.Format("Authenticate: {0} {1} PersonId: {2} Txn Id: {3} session Id: {4} Source IP: {5} Score Quiz Exception Code: {6} - Velocity (Authentication tries > 2) exceeded.",
                sessionData.Person.FirstName, sessionData.Person.LastName, sqe.PersonId, sqe.TxnId, HttpContext.Session.SessionID, HttpContext.Request.UserHostAddress, sqe.Code));
                return View("Error");             
            }
            catch (Exception ex)
            {
                log.Error(string.Format("Authenticate: System error SeesionId: {0} TxnId: {1} Message: {2} Stack: {3}",
                    HttpContext.Session.SessionID, sessionData.TxnId, ex.Message, ex.StackTrace));
                return View("Error");
            }
            finally
            {
                //placeholder for now
            }
        }


        /// <summary>
        /// Authenticate Again by requesting Quiz using data captured earlier in the session.
        /// This url is directly accessible. If accessed directly or after session timeout then user is redirected to Verify Action.
        /// </summary>
        /// <returns></returns>
        public ActionResult VerifyAgainAsync()
        {
            SessionData sessionData = HttpContext.GetSessionData();
            if (sessionData.VerifyCount == 0)
            {
                //this is an indication that session expired and a new session was created thus initializing the VerifyCount to 0
                // re-direct to person page after logging
                log.Warn("VerifyAgainAsync: New session created due to session expiry, lost the earlier session data, need to start from verify");
                log.Warn(string.Format("VerifyAgainAsync: New session details - SessionId: {0} {1}", HttpContext.Session.SessionID, sessionData));
                return RedirectToAction("Verify");
            }
            VerifyViewModel verifyViewModel = new VerifyViewModel();
            verifyViewModel.Person = sessionData.Person;
            return VerifyAsync(verifyViewModel);
        }
        private bool ValidatePerson(PersonDTO person)
        {
            bool dataIsValid = true;
            if (string.IsNullOrEmpty(person.FirstName))
            {
                log.Warn(string.Format("Empty First Name - Session Id: {0}.", HttpContext.Session.SessionID));
                dataIsValid = false;
            }
            if (string.IsNullOrEmpty(person.LastName))
            {
                log.Warn(string.Format("Empty Last Name - Session Id: {0}.", HttpContext.Session.SessionID));
                dataIsValid = false;
            }
            if (string.IsNullOrEmpty(person.StreetAddress))
            {
                log.Warn(string.Format("Empty Street Address - Session Id: {0}.", HttpContext.Session.SessionID));
                dataIsValid = false;
            }
            if (string.IsNullOrEmpty(person.City))
            {
                log.Warn(string.Format("Empty City - Session Id: {0}.", HttpContext.Session.SessionID));
                dataIsValid = false;
            }
            if (string.IsNullOrEmpty(person.StateCode))
            {
                log.Warn(string.Format("Empty State Code - Session Id: {0}.", HttpContext.Session.SessionID));
                dataIsValid = false;
            }
            if (string.IsNullOrEmpty(person.Ssn))
            {
                log.Warn(string.Format("Empty Ssn - Session Id: {0}.", HttpContext.Session.SessionID));
                dataIsValid = false;
            }
            if (string.IsNullOrEmpty(person.SsnRepeat))
            {
                log.Warn(string.Format("Empty SsnRepeat - Session Id: {0}.", HttpContext.Session.SessionID));
                dataIsValid = false;
            }
            else if (!person.SsnRepeat.Equals(person.Ssn))
            {
                log.Warn(string.Format("SSN does not match re-entered SSN - Session Id: {0}.", HttpContext.Session.SessionID));
                dataIsValid = false;
            }
            if (!person.DateOfBirth.HasValue)
            {
                log.Warn(string.Format("Invalid Date of birth - Session Id: {0}.", HttpContext.Session.SessionID));
                dataIsValid = false;
            }
            return dataIsValid;
        }

        private bool ValidateQuizAnswers(Quiz authenticationQuiz)
        {
            bool dataIsValid = true;
            foreach(QuizQuestion question in authenticationQuiz.Questions)
            {
                if (string.IsNullOrEmpty(question.ChoiceId))
                {
                    log.Warn(string.Format("Question with Id: {0} not answered -Txn Id: {1} Lex Id: {2}.",
                        question.QuestionId, authenticationQuiz.TxnId, authenticationQuiz.LexId));
                    if (dataIsValid) //if data is not already set as invalid, set it now as invalid.
                        dataIsValid = false;
                }
            }
            return dataIsValid;
        }
    }
}
