{
Copyright  DSS, Inc.
All Rights Reserved.

PURPOSE:  A collection of routines that handle FileMan date/time format and conversion.
          Lifted from FMUtils.pas and given their own space.  Also some general use
          Date/Time routines.
 AUTHOR:  Brian Smith
  DATES:  06/01/04 BS 1.0 1st version.
          11/07/08 JAC 2.0 Added functions IsDate, LongDateStringToNumericDateString,
                           MonthNameToNumber, TodayIfNull from dssLibrary.  Added
                           uDSSPiece to uses.  Added functions FMDateTimeToSQLString &
                           TDateTimeTOSQLString.
          12/03/08 MSH 2.1 Added additional CPRS date/time functions.
          04/06/10 JAC 2.2 Fix in LongDateStringToNumericDateString to raise exception properly.
                           Added function LastDayInMonth.


  NOTES:
}
unit fmDateTime;

interface

uses
  Trpcb, DSSRPC;

type
  TFMDateTime = Double;
  TVistaDateTimeFormat = (dfInternal, dfExternal);

function FMDateTimeToString(FMDateTime: string;bIncludeTime:boolean=true;bExceptionIfEmpty:boolean=false): string;
function FMDateTimeToSQLString(FMDateTime:string;bIncludeTime:boolean=false;bIncludeQuotes:boolean=true):string;
function FMDateTimeToTDateTime(FMDateTime:string;bIncludeTime:boolean=true;bExceptionIfEmpty:boolean=false):TDateTime;
function FMLongDateToTDateTime(Value: string; SendTime: boolean = false;bExceptionIfEmpty:boolean=false): TDateTime;
function FMTimeToTTime(FMDateTime: string): TDateTime;

function IsDate(sDateString: string): boolean;
function LongDateStringToNumericDateString(sLongDate:string):string;
function MonthNameToNumber(sMonthName:string):integer;

function StringToFMDateTime(aValue: string): string;
function TDateTimeToSQLString(aDateTime:TDateTime;bIncludeTime:boolean=false;bIncludeSingleQuotes:boolean=true):string;
function TDateTimeToVistaDateTime(ADateTime: TDateTime; AFormat: TVistaDateTimeFormat=dfInternal; AIncludeTime: boolean=false): string;
function TodayIfNull(dtDate:TDateTime):TDateTime;
function TTimeToFMTime(dtTime:TDateTime):string;
{!!2.1 BEGIN - FOIA functions from CPRS source}
function CharAt(const x: string; APos: Integer): Char;
function FormatFMDateTime(AFormat: string; ADateTime: TFMDateTime): string;
function FormatFMDateTimeStr(const AFormat, ADateTime: string): string;
function IsFMDateTime(x: string): Boolean;
function MakeFMDateTime(const AString: string): TFMDateTime;
function FMDateTimeToDateTime(ADateTime: TFMDateTime): TDateTime;
{END - FOIA functions from CPRS source}
function LastDayInMonth(const Year, Month: word): TDateTime;    {!!2.2}

implementation

uses
  SysUtils,dialogs, dssLibrary, fmUtils, uDSSPiece, dssConst;

{!!2.1 BEGIN - FOIA functions from CPRS source}
type
  EFMDateTimeError = class(Exception);

const
  { names of months used by FormatFMDateTime }
  MONTH_NAMES_SHORT: array[1..12] of string[3] =
    ('Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec');
  MONTH_NAMES_LONG:  array[1..12] of string[9] =
    ('January','February','March','April','May','June','July','August','September','October',
     'November', 'December');

function CharAt(const x: string; APos: Integer): Char;
{ Returns a character at a given position in a string or the null character if past the end }
begin
  if Length(x) < APos then Result := #0 else Result := x[APos];
end;

function FormatFMDateTime(AFormat: string; ADateTime: TFMDateTime): string;
{ Formats a Fileman Date/Time using (mostly) the same format string as Delphi FormatDateTime }
var
  x: string;
  y, m, d, h, n, s: Integer;

  function TrimFormatCount: Integer;
  { Delete repeating characters and count how many were deleted }
  var
    c: Char;
  begin
    result := 0;
    c := AFormat[1];
    repeat
      Delete(AFormat, 1, 1);
      Inc(result);
    until CharAt(AFormat, 1) <> c;
  end;

begin                                            {FormatFMDateTime}
  result := '';
  if not(ADateTime > 0) then Exit;
  x := FloatToStrF(ADateTime, ffFixed, 15, 6) + '0000000';
  y := StrToIntDef(Copy(x,  1, 3), 0) + 1700;
  m := StrToIntDef(Copy(x,  4, 2), 0);
  d := StrToIntDef(Copy(x,  6, 2), 0);
  h := StrToIntDef(Copy(x,  9, 2), 0);
  n := StrToIntDef(Copy(x, 11, 2), 0);
  s := StrToIntDef(Copy(x, 13, 2), 0);
  while Length(AFormat) > 0 do
    case UpCase(AFormat[1]) of
    '"': begin                                                                 // literal
           Delete(AFormat, 1, 1);
           while not (CharAt(AFormat, 1) in [#0, '"']) do
           begin
             result := result + AFormat[1];
             Delete(AFormat, 1, 1);
           end;
           if CharAt(AFormat, 1) = '"' then Delete(AFormat, 1, 1);
         end;
    'D': case TrimFormatCount of                                               // day/date
         1: if (d > 0) then result := result + IntToStr(d);
         2: if (d > 0) then result := result + FormatFloat('00', d);
         end;
    'H': case TrimFormatCount of                                               // hour
         1: result := result + IntToStr(h);
         2: result := result + FormatFloat('00', h);
         end;
    'M': case TrimFormatCount of                                               // month
         1: if (m > 0) then result := result + IntToStr(m);
         2: if (m > 0) then result := result + FormatFloat('00', m);
         3: if (m in [1..12]) then result := result + MONTH_NAMES_SHORT[m];
         4: if (m in [1..12]) then result := result + MONTH_NAMES_LONG[m];
         end;
    'N': case TrimFormatCount of                                               // minute
         1: result := result + IntToStr(n);
         2: result := result + FormatFloat('00', n);
         end;
    'S': case TrimFormatCount of                                               // second
         1: result := result + IntToStr(s);
         2: result := result + FormatFloat('00', s);
         end;
    'Y': case TrimFormatCount of                                               // year
         2: if (y > 0) then result := result + Copy(IntToStr(y), 3, 2);
         4: if (y > 0) then result := result + IntToStr(y);
         end;
    else begin                                                                 // other
           result := result + AFormat[1];
           Delete(AFormat, 1, 1);
         end;
    end; {case}
end;                                             {FormatFMDateTime}

function FormatFMDateTimeStr(const AFormat, ADateTime: string): string;
var
  FMDateTime: TFMDateTime;
begin
  result := ADateTime;
  if IsFMDateTime(ADateTime) then
  begin
    FMDateTime := MakeFMDateTime(ADateTime);
    result := FormatFMDateTime(AFormat, FMDateTime);
  end;
end;

function IsFMDateTime(x: string): Boolean;
var
  i: Integer;
begin
  result := FALSE;
  if (Length(x) < 7) then Exit;
  for i := 1 to 7 do if not(x[i] in ['0'..'9']) then Exit;
  if (Length(x) > 7) and (x[8] <> '.') then Exit;
  if (Length(x) > 8) and not (x[9] in ['0'..'9']) then Exit;
  result := TRUE;
end;

function MakeFMDateTime(const AString: string): TFMDateTime;
begin
  Result := -1;
  if (Length(AString) > 0) and IsFMDateTime(AString) then Result := StrToFloat(AString);
end;

function FMDateTimeToDateTime(ADateTime: TFMDateTime): TDateTime;
{ Converts a Fileman date/time (type double) to a Delphi date/time }
var
  ADate, ATime: TDateTime;
  DatePart, TimePart: string;
begin
  DatePart := Piece(FloatToStrF(ADateTime, ffFixed, 14, 6), '.', 1);
  TimePart := Piece(FloatToStrF(ADateTime, ffFixed, 14, 6), '.', 2) + '000000';
  if (Length(DatePart) <> 7) then raise EFMDateTimeError.Create('Invalid Fileman Date');
  if (Copy(TimePart, 1, 2) = '24') then TimePart := '23595959';
  ADate := EncodeDate(StrToInt(Copy(DatePart, 1, 3)) + 1700,
                      StrToInt(Copy(DatePart, 4, 2)),
                      StrToInt(Copy(DatePart, 6, 2)));
  ATime := EncodeTime(StrToInt(Copy(TimePart, 1, 2)),
                      StrToInt(Copy(TimePart, 3, 2)),
                      StrToInt(Copy(TimePart, 5, 2)), 0);
  Result := ADate + ATime;
end;
{END - FOIA functions from CPRS source}

//******************************************************************************
//  NOTE - There are several issues with the function FMDateTimeToString concerning
//  error checking and zero padding.  Please move to the function FormatFMDateTimeStr
//  above.  It uses similar formatting syntax as Delphi for the formatting of the
//  date.  Should it encounter an invalid date string, it simply returns the invalid
//  string without error.  This should allow the function to continue without

//  Also the common conjunction of FMLongDateToTDateTime(FMDateTimeToString(StringValue))
//  Should be replaced with  FMDateTimeToDateTime(StrToFloat(StringValue)).
//  In conjunction with this replacement, an error check should be used - something
//  along the lines of
//
//  if not(IsFMDateTime(sValue)) then
//    ShowErrorMessage('WARNING:  A document in this section contains an invalid date of "' + sValue + '" in VistA', 'VistA Data Error')
//  else ... ;
//******************************************************************************
function FMDateTimeToString(FMDateTime: string; bIncludeTime: boolean = TRUE; bExceptionIfEmpty: boolean = FALSE): string;
var
  sYear, sMonth, sDay, sHour, sMinute: string;
begin  //  Sample result: JAN 01,2004 @ 08:15
  if bExceptionIfEmpty then
    assert(FMDateTime > '', 'Blank date was passed to FMDateTimeToString');

  //????  MSH - Incomplete error checking for valid date.time string.
  if (trim(FMDateTime) = '') then
  begin
    result := '';
    exit
  end;

  sYear  := copy(FMDateTime, 0, 3);
  sYear  := inttostr(1700 + strtoint(sYear));

  //????  MSH - Some error checking for Month, but incomplete overall.
  sMonth := copy(FMDateTime, 4, 2);
  if (sMonth = '0') or (sMonth = '00') or (sMonth = '') then sMonth := '01';
  case StrToInt(sMonth) of
    1: sMonth  := 'JAN';
    2: sMonth  := 'FEB';
    3: sMonth  := 'MAR';
    4: sMonth  := 'APR';
    5: sMonth  := 'MAY';
    6: sMonth  := 'JUN';
    7: sMonth  := 'JUL';
    8: sMonth  := 'AUG';
    9: sMonth  := 'SEP';
    10: sMonth := 'OCT';
    11: sMonth := 'NOV';
    12: sMonth := 'DEC';
  else
    begin
      MessageDlg('Month (' + sMonth + ') is invalid!', mtCustom, [mbOK], 0);
      exit;
    end;
  end;

  //????  MSH - Missing error capturing invalid day conversions...
  sDay := copy(FMDateTime, 6, 2);
  Result := sMonth + ' ' + sDay + ', ' + sYear;

  //????  MSH - Not enough zero padding for FMDate.Times  example:  3080101.103
  if bIncludeTime then
  begin
    sHour := copy(FMDateTime, 9, 2);
    if sHour = '' then
      sHour := '00';
    sMinute := copy(FMDateTime, 11, 2);
    if sMinute = '' then
      sMinute := '00';
    result := result + ' @ ' + sHour + ':' + sMinute;
  end;
end;

function FMDateTimeToSQLString(FMDateTime:string;bIncludeTime:boolean=false;bIncludeQuotes:boolean=true):string;
begin
  result := TDateTimeToSQLString(FMDateTimeToTDateTime(FMDateTime,bIncludeTime),bIncludeTime,bIncludeQuotes);
end;

function FMLongDateToTDateTime(Value: string; SendTime: boolean = false;bExceptionIfEmpty:boolean=false): TDateTime;
var
  Hour:    integer;
  iMonth:integer;
  Month, Day, Time, AMPM, HourString, Minute, Second, Year: string;
  //nowtime: TDateTime;
  nowdate: TDateTime;
  nHour, nMinute, nSecond, nMs: word;
  sFinalTotry:string;
begin
  //Was FMDateToTDate
  //Handles "JUN 25,2004@hh:mm:ss" dates and turns them into TDateTime
  //The "time" part is optional and you can opt to not have the time encoded
  //into the returned TDateTime.
  if bExceptionIfEmpty then
    assert(Value > '', 'Blank date was passed to FMLongDateToTDateTime');

  if Value = '' then
    begin
      Result := 0;
      exit;
    end;

  DecodeTime(Now, nHour, nMinute, nSecond, nMs);
  nowdate := StrToTime(IntToStr(nHour) + ':' + IntToStr(nMinute) + ':' + IntToStr(nSecond));

  Month := piece(UpperCase(Value), ' ', 1);
  Day   := piece(piece(UpperCase(Value), ',', 1), ' ', 2);
  Year  := piece(UpperCase(Value), ',', 2);
  Year  := Trim(Year);
  if pos('@', Year) > 0 then
    begin
      //We need to deal with the time
      Time := piece(Year, '@', 2);
      Year := piece(Year, '@', 1);
      Hour := StrToIntDef(piece(Time, ':', 1), 0);
      AMPM := 'AM';
      case Hour of
        0: Time := '12';
        12: AMPM := 'PM';
        13..24:
          begin
              Hour := Hour - 12;
              AMPM := 'PM';
          end;
      end;
      HourString := IntToStr(Hour);
      if (HourString = '00') or (HourString = '0') then
        begin
          HourString := '12';
          AMPM := 'AM';
        end;
      Minute := piece(Time, ':', 2);
      if Minute = '' then
        Minute := '00';
      Second := piece(Time, ':', 3);
      if Second = '' then
        Second := '00';
      Time := HourString + ':' + Minute + ':' + Second + ' ' + AMPM;
    end;

  Month := inttostr(MonthNameToNumber(Month));

  //begin range checking/blocking
  if StrToInt(Day) < 1 then
    Day := '01';
  if StrToInt(Day) > 31 then
    Day := '31';

  if SendTime then
    sFinalToTry := Month + '/' + Day + '/' + Year + ' ' + Time
  else
    sFinalToTry := Month + '/' + Day + '/' + Year;

  try
    Result := StrToDateTime(sFinalToTry);
  except
    raise Exception.Create('Bad date/time found. Date/time passed in was ' +
      Month + '/' + Day + '/' + Year + ' ' + Time);
  end;
end;

function FMTimeToTTime(FMDateTime: string): TDateTime;
var
  sHour, sMin, sSec: string;
  Hour, Min, Sec:    word;
begin
  //was ConvertFileManTimeToDelphiDate
  if Piece(FMDateTime, '.', 2) <> '' then
    begin
      if Piece(FMDateTime, '.', 2) = '24' then
        begin
          Hour := 0;
          Min  := 0;
          Sec  := 0;
        end
      else
        begin
          sHour := Copy(Piece(FMDateTime, '.', 2), 1, 2);
          if Length(sHour) < 2 then
            sHour := sHour + '0';
          sMin := Copy(Piece(FMDateTime, '.', 2), 3, 2);
          if Length(sMin) < 2 then
            sMin := sMin + '0';
          sSec := Copy(Piece(FMDateTime, '.', 2), 5, 2);
          if Length(sSec) < 2 then
            sSec := sSec + '0';
          if sHour <> '' then
            Hour := StrToIntDef(sHour, 0)
          else
            Hour := 0;
          if sMin <> '' then
            Min := StrToIntDef(sMin, 0)
          else
            Min := 0;
          if sSec <> '' then
            Sec := StrToIntDef(sSec, 0)
          else
            Sec := 0;
        end;
    end
  else
    begin
      if FMDateTime = '24' then
        begin
          Hour := 0;
          Min  := 0;
          Sec  := 0;
        end
      else
        begin
          sHour := Copy(FMDateTime, 1, 2);
          if Length(sHour) < 2 then
            sHour := sHour + '0';
          sMin := Copy(FMDateTime, 3, 2);
          if Length(sMin) < 2 then
            sMin := sMin + '0';
          sSec := Copy(FMDateTime, 5, 2);
          if Length(sSec) < 2 then
            sSec := sSec + '0';
          if sHour <> '' then
            Hour := StrToIntDef(sHour, 0)
          else
            Hour := 0;
          if sMin <> '' then
            Min := StrToIntDef(sMin, 0)
          else
            Min := 0;
          if sSec <> '' then
            Sec := StrToIntDef(sSec, 0)
          else
            Sec := 0;
        end;
    end;
  try
    Result := EncodeTime(Hour, Min, Sec, 0);
  except
    on Exception do
      Result := Now;
  end;
end;

function FMDateTimeToTDateTime(FMDateTime:string;bIncludeTime:boolean=true;bExceptionIfEmpty:boolean=false):TDateTime;
//New name for:
//ConvertFileManDateToDelphiDate
begin
  if bExceptionIfEmpty then
    assert(FMDateTime > '', 'An empty date was found where one was expected.  It was passed to FMDateTimeToTDateTime.');

  result := FMLongDateToTDateTime(FMDateTimeToString(FMDateTime,bIncludeTime),bIncludeTime,bExceptionIfEmpty);
end;

function IsDate(sDateString: string): boolean;
// Returns true if a string can be successfuly put through StrToDate.
var
  dt: TDateTime;
begin
  Result := true;
  try
    dt := StrToDate(sDateString);
  except
    on EConvertError do
      Result := false
    else
      raise;
  end;
end;

function LongDateStringToNumericDateString(sLongDate:string): string;
// Got tired of fighting with long date format dates that I couldn't
// do computations on (ie "Jun 25,2004") so this turns it into mm/dd/yyyy
var
  sWork: string;
  sMonth,sDay,sYear: string;
begin
  //first put ^ as seperators instead of , and SPACE
  //will now handle "Jun 25, 2004" as well - the double space was causing a problem
  sWork := StringReplace(sLongdate,',',' ',[rfReplaceAll]);
  //replace double space with single so the "Piece" call works on year
  sWork := StringReplace(sWork,'  ',' ',[rfReplaceAll]);
  sMonth := Piece(sWork,' ',1);
  sDay := Piece(sWork,' ',2);
  sYear := Piece(sWork,' ',3);
  if (sMonth = '') or (sDay = '') or (sYear = '') then
  begin                                          {!!2.2} 
    raise Exception.Create('Blank month, day or year found in LongDateStringToNumericDateString: '+crlf+crlf+sLongDate);
  end;
  sMonth := inttostr(MonthNameToNumber(sMonth));
  result := sMonth + DateSeparator + sDay + DateSeparator + sYear;
end;

function MonthNameToNumber(sMonthName:string):integer;
// Returns the number of a month based on its name
var
  ilp:integer;
begin
  result := 0;
  for ilp := low(LongMonthNames) to high(LongMonthNames) do
    begin
      if sametext(sMonthName,LongMonthNames[ilp]) or sametext(sMonthName,ShortMonthNames[ilp]) then
        begin
          result := ilp;
          break;
        end;
    end;
end;

function StringToFMDateTime(aValue: string): string;
var
  sFormat,Year, Month, Day, Time, Hour, Minute, Second, Value: string;
begin
  sFormat := 'mm/dd/yyyy hh:nn:ss';
  //If you want to work on a date like "Jun 25, 2004" please call LongDateStringToNumericDateString
    Time := ''; //default
  //Convert the str date/time they passed in to the format we can dissect later
  try
    Value := FormatDateTime(sFormat, strToDateTime(aValue));
  except
    raise Exception.Create('fmDateTime bad value passed to StringToFMDateTime(): ' + aValue);
  end;

  if piece(Value, ' ', 2) <> '' then
    begin
      Time  := piece(Value, ' ', 2);
      Value := piece(Value, ' ', 1); //We need to get rid of the time.
    end;
  Year   := piece(Value, '/', 3);
  Year   := IntToStr(StrToIntDef(piece(Value, '/', 3), -1) - 1700);
  Month  := piece(Value, '/', 1);
  Day    := piece(Value, '/', 2);
  Hour   := piece(Time, ':', 1);
  Minute := piece(Time, ':', 2);
  Second := piece(Time, ':', 3);
  Result := Year + Month + Day;
  if ((Hour <> '') and (Minute <> '') and (Second <> '')) then
    Result := Result + '.' + Hour + Minute + Second;
  Result := RemoveTrailing(Result, '0');
  Result := RemoveTrailing(Result, '.');
end;

function TDateTimeToSQLString(aDateTime:TDateTime;bIncludeTime:boolean=false;bIncludeSingleQuotes:boolean=true):string;
var
  sFormat:string;
begin
  //We put the quotes on in here because NULL needs to have no quotes around it,
  //but non-null values need quotes.  The calling routine should NOT apply its own quotes.
  //convert a tdatetime to a long string format useful in SQL, or NULL if = 0
  //(add bIncludeTime optional param later, default to false
  sformat := 'mm/dd/yyyy';
  if bIncludeTime then
    sformat := sFormat + ' hh:nn:ss';

  if aDateTime = 0 then
    result := 'NULL'
  else if bIncludeSingleQuotes then
    result := AnsiQuotedStr(FormatDateTime(sFormat,aDateTime),SQ)
  else
    result := FormatDateTime(sFormat,aDateTime);
end;

function TDateTimeToVistaDateTime(ADateTime: TDateTime;
    AFormat: TVistaDateTimeFormat=dfInternal; AIncludeTime: boolean=false): string;
var
  internalFormat: integer;
  Value, Year, Month, Day, Time, Hour, Minute, Second: string;
begin
  //AFormat = (dfInternal,dfExternal);

  internalFormat := Ord(AFormat);
  case Aformat of
    dfInternal:
      begin
        Value := FormatDateTime('mm/dd/yyyy hh:nn:ss', ADateTime);
        if piece(Value, ' ', 2) <> '' then
          begin
            Time  := piece(Value, ' ', 2);
            Value := piece(Value, ' ', 1); //We need to get rid of the time.
          end;
        Year   := piece(Value, '/', 3);
        Year   := IntToStr(StrToIntDef(piece(Value, '/', 3), -1) - 1700);
        Month  := piece(Value, '/', 1);
        Day    := piece(Value, '/', 2);
        Hour   := piece(Time, ':', 1);
        Minute := piece(Time, ':', 2);
        Second := piece(Time, ':', 3);
        Result := Year + Month + Day;
        if ((AIncludeTime) and (Hour <> '') and
            (Minute <> '') and (Second <> '')) then
          Result := Result + '.' + Hour + Minute + Second;
      end;
    dfExternal:
      begin
        try
          if AIncludeTime then
            Result := FormatDateTime('mmm dd, yyyy', ADateTime) +
                      '@' + FormatDateTime('hh:mm:ss', ADateTime)
          else
            Result := FormatDateTime('mmm dd, yyyy', ADateTime);
        except
          on e:exception do
            raise Exception.Create('Error in TDateTimeToVistaDateTime(): ' +
                      E.Message);
        end;
      end;
    else
      Result := 'Error!';
  end;
end;

function TodayIfNull(dtDate:TDateTime):TDateTime;
// Useful for situations where you want to use a date time value, but if it
// is null, then use today's date instead.
begin
  if dtDate = 0 then
    result := now
  else
    result := dtDate;
end;

function TTimeToFMTime(dtTime:TDateTime):string;
var
  Hour, Min, Sec, MSec: word;
  S, HourSubString, MinSubString, SecSubString: string;
begin
  DecodeTime(dtTime, Hour, Min, Sec, MSec);
  if length(IntToStr(Hour)) = 1 then
    HourSubString := '0' + IntToStr(Hour)
  else
    HourSubString := IntToStr(Hour);

  if length(IntToStr(Min)) = 1 then
    MinSubString := '0' + IntToStr(Min)
  else
    MinSubString := IntToStr(Min);

  if length(IntToStr(Sec)) = 1 then
    SecSubString := '0' + IntToStr(Sec)
  else
    SecSubString := IntToStr(Sec);

  if (HourSubString = '00') and (SecSubString = '00') and (MinSubString = '00') then
  begin
    //Default to first second after minute
    Result := '000001'
  end
  else if (SecSubString = '00') and (MinSubString = '00') then
  begin
    //Just put Hour.
    //  BUT check to see if hour has any trailing zeros
    //String will always be 2 chars in length -- see above construct
    if Copy(HourSubString, 2, 1) = '0' then
    begin
      Result := Copy(HourSubString, 1, 1);
    end
    else
    begin
      Result := HourSubString;
    end;
  end
  else if SecSubString = '00' then
  begin
    //Just put Hour+Minutes.
    //  BUT check to see if minutes has any trailing zeros
    //String will always be 4 chars in length -- see above construct
    S := HourSubString + MinSubString;
    if Copy(S, 4, 1) = '0' then
    begin
      Result := HourSubString + Copy(MinSubString, 1, 1);
    end
    else
    begin
      Result := HourSubString + MinSubString;
    end;
  end
  else
  begin
    //Just put Hour+Minutes+Seconds.
    //  BUT check to see if minutes has any trailing zeros
    //String will always be 6 chars in length -- see above construct
    S := HourSubString + MinSubString + SecSubString;

    if Copy(S, 6, 1) = '0' then
    begin
      Result := HourSubString + MinSubString + Copy(SecSubString, 1, 1);
    end
    else
    begin
      Result := HourSubString + MinSubString + SecSubString;
    end;
  end;
  Result := RemoveTrailing(Result,'0');
end;

function LastDayInMonth(const Year, Month: word): TDateTime;
//!!2.2 Returns the last day of a month, given the year and month.
begin
  if Month = 12 then
    Result := EncodeDate(Year+1, 1, 1) - 1
  else
    Result := EncodeDate(Year, Month+1, 1) - 1
end;

end.
