{
Copyright  DSS, Inc.
All Rights Reserved.

PURPOSE:  DSS's desendent RPC Broker.
 AUTHOR:  UNKNOWN
  DATES:  02/08/99 UNK 1.0 1st version.
          01/20/04 VWH 1.1 Added Vista call logging feature and property to suppress Message Dialogs.
          03/18/04 VWH 1.2 Added seconds, RPCTimeLimit, and RPCBError to logging feature.
          10/28/06 JBF 1.3 Took out all "VEJD" references.
          12/27/06 JAC 1.4 Moved Piece, NTGetUserName to DSSVistaUtils.
          08/31/07 VWH 2.2 Increase RPC logging from 5000 lines to 20000 lines.
                           Define this value as const LOG_SIZE.
          05/07/09 VWH 2.3 Add a broker procedure LogThis which writes a string
                           to the log file (if FLogCalls is TRUE).  This
                           can be used to log any significant events in your
                           Delphi code that you would like to see relative to
                           the sequence of RPC's executed.  For example, you
                           could log the name of the function you are in prior
                           to calling an RPC and then the results that function
                           returns after the RPC executes.
          06/26/09 VWH 2.4 Add logging of RPCBError to both Call and lstCall
                           procedures if an exception is raised.  This gives
                           a better indication as to why the RPC call failed.
          01/06/10 WS  2.41 Added some new types in SetParams.

  NOTES:
}
unit DSSRPC;

interface

uses
  Windows,Messages,SysUtils,Classes,Graphics,    { BORLAND UNITS }
  Controls,Forms,Dialogs,
  Trpcb, RPCConf1;                               { VA BROKER UNITS }

type
  TDSSRPCBroker = class(TRPCBroker)
  private                                        { Private declarations }
    function AddEntryToVEJDSystemUser(ArrayValues:TStringList):String;
    function GetVEJDSystemUserInfo(sUserIFN,FieldNumber:String):String;
    function EvaluateForFileEntry(FileNumber,EntryName:String;var EntryIEN:String):Boolean;
    function EvaluateForVEJDUserFile:Boolean;
    function ConnectToBroker(ServerName,Port:String):Boolean;
    procedure GetStationName(sStationIFN:String);
    procedure GetVISTAUserLocation(sUserIFN:String);
    procedure GetVISTAUserName(sUserIFN:String);
    procedure GetLocationName;
    function GetCalResults:TStrings;
    procedure SetCallResults(const Value: TStrings);

    procedure SetParams(const RPCName: string; const AParam: array of const);
    procedure CallBroker;
    procedure CallBrokerInContext;
    procedure SetList(AStringList: TStrings; ParamIndex: Integer);
  protected                                      { Protected declarations }
    FUserDuz,FUserName,FNTUserName,FDNSUserName,FMenuContext,FPatientIFN,FStationName,FStationIFN,FDefaultLocationIFN,FDefaultLocationName:String;
    FVEJDSystemUserDUZ:String;
    FUseCurrentServerNamePort:Boolean;
    FCallHistoryList:TStrings;
    FCallLog: TStrings;
    FLastErrorMessage: String;
    FLogCalls: Boolean;
    FShowErrorDialogs: Boolean;
    FVistaLogName: TFileName;
    procedure PUseCurrentServerNamePort(const Value: Boolean);
    procedure PVEJDSystemUserDUZ(const Value: string);
    procedure PDNSUserName(const Value: string);
    procedure PNTUserName(const Value: string);
    procedure PUserDuz(const Value: string);
    procedure PUserName(const Value: string);
    procedure PPatientIFN(const Value: string);
    procedure PMenuContext(const Value: string);
    procedure PStationIFN(const Value: string);
    procedure PStationName(const Value: string);
    procedure PDefaultLocationIFN(const Value: string);
    procedure PDefaultLocationName(const Value: string);
  public                                         { Public declarations }
    constructor Create(AOwner : TComponent); override;
    procedure Call; override;
    procedure lstCall(OutputBuffer: TStrings);
    function Execute: Boolean;
    destructor Destroy; override;
    procedure LogVistaCalls(VistaLogName: TFileName); //Set with path and name of log file to start logging, set to blank to stop logging.
    property LastErrorMessage: String read FLastErrorMessage;
    procedure CallV(const RPCName: string; const AParam: array of const);
    function sCallV(const RPCName: string; const AParam: array of const): string;
    procedure tCallV(ReturnData: TStrings; const RPCName: string; const AParam: array of const);
    procedure LogThis(LogEntry: WideString);     {!!2.3}
  published                                      { Published declarations }
    property UseCurrentServerNamePort: Boolean read FUseCurrentServerNamePort write PUseCurrentServerNamePort;
    property VEJDSystemUserDUZ: string read FVEJDSystemUserDUZ write PVEJDSystemUserDUZ;
    property DNSUserName: string read FDNSUserName write PDNSUserName;
    property NTUserName: string read FNTUserName write PNTUserName;
    property MenuContext: string read FMenuContext write PMenuContext;
    property UserDuz: string read FUserDuz write PUserDuz;
    property UserName: string read FUserName write PUserName;
    property PatientIFN: string read FPatientIFN write PPatientIFN;
    property StationIFN: string read FStationIFN write PStationIFN;
    property StationName: string read FStationName write PStationName;
    property DefaultLocationIFN: string read FDefaultLocationIFN write PDefaultLocationIFN;
    property DefaultLocationName: string read FDefaultLocationName write PDefaultLocationName;
    property CallHistoryList: TStrings read GetCalResults write SetCallResults;
    property ShowErrorDialogs: Boolean read FShowErrorDialogs write FShowErrorDialogs default TRUE;
  end;

procedure Register;

const
  XWB_M_REJECT =  20000 + 2;                     // M error
  XWB_BadSignOn = 20000 + 4;                     // SignOn 'Error' (happens when cancel pressed)
  CRLF = #13+#10;
  LOG_SIZE = 20000;                              //!!2.2 # of lines in an RPC log before oldest entries drop off.
implementation

uses
  DSSVistaUtils;

constructor TDSSRPCBroker.Create(AOwner : TComponent);
begin
inherited Create(AOwner);
  NTUserName := NTGetUserName;
  FCallHistoryList := TStringList.Create;
  //FUseCurrentServerNamePort:=FALSE;
  FCallLog := TStringList.Create;
  FLogCalls := FALSE;
end;

destructor TDSSRPCBroker.Destroy;
begin
  try
    FCallHistoryList.Free;
    FCallLog.Free;
  except on exception do;
  end;
inherited Destroy;
end;

function TDSSRPCBroker.GetCalResults:TStrings;
begin
  Result := FCallHistoryList;
end;

procedure TDSSRPCBroker.SetCallResults(const Value: TStrings);
begin
  FCallHistoryList.Assign(Value);
end;

procedure TDSSRPCBroker.lstCall(OutputBuffer: TStrings);
var
  CallList : TStringList;
  i, j: Integer;
  StartDateTime,EndDateTime,MultSub, MultValue: String;
begin
  CallList := TStringList.Create;
  try
    if Param.Count>0 then
      begin
        try
          for i := 0 to Param.Count - 1 do
            begin
              if Param[i].Value <> '' then CallList.Add('Param[' + IntToStr(i) +'].Value = ' + Param[i].Value)
              else CallList.Add('Param[' + IntToStr(i) +'].Value = nil');
              if Param[i].PType <> undefined then CallList.Add('Param[' + IntToStr(i) + '].PType = ' + IntToStr(Ord(Param[i].pType)))
              else CallList.Add('Param[' + IntToStr(i) + '].PType = undefined');
              if Param[i].Mult.Count > 0 then
                begin
                  for j := 0 to Param[i].Mult.Count - 1 do
                    begin
                      {Get Mult subscript}
                      MultSub   := Param[i].Mult.Subscript(j);
                      {Get Mult value}
                      MultValue := Param[i].Mult[MultSub];
                      CallList.Add('Param[' + IntToStr(i) + '].Mult[' + MultSub + '] = ' + MultValue);
                    end;
                end;
            end;
        except on exception do;
        end;
      end;
  finally
    StartDateTime:=FormatDateTime('mm/dd/yy hh:nn:ss',now)+' TimeLimit = '+IntToStr(RPCTimeLimit)+'sec.';

    if FLogCalls then
      begin
        FCallLog.Insert(0, 'Remote Procedure: '+RemoteProcedure);
        for I := 0 to CallList.Count-1 do
          FCallLog.Insert(0, CallList[I]);
        FCallLog.Insert(0, StartDateTime+' ---------------Start Call---------------');
      end;

    try
      inherited lstCall(OutputBuffer);
      if RPCBError <> '' then
        begin
          FCallLog.Insert(0, 'ERROR: '+RPCBError);
          FCallLog.Insert(0, FormatDateTime('mm/dd/yy hh:nn:ss',now)+' TimeLimit = '+IntToStr(RPCTimeLimit)+'sec. ---------------End Call---------------');
          FCallLog.Insert(0, '');
          if FileExists(FVistaLogName) = TRUE then FCallLog.SaveToFile(FVistaLogName);   {!!1.2}
        end;
    except on E: Exception do
      begin
        if FLogCalls then
          begin
            if RPCBError <> '' then
            begin
              FCallLog.Insert(0, 'ERROR: '+RPCBError);  {!!2.4}
            end;
            FCallLog.Insert(0, 'EXCEPTION: '+E.message);
            FCallLog.Insert(0, FormatDateTime('mm/dd/yy hh:nn:ss',now)+' TimeLimit = '+IntToStr(RPCTimeLimit)+'sec. ---------------End Call---------------');
            FCallLog.Insert(0, '');
            if FileExists(FVistaLogName) = TRUE then FCallLog.SaveToFile(FVistaLogName); {!!1.2}
          end;
        Raise;
      end;
    end;

    EndDateTime:=FormatDateTime('mm/dd/yy hh:nn:ss',now)+' TimeLimit = '+IntToStr(RPCTimeLimit)+'sec.';
    CallList.Insert(0,'  End Date/Time: '+EndDateTime);
    CallList.Insert(0,'  Start Date/Time: '+StartDateTime);
    CallList.Insert(0,'Remote Procedure: '+RemoteProcedure);
    CallList.Insert(0,'---------------Start Call------------------------------------------');
    CallList.Add('');

    if OutputBuffer.Count>0 then
      begin
        try
          for I:=0 to OutputBuffer.Count-1 do
            begin
              if FLogCalls then
                FCallLog.Insert(0, 'Result '+IntToStr(I)+' = '+OutputBuffer[I]);
              CallList.Add('Result '+IntToStr(I)+' = '+OutputBuffer[I]);
            end;
        except on exception do;
        end;
      end
    else
      begin
        if FLogCalls then
          FCallLog.Insert(0, 'No results returned.');
        CallList.Add('No results returned.');
      end;

    if FLogCalls then
      begin
        FCallLog.Insert(0, EndDateTime+' ---------------End Call---------------');
        FCallLog.Insert(0, '');
        if FCallLog.Count>LOG_SIZE then               {!!2.2}
        for I := FCallLog.Count-1 downto LOG_SIZE do  {!!2.2}
          FCallLog.Delete(I);
        if FileExists(FVistaLogName)=TRUE then FCallLog.SaveToFile(FVistaLogName);       {!!1.2}
      end;

    CallList.Add('---------------End Call--------------------------------------------');

    for I:=CallList.Count-1 downto 0 do
      begin
        FCallHistoryList.Insert(0,CallList.Strings[I]);
      end;
    CallList.Free;
    if FCallHistoryList.Count > 1500 then                  {!!1.1}
      begin
        for I:=FCallHistoryList.Count - 1 downto 1500 do   {!!1.1}
          begin
            FCallHistoryList.Delete(I);
          end;
      end;
  end;                                           { END FINALLY }
end;

procedure TDSSRPCBroker.Call;
var
  CallList:TStringList;
  i, j: Integer;
  StartDateTime,EndDateTime,MultSub, MultValue: String;
begin
  CallList := TStringList.Create;
  try
    if Param.Count>0 then
      begin
        try
          for i := 0 to Param.Count - 1 do
            begin
              if Param[i].Value <> '' then CallList.Add('Param[' + IntToStr(i) +'].Value = ' + Param[i].Value)
              else CallList.Add('Param[' + IntToStr(i) +'].Value = nil');
              if Param[i].PType <> undefined then CallList.Add('Param[' + IntToStr(i) + '].PType = ' + IntToStr(Ord(Param[i].pType)))
              else CallList.Add('Param[' + IntToStr(i) + '].PType = undefined');
              if Param[i].Mult.Count > 0 then
                begin
                  for j := 0 to Param[i].Mult.Count - 1 do
                    begin
                      {Get Mult subscript}
                      MultSub   := Param[i].Mult.Subscript(j);
                      {Get Mult value}
                      MultValue := Param[i].Mult[MultSub];
                      CallList.Add('Param[' + IntToStr(i) + '].Mult[' + MultSub + '] = ' + MultValue);
                    end;
                end;
            end;
        except on exception do;
        end;
      end;
  finally
    StartDateTime:=FormatDateTime('mm/dd/yy hh:nn:ss',now)+'  TimeLimit = '+IntToStr(RPCTimeLimit)+'sec.';

    if FLogCalls then
      begin
        FCallLog.Insert(0, 'Remote Procedure: '+RemoteProcedure);
        for I := 0 to CallList.Count-1 do
          FCallLog.Insert(0, CallList[I]);
        FCallLog.Insert(0, StartDateTime+' ---------------Start Call---------------');
      end;

    try
      inherited Call;
      if RPCBError <> '' then
        begin
          FCallLog.Insert(0, 'ERROR: '+RPCBError);
          FCallLog.Insert(0, FormatDateTime('mm/dd/yy hh:nn:ss',now)+'  TimeLimit = '+IntToStr(RPCTimeLimit)+'sec. ---------------End Call---------------');
          FCallLog.Insert(0, '');
          if FLogCalls then
            begin
              if FileExists(FVistaLogName) = TRUE then FCallLog.SaveToFile(FVistaLogName); {!!1.2}
            end;
        end;
    except on E: Exception do
      begin
        if FLogCalls then
          begin
            if RPCBError <> '' then
            begin
              FCallLog.Insert(0, 'ERROR: '+RPCBError);     {!!2.4}
            end;
            FCallLog.Insert(0, 'EXCEPTION: '+E.message);
            FCallLog.Insert(0, FormatDateTime('mm/dd/yy hh:nn:ss',now)+' TimeLimit = '+IntToStr(RPCTimeLimit)+'sec. ---------------End Call---------------');
            FCallLog.Insert(0, '');
            if FileExists(FVistaLogName) = TRUE then FCallLog.SaveToFile(FVistaLogName); {!!1.2}
          end;
        Raise;
      end;
    end;

    EndDateTime:=FormatDateTime('mm/dd/yy hh:nn:ss',now)+'  TimeLimit = '+IntToStr(RPCTimeLimit)+'sec.';
    CallList.Insert(0,'  End Date/Time: '+EndDateTime);
    CallList.Insert(0,'  Start Date/Time: '+StartDateTime);
    CallList.Insert(0,'Remote Procedure: '+RemoteProcedure);
    CallList.Insert(0,'---------------Start Call------------------------------------------');
    CallList.Add('');
    if Results.Count>0 then
      begin
        try
          for I:=0 to Results.Count-1 do
            begin
              if FLogCalls then
                FCallLog.Insert(0, 'Result '+IntToStr(I)+' = '+Results[I]);
              CallList.Add('Result '+IntToStr(I)+' = '+Results[I]);
            end;
        except on exception do;
        end;
      end
    else
      begin
        if FLogCalls then
          FCallLog.Insert(0, 'No results returned.');
        CallList.Add('No results returned.');
      end;

    if FLogCalls then
      begin
        FCallLog.Insert(0, EndDateTime+' ---------------End Call---------------');
        FCallLog.Insert(0, '');
        if FCallLog.Count>LOG_SIZE then               {!!2.2}
        for I := FCallLog.Count-1 downto LOG_SIZE do  {!!2.2}
          FCallLog.Delete(I);
        if FileExists(FVistaLogName) = TRUE then FCallLog.SaveToFile(FVistaLogName);     {!!1.2}
      end;

    CallList.Add('---------------End Call--------------------------------------------');

    for I:=CallList.Count-1 downto 0 do
      begin
        FCallHistoryList.Insert(0,CallList.Strings[I]);
      end;
    CallList.Free;
    if FCallHistoryList.Count > 1500 then                  {!!1.1}
      begin
        for I := FCallHistoryList.Count - 1 downto 1500 do {!!1.1}
          begin
            FCallHistoryList.Delete(I);
          end;
      end;
  end;
end;

function TDSSRPCBroker.EvaluateForVEJDUserFile:Boolean;
var
  I:Integer;
begin
  RemoteProcedure := 'DSIC DDR FINDER';          {!!1.3 CONVERT FROM VEJD }
  Param[0].pType:=literal;
  Param[0].Value:='1';
  Param[1].pType:=literal;
  Param[1].Value:='';                            //For Mutliple fields within file
  Param[2].pType:=literal;
  Param[2].Value:='.01';
  Param[3].pType:=literal;
  Param[3].Value:='25';
  Param[4].pType:=literal;
  Param[4].Value:='19619.1';
  try
    Call;
    if Results.Count>1 then
      begin
        for I:=1 to Results.Count-1 do
          begin
            if Piece(Results[I],'^',1)='19619.1' then
              begin
                Result:=TRUE;
                Break;
              end;
          end;
      end
    else
      begin
        Result:=FALSE;
      end;
  except on Exception do
    begin
      Result:=FALSE;
    end;
  end;
end;

function TDSSRPCBroker.EvaluateForFileEntry(FileNumber,EntryName:String;var EntryIEN:String):Boolean;
var
  I:Integer;
begin
  EntryIEN := '';
  RemoteProcedure:='DSIC DDR FINDER';            {!!1.3 CONVERT FROM VEJD }
  Param[0].pType:=literal;
  Param[0].Value:=FileNumber;
  Param[1].pType:=literal;
  Param[1].Value:='';                            //For Mutliple fields within file
  Param[2].pType:=literal;
  Param[2].Value:='.01';
  Param[3].pType:=literal;
  Param[3].Value:='25';
  Param[4].pType:=literal;
  Param[4].Value:=UpperCase(EntryName);
  try
    Call;
    if Results.Count>1 then
      begin
        //IFN^.01Descr
        for I:=1 to Results.Count-1 do
          begin
            if UpperCase(EntryName)=UpperCase(Piece(Results[I],'^',2)) then
              begin
                EntryIEN:=Piece(Results[I],'^',1);
                Result:=TRUE;
                Break;
              end;
          end;
      end
    else
      begin
        EntryIEN:='';
        Result:=FALSE;
      end;
  except on Exception do
    begin
      EntryIEN:='';
      Result:=FALSE;
    end;
  end;
end;

function TDSSRPCBroker.GetVEJDSystemUserInfo(sUserIFN,FieldNumber:String):String;
begin
  RemoteProcedure := 'DSIC DDR GETS ENTRY DATA'; {!!1.3 CONVERT FROM VEJD }
  Param[0].pType := literal;
  Param[0].Value := '19619.1';
  Param[1].pType := literal;
  Param[1].Value := sUserIFN;                    //Which entry do we want?
  Param[2].pType := literal;
  Param[2].Value := FieldNumber;
  Param[3].pType := literal;
  Param[3].Value := 'E';                         //External format
  try
    Call;
    if Results.Count>0 then
      begin
        if Piece(Results[0],'^',1)<>'-1' then Result := Piece(Results[0],'^',4) {!!1.2}
        else Result := '';
      end
    else
      begin
        Result:='';
      end;
  except on Exception do Result:='';
  end;
end;

procedure TDSSRPCBroker.GetLocationName;
begin
  if DefaultLocationIFN <> '' then
    begin
      RemoteProcedure := 'DSIC DDR GETS ENTRY DATA';  {!!1.3 CONVERT FROM VEJD }
      Param[0].pType := literal;
      Param[0].Value := '44';                    //FileNumber
      Param[1].pType := literal;
      Param[1].Value := DefaultLocationIFN;      //Which entry do we want?
      Param[2].pType := literal;
      Param[2].Value := '.01';
      Param[3].pType := literal;
      Param[3].Value := 'E';                     //External format
      try
        Call;
        if Results.Count>0 then
          begin
            if Piece(Results[0],'^',1) <> '-1' then DefaultLocationName := Piece(Results[0],'^',4) {!!1.2}
            else DefaultLocationName := '';
          end
        else
          begin
            DefaultLocationName:='';
          end;
      except on Exception do DefaultLocationName:='';
      end;
    end
  else
    begin
      DefaultLocationName := '';
    end;
end;

function TDSSRPCBroker.AddEntryToVEJDSystemUser(ArrayValues:TStringList):String;
var
  I : Integer;
begin
  RemoteProcedure := 'DSIC DDR UPDATE FILE';     {!!1.3 CONVERT FROM VEJD }
  Param[0].pType := literal;
  Param[0].Value := '19619.1';
  Param[1].pType := list;
  for I:=0 to ArrayValues.Count-1 do
    Param[1].Mult['"'+Piece(ArrayValues.Strings[I],'^',1)+'"']:=Piece(ArrayValues.Strings[I],'^',2);
  try
    Call;
    if Results.Count>0 then
      begin
        if Piece(Results[0],'^',1)='-1' then
          begin
            Result:='';
          end
        else
          begin
            Result := Piece(Results[0],'^',2);
          end;
      end
    else
      begin
        Result:='';
      end;
  except on Exception do Result:='';
  end;
end;

procedure TDSSRPCBroker.GetVISTAUserLocation(sUserIFN:String);
begin
  if MenuContext='VEJDWPB CORE RPCS' then
    begin                                        //TIU GET PERSONAL PREFERENCES
      RemoteProcedure := 'TIU GET PERSONAL PREFERENCES';
      Param[0].pType := literal;
      Param[0].Value := sUserIFN;
      try
        Call;
        if Results.Count>0 then
          begin
            DefaultLocationIFN:=Piece(Results[0],'^',2);
            if DefaultLocationIFN<>'' then
              begin
                GetLocationName;
              end
            else
              begin
                DefaultLocationIFN:='';
                DefaultLocationName:='';
              end;
          end
        else
          begin
            DefaultLocationIFN:='';
            DefaultLocationName:='';
          end;
      except on Exception do
        begin
          DefaultLocationIFN:='';
          DefaultLocationName:='';
        end;
      end;
    end
  else
    begin
      DefaultLocationIFN:='';
      DefaultLocationName:='';
    end;
end;

procedure TDSSRPCBroker.GetVISTAUserName(sUserIFN:String);
begin
  RemoteProcedure := 'DSIC DDR GETS ENTRY DATA'; {!!1.3 CONVERT FROM VEJD }
  Param[0].pType := literal;
  Param[0].Value := '200';
  Param[1].pType := literal;
  Param[1].Value := sUserIFN;                    //Which entry do we want?
  Param[2].pType := literal;
  Param[2].Value := '.01';
  Param[3].pType := literal;
  Param[3].Value := 'E';                         //External format
  try
    Call;
    if Results.Count > 0 then
      begin
        if Piece(Results[0],'^',1) <> '-1' then UserName := Piece(Results[0],'^',4) {!!1.2}
        else UserName := '';
        if UserName = '' then UserName := 'Unknown User';
      end
    else
      begin
        UserName := 'Unknown User';
      end;
  except on Exception do UserName := 'Unknown User';
  end;
end;

procedure TDSSRPCBroker.GetStationName(sStationIFN:String);
begin
  RemoteProcedure := 'DSIC DDR GETS ENTRY DATA'; {!!1.3 CONVERT FROM VEJD }
  Param[0].pType := literal;
  Param[0].Value := '4';//
  Param[1].pType := literal;
  Param[1].Value := sStationIFN;                 //Which entry do we want?
  Param[2].pType := literal;
  Param[2].Value := '.01';
  Param[3].pType := literal;
  Param[3].Value := 'E';                         //External format
  try
    Call;
    if Results.Count>0 then
      begin
        if Piece(Results[0],'^',1) <> '-1' then StationName := Piece(Results[0],'^',4)   {!!1.2}
        else StationName := '';
        if StationName = '' then StationName := 'Unknown Station';
      end
    else
      begin
        StationName := 'Unknown Station';
      end;
  except on Exception do StationName := 'Unknown Station';
  end;
end;

procedure TDSSRPCBroker.PUseCurrentServerNamePort(const Value: boolean);
begin
  FUseCurrentServerNamePort := Value;
end;

procedure TDSSRPCBroker.PStationIFN(const Value: string);
begin
  FStationIFN := Value;
end;

procedure TDSSRPCBroker.PStationName(const Value: string);
begin
  FStationName := Value;
end;

procedure TDSSRPCBroker.PNTUserName(const Value: string);
begin
  FNTUserName := Value;
end;

procedure TDSSRPCBroker.PVEJDSystemUserDUZ(const Value: string);
begin
  FVEJDSystemUserDUZ := Value;
end;

procedure TDSSRPCBroker.PUserName(const Value: string);
begin
  FUserName := Value;
end;

procedure TDSSRPCBroker.PDNSUserName(const Value: string);
begin
  FDNSUserName := Value;
end;

procedure TDSSRPCBroker.PUserDuz(const Value: string);
begin
  FUserDuz := Value;
end;

procedure TDSSRPCBroker.PPatientIFN(const Value: string);
begin
  FPatientIFN := Value;
end;

procedure TDSSRPCBroker.PDefaultLocationIFN(const Value: string);
begin
  FDefaultLocationIFN := Value;
end;

procedure TDSSRPCBroker.PDefaultLocationName(const Value: string);
begin
  FDefaultLocationName := Value;
end;

procedure TDSSRPCBroker.PMenuContext(const Value: string);
begin
  FMenuContext := Value;
end;

function TDSSRPCBroker.ConnectToBroker(ServerName,Port:String):Boolean;
var
  EntryIFN,sNTUserName:String;
  ArrayValues:TStringList;
begin
  VEJDSystemUserDUZ:='';
  DNSUserName:='';
  UserDuz:='';
  StationIFN:='';
  StationName:='';
  Server:=ServerName;
  ListenerPort:=StrToIntDef(Port,9200);
  //ShowMessage(Server+#13+IntToStr(ListenerPort));
  try
    //ShowMessage('Connecting.');
    Connected := TRUE;
    //ShowMessage('Connected.');
    if Socket > 0 then
      begin
        //ShowMessage('Socket > 0 ');
        try
          if CreateContext(MenuContext)=FALSE then
            begin
              Connected := FALSE;//Per discussion with SD - Current Connection could disconnect from other apps like CPRS????
              FLastErrorMessage := 'You are not a vaild user of the option "'+MenuContext+'".';
              if ShowErrorDialogs then
                MessageDlg('You are not a vaild user of the option "'+MenuContext+'".',mtinformation,[mbok],0);
              Result := FALSE;
            end
          else
            begin
              Param[0].PType:=reference;
              Param[0].Value:='DUZ';
              RemoteProcedure:='XWB GET VARIABLE VALUE';
              Call;
              if Results.Count>0 then UserDuz:=Results[0];
              Param[0].PType:=reference;
              Param[0].Value:='DUZ(2)';
              RemoteProcedure:='XWB GET VARIABLE VALUE';
              Call;
              if Results.Count>0 then StationIFN:=Results[0];
              if StationIFN <> '' then GetStationName(StationIFN);
              if UserDUZ <> '' then GetVISTAUserName(UserDUZ);
              if UserDUZ<>'' then GetVISTAUserLocation(UserDUZ);
              sNTUserName:=NTGetUserName;
              //ShowMessage(sNTUserName);
              //FM File Evaluation No Longer Needed??
              if EvaluateForVEJDUserFile=TRUE then
                begin//System User File Present
                  EntryIFN:='';
                  if EvaluateForFileEntry('19619.1',UserName,EntryIFN) = TRUE then
                    begin
                      //System User Entry Present -- Shall we update the NT User Name??
                      VEJDSystemUserDUZ:=EntryIFN;
                      DNSUserName:=GetVEJDSystemUserInfo(EntryIFN,'1');
                      RemoteProcedure := 'DSIC DDR FILER'; {!!1.3 CONVERT FROM VEJD }
                      Param[0].pType := literal;
                      Param[0].Value := '19619.1';
                      Param[1].pType := literal;
                      Param[1].Value := EntryIFN+',';
                      Param[2].pType := literal;
                      Param[2].Value := '';
                      Param[3].pType := list;
                      Param[3].Mult['"2"']:=sNTUserName;
                      try
                        Call;
                      except on exception do;
                      end;
                    end
                  else
                    begin                        //System User Entry Not Present -- Make One
                      EntryIFN:='';
                      ArrayValues:=TStringList.Create;
                      ArrayValues.Add('.01^'+UserDUZ);
                      ArrayValues.Add('2^'+sNTUserName);
                      try
                        EntryIFN:=AddEntryToVEJDSystemUser(ArrayValues);
                      finally
                        if EntryIFN<>'' then
                          begin
                            VEJDSystemUserDUZ:=EntryIFN;
                            DNSUserName:='';
                          end
                        else
                          begin
                            VEJDSystemUserDUZ:='';
                            DNSUserName:='';
                          end;
                        ArrayValues.Free;
                      end;
                    end;
                end
              else
                begin                            //System User File Not Present
                  VEJDSystemUserDUZ:='';
                  DNSUserName:='';
                end;
              Result:=TRUE;
            end;
          except on E:Exception do
            begin
              FLastErrorMessage := E.Message;
              if ShowErrorDialogs then
                MessageDlg(E.Message,mtinformation,[mbok],0);
              Connected := FALSE;
              Result := FALSE;
            end;
          end;
      end
    else
      begin
        //ShowMessage('Socket = 0 ');
        //ShowMessage('Invalid Log On.');
        Connected := FALSE;
        Result := FALSE;
      end;
  except on Exception do
    begin
      //ShowMessage(E.Message);
      //MessageDlg('Connection to the server could not be established.',mterror,[mbok],0);
      Connected := FALSE;
      Result := FALSE;
    end;
  end;
end;

function TDSSRPCBroker.Execute:Boolean;
var
  PortString,ServerString:String;
  J:Integer;
begin
  VEJDSystemUserDUZ:='';
  DNSUserName:='';
  ServerString:='';
  PortString:='0';
  PatientIFN:='';
  if FUseCurrentServerNamePort=TRUE then
    begin
      if Server='' then Server:='BROKERSERVER';
      if ListenerPort<1 then ListenerPort:=9200;
    end
  else
    begin
      if ParamCount > 0 then
        begin
          for J:=0 to ParamCount do
            begin
              if UpperCase(Piece(ParamStr(J),'=',1))='S' then ServerString:=UpperCase(Piece(ParamStr(J),'=',2));
              if UpperCase(Piece(ParamStr(J),'=',1))='P' then PortString:=UpperCase(Piece(ParamStr(J),'=',2));
              if UpperCase(Piece(ParamStr(J),'=',1))='D' then PatientIFN:=UpperCase(Piece(ParamStr(J),'=',2));
            end;
        end;
      if PortString <> '' then ListenerPort := StrToInt(PortString);
      Server := ServerString;
    end;
  if MenuContext = '' then MenuContext := 'VEJDWPB CORE RPCS';
  if (Server<>'') and (ListenerPort>1) then
    begin
      if ConnectToBroker(Server,IntToStr(ListenerPort))=TRUE then
        begin
          Result := TRUE;
        end
      else
        begin
          DefaultLocationIFN:='';
          DefaultLocationName:='';
          VEJDSystemUserDUZ:='';
          DNSUserName:='';
          UserDuz:='';
          UserName:='';
          StationIFN:='';
          StationName:='';
          Result:=FALSE;
        end;
    end
  else
    begin
      //Set Defaults
      Server:='BROKERSERVER';
      ListenerPort:=9200;
      try
        if RPCConf1.GetServerInfo(ServerString,PortString) <> 1 then
          begin
            DefaultLocationIFN:='';
            DefaultLocationName:='';
            VEJDSystemUserDUZ:='';
            DNSUserName:='';
            UserDuz:='';
            UserName:='';
            StationIFN:='';
            StationName:='';
            Result:=FALSE;
          end
        else
          begin
            if ConnectToBroker(ServerString,PortString)=TRUE then
              begin
                Result := TRUE
              end
            else
              begin
                DefaultLocationIFN:='';
                DefaultLocationName:='';
                VEJDSystemUserDUZ:='';
                DNSUserName:='';
                UserDuz:='';
                UserName:='';
                StationIFN:='';
                StationName:='';
                Result:=FALSE;
              end;
          end;
      except on exception do
        begin
          DefaultLocationIFN := '';
          DefaultLocationName := '';
          VEJDSystemUserDUZ := '';
          DNSUserName := '';
          UserDuz := '';
          UserName := '';
          StationIFN := '';
          StationName := '';
          Result := FALSE;
        end;
      end;
    end;
end;

procedure Register;
begin
  RegisterComponents('DSSVista', [TDSSRPCBroker]);
end;

procedure TDSSRPCBroker.LogThis(LogEntry: WideString);
{!!2.3 }
var
  I: Integer;
begin
  if FLogCalls then
  begin
    try
      FCallLog.Insert(0, LogEntry);
      FCallLog.Insert(0, '');
      if FCallLog.Count>LOG_SIZE then
      for I := FCallLog.Count-1 downto LOG_SIZE do
        FCallLog.Delete(I);
      if FileExists(FVistaLogName)=TRUE then FCallLog.SaveToFile(FVistaLogName);
    except on exception do //Do nothing except prevent application crash
    end;
  end;
end;

procedure TDSSRPCBroker.LogVistaCalls(VistaLogName: TFileName);
var
  FileHandle: Integer;
begin
  FCallLog.Clear;

  if VistaLogName = '' then
    begin
      FLogCalls := FALSE;
      Exit;
    end;

  if FileExists(VistaLogName) then
    begin
      FVistaLogName := VistaLogName;
      if FileExists(FVistaLogName) = TRUE then FCallLog.LoadFromFile(FVistaLogName);  {!!1.2}
      FLogCalls := TRUE;
    end
  else
    begin
      FileHandle := FileCreate(PChar(VistaLogName));
      if FileHandle > 0 then
        begin
          FileClose(FileHandle);
          FVistaLogName := VistaLogName;
          if FileExists(FVistaLogName) = TRUE then FCallLog.LoadFromFile(FVistaLogName); {!!1.2}
          FLogCalls := TRUE;
        end
      else
        begin
          FLastErrorMessage := 'Invalid Vista Calls Log name: '+#13+
              VistaLogName+#13+'Vista calls are NOT being logged.';
          if ShowErrorDialogs then
            MessageDlg('Invalid Vista Calls Log name: '+#13+
                VistaLogName+#13+
               'Vista calls are NOT being logged.', mtinformation, [mbok], 0);
          FLogCalls := FALSE;
        end;
    end;
end;

procedure TDSSRPCBroker.CallV(const RPCName: string; const AParam: array of const);
var
  SavedCursor: TCursor;
begin
  SavedCursor := Screen.Cursor;
  Screen.Cursor := crHourGlass;
  try
    SetParams(RPCName, AParam);
    CallBroker;  //RPCBrokerV.Call;
  finally
    Screen.Cursor := SavedCursor;
  end;
end;

function TDSSRPCBroker.sCallV(const RPCName: string; const AParam: array of const): string;
{ Calls the broker and returns a scalar value - the first row of the results. }
var
  SavedCursor: TCursor;
begin
  SavedCursor := Screen.Cursor;
  Screen.Cursor := crHourGlass;
  try
    SetParams(RPCName, AParam);
    CallBroker;  
    if self.Results.Count > 0 then
      Result := self.Results[0]
    else
      Result := '';
  finally
    Screen.Cursor := SavedCursor;
  end;
end;

procedure TDSSRPCBroker.tCallV(ReturnData: TStrings; const RPCName: string;
  const AParam: array of const);
{ calls the broker and returns TStrings data - the whole set of results }
var
  SavedCursor: TCursor;
begin
  if ReturnData = nil then
    raise Exception.Create('DSSRPCBroker.TCAllV:TStrings not created before calling tCallV');
  SavedCursor := Screen.Cursor;
  try
    Screen.Cursor := crHourGlass;
    SetParams(RPCName, AParam);
    CallBroker;  //RPCBrokerV.Call;
    ReturnData.Assign(self.Results);
  finally
    Screen.Cursor := SavedCursor;
  end;
end;

procedure TDSSRPCBroker.SetParams(const RPCName: string; const AParam: array of const);
const
  BoolChar: array[boolean] of char = ('0', '1');
var
  i: integer;
  TmpExt: Extended;
begin
  if Length(RPCName) = 0 then raise Exception.Create('No RPC Name');
  with self do
    begin
      ClearParameters := TRUE;
      RemoteProcedure := RPCName;
      for i := 0 to High(AParam) do with AParam[i] do
        begin
          Param[i].PType := literal;
          case VType of
          vtInteger:    Param[i].Value := IntToStr(VInteger);
          vtBoolean:    Param[i].Value := BoolChar[VBoolean];
          vtChar:       Param[i].Value := VChar;
          vtWideChar:   Param[i].Value := VWideChar;       {!!2.41}
          //vtExtended:   Param[i].Value := FloatToStr(VExtended^);
          vtExtended:   begin
                          TmpExt := VExtended^;
                          if(abs(TmpExt) < 0.0000000000001) then TmpExt := 0;
                          Param[i].Value := FloatToStr(TmpExt);
                        end;
          vtString:     with Param[i] do
                        begin
                          Value := VString^;
                          if (Length(Value) > 0) and (Value[1] = #1) then
                          begin
                            Value := Copy(Value, 2, Length(Value));
                            PType := reference;
                          end;
                        end;
          vtPChar:      Param[i].Value := StrPas(VPChar);
          vtPointer:    if VPointer = nil
                          then ClearParameters := TRUE {Param[i].PType := null}
                          else raise Exception.Create('Pointer type must be nil.');
          vtObject:     if VObject is TStrings then
                            SetList(TStrings(VObject), i);
          vtAnsiString: with Param[i] do
                        begin
                          Value := string(VAnsiString);
                          if (Length(Value) > 0) and (Value[1] = #1) then
                          begin
                            Value := Copy(Value, 2, Length(Value));
                            PType := reference;
                          end;
                        end;
          vtInt64:      Param[i].Value := IntToStr(VInt64^);
{          
          vtUnicodeString: with Param[i] do
                        begin
                          Value := string(VUnicodeString);
                          if (Length(Value) > 0 )and (Value[1] = #1) then
                          begin
                            Value := Copy(Value, 2, Length(Value));
                            PType := reference;
                          end;
                        end;
}
            else raise Exception.Create('Unable to pass parameter type to Broker.');
        end;                                     {case}
      end;                                       {for}
    end;                                         {with}
  //RPCLastCall := RPCName + ' (SetParam end)';
end;

procedure TDSSRPCBroker.CallBroker;
begin
//  UpdateContext(uBaseContext);
  CallBrokerInContext;
end;

procedure TDSSRPCBroker.CallBrokerInContext;
var
  x : string;
begin
  with self do if not Connected then  // happens if broker connection is lost
    begin
      ClearResults := TRUE;
      Exit;
    end;
  //RPCBrokerV.Call;
  try
    self.Call;
  except
    // The broker erroneously sets connected to FALSE if there is any error (including an
    // error on the M side). It should only set connection to FALSE if there is no connection.
    on E:EBrokerError do
    begin
      if E.Code = XWB_M_REJECT then
        begin
          x := 'An error occurred on the server.' + CRLF + CRLF + E.Action;
          Application.MessageBox(PChar(x), 'Server Error', MB_OK);
        end
      else raise;
    (*
      case E.Code of
      XWB_M_REJECT:  begin
                       x := 'An error occurred on the server.' + CRLF + CRLF + E.Action;
                       Application.MessageBox(PChar(x), 'Server Error', MB_OK);
                     end;
      else           begin
                       x := 'An error occurred with the network connection.' + CRLF +
                            'Action was: ' + E.Action + CRLF + 'Code was: ' + E.Mnemonic +
                            CRLF + CRLF + 'Application cannot continue.';
                       Application.MessageBox(PChar(x), 'Network Error', MB_OK);
                     end;
      end;
      *)
    end;
  end;
end;

procedure TDSSRPCBroker.SetList(AStringList: TStrings; ParamIndex: Integer);
{ places TStrings into RPCBrokerV.Mult[n], where n is a 1-based (not 0-based) index }
var
  i: Integer;
begin
  with self.Param[ParamIndex] do
    begin
      PType := list;
      with AStringList do
        begin
          for i := 0 to Count - 1 do
            Mult[IntToStr(i+1)] := Strings[i];
        end;
    end;
end;

end.
