/*
 * Copyright (c) Mirth Corporation. All rights reserved.
 * http://www.mirthcorp.com
 *
 * The software in this package is published under the terms of the MPL
 * license a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package com.mirth.connect.manager;

import java.awt.Cursor;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.HashMap;
import java.util.Map;

import javax.swing.DefaultComboBoxModel;

import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.lang.ArrayUtils;

import com.mirth.connect.manager.components.MirthFieldConstraints;

public class ManagerDialog extends javax.swing.JDialog {

    private ServiceController serviceController = null;
    private ManagerController managerController = null;
    private boolean loading = false;
    private String lastSelectedDatabaseType;
    private Map<String, String> databaseUrls = new HashMap<String, String>();

    public ManagerDialog() {
        try {
            serviceController = ServiceControllerFactory.getServiceController();
            managerController = ManagerController.getInstance();
        } catch (Exception e) {
            e.printStackTrace();
            System.exit(1);
        }
    }

    public void setupDialog() {
        initComponents();

        databaseUrls.put("derby", "jdbc:derby:${dir.base}/mirthdb;create=true");
        databaseUrls.put("postgres", "jdbc:postgresql://localhost:5432/mirthdb");
        databaseUrls.put("mysql", "jdbc:mysql://localhost:3306/mirthdb");
        databaseUrls.put("oracle", "jdbc:oracle:thin:@localhost:1521:DB");
        databaseUrls.put("sqlserver2000", "jdbc:jtds:sqlserver://localhost:1433/mirthdb");
        databaseUrls.put("sqlserver", "jdbc:jtds:sqlserver://localhost:1433/mirthdb");

        // Remove the service tab if it's not supported
        if (!serviceController.isShowServiceTab()) {
            tabPanel.removeTabAt(0);
        }

        pack();
        setLocationRelativeTo(null);
        setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
        addWindowListener(new WindowAdapter() {

            public void windowClosing(WindowEvent e) {
                close();
            }
        });

        // listen for trigger button and double click to edit channel.
        serverLogFiles.addMouseListener(new java.awt.event.MouseAdapter() {

            public void mousePressed(java.awt.event.MouseEvent evt) {
            }

            public void mouseReleased(java.awt.event.MouseEvent evt) {
            }

            public void mouseClicked(java.awt.event.MouseEvent evt) {
                if (evt.getClickCount() >= 2) {
                    if (serverLogFiles.getSelectedIndex() != -1) {
                        viewFileButtonActionPerformed(null);
                    }
                }
            }
        });

        mirthSupportLink.setToolTipText("Visit Mirth Corporation's website.");
        mirthSupportLink.setCursor(new Cursor(Cursor.HAND_CURSOR));

        serverMemoryField.setDocument(new MirthFieldConstraints(0, false, false, true));

        loadProperties();
    }

    public void open() {
        managerController.updateMirthServiceStatus();
        loadProperties();
        if (serviceController.isStartupPossible()) {
            startup.setEnabled(true);

            if (serviceController.isStartup()) {
                startup.setSelected(true);
            } else {
                startup.setSelected(false);
            }
        } else {
            startup.setEnabled(false);
        }
        setVisible(true);
    }

    public void close() {
        setVisible(false);

        // if there is no tray icon, shutdown and exit the manager
        if (!serviceController.isShowTrayIcon()) {
            Manager.shutdown();
        }
    }

    public void setApplyEnabled(boolean enabled) {
        applyButton.setEnabled(enabled);
    }

    public boolean isApplyEnabled() {
        return applyButton.isEnabled();
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc=" Generated Code
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        jPanel3 = new javax.swing.JPanel();
        heading = new com.mirth.connect.manager.MirthHeadingPanel();
        headingLabel = new javax.swing.JLabel();
        tabPanel = new javax.swing.JTabbedPane();
        servicePanel = new javax.swing.JPanel();
        serviceButtonContainer = new javax.swing.JPanel();
        jLabel11 = new javax.swing.JLabel();
        startButton = new javax.swing.JButton();
        stopButton = new javax.swing.JButton();
        jLabel12 = new javax.swing.JLabel();
        jLabel13 = new javax.swing.JLabel();
        restartButton = new javax.swing.JButton();
        startup = new javax.swing.JCheckBox();
        refreshServiceButton = new javax.swing.JButton();
        jLabel20 = new javax.swing.JLabel();
        serverPanel = new javax.swing.JPanel();
        serverWebstartPort = new com.mirth.connect.manager.components.MirthTextField();
        serverAdministratorPort = new com.mirth.connect.manager.components.MirthTextField();
        jLabel3 = new javax.swing.JLabel();
        jLabel1 = new javax.swing.JLabel();
        jScrollPane1 = new javax.swing.JScrollPane();
        serverLogFiles = new javax.swing.JList();
        jLabel4 = new javax.swing.JLabel();
        viewFileButton = new javax.swing.JButton();
        refreshButton = new javax.swing.JButton();
        jLabel5 = new javax.swing.JLabel();
        jLabel6 = new javax.swing.JLabel();
        mirthLogLevel = new com.mirth.connect.manager.components.MirthComboBox();
        databaseLogLevel = new com.mirth.connect.manager.components.MirthComboBox();
        jLabel18 = new javax.swing.JLabel();
        serverJmxPort = new com.mirth.connect.manager.components.MirthTextField();
        serverMemoryLabel = new javax.swing.JLabel();
        serverMemoryField = new com.mirth.connect.manager.components.MirthTextField();
        channelLogLevel = new com.mirth.connect.manager.components.MirthComboBox();
        jLabel19 = new javax.swing.JLabel();
        databasePanel = new javax.swing.JPanel();
        jLabel7 = new javax.swing.JLabel();
        jLabel8 = new javax.swing.JLabel();
        databaseUrl = new com.mirth.connect.manager.components.MirthTextField();
        databaseType = new com.mirth.connect.manager.components.MirthComboBox();
        jLabel9 = new javax.swing.JLabel();
        jLabel10 = new javax.swing.JLabel();
        databaseUsername = new com.mirth.connect.manager.components.MirthTextField();
        databasePassword = new com.mirth.connect.manager.components.MirthPasswordField();
        infoPanel = new javax.swing.JPanel();
        jLabel2 = new javax.swing.JLabel();
        version = new javax.swing.JLabel();
        jLabel15 = new javax.swing.JLabel();
        jLabel16 = new javax.swing.JLabel();
        serverId = new javax.swing.JLabel();
        javaVersion = new javax.swing.JLabel();
        jLabel14 = new javax.swing.JLabel();
        mirthSupportLink = new javax.swing.JLabel();
        jLabel17 = new javax.swing.JLabel();
        applyButton = new javax.swing.JButton();
        launchButton = new javax.swing.JButton();
        cancelButton = new javax.swing.JButton();
        okButton = new javax.swing.JButton();

        setTitle("Mirth Connect Server Manager");
        setResizable(false);

        jPanel3.setBackground(new java.awt.Color(255, 255, 255));

        headingLabel.setFont(new java.awt.Font("Tahoma", 1, 18));
        headingLabel.setForeground(new java.awt.Color(255, 255, 255));
        headingLabel.setText("Mirth Connect Server Manager");

        javax.swing.GroupLayout headingLayout = new javax.swing.GroupLayout(heading);
        heading.setLayout(headingLayout);
        headingLayout.setHorizontalGroup(
            headingLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(headingLayout.createSequentialGroup()
                .addContainerGap()
                .addComponent(headingLabel, javax.swing.GroupLayout.DEFAULT_SIZE, 382, Short.MAX_VALUE)
                .addContainerGap())
        );
        headingLayout.setVerticalGroup(
            headingLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, headingLayout.createSequentialGroup()
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                .addComponent(headingLabel, javax.swing.GroupLayout.PREFERRED_SIZE, 29, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addContainerGap())
        );

        tabPanel.setBackground(new java.awt.Color(255, 255, 255));
        tabPanel.setBorder(javax.swing.BorderFactory.createEmptyBorder(0, 0, 0, 0));
        tabPanel.setFocusable(false);

        servicePanel.setBackground(new java.awt.Color(255, 255, 255));
        servicePanel.setBorder(javax.swing.BorderFactory.createEmptyBorder(0, 0, 0, 0));
        servicePanel.setFocusable(false);

        serviceButtonContainer.setBackground(new java.awt.Color(255, 255, 255));

        jLabel11.setText("Starts the Mirth Connect service");

        startButton.setText("Start");
        startButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                startButtonActionPerformed(evt);
            }
        });

        stopButton.setText("Stop");
        stopButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                stopButtonActionPerformed(evt);
            }
        });

        jLabel12.setText("Stops the Mirth Connect service");

        jLabel13.setText("Restarts the Mirth Connect service");

        restartButton.setText("Restart");
        restartButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                restartButtonActionPerformed(evt);
            }
        });

        startup.setBackground(new java.awt.Color(255, 255, 255));
        startup.setText("Start Mirth Connect Server Manager on system startup");
        startup.setToolTipText("Starts this application when logging into the operating system. Currently only enabled for Windows.");
        startup.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                startupActionPerformed(evt);
            }
        });

        refreshServiceButton.setText("Refresh");
        refreshServiceButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                refreshServiceButtonActionPerformed(evt);
            }
        });

        jLabel20.setText("Refreshes the Mirth Connect service status");

        javax.swing.GroupLayout serviceButtonContainerLayout = new javax.swing.GroupLayout(serviceButtonContainer);
        serviceButtonContainer.setLayout(serviceButtonContainerLayout);
        serviceButtonContainerLayout.setHorizontalGroup(
            serviceButtonContainerLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(serviceButtonContainerLayout.createSequentialGroup()
                .addGroup(serviceButtonContainerLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(serviceButtonContainerLayout.createSequentialGroup()
                        .addContainerGap()
                        .addGroup(serviceButtonContainerLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addComponent(startButton)
                            .addComponent(restartButton)
                            .addComponent(refreshServiceButton)
                            .addComponent(stopButton, javax.swing.GroupLayout.PREFERRED_SIZE, 57, javax.swing.GroupLayout.PREFERRED_SIZE))
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                        .addGroup(serviceButtonContainerLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addComponent(jLabel12)
                            .addComponent(jLabel20)
                            .addComponent(jLabel13)
                            .addComponent(jLabel11)))
                    .addGroup(serviceButtonContainerLayout.createSequentialGroup()
                        .addGap(10, 10, 10)
                        .addComponent(startup)))
                .addContainerGap(26, Short.MAX_VALUE))
        );

        serviceButtonContainerLayout.linkSize(javax.swing.SwingConstants.HORIZONTAL, new java.awt.Component[] {refreshServiceButton, restartButton, startButton, stopButton});

        serviceButtonContainerLayout.setVerticalGroup(
            serviceButtonContainerLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(serviceButtonContainerLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(serviceButtonContainerLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(startButton)
                    .addComponent(jLabel11))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(serviceButtonContainerLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(restartButton)
                    .addComponent(jLabel13))
                .addGap(7, 7, 7)
                .addGroup(serviceButtonContainerLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(stopButton)
                    .addComponent(jLabel12))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(serviceButtonContainerLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(refreshServiceButton)
                    .addComponent(jLabel20))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, 28, Short.MAX_VALUE)
                .addComponent(startup)
                .addContainerGap())
        );

        javax.swing.GroupLayout servicePanelLayout = new javax.swing.GroupLayout(servicePanel);
        servicePanel.setLayout(servicePanelLayout);
        servicePanelLayout.setHorizontalGroup(
            servicePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(servicePanelLayout.createSequentialGroup()
                .addContainerGap()
                .addComponent(serviceButtonContainer, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addContainerGap(42, Short.MAX_VALUE))
        );
        servicePanelLayout.setVerticalGroup(
            servicePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addComponent(serviceButtonContainer, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
        );

        tabPanel.addTab("Service", servicePanel);

        serverPanel.setBackground(new java.awt.Color(255, 255, 255));
        serverPanel.setBorder(javax.swing.BorderFactory.createEmptyBorder(0, 0, 0, 0));
        serverPanel.setFocusable(false);

        jLabel3.setText("Administrator Port:");

        jLabel1.setText("Web Start Port:");

        serverLogFiles.addListSelectionListener(new javax.swing.event.ListSelectionListener() {
            public void valueChanged(javax.swing.event.ListSelectionEvent evt) {
                serverLogFilesValueChanged(evt);
            }
        });
        jScrollPane1.setViewportView(serverLogFiles);

        jLabel4.setText("Log Files:");

        viewFileButton.setText("View File");
        viewFileButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                viewFileButtonActionPerformed(evt);
            }
        });

        refreshButton.setText("Refresh");
        refreshButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                refreshButtonActionPerformed(evt);
            }
        });

        jLabel5.setText("Main Log Level:");

        jLabel6.setText("Database Log Level:");

        mirthLogLevel.setModel(new javax.swing.DefaultComboBoxModel(new String[] { "Item 1", "Item 2", "Item 3", "Item 4" }));

        databaseLogLevel.setModel(new javax.swing.DefaultComboBoxModel(new String[] { "Item 1", "Item 2", "Item 3", "Item 4" }));

        jLabel18.setText("JMX Port:");

        serverMemoryLabel.setText("Server Memory (mb):");

        channelLogLevel.setModel(new javax.swing.DefaultComboBoxModel(new String[] { "Item 1", "Item 2", "Item 3", "Item 4" }));

        jLabel19.setText("Channel Log Level:");

        javax.swing.GroupLayout serverPanelLayout = new javax.swing.GroupLayout(serverPanel);
        serverPanel.setLayout(serverPanelLayout);
        serverPanelLayout.setHorizontalGroup(
            serverPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, serverPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(serverPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addComponent(jLabel4, javax.swing.GroupLayout.Alignment.TRAILING)
                    .addComponent(serverMemoryLabel, javax.swing.GroupLayout.Alignment.TRAILING)
                    .addComponent(jLabel18, javax.swing.GroupLayout.Alignment.TRAILING)
                    .addComponent(jLabel3, javax.swing.GroupLayout.Alignment.TRAILING)
                    .addComponent(jLabel1, javax.swing.GroupLayout.Alignment.TRAILING))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(serverPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING)
                    .addGroup(serverPanelLayout.createSequentialGroup()
                        .addGroup(serverPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addGroup(serverPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING, false)
                                .addComponent(serverWebstartPort, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                                .addComponent(serverAdministratorPort, javax.swing.GroupLayout.DEFAULT_SIZE, 60, Short.MAX_VALUE))
                            .addGroup(serverPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING, false)
                                .addComponent(serverMemoryField, javax.swing.GroupLayout.Alignment.LEADING, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                                .addComponent(serverJmxPort, javax.swing.GroupLayout.Alignment.LEADING, javax.swing.GroupLayout.DEFAULT_SIZE, 60, Short.MAX_VALUE)))
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, 11, Short.MAX_VALUE)
                        .addGroup(serverPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addComponent(jLabel19, javax.swing.GroupLayout.Alignment.TRAILING)
                            .addComponent(jLabel6, javax.swing.GroupLayout.Alignment.TRAILING)
                            .addComponent(jLabel5, javax.swing.GroupLayout.Alignment.TRAILING)))
                    .addComponent(jScrollPane1, javax.swing.GroupLayout.DEFAULT_SIZE, 169, Short.MAX_VALUE))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(serverPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(serverPanelLayout.createSequentialGroup()
                        .addComponent(viewFileButton, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                        .addContainerGap())
                    .addGroup(serverPanelLayout.createSequentialGroup()
                        .addComponent(refreshButton, javax.swing.GroupLayout.DEFAULT_SIZE, 73, Short.MAX_VALUE)
                        .addContainerGap())
                    .addGroup(serverPanelLayout.createSequentialGroup()
                        .addGroup(serverPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addComponent(channelLogLevel, javax.swing.GroupLayout.DEFAULT_SIZE, 73, Short.MAX_VALUE)
                            .addComponent(databaseLogLevel, javax.swing.GroupLayout.DEFAULT_SIZE, 73, Short.MAX_VALUE)
                            .addComponent(mirthLogLevel, javax.swing.GroupLayout.DEFAULT_SIZE, 73, Short.MAX_VALUE))
                        .addGap(13, 13, 13))))
        );

        serverPanelLayout.linkSize(javax.swing.SwingConstants.HORIZONTAL, new java.awt.Component[] {refreshButton, viewFileButton});

        serverPanelLayout.setVerticalGroup(
            serverPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(serverPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(serverPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jLabel1)
                    .addComponent(jLabel5)
                    .addComponent(mirthLogLevel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(serverWebstartPort, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(serverPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jLabel3)
                    .addComponent(databaseLogLevel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(serverAdministratorPort, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(jLabel6))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(serverPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jLabel18)
                    .addComponent(serverJmxPort, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(channelLogLevel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(jLabel19))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(serverPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(serverMemoryLabel)
                    .addComponent(serverMemoryField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(serverPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addComponent(jLabel4)
                    .addGroup(serverPanelLayout.createSequentialGroup()
                        .addComponent(refreshButton)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(viewFileButton))
                    .addComponent(jScrollPane1, javax.swing.GroupLayout.DEFAULT_SIZE, 54, Short.MAX_VALUE))
                .addContainerGap())
        );

        tabPanel.addTab("Server", serverPanel);

        databasePanel.setBackground(new java.awt.Color(255, 255, 255));
        databasePanel.setBorder(javax.swing.BorderFactory.createEmptyBorder(0, 0, 0, 0));
        databasePanel.setFocusable(false);

        jLabel7.setText("Type:");

        jLabel8.setText(" URL:");

        databaseType.setModel(new javax.swing.DefaultComboBoxModel(new String[] { "derby", "postgres", "mysql", "oracle", "sqlserver2000", "sqlserver" }));
        databaseType.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                databaseTypeActionPerformed(evt);
            }
        });

        jLabel9.setText("Username:");

        jLabel10.setText("Password:");

        javax.swing.GroupLayout databasePanelLayout = new javax.swing.GroupLayout(databasePanel);
        databasePanel.setLayout(databasePanelLayout);
        databasePanelLayout.setHorizontalGroup(
            databasePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(databasePanelLayout.createSequentialGroup()
                .addGap(23, 23, 23)
                .addGroup(databasePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addComponent(jLabel8, javax.swing.GroupLayout.Alignment.TRAILING)
                    .addComponent(jLabel7, javax.swing.GroupLayout.Alignment.TRAILING)
                    .addComponent(jLabel9, javax.swing.GroupLayout.Alignment.TRAILING)
                    .addComponent(jLabel10, javax.swing.GroupLayout.Alignment.TRAILING))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(databasePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addComponent(databaseType, javax.swing.GroupLayout.PREFERRED_SIZE, 100, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(databaseUrl, javax.swing.GroupLayout.DEFAULT_SIZE, 288, Short.MAX_VALUE)
                    .addComponent(databaseUsername, javax.swing.GroupLayout.PREFERRED_SIZE, 150, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(databasePassword, javax.swing.GroupLayout.PREFERRED_SIZE, 150, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addContainerGap())
        );
        databasePanelLayout.setVerticalGroup(
            databasePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(databasePanelLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(databasePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jLabel7)
                    .addComponent(databaseType, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(databasePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jLabel8)
                    .addComponent(databaseUrl, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(databasePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jLabel9)
                    .addComponent(databaseUsername, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(databasePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jLabel10)
                    .addComponent(databasePassword, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addContainerGap(71, Short.MAX_VALUE))
        );

        tabPanel.addTab("Database", databasePanel);

        infoPanel.setBackground(new java.awt.Color(255, 255, 255));
        infoPanel.setBorder(javax.swing.BorderFactory.createEmptyBorder(0, 0, 0, 0));

        jLabel2.setText("Server Version:");

        version.setText("version");

        jLabel15.setText("Server ID:");

        jLabel16.setText("Java Version:");

        serverId.setText("serverId");

        javaVersion.setText("javaVersion");

        jLabel14.setText("Need Help?  Contact");

        mirthSupportLink.setText("<html><font color=blue><u>Mirth Corporation</u></font></html>");
        mirthSupportLink.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                mirthSupportLinkMouseClicked(evt);
            }
        });

        jLabel17.setText("for professional support.");

        javax.swing.GroupLayout infoPanelLayout = new javax.swing.GroupLayout(infoPanel);
        infoPanel.setLayout(infoPanelLayout);
        infoPanelLayout.setHorizontalGroup(
            infoPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(infoPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(infoPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(infoPanelLayout.createSequentialGroup()
                        .addComponent(jLabel14)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(mirthSupportLink, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(jLabel17))
                    .addGroup(infoPanelLayout.createSequentialGroup()
                        .addGroup(infoPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING)
                            .addComponent(jLabel16)
                            .addComponent(jLabel15)
                            .addComponent(jLabel2))
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addGroup(infoPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addComponent(version)
                            .addComponent(serverId)
                            .addComponent(javaVersion))))
                .addContainerGap(54, Short.MAX_VALUE))
        );
        infoPanelLayout.setVerticalGroup(
            infoPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, infoPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(infoPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jLabel2)
                    .addComponent(version))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(infoPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jLabel15)
                    .addComponent(serverId))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(infoPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jLabel16)
                    .addComponent(javaVersion))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, 90, Short.MAX_VALUE)
                .addGroup(infoPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jLabel14)
                    .addComponent(mirthSupportLink, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(jLabel17))
                .addContainerGap())
        );

        tabPanel.addTab("Info", infoPanel);

        applyButton.setText("Apply");
        applyButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                applyButtonActionPerformed(evt);
            }
        });

        launchButton.setText("Administrator");
        launchButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                launchButtonActionPerformed(evt);
            }
        });

        cancelButton.setText("Cancel");
        cancelButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                cancelButtonActionPerformed(evt);
            }
        });

        okButton.setText("Ok");
        okButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                okButtonActionPerformed(evt);
            }
        });

        javax.swing.GroupLayout jPanel3Layout = new javax.swing.GroupLayout(jPanel3);
        jPanel3.setLayout(jPanel3Layout);
        jPanel3Layout.setHorizontalGroup(
            jPanel3Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addComponent(heading, javax.swing.GroupLayout.DEFAULT_SIZE, 402, Short.MAX_VALUE)
            .addGroup(jPanel3Layout.createSequentialGroup()
                .addContainerGap()
                .addComponent(tabPanel)
                .addContainerGap())
            .addGroup(jPanel3Layout.createSequentialGroup()
                .addContainerGap()
                .addComponent(launchButton)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, 57, Short.MAX_VALUE)
                .addComponent(okButton, javax.swing.GroupLayout.PREFERRED_SIZE, 72, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(cancelButton)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(applyButton, javax.swing.GroupLayout.PREFERRED_SIZE, 72, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addContainerGap())
        );

        jPanel3Layout.linkSize(javax.swing.SwingConstants.HORIZONTAL, new java.awt.Component[] {applyButton, cancelButton});

        jPanel3Layout.setVerticalGroup(
            jPanel3Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(jPanel3Layout.createSequentialGroup()
                .addComponent(heading, javax.swing.GroupLayout.PREFERRED_SIZE, 49, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(tabPanel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(jPanel3Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(launchButton)
                    .addComponent(applyButton)
                    .addComponent(cancelButton)
                    .addComponent(okButton))
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(getContentPane());
        getContentPane().setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addComponent(jPanel3, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addComponent(jPanel3, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
        );
    }// </editor-fold>//GEN-END:initComponents

    private void mirthSupportLinkMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_mirthSupportLinkMouseClicked
        BareBonesBrowserLaunch.openURL("http://www.mirthcorp.com/services/support");
}//GEN-LAST:event_mirthSupportLinkMouseClicked

private void startupActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_startupActionPerformed
    if (startup.isSelected()) {
        serviceController.setStartup(true);
    } else {
        serviceController.setStartup(false);
    }
}//GEN-LAST:event_startupActionPerformed

private void refreshServiceButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_refreshServiceButtonActionPerformed
    managerController.updateMirthServiceStatus();
}//GEN-LAST:event_refreshServiceButtonActionPerformed

private void okButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_okButtonActionPerformed
    saveProperties();
    close();
}//GEN-LAST:event_okButtonActionPerformed

private void cancelButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_cancelButtonActionPerformed
    close();
}//GEN-LAST:event_cancelButtonActionPerformed

private void applyButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_applyButtonActionPerformed
    saveProperties();
}//GEN-LAST:event_applyButtonActionPerformed

private void serverLogFilesValueChanged(javax.swing.event.ListSelectionEvent evt) {//GEN-FIRST:event_serverLogFilesValueChanged
    if (serverLogFiles.getSelectedIndex() != -1) {
        viewFileButton.setEnabled(true);
    } else {
        viewFileButton.setEnabled(false);
    }
}//GEN-LAST:event_serverLogFilesValueChanged

private void refreshButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_refreshButtonActionPerformed
    refreshLogs();
}//GEN-LAST:event_refreshButtonActionPerformed

private void databaseTypeActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_databaseTypeActionPerformed
    // If the properties are loading, don't do anything here.  If the database 
    // value is changing, then set the default values.
    if (loading) {
        lastSelectedDatabaseType = (String) databaseType.getSelectedItem();
    } else if (!databaseType.getSelectedItem().equals(lastSelectedDatabaseType)) {

        // If the last selection was not using the default values, then prompt
        // to see if the user wants to continue.
        if (!databaseUrl.getText().equals(databaseUrls.get(lastSelectedDatabaseType)) || !databaseUsername.getText().equals("") || !new String(databasePassword.getPassword()).equals("")) {
            if (!managerController.alertOptionDialog("Changing your database type will clear your database URL, username, and password.\nAre you sure you want to continue?")) {
                databaseType.setSelectedItem(lastSelectedDatabaseType);
                return;
            }
        }
        lastSelectedDatabaseType = (String) databaseType.getSelectedItem();

        databaseUrl.setText(databaseUrls.get(lastSelectedDatabaseType));
        databaseUsername.setText("");
        databasePassword.setText("");

    }
}//GEN-LAST:event_databaseTypeActionPerformed

private void launchButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_launchButtonActionPerformed
    managerController.launchAdministrator();
}//GEN-LAST:event_launchButtonActionPerformed

private void viewFileButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_viewFileButtonActionPerformed
    managerController.openLogFile(managerController.getLog4jProperties().getString(ManagerConstants.DIR_LOGS) + System.getProperty("file.separator") + (String) serverLogFiles.getSelectedValue());
}//GEN-LAST:event_viewFileButtonActionPerformed

private void restartButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_restartButtonActionPerformed
    managerController.restartMirthWorker();
}//GEN-LAST:event_restartButtonActionPerformed

private void stopButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_stopButtonActionPerformed
    managerController.stopMirthWorker();
}//GEN-LAST:event_stopButtonActionPerformed

private void startButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_startButtonActionPerformed
    managerController.startMirthWorker();
}//GEN-LAST:event_startButtonActionPerformed

    private void loadProperties() {
        loading = true;

        serverId.setText(managerController.getServerId());
        serverMemoryField.setText(managerController.getServiceXmx());
        version.setText(managerController.getServerVersion());
        javaVersion.setText(System.getProperty("java.version"));
        serverWebstartPort.setText(managerController.getServerProperties().getString(ManagerConstants.SERVER_WEBSTART_PORT));
        serverAdministratorPort.setText(managerController.getServerProperties().getString(ManagerConstants.SERVER_ADMINISTRATOR_PORT));
        serverJmxPort.setText(managerController.getServerProperties().getString(ManagerConstants.SERVER_JMX_PORT));

        boolean applyEnabled = isApplyEnabled();

        databaseType.setSelectedItem(managerController.getServerProperties().getString(ManagerConstants.DATABASE_TYPE));
        databaseUrl.setText(managerController.getServerProperties().getString(ManagerConstants.DATABASE_URL));
        databaseUsername.setText(managerController.getServerProperties().getString(ManagerConstants.DATABASE_USERNAME));
        databasePassword.setText(managerController.getServerProperties().getString(ManagerConstants.DATABASE_PASSWORD));

        mirthLogLevel.setModel(new DefaultComboBoxModel(ManagerConstants.LOG4J_ERROR_CODES));
        databaseLogLevel.setModel(new DefaultComboBoxModel(ManagerConstants.LOG4J_ERROR_CODES));
        channelLogLevel.setModel(new DefaultComboBoxModel(ManagerConstants.LOG4J_ERROR_CODES_WITH_BLANK));

        String rootLogCode = managerController.getLog4jProperties().getStringArray(ManagerConstants.LOG4J_MIRTH_LOG_LEVEL)[0];
        if (ArrayUtils.contains(ManagerConstants.LOG4J_ERROR_CODES, rootLogCode)) {
            mirthLogLevel.setSelectedItem(rootLogCode);
        }

        databaseLogLevel.setSelectedItem(managerController.getLog4jProperties().getString(ManagerConstants.LOG4J_DATABASE_LOG_LEVEL));
        
        String channelLogCode = null;
        for (int i = 0; (i < ManagerConstants.LOG4J_CHANNEL_LOG_LEVELS.length) && !ManagerConstants.LOG4J_ERROR_CODES_WITH_BLANK[0].equals(channelLogCode); i++) {
            String tempLogCode = managerController.getLog4jProperties().getString(ManagerConstants.LOG4J_CHANNEL_LOG_LEVELS[i]);
            
            if (tempLogCode == null) { // log code not found, stop
                channelLogCode = ManagerConstants.LOG4J_ERROR_CODES_WITH_BLANK[0];
            } else if (channelLogCode == null) {  // first code found, save and continue
                channelLogCode = tempLogCode;
            } else if (!channelLogCode.equalsIgnoreCase(tempLogCode)) { // different code, stop
                channelLogCode = ManagerConstants.LOG4J_ERROR_CODES_WITH_BLANK[0];
            }            
        }
        
        channelLogLevel.setSelectedItem(channelLogCode);

        setApplyEnabled(applyEnabled);

        refreshLogs();
        serverLogFilesValueChanged(null);

        loading = false;
    }

    public void saveProperties() {
        managerController.getServerProperties().setProperty(ManagerConstants.SERVER_WEBSTART_PORT, serverWebstartPort.getText());
        managerController.getServerProperties().setProperty(ManagerConstants.SERVER_ADMINISTRATOR_PORT, serverAdministratorPort.getText());
        managerController.getServerProperties().setProperty(ManagerConstants.SERVER_JMX_PORT, serverJmxPort.getText());

        managerController.getServerProperties().setProperty(ManagerConstants.DATABASE_TYPE, ((String) databaseType.getSelectedItem()));
        managerController.getServerProperties().setProperty(ManagerConstants.DATABASE_URL, databaseUrl.getText());
        managerController.getServerProperties().setProperty(ManagerConstants.DATABASE_USERNAME, databaseUsername.getText());
        managerController.getServerProperties().setProperty(ManagerConstants.DATABASE_PASSWORD, new String(databasePassword.getPassword()));

        try {
            managerController.getServerProperties().save();
        } catch (ConfigurationException e) {
            managerController.alertErrorDialog("Error saving " + managerController.getServerProperties().getFile().getPath() + ":\n" + e.getMessage());
        }
        
        String[] logLevel = managerController.getLog4jProperties().getStringArray(ManagerConstants.LOG4J_MIRTH_LOG_LEVEL);
        logLevel[0] = (String) mirthLogLevel.getSelectedItem();
        managerController.getLog4jProperties().setProperty(ManagerConstants.LOG4J_MIRTH_LOG_LEVEL, logLevel);

        managerController.getLog4jProperties().setProperty(ManagerConstants.LOG4J_DATABASE_LOG_LEVEL, (String) databaseLogLevel.getSelectedItem());

        String channelLogLevelCode = (String) channelLogLevel.getSelectedItem();
        
        // Only set all of the log levels if the selected value is not blank
        if (!ManagerConstants.LOG4J_ERROR_CODES_WITH_BLANK[0].equals(channelLogLevelCode)) {
            for (String channelLogLevel : ManagerConstants.LOG4J_CHANNEL_LOG_LEVELS) {
                managerController.getLog4jProperties().setProperty(channelLogLevel, channelLogLevelCode);
            }
        }
        
        try {
            managerController.getLog4jProperties().save();
        } catch (ConfigurationException e) {
            managerController.alertErrorDialog("Error saving " + managerController.getLog4jProperties().getFile().getPath() + ":\n" + e.getMessage());
        }
        
        managerController.setServiceXmx(serverMemoryField.getText());

        setApplyEnabled(false);
    }

    private void refreshLogs() {
        String logPath = managerController.getLog4jProperties().getString(ManagerConstants.DIR_LOGS);
        serverLogFiles.setListData(managerController.getLogFiles(logPath).toArray());
    }

    public void setStartButtonActive(boolean active) {
        startButton.setEnabled(active);
    }

    public void setStopButtonActive(boolean active) {
        stopButton.setEnabled(active);
    }

    public void setRestartButtonActive(boolean active) {
        restartButton.setEnabled(active);
    }

    public void setLaunchButtonActive(boolean active) {
        launchButton.setEnabled(active);
    }
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton applyButton;
    private javax.swing.JButton cancelButton;
    private com.mirth.connect.manager.components.MirthComboBox channelLogLevel;
    private com.mirth.connect.manager.components.MirthComboBox databaseLogLevel;
    private javax.swing.JPanel databasePanel;
    private com.mirth.connect.manager.components.MirthPasswordField databasePassword;
    private com.mirth.connect.manager.components.MirthComboBox databaseType;
    private com.mirth.connect.manager.components.MirthTextField databaseUrl;
    private com.mirth.connect.manager.components.MirthTextField databaseUsername;
    private com.mirth.connect.manager.MirthHeadingPanel heading;
    private javax.swing.JLabel headingLabel;
    private javax.swing.JPanel infoPanel;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel jLabel10;
    private javax.swing.JLabel jLabel11;
    private javax.swing.JLabel jLabel12;
    private javax.swing.JLabel jLabel13;
    private javax.swing.JLabel jLabel14;
    private javax.swing.JLabel jLabel15;
    private javax.swing.JLabel jLabel16;
    private javax.swing.JLabel jLabel17;
    private javax.swing.JLabel jLabel18;
    private javax.swing.JLabel jLabel19;
    private javax.swing.JLabel jLabel2;
    private javax.swing.JLabel jLabel20;
    private javax.swing.JLabel jLabel3;
    private javax.swing.JLabel jLabel4;
    private javax.swing.JLabel jLabel5;
    private javax.swing.JLabel jLabel6;
    private javax.swing.JLabel jLabel7;
    private javax.swing.JLabel jLabel8;
    private javax.swing.JLabel jLabel9;
    private javax.swing.JPanel jPanel3;
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JLabel javaVersion;
    private javax.swing.JButton launchButton;
    private com.mirth.connect.manager.components.MirthComboBox mirthLogLevel;
    private javax.swing.JLabel mirthSupportLink;
    private javax.swing.JButton okButton;
    private javax.swing.JButton refreshButton;
    private javax.swing.JButton refreshServiceButton;
    private javax.swing.JButton restartButton;
    private com.mirth.connect.manager.components.MirthTextField serverAdministratorPort;
    private javax.swing.JLabel serverId;
    private com.mirth.connect.manager.components.MirthTextField serverJmxPort;
    private javax.swing.JList serverLogFiles;
    private com.mirth.connect.manager.components.MirthTextField serverMemoryField;
    private javax.swing.JLabel serverMemoryLabel;
    private javax.swing.JPanel serverPanel;
    private com.mirth.connect.manager.components.MirthTextField serverWebstartPort;
    private javax.swing.JPanel serviceButtonContainer;
    private javax.swing.JPanel servicePanel;
    private javax.swing.JButton startButton;
    private javax.swing.JCheckBox startup;
    private javax.swing.JButton stopButton;
    private javax.swing.JTabbedPane tabPanel;
    private javax.swing.JLabel version;
    private javax.swing.JButton viewFileButton;
    // End of variables declaration//GEN-END:variables
}
